/**************************************************************************************
* 
*	Project Name : S5PC100 Validation
*
*	Copyright 2006 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PC100.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : system.c
*  
*	File Description : This file implements exceptions and MMU/Cache control.
*
*	Author : Haksoo,Kim
*	Dept. : AP Development Team
*	Created Date : 2006/11/08
*	Version : 0.1 
* 
*	History
*	- Created(Haksoo,Kim 2006/11/08)
*  
**************************************************************************************/

#include <stdio.h>

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "intc.h"

////
// Exception
//

#define pExceptionRESET			(*(u32 *)(_Exception_Vector + 0x0))
#define pExceptionUNDEF			(*(u32 *)(_Exception_Vector + 0x4))
#define pExceptionSWI			(*(u32 *)(_Exception_Vector + 0x8))
#define pExceptionPABORT		(*(u32 *)(_Exception_Vector + 0xc))
#define pExceptionDABORT		(*(u32 *)(_Exception_Vector + 0x10))
#define pExceptionRESERVED		(*(u32 *)(_Exception_Vector + 0x14))
#define pExceptionIRQ			(*(u32 *)(_Exception_Vector + 0x18))
#define pExceptionFIQ			(*(u32 *)(_Exception_Vector + 0x1c))


////
// MMU/Cache
//
#define DESC_SEC        (0x2|(0<<4))

#if 0
#define CB				(3<<2)				//cache_on, write_back
#define CNB				(2<<2)				//cache_on, write_through 
#define NCB				(1<<2)				//cache_off,WR_BUF on
#define NCNB			(0<<2)				//cache_off,WR_BUF off
#define CBWA			(1<<12)|(3<<2)		//Outer and Inner Write-Back, Write-Allocate
#else
#define STSH				(0<<12)|(0<<2)		//Strongly-ordered, Shareable in C-A8. <> cache_off,WR_BUF off in ARM9
#define DESH				(0<<12)|(1<<2)		//Device, Shareable in C-A8 <> cache_off,WR_BUF on in ARM9
#define CNB				(0<<12)|(2<<2)		//Outer and Inner Write-Through, no Write-Allocate in C-A8 <> cache_on, write_through in ARM9
#define CB				(0<<12)|(3<<2)		//Outer and Inner Write-Back, no Write-Allocate in C-A8 <> cache_on, write_back
#define NCNB				(0<<12)|(0<<2)		//Outer and Inner Non-Cacheable
#define NCB				(1<<12)|(1<<2)		//Reserved
#define IMDF				(1<<12)|(2<<2)		//Implementation defined
#define CBWA			(1<<12)|(3<<2)		//Outer and Inner Write-Back, Write-Allocate
#define NSHD				(2<<12)|(0<<2)		//Non-shareable Device
#define RES2				(2<<12)|(1<<2)		//Reserved
#define RES3				(2<<12)|(2<<2)		//Reserved
#define RES4				(2<<12)|(3<<2)		//Reserved
/* Outer and Inner independantly control
#define NSHD				(1<<14)|(0<<12)|(0<<2)		//Non-shareable Device
#define RES2				(1<<14)|(0<<12)|(1<<2)		//Reserved
#define RES3				(1<<14)|(0<<12)|(2<<2)		//Reserved
#define RES4				(1<<14)|(0<<12)|(3<<2)		//Reserved
#define NSHD				(1<<14)|(1<<12)|(0<<2)		//Non-shareable Device
#define RES2				(1<<14)|(1<<12)|(1<<2)		//Reserved
#define RES3				(1<<14)|(1<<12)|(2<<2)		//Reserved
#define RES4				(1<<14)|(1<<12)|(3<<2)		//Reserved
#define NSHD				(1<<14)|(2<<12)|(0<<2)		//Non-shareable Device
#define RES2				(1<<14)|(2<<12)|(1<<2)		//Reserved
#define RES3				(1<<14)|(2<<12)|(2<<2)		//Reserved
#define RES4				(1<<14)|(2<<12)|(3<<2)		//Reserved
#define NSHD				(1<<14)|(3<<12)|(0<<2)		//Non-shareable Device
#define RES2				(1<<14)|(3<<12)|(1<<2)		//Reserved
#define RES3				(1<<14)|(3<<12)|(2<<2)		//Reserved
#define RES4				(1<<14)|(3<<12)|(3<<2)		//Reserved
*/
#endif

#define WA				(1<<12)				// write allocation

#define AP_RW			(3<<10)				//supervisor=RW, user=RW
#define AP_RO			(2<<10)				//supervisor=RW, user=RO
#define AP_NO			(1<<10)				//supervisor=RW, user=No access

#define DOMAIN_FAULT	(0x0)
#define DOMAIN_CHK		(0x1) 
#define DOMAIN_NOTCHK	(0x3) 
#define DOMAIN0			(0x0<<5)
#define DOMAIN1			(0x1<<5)

#define DOMAIN0_ATTR	(DOMAIN_CHK<<0) 
#define DOMAIN1_ATTR	(DOMAIN_FAULT<<2) 

#define RW_CB			(AP_RW|DOMAIN0|CB|DESC_SEC)
#define RW_CNB			(AP_RW|DOMAIN0|CNB|DESC_SEC)
#define RW_CBWA		(AP_RW|DOMAIN0|CBWA|DESC_SEC)
#define RW_NCB			(AP_RW|DOMAIN0|NCB|DESC_SEC)
#define RW_NCNB			(AP_RW|DOMAIN0|NCNB|DESC_SEC)
#define RW_FAULT		(AP_RW|DOMAIN1|NCNB|DESC_SEC)

#define RW_STSH			(AP_RW|DOMAIN0|STSH|DESC_SEC)

#define RW_NSNCNB		((0<<19)|AP_RW|DOMAIN0|NCNB|DESC_SEC)


u32 bDCacheEnabled = false;
u32 bICacheEnabled = false;


//////////
// Function Name : ExceptionUNDEF
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void ExceptionUNDEF(void)
{
	UART_Printf("Undefined instruction exception.\n");
	while(1);
}

//////////
// Function Name : ExceptionSWI
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void ExceptionSWI(void)
{
	UART_Printf("SWI exception.\n");
	while(1);
}

//////////
// Function Name : ExceptionPABORT
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void ExceptionPABORT(void)
{
	UART_Printf("IFSR=0x%x\n",CoReadIFSR());
	UART_Printf("FAR=0x%x\n",CoReadFAR());
	UART_Printf("Pabort exception.\n");
	while(1);
}

//////////
// Function Name : ExceptionDABORT
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void ExceptionDABORT(void)
{
	UART_Printf("DFSR=0x%x\n",CoReadDFSR());
	UART_Printf("FAR=0x%x\n",CoReadFAR());
	UART_Printf("Dabort exception.\n");
	while(1);
}

void (*IntHandlerTable[INT_LIMIT])(void);
//////////
// Function Name : ExceptionIRQ
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void ExceptionIRQ(void)
{
	u32 intNum;

	intNum=INTC_ReadIntSrc();
	IntHandlerTable[intNum]();

	return;
}

//////////
// Function Name : ExceptionFIQ
// Function Description : 
// Input : NONE
// Output : NONE
// Version : 
void ExceptionFIQ(void)
{
	UART_Printf("FIQ exception.\n");
	while(1);
}

//////////
// Function Name : SYSTEM_InitException
// Function Description : 
// Input : NONE
// Output : NONE
// Version
//#if	(VIC_MODE == 0)
extern void IsrIRQ(void);
//#endif

void SYSTEM_InitException( void)
{
	pExceptionUNDEF		=	(u32)ExceptionUNDEF; 
	pExceptionSWI  		=	(u32)ExceptionSWI;
	pExceptionPABORT		=	(u32)ExceptionPABORT;
	pExceptionDABORT		=	(u32)ExceptionDABORT;
	pExceptionIRQ		=	(u32)IsrIRQ;
	pExceptionFIQ			=	(u32)IsrIRQ;

	INTC_Init();

	CoEnableIrq();
	CoEnableFiq();
}

void CoStartICache(void)
{
	CoDisableICache();
	CoInvalidateICache();
	CoEnableICache();

	bICacheEnabled = true;
}

void CoStartICacheOnly(void)
{
	CoDisableDCache();
	CoDisableMmu();
	CoDisableICache();
	CoInvalidateICache();
	CoEnableICache();

	bDCacheEnabled = false;
	bICacheEnabled = true;
}

void SetTransTable(u32 uVaStart, u32 uVaEnd, u32 uPaStart, u32 attr)
{
	u32 *pTT;
	u32 i, nNumOfSec;

	Assert((uPaStart&0xfffff) == 0);
	Assert((uVaStart&0xfffff) == 0);

	pTT = (u32 *)_MMUTT_BaseAddress+(uVaStart>>20);

	nNumOfSec = (0x1000+(uVaEnd>>20)-(uVaStart>>20))%0x1000;

	for(i=0;i<=nNumOfSec;i++)*pTT++=attr |(((uPaStart>>20)+i)<<20);
}

void CoStopMmuAndCache(void)
{
	u32 i, j;
	
	CoDisableDCache();
	// ARM920T L1 Dcache - NumWays=64, SetsPerWay=8 (offset = 26,5)
	// C100 L1 Dcache - NumWays=4, SetsPerWay=128 (offset = 30,6)

	#ifdef Cortex
	
	for( i=0;i<4;i++) 
		for( j=0;j<128;j++)
			CoInvalidateDCacheIndex((i<<30)|(j<<6));
	#else
	for( i=0;i<4;i++) 
		for( j=0;j<128;j++)
			CoInvalidateDCacheIndex((i<<30)|(j<<5));	
	#endif
	CoDisableMmu();
	CoInvalidateDTlb();
}

void CoCleanAndInvalidateL1DCache(void)
{
	int i,j;

	// L1 Dcache 32KB [31:30]Way, [12:6]Set, [3:1]Level
	// Cortex A8 L1 Dcache - NumWays=4, SetsPerWay=128 (offset = 30,6)
	for(i=0;i<4;i++)
		for(j=0;j<128;j++)
			CoCleanAndInvalidateDCacheIndex((i<<30)|(j<<6)); // If write-back is used, the D-cache should be cleared.
}

void CoInvalidateL2Cache(void)
{
	int i,j;

	// L2 Dcache 512KB [31:29]Way, [15:6]Set, [3:1]Level
	for(i=0;i<8;i++)
		for(j=0;j<1024;j++)
			CoInvalidateDCacheIndex((i<<29)|(j<<6)|(1<<1)); // D-cache is cleared.
}

void CoStartL2Cache(void)
{
	Assert(bDCacheEnabled==false);
	Assert(bICacheEnabled==false);

	CoSetL2CacheAuxCrtlReg(0);
	CoEnableL2Cache();
}

void CoStartMmuAndCache(void)
{
	// L2 setting & Invalidate Entire D cache
	CoDisableL2Cache();
	CoInvalidateL2Cache();	
	CoStartL2Cache();	

	CoInvalidateICache();
	CoInvalidateDTlb();

	CoDisableICache();
	CoDisableDCache();
	CoDisableMmu();

	// min. table unit 0x100000 : 1MB)
	SetTransTable(0x00000000,0x08000000,0x00000000,RW_NCNB);	// iROM(0x0~0x7FFF) & iRAM(0x20000~0x37FFF) 
	SetTransTable(0x08000000,0x10000000,0x08000000,RW_FAULT); // Reserved
	SetTransTable(0x10000000,0x20000000,0x10000000,RW_FAULT); // Reserved

#ifdef FPGA

	SetTransTable(_DRAM_BaseAddress+0x00000000,_DRAM_BaseAddress+0x01000000,_DRAM_BaseAddress+0x00000000,RW_NCNB);
	SetTransTable(_DRAM_BaseAddress+0x01000000,_DRAM_BaseAddress+0x07f00000,_DRAM_BaseAddress+0x01000000,RW_NCNB);
	SetTransTable(_DRAM_BaseAddress+0x07f00000,_DRAM_BaseAddress+0x08000000,_DRAM_BaseAddress+0x07f00000,RW_NCNB);

	SetTransTable(_DRAM_BaseAddress+0x20000000,_DRAM_BaseAddress+0x28000000,_DRAM_BaseAddress+0x20000000,RW_NCNB);

#else

	// Memory Port0 96MB
	SetTransTable(_DRAM0_BaseAddress+0x00000000,_DRAM0_BaseAddress+0x04000000,_DRAM0_BaseAddress+0x00000000,RW_NCNB);
	SetTransTable(_DRAM0_BaseAddress+0x04000000,_DRAM0_BaseAddress+0x06000000,_DRAM0_BaseAddress+0x04000000,RW_CB);
	SetTransTable(0x26000000,0x30000000,0x26000000,RW_FAULT); // Reserved
	SetTransTable(0x30000000,0x40000000,0x30000000,RW_FAULT); // Reserved

	// Memory Port1 128MB		
	SetTransTable(_DRAM_BaseAddress+0x00000000,_DRAM_BaseAddress+0x01000000,_DRAM_BaseAddress+0x00000000,RW_CBWA);
	//SetTransTable(_DRAM_BaseAddress+0x00000000,_DRAM_BaseAddress+0x00800000,_DRAM_BaseAddress+0x00000000,RW_CBWA);
	//SetTransTable(_DRAM_BaseAddress+0x00800000,_DRAM_BaseAddress+0x01000000,_DRAM_BaseAddress+0x08000000,RW_NCNB);
	
	SetTransTable(_DRAM_BaseAddress+0x01000000,_DRAM_BaseAddress+0x04100000,_DRAM_BaseAddress+0x01000000,RW_NCNB);
	SetTransTable(_DRAM_BaseAddress+0x04100000,_DRAM_BaseAddress+0x07f00000,_DRAM_BaseAddress+0x04100000,RW_CBWA); // heap area set to CBWA
	SetTransTable(_DRAM_BaseAddress+0x07f00000,_DRAM_BaseAddress+0x08000000,_DRAM_BaseAddress+0x07f00000,RW_CBWA);
	SetTransTable(0x48000000,0x50000000,0x48000000,RW_FAULT); // Reserved
	SetTransTable(0x50000000,0x60000000,0x50000000,RW_FAULT); // Reserved

#endif
	
	SetTransTable(0x60000000,0x80000000,0x60000000,RW_FAULT); // Reserved
	
	SetTransTable(0x80000000,0x80100000,0x80000000,RW_CBWA); // SROMC Bank0 for 1MB code region using SRAM
	SetTransTable(0x80100000,0x88000000,0x80100000,RW_NCNB); // SROMC Bank0 remained region
	SetTransTable(0x88000000,0x90000000,0x88000000,RW_NCNB); // SROMC Bank1
	SetTransTable(0x90000000,0x98000000,0x90000000,RW_NCNB); // SROMC Bank2
	SetTransTable(0x98000000,0xa0000000,0x98000000,RW_NCNB); // SROMC Bank3
	SetTransTable(0xa0000000,0xa8000000,0xa0000000,RW_NCNB); // SROMC Bank4
	SetTransTable(0xa8000000,0xb0000000,0xa8000000,RW_NCNB); // SROMC Bank5

	SetTransTable(0xb0000000,0xb8000000,0xb0000000,RW_NCNB); // OneNAND0
	SetTransTable(0xb8000000,0xc0000000,0xb8000000,RW_NCNB); // OneNAND1
	SetTransTable(0xc0000000,0xd0000000,0xc0000000,RW_NCNB); // MP3_SRAM
	
	SetTransTable(0xd0000000,0xd8000000,0xd0000000,RW_NCNB); // IROMC,IRAMC
	SetTransTable(0xd8000000,0xe0000000,0xd8000000,RW_NCNB); // DMZ ROM

	SetTransTable(0xe0000000,0x00000000,0xe0000000,RW_STSH); // SFR

	CoSetTTBase(_MMUTT_BaseAddress);

	//DOMAIN1: no_access, DOMAIN0,2~15=client(AP is checked)
	CoSetDomain(0x55555550|DOMAIN1_ATTR|DOMAIN0_ATTR);

	//CoInvalidateICache();
	//CoCleanDCacheIndex(0);

	//CoSetProcessId(0x0);
	CoEnableAlignFault();	// not support unaligned data access. produce alignment fault	
	//CoDisableAlignFault();	// support unaligned data access	
	
	CoEnableMmu();
	CoEnableICache();
	CoEnableDCache(); // DCache should be turned on after MMU is turned on. 
	CoEnableBranchPrediction();
}

void TZPC_Init(void)
{	
	Outp32(TZPC0_BASE+0x000, 0x00); // TZPC Secure RAM region size

	Outp32(TZPC0_BASE+0x804, 0xff); // TZPCDECPROT0Set
	Outp32(TZPC0_BASE+0x810, 0xff); // TZPCDECPROT1Set

	Outp32(TZPC1_BASE+0x804, 0xff); // TZPCDECPROT0Set
	Outp32(TZPC1_BASE+0x810, 0xff); // TZPCDECPROT1Set
	Outp32(TZPC1_BASE+0x81c, 0xff); // TZPCDECPROT2Set

	Outp32(TZPC2_BASE+0x804, 0xff); // TZPCDECPROT1Set
	Outp32(TZPC2_BASE+0x810, 0xff); // TZPCDECPROT2Set
}

