/**************************************************************************************
* 
*	Project Name : S5PC100 Validation
*
*	Copyright 2006~2008 by Samsung Electronics, Inc.
*	All rights reserved.
*
*	Project Description :
*		This software is only for validating functions of the S5PC100.
*		Anybody can use this software without our permission.
*  
*--------------------------------------------------------------------------------------
* 
*	File Name : library.c
*  
*	File Description : This file implements library functions.
*
*	Author : Haksoo,Kim
*	Dept. : AP Development Team
*	Created Date : 2006/11/08
*	Version : 0.1 
* 
*	History
*	- Created(Haksoo,Kim 2006/11/08)
*	- Added DownloadImageThruUsbOtg function (Haksoo,Kim 2007/01/24)
*	- Edited RPRINTF (OnPil,Shin (SOP) 2008/03/01)
*  
**************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>

#include "option.h"
#include "library.h"
#include "v210_sfr.h"
#include "system.h"
#include "uart\uart.h"
#include "gpio.h"
#include "intc.h"
#include "register_addr.h"

#define IMAGE_MAXSIZE 1600


#define		NTESTIP			31
#define		MASKBIT		0xF

typedef  struct st_CLKPWR_Test
{
	char name[64];
	unsigned int uAddr;	
	unsigned int uWriteVal;
	unsigned int uMask;
}CLKPWR_TEST_REG;

CLKPWR_TEST_REG		sRegClkPowerTest[] = 
{ 
	{"CPTEST_FIMD     ",		0xf80000A8	,	0xF,  	0xF},
//	{"CPTEST_MIE      ",		0xf8100034	,	0xF,  	0xF},
//	{"CPTEST_MDNIE     ",		0xfae00134 	,	0xF,  	0xF},
//	{"CPTEST_IELCD     ",		0xf82000A8	,	0xF,  	0xF},		
	{"CPTEST_G3D      ",		0xf3000a88	,	0xF,  	0xF},
	{"CPTEST_MFC      ",		0xF1702000 	,	0xF,  	0xF},
	{"CPTEST_DMC1_SFR ",	0xF1400064	,	0xF,  	0xF},			//QOSCONFIG0
	{"CPTEST_DSIM     ",		0xFA50000C	,	0xF,  	0xF},			//DSIM_TIMEOUT		TimeOutRegister
	{"CPTEST_SYSCON   ",		0xE010F000	,	0xF,  	0xF},
	{"CPTEST_SPI0     ",		0xE1300020	,	0xF,  	0xF},			//PACKET_CNT_REG
	{"CPTEST_UART	",		0xE290042C	,	0xF,  	0xF},			//clock generator divide
	{"CPTEST_VIC0     ",		0xF2000024	,	0xF,  	0xF},	
	{"CPTEST_INTMEM   ",		0xD0020000	,	0xF,  	0xF},
	{"CPTEST_FIMC0	  ",		0xfb200018 	,	0xF,  	0xF},
	{"CPTEST_FIMC1	  ",		0xfb300018 	,	0xF,  	0xF},
	{"CPTEST_FIMC2	  ",		0xfb400018 	,	0xF,  	0xF},
	{"CPTEST_JPEG     ",		0xFB600058 	,	0xF,  	0xF},
	{"CPTEST_ROT      ",		0xfb100004	,	0xF,  	0xF},
	{"CPTEST_IPC	",		0xFB700218 	,	0xF,  	0xF},
	{"CPTEST_VP       ",		0xf9100028	,	0x8,		0x8},
	{"CPTEST_MIXER    ",		0xf9200024	,	0xC,  	0xC},
	{"CPTEST_TVENC    ",		0xf900018c	,	0xF,  	0xF},
	{"CPTEST_HDMI0    ",		0xfa100000	,	0xF,  	0xF},
	{"CPTEST_HDMI2    ",		0xfa110060 	,	0xB,  	0xB},	
	{"CPTEST_HDMI3    ",		0xfa150020 	,	0xF,  	0xF},	
	{"CPTEST_HDMI4    ",		0xe1b00010 	,	0x3,  	0x3},
	{"CPTEST_NANDXL   ",		0xB0E0000C	,	0xF,  	0xF},			//nand flash address
	{"CPTEST_CFCON    ",		0xE8200034	,	0xE,  	0xE},
	{"CPTEST_SROMC    ",		0xE8000018	,	0xF,  	0xF},		
	{"CPTEST_MODEMIF  ",		0XED008018	,	0xF,  	0xF},			//INT2AP
	{"CPTEST_SECSS	  ",		0xEA000020	,	0xF,  	0xF},
	{"CPTEST_TSI      ",		0xEB400024	,	0xF,  	0xF},		//TS_PID0
	{"CPTEST_HSMMC0   ",		0xEB000004	,	0xF,  	0xF},
//	{"CPTEST_USBHOST  ",		0xEC200038	,	0xE0,   	0xE0},
//	{"CPTEST_USTOTG   ",		0xec200014	,	0xF,    	0xF},
	{"CPTEST_AUDIO	  ",		0xEEE20014	,	0xF,	  	0xF},
};


//////////
// Function Name : InitUartPort
// Function Description : This function initializes gpio for debugging uart ch.
// Input :  ch, uart ch number
//			flowControl, whether flow control or not
// Output : NONE
// Version :
void InitUartPort(u8 ch, u8 flowControl)
{
	switch (ch)
	{
		default:
		case 0:
			if(flowControl == TRUE)
			{
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_0, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_1, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_2, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_3, eGFunc_0);
			}
			else
			{
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_0, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_1, eGFunc_0);
			}
			break;
			
		case 1:
			if(flowControl == TRUE)
			{
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_4, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_5, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_6, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_7, eGFunc_0);
			}
			else
			{
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_4, eGFunc_0);
				GPIO_SetFunctionEach(eGPIO_A0, eGPIO_5, eGFunc_0);
			}
			break;
			
		case 2:
			GPIO_SetFunctionEach(eGPIO_A1, eGPIO_0, eGFunc_0);
			GPIO_SetFunctionEach(eGPIO_A1, eGPIO_1, eGFunc_0);
			break;
			
		case 3:
			GPIO_SetFunctionEach(eGPIO_A1, eGPIO_2, eGFunc_0);
			GPIO_SetFunctionEach(eGPIO_A1, eGPIO_3, eGFunc_0);
			break;		
	}
	
	return;	
}

//////////
// Function Name : InitLED
// Function Desctiption : This function initializes gpio for debugging LED
// Input : NONE
// Output : NONE
// Version :
void InitLED(void)
{
#ifdef FPGA
	GPIO_SetFunctionEach(eGPIO_D0, eGPIO_0, eGPO);
	GPIO_SetFunctionEach(eGPIO_D0, eGPIO_1, eGPO);
	GPIO_SetFunctionEach(eGPIO_D0, eGPIO_2, eGPO);
#else // silicon
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_4, eGPO);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_5, eGPO);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_6, eGPO);
	GPIO_SetFunctionEach(eGPIO_H2, eGPIO_7, eGPO);
#endif
	return;
}

//////////
// Function Name : DisplayLED
// Function Desctiption : This function controls debugging LED
// Input : data, LED value 
// Output : NONE
// Version :
void DisplayLED(u8 data)
{
	u32 temp;

#ifdef FPGA
	temp = GPIO_GetDataAll(eGPIO_D0);
	temp = (temp & ~(0x7))|(data&0x7);	
	GPIO_SetDataAll(eGPIO_D0, temp);
#else // silicon
	temp = GPIO_GetDataAll(eGPIO_H2);
	temp = (temp & ~(0xF<<4))|((data&0xF)<<4);	
	GPIO_SetDataAll(eGPIO_H2, temp);
#endif

	return;
}


//////////
// Function Name : OpenConsole
// Function Description : This function opens uart and LED for debugging
// Input : NONE
// Output : NONE
// Version : 
void OpenConsole( u8 ch)
{	
//	u8 ch=0;	//uart channel for debugging
	
	InitUartPort(ch, FALSE);
	UART_InitDebugCh(ch, 115200);
	InitLED();

	return;
}

//////////
// Function Name : GetIntNum
// Function Description : This function gets the number which a user enters
// Input : NONE
// Output : number, number which a user enters
// Version : 
s32 GetIntNum( void)
{
    char str[30];
    char *string = str;
    int base     = 10;
    int minus    = 0;
    int result   = 0;
    int lastIndex;    
    int i,j;
    
    gets(string);


	i=0; j=0;
	do {
		if (string[j]==0x8) {
			if (i>0)	i--; 
		} else 
			string[i++]=string[j];
	} while(string[j++]!=0);


    if(string[0]=='-') {
        minus = 1;
        string++;
    }
    
    if(string[0]=='0' && (string[1]=='x' || string[1]=='X')) {
        base    = 16;
        string += 2;
    }
    
    lastIndex = strlen(string) - 1;
    
    if(lastIndex<0)
        return -1;
    
    if(string[lastIndex]=='h' || string[lastIndex]=='H' ) {
        base = 16;
        string[lastIndex] = 0;
        lastIndex--;
    }

    if(base==10) {
        result = atoi(string);
        result = minus ? (-1*result):result;
    }
    else {
        for(i=0;i<=lastIndex;i++) {
            if(isalpha(string[i])) {
                if(isupper(string[i]))
                    result = (result<<4) + string[i] - 'A' + 10;
                else
                    result = (result<<4) + string[i] - 'a' + 10;
            }
            else
                result = (result<<4) + string[i] - '0';
        }
        result = minus ? (-1*result):result;
    }
    return result;
}

//////////
// Function Name : DownloadImageThruUart
// Function Description : This function downloads a certain image through uart
// Input : DownloadAddress, address to download the image 
// Output : FileSize, size of downloaded image
// Version : 
u32 DownloadImageThruUart(u8 *DownloadAddress)
{
	char buf[4];
	int i;
	u32	FileSize;
	u16	CheckSum=0,dnCS;

	UART_Printf("\nDownloadAddress : 0x%08x\n",DownloadAddress);
	UART_Printf("STATUS : ");

	//	To get the file size.
	for(i=0;i<4;i++)
		buf[i]=UART_Getc();

	FileSize=(buf[0])+(buf[1]<<8)+(buf[2]<<16)+(buf[3]<<24);
	FileSize-=4;
	
	for(i=0;i<FileSize;i++) {
		*(DownloadAddress+i)=UART_Getc();
		if((i&0x3ff)==0)
			putchar('#');
	}

	for(i=0;i<(FileSize-2);i++)
		CheckSum+=*((u8 *)(DownloadAddress+i));

	dnCS=*((u8 *)(DownloadAddress+FileSize-2))+
		(*( (u8 *)(DownloadAddress+FileSize-1) )<<8);

	if(CheckSum!=dnCS) {
		UART_Printf("\nChecksum Error!!! MEM : %x  DN : %x\n",CheckSum,dnCS);
		FileSize=0;
	} else {
		FileSize-=2;
		UART_Printf("\n%d bytes downloaded OK.\n",FileSize);
	}
    	
	return FileSize;
}


//////////
// Function Name : Delay
// Function Description : 
// Input : usec, time in 100us unit
// Output : NONE
// Version : 
u32 delayLoopCount;
void Delay(u32 usec)
{
	u32 i=0;
    
    for(;usec>0;usec--)
    {
    	for(i=0;i<delayLoopCount;i++);
    }
}

//////////
// Function Name : Pow
// Function Description : Calculates x raise to the power of y
// Input : x, y
// Output : uResult - result value
// Version : 
u32 Pow(u32 x, u32 y)
{
    u32 i, uResult = 1;
    
    for(i=0 ; i<y ; i++)
    {
    	uResult = uResult*x;
    }
    
    return uResult;
}  

//////////
// Function Name : Copy
// Function Description : Copy from src address to dst address by words count
// Input : sa, da, words
// Output : void
// Version : 
void Copy(u32 sa, u32 da, u32 words)
{
	u32 i;	
	for (i=0; i<words; i++)
		*(u32 *)(da+i*4) = *(u32 *)(sa+i*4);
}

//////////
// Function Name : Copy8
// Function Description : Copy from src address to dst address by bytes count
// Input : sa, da, bytes
// Output : void
// Version : 
void Copy8(u32 sa, u32 da, u32 bytes)
{
	u32 i;
	for (i=0; i<bytes; i++)
		*(u8 *)(da+i) = *(u8 *)(sa+i);
}

//////////
// Function Name : Copy16
// Function Description : Copy from src address to dst address by bytes count
// Input : sa, da, bytes
// Output : void
// Version : 
void Copy16(u32 sa, u32 da, u32 Hwords)
{
	u32 i;
	for (i=0; i<Hwords; i++)
		*(u16 *)(da+i*2) = *(u16 *)(sa+i*2);
}

//////////
// Function Name : Compare
// Function Description : compare data
// Input : a0, a1, words
// Output : ret
// Version : 
u8 Compare( u32 a0,  u32 a1,  u32 words)
{
	volatile u32 d0,d1;
	volatile u32 *pD0, *pD1;
	u8 ret = true;
	u32  ecnt = 0;
	u32 i;
	
	pD0 = (volatile u32 *)(a0);
	pD1 = (volatile u32 *)(a1);
	
	//UART_Printf("\n");	

	for (i=0; i<words; i++)
	{
		
		d0=*pD0;
		d1=*pD1;

		if (d0!= d1) 
		{
			ret = false;
			UART_Printf(" %08x=%08x <-> %08x=%08x\n", pD0, d0, pD1, d1);
			ecnt++;
		}
		pD0++;
		pD1++;
	}
/*
	if (ret == false)
	{
		Assert(0);		
		UART_Printf("\n");
	}
*/
	return ret;
}

//////////
// Function Name : Dump32
// Function Description : dump data
// Input : addr, words
// Output : ret
// Version : 
void Dump32(u32 addr, u32 words)
{
	int i, j;
	u32 *pMem;

	pMem = (u32 *)(addr/4*4);

	UART_Printf("\n");
	for(i=0; i<words; )
	{
		UART_Printf(" %04x: ", i);

		for(j=0; j<8; j++)
			UART_Printf("%08x ", *pMem++),
			i++;
		UART_Printf("\n");
	}
}

//////////
// Function Name : Dump16
// Function Description : dump data
// Input : addr, hwords
// Output : ret
// Version : 
void Dump16(u32 addr, u32 hwords)
{
	int i, j;
	u16 *pMem;

	pMem = (u16 *)(addr/2*2);

	UART_Printf("\n");
	for(i=0; i<hwords; )
	{
		UART_Printf(" %04x: ", i);

		for(j=0; j<16; j++)
			UART_Printf("%04x ", *pMem++),
			i++;
		UART_Printf("\n");
	}
}


//////////
// Function Name : Dump8
// Function Description : dump data
// Input : addr, bytes
// Output : ret
// Version :
void Dump8(u32 addr, u32 bytes)
{
	int i, j;
	u8 *pMem;

	pMem = (u8 *)addr;

	UART_Printf("\n");
	for(i=0; i<bytes; )
	{
		UART_Printf(" %04x: ", i);

		for(j=0; j<16; j++)
			UART_Printf("%02x ", *pMem++),
			i++;
		UART_Printf("\n");
	}
}

/*
//////////
// Function Name : Stop
// Function Description : This function is called from "Assert"
// Version : 
void Stop(void) 
{ 
	while(1); 
}
*/

//////////
// Function Name : LoadFromFile
// Function Description : This function download the file from Host(PC)
// Input :  fileName - file name
//			uDstAddr - address to download
// Output : file size
// Version :
// added by rb1004
u32 LoadFromFile(const char* fileName, u32 uDstAddr)
{
	u32 uFileSize;
	
#if (SEMIHOSTING)
	u32 read;
	u32 i, uQuotient, uRemainder, uWriteAddr;

	FILE* fp = fopen(fileName, "rb");
//	Assert(fp);
	fseek(fp, 0, SEEK_END);
	uFileSize = ftell(fp);
	fseek(fp, 0, SEEK_SET);
    //rewind(fp);
	uWriteAddr = uDstAddr;
    	uQuotient = uFileSize/10000;
	uRemainder = uFileSize%10000;

	for(i=0; i<uQuotient ; i++)
	{
		read = fread((void *)uWriteAddr, sizeof(unsigned char), 10000, fp);
//		Assert(read==10000);
		uWriteAddr += 10000;
	}
	read = fread((void *)uWriteAddr, sizeof(unsigned char), uRemainder, fp);
//	Assert(read==uRemainder);	
	
	fclose(fp);
	
#else

	UART_Printf(" Sorry! It is not ready.\n");

#endif

	return uFileSize;
}

void LoadFromFile1(const char* fileName, u32 uDstAddr, u32* uFileSize)
{

#if SEMIHOSTING

	u32 read, i, uQuotient, uRemainder, uWriteAddr;

	FILE* fp = fopen(fileName, "rb");
//	Assert(fp);
	fseek(fp, 0, SEEK_END);
	*uFileSize = ftell(fp);
    	rewind(fp);

	uWriteAddr = uDstAddr;
    	uQuotient = *uFileSize/10000;
	uRemainder = *uFileSize%10000;

	for(i=0; i<uQuotient ; i++)
	{
		read = fread((void *)uWriteAddr, sizeof(unsigned char), 10000, fp);
//		Assert(read==10000);
		uWriteAddr += 10000;
	}
	read = fread((void *)uWriteAddr, sizeof(unsigned char), uRemainder, fp);
//	Assert(read==uRemainder);	
	
	fclose(fp);


	//read = fread((void *)uDstAddr, sizeof(unsigned char), *uFileSize, fp);
	Assert(read==*uFileSize);
	
	//fclose(fp);
	
#else

	UART_Printf(" Sorry! It is not ready.\n");

#endif
}


//////////
// Function Name : SaveToFile
// Function Description : This function upload the file to Host(PC)
// Input :  fileName - file name
//		  fileSize - upload file size
//		  uSrcAddr - address to download
// Output : None
// Version :
// added by rb1004
void SaveToFile(const char* fileName, u32 fileSize, u32 uSrcAddr)
{
#if (SEMIHOSTING)
	u32 written;
	u32 i, uQuotient, uRemainder, uWriteAddr;

	FILE* fp = fopen(fileName, "wb");
//	Assert(fp);

	uWriteAddr = uSrcAddr;
    	uQuotient = fileSize/10000;
	uRemainder = fileSize%10000;


	//written = fwrite((void *)uSrcAddr, sizeof(unsigned char), fileSize, fp);
	Assert(written == fileSize);
	
	for(i=0; i<uQuotient ; i++)
	{
		written = fwrite((void *)uWriteAddr, sizeof(unsigned char), 10000, fp);
//		Assert(written==10000);
		uWriteAddr += 10000;
	}
	written = fwrite((void *)uWriteAddr, sizeof(unsigned char), uRemainder, fp);
//	Assert(written==uRemainder);	

	fclose(fp);
#else

	UART_Printf(" Sorry! It is not ready.\n");

#endif
}


//////////
// Function Name : ConvertBmpToRgb16bpp
// Function Description : This function converts the BMP format to 16RGB format
// Input :  fromAddr - source address
//			toAddr - destination address
//			xSize - image x size
//			ySize - image y size
// Output : file size
// Version :
void ConvertBmpToRgb16bpp(u32 fromAddr, u32 toAddr, u32 xSize, u32 ySize)
{
	u32 i,x,y;
	u16 *lcdFramePtr;
	u16 *tempPtr;
//	u16 temp[IMAGE_MAXSIZE];
	u16 *temp;	
	u32 tempBuffer = fromAddr + 0x400000; // up to 1600x1200
	u8 b,g,r;
	u8 *srcPtr=(u8 *)(fromAddr+54);

	temp = malloc(IMAGE_MAXSIZE);	
	lcdFramePtr=(u16 *)(toAddr);

	for(y=0;y<ySize;y++) {
		for(x=0;x<xSize*3;x++) {
			b=*srcPtr++;
			g=*srcPtr++;
			r=*srcPtr++;

			*lcdFramePtr++=(r>>3)<<11|(g>>2)<<5|(b>>3)<<0;
		}
	}

	lcdFramePtr=(u16 *)(toAddr+xSize*ySize*2-2);
	tempPtr=(u16 *)(tempBuffer);
	for(y=0;y<xSize*ySize;y++) {
		*tempPtr++=*lcdFramePtr--;
	}

	lcdFramePtr=(u16 *)(toAddr);
	tempPtr=(u16 *)(tempBuffer);
	for(y=0;y<ySize;y++) {
		for(x=0;x<xSize;x++) {
			temp[(xSize-1)-x]=*tempPtr++;
		}
		if(x==xSize) {
			for(i=0;i<xSize;i++)
				*lcdFramePtr++=temp[i];
		}
	}
	free(temp);	
}


//////////
// Function Name : ConvertBmpToRgb24bpp
// Function Description : This function converts the BMP format to 24RGB format
// Input :  fromAddr - source address
//			toAddr - destination address
//			xSize - image x size
//			ySize - image y size
// Output : file size
// Version :
void ConvertBmpToRgb24bpp(u32 fromAddr, u32 toAddr, u32 xSize, u32 ySize)
{
	u32 i,x,y;
	u32 *lcdFramePtr,*tempPtr;
//	u32 temp[IMAGE_MAXSIZE];
	u32 *temp;	
	u32 tempBuffer = fromAddr + 0x760000; // up to 1600x1200
   	u8 b,g,r;
	u8 *srcPtr=(u8 *)(fromAddr+54);

	temp = malloc(IMAGE_MAXSIZE);
	
	lcdFramePtr=(u32 *)(toAddr);
   	for(y=0;y<xSize*ySize;y++) {
    	b=*srcPtr++;
    	g=*srcPtr++;
	    r=*srcPtr++;

	    *lcdFramePtr++=(r<<16)|(g<<8)|(b<<0);
	}

	lcdFramePtr=(u32 *)(toAddr+xSize*ySize*4-4);
	tempPtr=(u32 *)(tempBuffer);
    for(y=0;y<xSize*ySize;y++) {
	    *tempPtr++=*lcdFramePtr--;
    }

	lcdFramePtr=(u32 *)(toAddr);
	tempPtr=(u32 *)(tempBuffer);
    for(y=0;y<ySize;y++) {
	    for(x=0;x<xSize;x++) {
			temp[(xSize-1)-x]=*tempPtr++;
	    }
	    if(x==xSize) {	    	
		   	for(i=0;i<xSize;i++)
		    	*lcdFramePtr++=temp[i];
		}
	}	
	free(temp);
}

// Function Name : ConvertBmpToPackedRgb24bpp
// Function Description : This function converts the BMP format to 24RGB format
// Input :  fromAddr - source address
//			toAddr - destination address
//			xSize - image x size
//			ySize - image y size
// Output : file size
// Version :
void ConvertBmpToPackedRgb24bpp(u32 fromAddr, u32 toAddr, u32 xSize, u32 ySize)
{
	u32 i,x,y;
	u8 *lcdFramePtr,*tempPtr;
	u32 tempBuffer = fromAddr + 0x760000; // up to 1600x1200
   	u8 b,g,r;
	u8 *srcPtr=(u8 *)(fromAddr+54);
	
	lcdFramePtr=(u8 *)(toAddr);
   	for(y=0;y<xSize*ySize;y++)
	{
    		*lcdFramePtr++=*srcPtr++;
	    	*lcdFramePtr++=*srcPtr++;
		*lcdFramePtr++=*srcPtr++;		
	}
}

void ConvertImgSzToNumber(IMG_RESOLUTION eSize, u32* uHsz, u32* uVsz)
{
	if (eSize == UXGA)
		*uHsz = 1600, *uVsz = 1200; 
	else if (eSize == SXGA)
		*uHsz = 1280, *uVsz = 1024; 
	else if (eSize == SVGA)
		*uHsz = 800, *uVsz = 600;		
	else if (eSize == VGA)
		*uHsz = 640, *uVsz = 480; 
	else if (eSize == QVGA)
		*uHsz = 320, *uVsz = 240; 
	else if (eSize == QQVGA)
		*uHsz = 160, *uVsz = 120; 
	else if (eSize == CIF)
		*uHsz = 352, *uVsz = 288; 
	else if (eSize == QCIF)
		*uHsz = 172, *uVsz = 144; 
	else
	{
		Assert(0);
	}
			
}


REGINFO		sRegInfo[] = 
{

	// SROMC
	{"rSROM_BW			", SROM_BASE+0x00, 24, RW, DPDB, 0, 0x0},   	

	{"rSROM_BC0			", SROM_BASE+0x04, 32, RW, DPDB, 0, 0x0},   	
	{"rSROM_BC1			", SROM_BASE+0x08, 32, RW, DPDB, 0, 0x0},   	
	{"rSROM_BC2			", SROM_BASE+0x0C, 32, RW, DPDB, 0, 0x0},   	
	{"rSROM_BC3			", SROM_BASE+0x10, 32, RW, DPDB, 0, 0x0},   	
	{"rSROM_BC4			", SROM_BASE+0x14, 32, RW, DPDB, 0, 0x0},   	
	{"rSROM_BC5			", SROM_BASE+0x18, 32, RW, DPDB, 0, 0x0},   	

};

void RPRINTF(REGINFO sReg, u32 uPattern, u8 uRegStatus, s32 Nth,u8 uInDetail)
{
#if 0
	
	if(uInDetail)
		UART_Printf("\n%s,0x%X,0x%X,0x%X", sReg.name, sReg.uAddr, uPattern, sReg.rValue);	
	else
	{	
		if(!Nth)	
		UART_Printf("\n%s,0x%X\t", sReg.name, sReg.uAddr);
	}
	if(uRegStatus)
	{
		if(uInDetail)
			UART_Printf("\t[%d]",Nth);
		else
			UART_Printf("%d)",Nth);			
	}
#else		// Edited by SOP on 2008/03/01
	if(uInDetail)
		UART_Printf("\n%20s (0x%08X): Pattern 0x%08X, Read 0x%08X", sReg.name, sReg.uAddr, uPattern, sReg.rValue);	
	else
	{	
		if(!Nth)	
		UART_Printf("\n%20s, 0x%08X\t", sReg.name, sReg.uAddr);
	}
	if(uRegStatus)
	{
		if(uInDetail)
			UART_Printf(", Error Pattern Bit[%d]",Nth);
		else
			UART_Printf("%d)",Nth);			
	}

#endif	
	return;
}

void TestSFR(void)
{
	volatile u32 *pAddr;	
	s32 i,j, count = sizeof(sRegInfo)/sizeof(REGINFO), nPattern;
	u32 uRegStatus, uWritePatternWithMask, uReadWithMask;	
	
	
	
	u32 uPatternShitfOneBit[] = 
	{
		0x00000001,		0x00000002,		0x00000004,		0x00000008,		0x00000010,		0x00000020,		0x00000040,		0x00000080,
		0x00000100,		0x00000200,		0x00000400,		0x00000800,		0x00001000,		0x00002000,		0x00004000,		0x00008000,
		0x00010000,		0x00020000,		0x00040000,		0x00080000,		0x00100000,		0x00200000,		0x00400000,		0x00800000,
		0x01000000,		0x02000000,		0x04000000,		0x08000000,		0x10000000,		0x20000000,		0x40000000,		0x80000000,
	};
	
	u32 uDefaultBitMask[] =
	{
		0x00000001,		0x00000003,		0x00000007,		0x0000000F,		0x0000001F,		0x0000003F,		0x0000007F,		0x000000FF,
		0x000001FF,		0x000003FF,		0x000007FF,		0x00000FFF,		0x00001FFF,		0x00003FFF,		0x00007FFF,		0x0000FFFF,
		0x0001FFFF,		0x0003FFFF,		0x0007FFFF,		0x000FFFFF,		0x001FFFFF,		0x003FFFFF,		0x007FFFFF,		0x00FFFFFF,
		0x01FFFFFF,		0x03FFFFFF,		0x07FFFFFF,		0x0FFFFFFF,		0x1FFFFFFF,		0x3FFFFFFF,		0x7FFFFFFF,		0xFFFFFFFF,
	};

	
	for(i=0; i<count; i++)
	{		
		pAddr = (u32*)sRegInfo[i].uAddr;		
		
		if(sRegInfo[i].uFlag == DPDB)
		{		
			if(sRegInfo[i].uRWType == RW)
			{
				for(j=0; j< sRegInfo[i].uBitLen ; j++)
				{			
					uWritePatternWithMask = (uPatternShitfOneBit[j] &uDefaultBitMask[j]);
					*pAddr = uWritePatternWithMask;					
					
					uReadWithMask = (*pAddr & uDefaultBitMask[j]);	
					sRegInfo[i].rValue = uReadWithMask;
					
					uRegStatus = (	uWritePatternWithMask !=uReadWithMask );					
					RPRINTF(sRegInfo[i], uPatternShitfOneBit[j], uRegStatus, j, 1);
				}
			}
		}		
		else if(sRegInfo[i].uFlag == DPPB)
		{		
			nPattern = (sizeof(uPatternShitfOneBit)/sizeof(uPatternShitfOneBit[0]));			
			if(sRegInfo[i].uRWType == RW)
			{
				for(j=0; j<nPattern; j++)
				{
					if(uPatternShitfOneBit[j] & sRegInfo[i].uPrivateBitMask)
					{
						uWritePatternWithMask = (uPatternShitfOneBit[j] & sRegInfo[i].uPrivateBitMask);
						*pAddr = uWritePatternWithMask;
						
						uReadWithMask = (*pAddr & sRegInfo[i].uPrivateBitMask);
						sRegInfo[i].rValue = uReadWithMask;
						
						uRegStatus = (	uWritePatternWithMask != uReadWithMask );												
						RPRINTF(sRegInfo[i], uPatternShitfOneBit[j], uRegStatus, j, 1);					
					}
				}
			}			
		}					
	}
	UART_Printf("\n\nTest Complete!\n\n");

	
	return ;
}

//////////
// Function Name : GetBitPosition
// Function Description : This function return the Bit Position of uValue
// Input :  	uValue - Value
// Output : 	Bit position
// Version :
// added by rb1004
u32 GetBitPosition(u32 uValue)
{
	u8 bitshift;

	for(bitshift=0 ; bitshift<32 ; bitshift++)
	{
	  if(uValue & (1<<bitshift))
	  	return bitshift;
	}
	return 0;  
}


//added by rb1004...2007.03.09
void UART_Printf(const char *fmt,...)
{
    va_list ap;
    char string[256];
	int i;

    va_start(ap, fmt);
    vsprintf(string, fmt, ap);
    for (i = 0; string[i]; i++)
		UART_Putc(string[i]);
    va_end(ap);
}

//////////
// Function Name : Inp32Not4ByteAlign
// Function Description : This function plays a similar role to Inp32(addr) when address is not aligned with 4bytes
// Input :  addr
// Output : data
// Version :
u32 Inp32Not4ByteAlign(u32 addr)
{
	u32 Temp1;
	u32 Temp2;
	u32 Temp3;
	u32 Temp4;
	
	Temp1 = Inp8(addr);
	addr++;
	Temp2 = Inp8(addr);
	Temp2 = Temp2<<8;
	addr++;
	Temp3 = Inp8(addr);
	Temp3 = Temp3<<16;
	addr++;
	Temp4 = Inp8(addr);
	Temp4 = Temp4<<24;
	return (Temp1 | Temp2 |Temp3 |Temp4 );
}

//////////
// Function Name : Pause
// Function Description : This function is Pause
// Input :  none
// Output : none
// Version : 0.1
void Pause(void)
{
	UART_Printf("\nPress any key to continue ...\n");
	UART_Getc();
}

//////////
// Function Name : TestRegAccessInClockPowerOff
// Function Description : Accesses its register in Clock/Power Off
// Input : 
//		 eTESTIP - the IP in Clock or Power Off state
//		MaskTestIP - 0(Mask: no test), 1(unMask : test)
// Output : 1
// Version : Number of fails


int AccessOtherRegsInClockPowerOff(CLKPWR_TEST eTESTIP, int MaskTestIP)
{
	unsigned int nTestIP;
	unsigned int *pRegAddr;
	CLKPWR_TEST	eTestIP	= eTESTIP;
	unsigned int uInitVal, uReadVal, nFails=0;

	Outp32(0xE010E804, 0x960001);			//HDMI Clock Setup

//	USB Setup
//	SYSC_SetOscPadOnOff(eMODE_NORMAL, eOSC_USB, true);	//enable OTG clock pad
//	SYSC_SetUSBPHYControl(true);	//unmask usb signal
// . phy-power on 
//---------------
//	Outp32(0xEC100000, Inp32(0xEC100000)&~(0x7<<6)|(0x4<<6));
// . set clock source for PLL
//-----------------------	
//	Outp32(0xEC100004, Inp32(0xEC100004)&~(0x3<<0)|(0x3<<0));	//24MHz
	
	for(nTestIP = 0; nTestIP<NTESTIP; nTestIP++)
	{
		pRegAddr = (unsigned int*)sRegClkPowerTest[nTestIP].uAddr;
		UART_Printf("\n%s is being tested now  ", sRegClkPowerTest[nTestIP].name);
		if(nTestIP != eTestIP)
		{
			uInitVal = *pRegAddr;							//Read Init value
			*pRegAddr &= ~(sRegClkPowerTest[nTestIP].uMask<< 0);
			*pRegAddr |= ((sRegClkPowerTest[nTestIP].uWriteVal & sRegClkPowerTest[nTestIP].uMask)<< 0);		//Write value
			uReadVal = (sRegClkPowerTest[nTestIP].uMask& ( *pRegAddr));				//Read value			
			if(sRegClkPowerTest[nTestIP].uWriteVal != uReadVal)			//Compare
			{
				UART_Printf("\n%s Error : R/W, R-0x%x, W-0x%x", 
					sRegClkPowerTest[nTestIP].name, uReadVal, 
					(sRegClkPowerTest[nTestIP].uWriteVal & sRegClkPowerTest[nTestIP].uMask));
				nFails++;
			}
			*pRegAddr = uInitVal;							//Restore					
		}
		else
		{
			if(MaskTestIP)
				uInitVal = *pRegAddr;							//Read Init value
			//*pRegAddr &= ~(MASKBIT << 0);
			//*pRegAddr |= (sRegClkPowerTest[nTestIP].uWriteVal << 0);		//Write value			
			//*pRegAddr = uInitVal;							//Restore		
		}
			
	}
	if(!(nFails))
		UART_Printf("\nTest is Passed : R/W ");
	else
		UART_Printf("\nNumber of fails : 0x%x ", nFails);
	return nFails;
}

int DispGetIntNum(const char *fmt,...)
{
	va_list ap;
	char string[256];
	int i;
	int uReturnNum;

	va_start(ap, fmt);
	vsprintf(string, fmt, ap);
	for (i = 0; string[i]; i++)
		UART_Putc(string[i]);
	va_end(ap);

	uReturnNum = UART_GetIntNum();
	return uReturnNum;
}