//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//

/*++

THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:

    Drv.c   Power Controller Driver

Abstract:

   Streams interface driver (MDD)

Functions:

Notes:

--*/

#include <bsp.h>
#include <pmpolicy.h>
#include "button.h"
#include <drvmsg.h>
#include <drvlib_mem.h>

#ifdef DEEPIDLE_TEST
#define LPAUDIO_TEST
#endif

// In case of Voltage Control through PMIC, Reset button must not be used.
// Because GPIO(GPH04) for controling Internal voltage in PMIC is shared to reset button

static volatile GPIO_REG *g_pGPIOReg = NULL;
static DWORD g_dwSysIntrPowerBtn = SYSINTR_UNDEFINED;
static HANDLE g_hEventPowerBtn = NULL;
static HANDLE g_hThreadPowerBtn = NULL;

#ifndef ENABLE_VOLTAGE_CONTROL
static HANDLE g_hEventResetBtn = NULL;
static DWORD g_dwSysIntrResetBtn = SYSINTR_UNDEFINED;
static HANDLE g_hThreadResetBtn = NULL;
#endif

static BOOL g_bExitThread = FALSE;


HANDLE PWC_msgQue = NULL;


INT WINAPI PowerButtonThread(void)
{
    DWORD nBtnCount = 0;
	DWORD dwErr;	
    WCHAR  state[1024] = {0};
	LPWSTR pState = &state[0];
	DWORD dwBufChars = (sizeof(state) / sizeof(state[0]));
	DWORD  dwStateFlags = 0;   
#ifdef LPAUDIO_TEST
	MSG_DEV_STATE   txmqParam;
	MSGQUEUEOPTIONS msgOptions;

    // 2. Message Queue initialization for Device Power State transition
    memset((void *)&msgOptions, 0x0, sizeof(msgOptions));
    msgOptions.dwSize = sizeof(MSGQUEUEOPTIONS);
    msgOptions.dwFlags = 0;
    msgOptions.dwMaxMessages = 1024;        // Max number of MSG queue entries
    msgOptions.cbMaxMessage = sizeof(MSG_DEV_STATE);
    msgOptions.bReadAccess = FALSE;

    PWC_msgQue = CreateMsgQueue(OEMPM_MSGQ_NAME, &msgOptions);
#endif

    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PowerButtonThread()\r\n")));

    while(!g_bExitThread)
    {
        WaitForSingleObject(g_hEventPowerBtn, INFINITE);

        if(g_bExitThread)
        {
            break;
        }

		dwErr = GetSystemPowerState(pState, dwBufChars, &dwStateFlags);
		if (ERROR_SUCCESS != dwErr)
		{
		    ERRMSG((_T("PMGET!GetSystemPowerState:ERROR:%d\n"), dwErr));
		}
		else
		{
		    DBGMSG(PWRBTN_INFO, (TEXT("PMGET! System Power state is '%s', flags 0x%08x\n"), state, dwStateFlags));
		}
        
        Button_pwrbtn_disable_interrupt();            // Mask EINT
        Button_pwrbtn_clear_interrupt_pending();        // Clear Interrupt Pending

        InterruptDone(g_dwSysIntrPowerBtn);

#ifndef    SLEEP_AGING_TEST
        while(Button_pwrbtn_is_pushed())
        {
            // Wait for Button Released...
            Sleep(10);
        }
#endif

        nBtnCount++;
        DBGMSG(PWRBTN_INFO, (TEXT("[PWRBTN] Power Button Event [%d]\r\n"), nBtnCount));

#ifdef LPAUDIO_TEST

        GetSystemPowerState(pState, dwBufChars, &dwStateFlags);
        DBGMSG(PWRBTN_INFO, (TEXT("PBT_THREAD Power State : %s(0x%x)\r\n"), state,dwStateFlags));

        if(wcscmp(state,TEXT("useridle")) == 0)
        {
            SetSystemPowerState(NULL, POWER_STATE_ON, POWER_FORCE);

            RETAILMSG(1,(_T("Send LP Mode escaping\r\n")));

        	txmqParam.bLPModeSupport = FALSE;
        	txmqParam.dwDeviceID = IRQ_EINT31;
        	txmqParam.dwMsg = DTOP_POWER_SET;
        	txmqParam.dwLParam = D2;
        	txmqParam.dwRParam = 0;
            if(!WriteMsgQueue(PWC_msgQue, &txmqParam, sizeof(MSG_DEV_STATE), INFINITE, 0))
            {
                RETAILMSG(1,(TEXT(" PWC:WriteMsgQueue Error(0x%x)\r\n"),GetLastError()));
            }

        }
        else
        {
            SetSystemPowerState(NULL, POWER_STATE_IDLE, POWER_FORCE);
            RETAILMSG(1,(_T("Send LP Mode entering\r\n")));

        	txmqParam.bLPModeSupport = FALSE;
        	txmqParam.dwDeviceID = IRQ_EINT31;
        	txmqParam.dwMsg = DTOP_POWER_SET;
        	txmqParam.dwLParam = D2;
        	txmqParam.dwRParam = 0;
            if(!WriteMsgQueue(PWC_msgQue, &txmqParam, sizeof(MSG_DEV_STATE), INFINITE, 0))
            {
                RETAILMSG(1,(TEXT(" PWC:WriteMsgQueue Error(0x%x)\r\n"),GetLastError()));
            }

        }
#else
        GetSystemPowerState(pState, dwBufChars, &dwStateFlags);
        DBGMSG(PWRBTN_INFO, (TEXT("PBT_THREAD Power State : %s(0x%x)\r\n"), state,dwStateFlags));

		if ( wcscmp(state,TEXT("screenoff")) == 0 )
		{
		    DBGMSG(PWRBTN_INFO,(TEXT("setsystempowerstate to ON from screen off\r\n"))); 
			SetSystemPowerState( NULL, POWER_STATE_ON, POWER_FORCE );
		}
		else if ( wcscmp(state,TEXT("systemidle")) == 0 )
		{
		    DBGMSG(PWRBTN_INFO,(TEXT("setsystempowerstate to ON from systemidle\r\n"))); 
			SetSystemPowerState( NULL, POWER_STATE_ON, POWER_FORCE );
		} 
		else if ( wcscmp(state,TEXT("useridle")) == 0 )
		{
		    DBGMSG(PWRBTN_INFO,(TEXT("setsystempowerstate to ON from useridle\r\n"))); 
			SetSystemPowerState( NULL, POWER_STATE_ON, POWER_FORCE );
		}         
		else if ( wcscmp(state,TEXT("backlightoff")) == 0 )
		{
            DBGMSG(PWRBTN_INFO,(TEXT("setsystempowerstate to ON from backlight off\r\n"))); 
			SetSystemPowerState( NULL, POWER_STATE_ON, POWER_FORCE );
		}
		else
		{
			// CE device
			//PowerPolicyNotify(PPN_POWERBUTTONPRESSED, 0);
			SetSystemPowerState(NULL, POWER_STATE_SUSPEND, POWER_FORCE);
		}      
#endif
        Button_pwrbtn_enable_interrupt();            // UnMask EINT
        
#ifdef    SLEEP_AGING_TEST
        Sleep(7000);
        SetEvent(g_hEventPowerBtn);
#endif
    }

    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PowerButtonThread()\r\n")));
	
    return 0;
}


#ifndef ENABLE_VOLTAGE_CONTROL
INT WINAPI ResetButtonThread(void)
{

#ifdef LPAUDIO_TEST
	WCHAR  state[1024] = {0};
	LPWSTR pState = &state[0];
    DWORD dwBufChars = (sizeof(state) / sizeof(state[0]));
   	DWORD  dwStateFlags = 0;
	MSG_DEV_STATE   txmqParam;
	MSGQUEUEOPTIONS msgOptions;

    // 2. Message Queue initialization for Device Power State transition
    memset((void *)&msgOptions, 0x0, sizeof(msgOptions));
    msgOptions.dwSize = sizeof(MSGQUEUEOPTIONS);
    msgOptions.dwFlags = 0;
    msgOptions.dwMaxMessages = 1024;        // Max number of MSG queue entries
    msgOptions.cbMaxMessage = sizeof(MSG_DEV_STATE);
    msgOptions.bReadAccess = FALSE;

    PWC_msgQue = CreateMsgQueue(OEMPM_MSGQ_NAME, &msgOptions);    
#endif

    DBGMSG(PWRBTN_FUNC, (TEXT("[RSTBTN] ++ResetButtonThread()\r\n")));
    
    while(!g_bExitThread)
    {
        WaitForSingleObject(g_hEventResetBtn, INFINITE);

        if(g_bExitThread)
        {
            break;
        }

        Button_rstbtn_disable_interrupt();            // Mask EINT
        Button_rstbtn_clear_interrupt_pending();        // Clear Interrupt Pending

        InterruptDone(g_dwSysIntrResetBtn);

        DBGMSG(PWRBTN_INFO, (TEXT("[RSTBTN] Reset Button Event\r\n")));
        
#ifdef LPAUDIO_TEST

        GetSystemPowerState(pState, dwBufChars, &dwStateFlags);
        DBGMSG(PWRBTN_INFO, (TEXT("PBT_THREAD Power State : %s(0x%x)\r\n"), state,dwStateFlags));

        if(wcscmp(state,TEXT("useridle")) == 0)
        {
            SetSystemPowerState(NULL, POWER_STATE_ON, POWER_FORCE);

            RETAILMSG(1,(_T("Send LP Mode escaping\r\n")));

        	txmqParam.bLPModeSupport = FALSE;
        	txmqParam.dwDeviceID = IRQ_EINT4;
        	txmqParam.dwMsg = DTOP_POWER_SET;
        	txmqParam.dwLParam = D2;
        	txmqParam.dwRParam = 0;
            if(!WriteMsgQueue(PWC_msgQue, &txmqParam, sizeof(MSG_DEV_STATE), INFINITE, 0))
            {
                RETAILMSG(1,(TEXT(" PWC:WriteMsgQueue Error(0x%x)\r\n"),GetLastError()));
            }

        }
        else
        {
            SetSystemPowerState(NULL, POWER_STATE_IDLE, POWER_FORCE);
            RETAILMSG(1,(_T("Send LP Mode entering\r\n")));

        	txmqParam.bLPModeSupport = FALSE;
        	txmqParam.dwDeviceID = IRQ_EINT4;
        	txmqParam.dwMsg = DTOP_POWER_SET;
        	txmqParam.dwLParam = D2;
        	txmqParam.dwRParam = 0;
            if(!WriteMsgQueue(PWC_msgQue, &txmqParam, sizeof(MSG_DEV_STATE), INFINITE, 0))
            {
                RETAILMSG(1,(TEXT(" PWC:WriteMsgQueue Error(0x%x)\r\n"),GetLastError()));
            }

        }
#else
        SetSystemPowerState(NULL, POWER_STATE_RESET, POWER_FORCE);
        //KernelIoControl(IOCTL_HAL_REBOOT, NULL, 0, NULL, 0, NULL);
        ERRMSG((_T("[RSTBTN] Soft Reset Failed\r\n")));
#endif
        Button_rstbtn_enable_interrupt();                // UnMask EINT
    }

    DBGMSG(PWRBTN_FUNC, (TEXT("[RSTBTN] --ResetButtonThread()\r\n")));
    return 0;
}
#endif

static BOOL
PWR_AllocResources(void)
{
    DWORD dwIRQ;
    PHYSICAL_ADDRESS    ioPhysicalBase = {0,0};

    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PWR_AllocResources()\r\n")));

    //------------------
    // GPIO Controller SFR
    //------------------
    g_pGPIOReg = (volatile GPIO_REG *)DrvLib_MapIoSpace(BASE_REG_PA_GPIO, sizeof(GPIO_REG), FALSE);
    if (g_pGPIOReg == NULL)
    {
        ERRMSG((_T("[PWRBTN] PWR_AllocResources() : pGPIOReg DrvLib_MapIoSpace() Failed \n\r")));
        return FALSE;
    }

    //--------------------
    // Power Button Interrupt
    //--------------------
    dwIRQ = IRQ_EINT31;
    g_dwSysIntrPowerBtn = SYSINTR_UNDEFINED;
    g_hEventPowerBtn = NULL;

    if (!KernelIoControl(IOCTL_HAL_REQUEST_SYSINTR, &dwIRQ, sizeof(DWORD), &g_dwSysIntrPowerBtn, sizeof(DWORD), NULL))
    {
        ERRMSG((_T("[PWRBTN:ERR] PWR_AllocResources() : IOCTL_HAL_REQUEST_SYSINTR Power Button Failed \n\r")));
        g_dwSysIntrPowerBtn = SYSINTR_UNDEFINED;
        return FALSE;
    }

    g_hEventPowerBtn = CreateEvent(NULL, FALSE, FALSE, NULL);
    if(NULL == g_hEventPowerBtn)
    {
        ERRMSG((_T("[PWRBTN:ERR] PWR_AllocResources() : CreateEvent() Power Button Failed \n\r")));
        return FALSE;
    }

    if (!(InterruptInitialize(g_dwSysIntrPowerBtn, g_hEventPowerBtn, 0, 0)))
    {
        ERRMSG((_T("[PWRBTN:ERR] PWR_AllocResources() : InterruptInitialize() Power Button Failed \n\r")));
        return FALSE;
    }

#ifndef ENABLE_VOLTAGE_CONTROL
    //--------------------
    // Reset Button Interrupt
    //--------------------
    dwIRQ = IRQ_EINT4;
    g_dwSysIntrResetBtn = SYSINTR_UNDEFINED;
    g_hEventResetBtn = NULL;

    if (!KernelIoControl(IOCTL_HAL_REQUEST_SYSINTR, &dwIRQ, sizeof(DWORD), &g_dwSysIntrResetBtn, sizeof(DWORD), NULL))
    {
        ERRMSG((_T("[PWRBTN:ERR] PWR_AllocResources() : IOCTL_HAL_REQUEST_SYSINTR Reset Button Failed \n\r")));
        g_dwSysIntrResetBtn = SYSINTR_UNDEFINED;
        return FALSE;
    }

    g_hEventResetBtn = CreateEvent(NULL, FALSE, FALSE, NULL);
    if(NULL == g_hEventResetBtn)
    {
        ERRMSG((_T("[PWRBTN:ERR] PWR_AllocResources() : CreateEvent() Reset Button Failed \n\r")));
        return FALSE;
    }

    if (!(InterruptInitialize(g_dwSysIntrResetBtn, g_hEventResetBtn, 0, 0)))
    {
        ERRMSG((_T("[PWRBTN:ERR] PWR_AllocResources() : InterruptInitialize() Reset Button Failed \n\r")));
        return FALSE;
    }
#endif

    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PWR_AllocResources()\r\n")));

    return TRUE;
}

static void
PWR_ReleaseResources(void)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PWR_ReleaseResources()\r\n")));

    if (g_pGPIOReg != NULL)
    {
        DrvLib_UnmapIoSpace((PVOID)g_pGPIOReg);
        g_pGPIOReg = NULL;
    }

    if (g_dwSysIntrPowerBtn != SYSINTR_UNDEFINED)
    {
        InterruptDisable(g_dwSysIntrPowerBtn);
    }

    if (g_hEventPowerBtn != NULL)
    {
        CloseHandle(g_hEventPowerBtn);
    }

    if (g_dwSysIntrPowerBtn != SYSINTR_UNDEFINED)
    {
        KernelIoControl(IOCTL_HAL_RELEASE_SYSINTR, &g_dwSysIntrPowerBtn, sizeof(DWORD), NULL, 0, NULL);
    }

#ifndef ENABLE_VOLTAGE_CONTROL
    if (g_dwSysIntrResetBtn != SYSINTR_UNDEFINED)
    {
        InterruptDisable(g_dwSysIntrResetBtn);
    }

    if (g_hEventResetBtn != NULL)
    {
        CloseHandle(g_hEventResetBtn);
    }

    if (g_dwSysIntrResetBtn != SYSINTR_UNDEFINED)
    {
        KernelIoControl(IOCTL_HAL_RELEASE_SYSINTR, &g_dwSysIntrResetBtn, sizeof(DWORD), NULL, 0, NULL);
    }
#endif

    g_pGPIOReg = NULL;

    g_dwSysIntrPowerBtn = SYSINTR_UNDEFINED;
    g_hEventPowerBtn = NULL;

#ifndef ENABLE_VOLTAGE_CONTROL    
    g_dwSysIntrResetBtn = SYSINTR_UNDEFINED;
    g_hEventResetBtn = NULL;
#endif    

    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PWR_ReleaseResources()\r\n")));
}

BOOL
DllMain(HINSTANCE hinstDll, DWORD dwReason, LPVOID lpReserved)
{
    if (dwReason == DLL_PROCESS_ATTACH)
    {
        //DEBUGREGISTER(hinstDll);
        DBGMSG(PWRBTN_INFO, (TEXT("[PWRBTN] DllEntry() : Process Attach\r\n")));
    }
    else if (dwReason == DLL_PROCESS_DETACH)
    {
        DBGMSG(PWRBTN_INFO, (TEXT("[PWRBTN] DllEntry() : Process Detach\r\n")));
    }

    return TRUE;
}

BOOL
PWR_PowerUp(DWORD pContext)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PWR_PowerUp(0x%08x)\r\n"), pContext));

    // Do not Disable Interrupt for Power Bottun
    // Also Do not Clear Pending.
    // After Wake Up. Button Driver Should Get Interrupt
    // and Set System Power State to Resume...
    // Interrupt Disable
    Button_pwrbtn_disable_interrupt();
#ifndef ENABLE_VOLTAGE_CONTROL    
    Button_rstbtn_disable_interrupt();
#endif

    // Port Initialize as External Interrupt
    Button_port_initialize();

    // Interrupt Siganl Method and Filtering
    Button_pwrbtn_set_interrupt_method(EINT_SIGNAL_FALL_EDGE);
    Button_pwrbtn_set_filter_method(EINT_FILTER_DELAY, 0);
#ifndef ENABLE_VOLTAGE_CONTROL
    Button_rstbtn_set_interrupt_method(EINT_SIGNAL_FALL_EDGE);
    Button_rstbtn_set_filter_method(EINT_FILTER_DELAY, 0);
#endif
    // Clear Interrupt Pending
    Button_pwrbtn_clear_interrupt_pending();
#ifndef ENABLE_VOLTAGE_CONTROL
    Button_rstbtn_clear_interrupt_pending();
#endif

    // Enable Interrupt
    Button_pwrbtn_enable_interrupt();
#ifndef ENABLE_VOLTAGE_CONTROL
    Button_rstbtn_enable_interrupt();
#endif
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PWR_PowerUp(0x%08x)\r\n"), pContext));
    return TRUE;
}

BOOL
PWR_PowerDown(DWORD pContext)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PWR_PowerDown(0x%08x)\r\n"), pContext));

    // Interrupt Disable and Clear Pending
    Button_pwrbtn_disable_interrupt();
    Button_pwrbtn_clear_interrupt_pending();
#ifndef ENABLE_VOLTAGE_CONTROL    
    Button_rstbtn_disable_interrupt();
    Button_rstbtn_clear_interrupt_pending();
#endif
	DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PWR_PowerDown(0x%08x)\r\n"), pContext));
    return TRUE;
}

BOOL PWR_Deinit(DWORD pContext)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PWR_Deinit(0x%08x)\r\n"), pContext));

    g_bExitThread = TRUE;

    if (g_hThreadPowerBtn)        // Make Sure if thread is exist
    {
        Button_pwrbtn_disable_interrupt();
        Button_pwrbtn_clear_interrupt_pending();

        // Signal Thread to Finish
        SetEvent(g_hEventPowerBtn);
        // Wait for Thread to Finish
        WaitForSingleObject(g_hThreadPowerBtn, INFINITE);
        CloseHandle(g_hThreadPowerBtn);
        g_hThreadPowerBtn = NULL;
    }
#ifndef ENABLE_VOLTAGE_CONTROL
    if (g_hThreadResetBtn)        // Make Sure if thread is exist
    {
        Button_rstbtn_disable_interrupt();
        Button_rstbtn_clear_interrupt_pending();

        // Signal Thread to Finish
        SetEvent(g_hEventResetBtn);
        // Wait for Thread to Finish
        WaitForSingleObject(g_hThreadResetBtn, INFINITE);
        CloseHandle(g_hThreadResetBtn);
        g_hThreadResetBtn = NULL;
    }
#endif
    PWR_ReleaseResources();

    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PWR_Deinit(0x%08x)\r\n")));
    return TRUE;
}

DWORD
PWR_Init(DWORD dwContext)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] ++PWR_Init(0x%08x)\r\n"), dwContext));

    if (PWR_AllocResources() == FALSE)
    {
        ERRMSG((_T("[PWR:ERR] PWR_Init() : PWR_AllocResources() Failed \n\r")));
        goto CleanUp;
    }

    Button_initialize_register_address((void *)g_pGPIOReg);

    // Interrupt Disable and Clear Pending
    Button_pwrbtn_disable_interrupt();

#ifndef ENABLE_VOLTAGE_CONTROL    
    Button_rstbtn_disable_interrupt();
#endif
    // Initialize Port as External Interrupt
    Button_port_initialize();

    // Create Power Button Thread
    g_hThreadPowerBtn = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) PowerButtonThread, NULL, 0, NULL);
    if (g_hThreadPowerBtn == NULL )
    {
        ERRMSG((_T("[PWR:ERR] PWR_Init() : CreateThread() Power Button Failed \n\r")));
        goto CleanUp;
    }
#ifndef ENABLE_VOLTAGE_CONTROL
    // Create Reset Button Thread
    g_hThreadResetBtn = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) ResetButtonThread, NULL, 0, NULL);
    if (g_hThreadResetBtn == NULL )
    {
        ERRMSG((_T("[PWR:ERR] PWR_Init() : CreateThread() Reset Button Failed \n\r")));
        goto CleanUp;
    }
#endif

    // Interrupt Siganl Method and Filtering
    Button_pwrbtn_set_interrupt_method(EINT_SIGNAL_FALL_EDGE);
    Button_pwrbtn_set_filter_method(EINT_FILTER_DELAY, 0);
    
#ifndef ENABLE_VOLTAGE_CONTROL
    Button_rstbtn_set_interrupt_method(EINT_SIGNAL_FALL_EDGE);
    Button_rstbtn_set_filter_method(EINT_FILTER_DELAY, 0);
#endif

    // Clear Interrupt Pending
    Button_pwrbtn_clear_interrupt_pending();
#ifndef ENABLE_VOLTAGE_CONTROL
    Button_rstbtn_clear_interrupt_pending();
#endif
    // Enable Interrupt
    Button_pwrbtn_enable_interrupt();
#ifndef ENABLE_VOLTAGE_CONTROL
    Button_rstbtn_enable_interrupt();
#endif
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] --PWR_Init(0x%08x)\r\n")));
    return TRUE;

CleanUp:

    ERRMSG((_T("[PWR:ERR] --PWR_Init() : Failed\r\n")));
    PWR_Deinit(0);

    return FALSE;
}

DWORD
PWR_Open(DWORD pContext, DWORD dwAccess, DWORD dwShareMode)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] PWR_Open(0x%08x, 0x%08x, 0x%08x)\r\n"), pContext, dwAccess, dwShareMode));    return (0x12345678);
}

BOOL
PWR_Close(DWORD pContext)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] PWR_Close(0x%08x)\r\n"), pContext));

    return TRUE;
}

DWORD
PWR_Read (DWORD pContext,  LPVOID pBuf, DWORD Len)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] PWR_Read(0x%08x, 0x%08x, 0x%08x)\r\n"), pContext, pBuf, Len));

    return (0);    // End of File
}

DWORD
PWR_Write(DWORD pContext, LPCVOID pBuf, DWORD Len)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] PWR_Write(0x%08x, 0x%08x, 0x%08x)\r\n"), pContext, pBuf, Len));

    return (0);    // Number of Byte
}

DWORD
PWR_Seek (DWORD pContext, long pos, DWORD type)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] PWR_Seek(0x%08x, 0x%08x, 0x%08x)\r\n"), pContext, pos, type));

    return (DWORD)-1;    // Failure
}

BOOL
PWR_IOControl(DWORD pContext, DWORD dwCode, PBYTE pBufIn, DWORD dwLenIn, PBYTE pBufOut, DWORD dwLenOut, PDWORD pdwActualOut)
{
    DBGMSG(PWRBTN_FUNC, (TEXT("[PWRBTN] PWR_IOControl(0x%08x, 0x%08x)\r\n"), pContext, dwCode));

    return FALSE;    // Failure
}

