//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this sample source code is subject to the terms of the Microsoft
// license agreement under which you licensed this sample source code. If
// you did not accept the terms of the license agreement, you are not
// authorized to use this sample source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the LICENSE.RTF on your install media or the root of your tools installation.
// THE SAMPLE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES OR INDEMNITIES.
//
//
// Copyright (c) Samsung Electronics. Co. LTD.  All rights reserved.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    DispConAPI.c

Abstract:       Implementation of Video Driver
                This module handle Display Controller IOCTLs

Functions:


Notes:


--*/


#include <bsp.h>
#include "SVEngine.h"

#define GAMMA_LUT_SIZE  DISP_GAMMA_VALUE_NUM

BOOL SVE_DispCon_API_Proc(
    DWORD hOpenContext,
    DWORD dwCode,
    PBYTE pBufIn,
    DWORD dwLenIn,
    PBYTE pBufOut,
    DWORD dwLenOut,
    PDWORD pdwActualOut
    )
{
    SVEngineContext *pCtxt;
    SVEnginePowerContext *pPMCtxt;
    BOOL bRet = TRUE;

    VDE_MSG((_T("[VDE] ++SVE_DispCon_API_Proc()\n\r")));

    pCtxt = SVE_get_context();
    pPMCtxt = SVE_get_power_context();

    //-------------------------------------------------------------
    // Check OpenContext of Caller have the right to access to H/W Resource
    //-------------------------------------------------------------
    switch(dwCode)
    {
        case SVE_FIMD_SET_INTERFACE_PARAM:
        case SVE_FIMD_SET_OUTPUT_RGBIF:
        case SVE_FIMD_SET_OUTPUT_ENABLE:
        case SVE_FIMD_SET_OUTPUT_DISABLE:
        case SVE_FIMD_GET_OUTPUT_STATUS:
        {
            bRet = SVE_resource_compare_FIMD_interface(hOpenContext);
            break;
        }
        case SVE_FIMD_SET_WINDOW_MODE:
        {
            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
            }
            else
            {
            SVEARG_FIMD_WIN_MODE *pArg = (SVEARG_FIMD_WIN_MODE *)pBufIn;
            bRet = SVE_resource_compare_FIMD_window( \
                        Disp_get_win_num_from_win_mode(pArg->dwWinMode), \
                        hOpenContext);
            }
            break;
        }
        case SVE_FIMD_SET_WINDOW_POSITION:
        {
            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
            }
            else
            {
            SVEARG_FIMD_WIN_POS *pArg = (SVEARG_FIMD_WIN_POS *)pBufIn;
            bRet = SVE_resource_compare_FIMD_window( \
                                                pArg->dwWinNum, hOpenContext);
            }
            break;
        }
        case SVE_FIMD_SET_WINDOW_FRAMEBUFFER:
        {
            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
            }
            else
            {
                SVEARG_FIMD_WIN_FRAMEBUFFER *pArg = \
                                        (SVEARG_FIMD_WIN_FRAMEBUFFER *)pBufIn;
                bRet = SVE_resource_compare_FIMD_window( \
                                                pArg->dwWinNum, hOpenContext);
            }
            break;
        }
        case SVE_FIMD_SET_WINDOW_ENABLE:
        case SVE_FIMD_SET_WINDOW_DISABLE:
        {
            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
            }
            else
            {
                DWORD dwWinNum = *((DWORD *)pBufIn);
                bRet = SVE_resource_compare_FIMD_window(dwWinNum, \
                                                            hOpenContext);
            }
            break;
        }
        case SVE_FIMD_SET_WINDOW_COLORMAP:
        {
            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
            }
            else
            {
                SVEARG_FIMD_COLORMAP_CONTROL *pArg = \
                                        (SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn;
                bRet = SVE_resource_compare_FIMD_window(pArg->eWinNum, \
                                                                hOpenContext);
            }
            break;
        }
        case SVE_FIMD_SET_WONDOW_QOS:
        {
            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
            }
            else
            {
                SVEARG_FIMD_WINDOW_QOS_CONTROL *pArg = \
                                    (SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn;
                bRet = SVE_resource_compare_FIMD_window(pArg->eWinNum, \
                                                                hOpenContext);
            }
            break;
        }

        // ColorKey and AlphaBlend setting depends on two adjacent windows's setting
        case SVE_FIMD_SET_WINDOW_BLEND_COLORKEY:
        case SVE_FIMD_SET_WINDOW_BLEND_ALPHA:
        case SVE_FIMD_SET_WINDOW_BLEND_ALPHA_EX:

        case SVE_FIMD_WAIT_FRAME_INTERRUPT:
        case SVE_FIMD_GET_WINDOW_STATUS:
        case SVE_FIMD_SET_WINDOW_BLEND_DISABLE:
        case SVE_FIMD_SET_WB_ENABLE:
        case SVE_FIMD_SET_WB_DISABLE:

        case SVE_FIMD_SET_GAMMA:
        case SVE_FIMD_SET_COLORGAIN:
        case SVE_FIMD_SET_HUE:
        case SVE_FIMD_SET_DITHERING:
        case SVE_FIMD_SET_ALPS:
            bRet = TRUE;
            break;
        default:
            VDE_ERR( (TEXT("[VDE:ERR] Invalid IOCTL code\n")));
            bRet = FALSE;
            goto CleanUp;
    }

    if (!bRet)
    {
        VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc(0x%08x) : No Right to Access to H/W Resource\r\n"), dwCode));
        goto CleanUp;
    }

    //-----------------------------------
    // Processing IOCTL for Display Controller
    //-----------------------------------
    switch(dwCode)
    {
        case SVE_FIMD_SET_INTERFACE_PARAM:
        {
            SVEARG_FIMD_OUTPUT_IF *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_INTERFACE_PARAM\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            pArg = (SVEARG_FIMD_OUTPUT_IF *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                // Setup Output Device Information fot LCD
                Disp_set_output_device_information(&pArg->tDispDevInfo);
            }

            // Backup for PM
            memcpy(&pPMCtxt->tFIMDOutputParam, pArg, sizeof(SVEARG_FIMD_OUTPUT_IF));

            break;
        }

        case SVE_FIMD_SET_OUTPUT_RGBIF:
        {
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_OUTPUT_RGBIF\n\r")));

            if (pPMCtxt->bPowerOn)
            {
                // Initialize Display Controller for RGB I/F
                Disp_initialize_output_interface(DISP_VIDOUT_RGBIF);
                Disp_set_output_path(pPMCtxt->eVideoOutPath);
            }

            break;
        }

        case SVE_FIMD_SET_WB_ENABLE:
        {
            DWORD* pdwCaptureFrameRate;
            DISP_VIDOUT_MODE tDIspMode = \
                            pPMCtxt->tFIMDOutputParam.tDispDevInfo.VideoOutMode;
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WB_ENABLE\n\r")));

            if(tDIspMode != DISP_VIDOUT_RGBIF)// && \
                //tDIspMode != DISP_VIDOUT_I80IF_LDI0 && \
                //tDIspMode != DISP_VIDOUT_I80IF_LDI1)
            {
                VDE_ERR((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WB_ENABLE not allowed current mode for WB(%d)\n\r"),tDIspMode));
                break;
            }

            pdwCaptureFrameRate = (DWORD *)pBufIn;

            if(*pdwCaptureFrameRate > \
                    pPMCtxt->tFIMDOutputParam.tDispDevInfo.RGBINFO.Frame_Rate)
            {
                *pdwCaptureFrameRate = \
                    pPMCtxt->tFIMDOutputParam.tDispDevInfo.RGBINFO.Frame_Rate;
            }

            if (pPMCtxt->bPowerOn)
            {
                // Enable writeback interface
                Disp_enable_WB_interface( \
                    pPMCtxt->tFIMDOutputParam.tDispDevInfo.RGBINFO.Frame_Rate/ \
                        (*pdwCaptureFrameRate));
            }

            // Backup for PM
            pPMCtxt->dwCaptureFrameRate = *pdwCaptureFrameRate;
            pPMCtxt->bFIMDOutputWB = TRUE;

            break;
        }

        case SVE_FIMD_SET_WB_DISABLE:
        {
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WB_DISABLE\n\r")));

            if(pPMCtxt->bFIMDOutputWB == FALSE)
            {
                VDE_ERR((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WB_DISABLE current status is not WB state\n\r")));
                break;
            }

            if (pPMCtxt->bPowerOn)
            {
                // Enable writeback interface
                Disp_disable_WB_interface(pPMCtxt->eVideoOutPath);
            }

            // Backup for PM
            pPMCtxt->bFIMDOutputWB = FALSE;

            break;
        }

        case SVE_FIMD_SET_OUTPUT_ENABLE:
        {
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_OUTPUT_ENABLE\n\r")));

            if (pPMCtxt->bPowerOn)
            {
                // Enable Interrupt
                Disp_set_frame_interrupt(DISP_FRMINT_FRONTPORCH);
                Disp_enable_frame_interrupt();

                // Video Output Enable
                Disp_envid_onoff(DISP_ENVID_ON);
                pCtxt->bVideoEnable = TRUE;
            }

            // Backup for PM
            pPMCtxt->bFIMDOutputEnable = TRUE;

            break;
        }

        case SVE_FIMD_SET_OUTPUT_DISABLE:
        {
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_OUTPUT_DISABLE\n\r")));
            //Sleep(10); // Because of HW Bug for TVSC off.

            if (pPMCtxt->bPowerOn)
            {
                // Video Output Disable
                Disp_envid_onoff(DISP_ENVID_OFF);
                pCtxt->bVideoEnable = FALSE;
            }

            // Backup for PM
            pPMCtxt->bFIMDOutputEnable = FALSE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_MODE:
        {
            SVEARG_FIMD_WIN_MODE *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_MODE\n\r")));

            pArg = (SVEARG_FIMD_WIN_MODE *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                Disp_set_window_mode(pArg->dwWinMode, pArg->dwBPP, pArg->dwWidth, pArg->dwHeight, pArg->dwOffsetX, pArg->dwOffsetY);
            }

            // Backup for PM
            memcpy(&pPMCtxt->tFIMDWinMode[Disp_get_win_num_from_win_mode(pArg->dwWinMode)], pArg, sizeof(SVEARG_FIMD_WIN_MODE));
            pPMCtxt->bFIMDWinMode[Disp_get_win_num_from_win_mode(pArg->dwWinMode)] = TRUE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_POSITION:
        {
            SVEARG_FIMD_WIN_POS *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_POSITION\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            pArg = (SVEARG_FIMD_WIN_POS *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                Disp_set_window_position(pArg->dwWinNum, pArg->dwOffsetX, pArg->dwOffsetY);
                if (pArg->bWaitForVSync && pCtxt->bVideoEnable)
                {
                    if (WAIT_TIMEOUT == SVE_wait_disp_cmd_done())
                    {
                        VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_FRAMEBUFFER : SVE_wait_disp_cmd_done() TimeOut\n\r")));
                        //bRet = FALSE;
                    }
                }
            }

            // Backup for PM
            pPMCtxt->tFIMDWinMode[pArg->dwWinNum].dwOffsetX = pArg->dwOffsetX;
            pPMCtxt->tFIMDWinMode[pArg->dwWinNum].dwOffsetY = pArg->dwOffsetY;
            pPMCtxt->bFIMDWinMode[pArg->dwWinNum] = TRUE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_FRAMEBUFFER:
        {
            SVEARG_FIMD_WIN_FRAMEBUFFER *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_FRAMEBUFFER\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            pArg = (SVEARG_FIMD_WIN_FRAMEBUFFER *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                Disp_set_framebuffer(pArg->dwWinNum, pArg->dwFrameBuffer);
                if (pArg->bWaitForVSync && pCtxt->bVideoEnable)
                {
                    if (WAIT_TIMEOUT == SVE_wait_disp_cmd_done())
                    {
                        VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_FRAMEBUFFER : SVE_wait_disp_cmd_done() TimeOut\n\r")));
                        //bRet = FALSE;
                    }
                }
            }

            // Backup for PM
            memcpy(&pPMCtxt->tFIMDWinFBuffer[pArg->dwWinNum], pArg, sizeof(SVEARG_FIMD_WIN_FRAMEBUFFER));
            pPMCtxt->bFIMDWinFBuffer[pArg->dwWinNum] = TRUE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_ENABLE:
        {
            DWORD dwWinNum = 0;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_ENABLE\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            if (pPMCtxt->bPowerOn)
            {
                dwWinNum = *((DWORD *)pBufIn);
                Disp_window_onfoff(dwWinNum, DISP_WINDOW_ON);
                pCtxt->bWindowEnable[dwWinNum] = TRUE;
            }

            // Backup for PM
            pPMCtxt->bFIMDWinEnable[dwWinNum] = TRUE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_DISABLE:
        {
            DWORD dwWinNum;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_DISABLE\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                return FALSE;
            }

            dwWinNum = *((DWORD *)pBufIn);

            if (pPMCtxt->bPowerOn)
            {
                Disp_window_onfoff(dwWinNum, DISP_WINDOW_OFF);
                pCtxt->bWindowEnable[dwWinNum] = FALSE;
            }

            // Backup for PM
            pPMCtxt->bFIMDWinEnable[dwWinNum] = FALSE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_BLEND_COLORKEY:
        {
            SVEARG_FIMD_WIN_COLORKEY *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_BLEND_COLORKEY\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            pArg = (SVEARG_FIMD_WIN_COLORKEY *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                Disp_set_color_key(pArg->dwWinNum, pArg->bOnOff, pArg->bBlend, pArg->dwDirection, pArg->dwColorKey, pArg->dwCompareKey, pArg->dwKeyAlpha);
            }

            // Backup for PM
            memcpy(&pPMCtxt->tFIMDColorKey[pArg->dwWinNum], pArg, sizeof(SVEARG_FIMD_WIN_COLORKEY));
            pPMCtxt->bFIMDColorKey[pArg->dwWinNum] = TRUE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_BLEND_ALPHA:
        {
            SVEARG_FIMD_WIN_ALPHA *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_BLEND_ALPHA\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            pArg = (SVEARG_FIMD_WIN_ALPHA *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                Disp_set_alpha_blending(pArg->dwWinNum, pArg->dwMethod, pArg->dwAlpha0, pArg->dwAlpha1);
            }

            // Backup for PM
            memcpy(&pPMCtxt->tFIMDAlpha[pArg->dwWinNum], pArg, sizeof(SVEARG_FIMD_WIN_ALPHA));
            pPMCtxt->bFIMDAlpha[pArg->dwWinNum] = TRUE;

            break;
        }

        case SVE_FIMD_SET_WINDOW_BLEND_ALPHA_EX:
        {
            SVEARG_FIMD_WIN_ALPHA_EX *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_BLEND_ALPHA\n\r")));

            if(!pBufIn)
            {
                VDE_ERR((TEXT("Invalid Buffer : pBufIn:0x%x\n"), pBufIn));
                bRet = FALSE;
                break;
            }

            pArg = (SVEARG_FIMD_WIN_ALPHA_EX *)pBufIn;

            if (pPMCtxt->bPowerOn)
            {
                Disp_set_alpha_blending_ex(pArg->dwWinNum, pArg->dwMethod,
                                    pArg->dwABlendEq,
                                    pArg->dwBBlendEq,
                                    pArg->dwPBlendEq,
                                    pArg->dwQBlendEq,
                                    pArg->dwAlpha0, pArg->dwAlpha1);
            }

            // Backup for PM
            memcpy(&pPMCtxt->tFIMDAlphaEx[pArg->dwWinNum], pArg, sizeof(SVEARG_FIMD_WIN_ALPHA_EX));
            pPMCtxt->bFIMDAlpha[pArg->dwWinNum] = TRUE;

            break;
        }

        case SVE_FIMD_WAIT_FRAME_INTERRUPT:
        {
            if (pPMCtxt->bPowerOn && pCtxt->bVideoEnable)
            {
                if (WAIT_TIMEOUT == SVE_wait_disp_cmd_done())
                {
                    VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_WAIT_FRAME_INTERRUPT : SVE_wait_disp_cmd_done() TimeOut\n\r")));
                    bRet = FALSE;
                }
            }
            else
            {
                VDE_INF((_T("[VDE:INF] SVE_DispCon_API_Proc() : SVE_FIMD_WAIT_FRAME_INTERRUPT : Video Output is Disabled or Power Off Mode\n\r")));
            }
            break;
        }

        case SVE_FIMD_GET_OUTPUT_STATUS:
        {
            SVEARG_FIMD_OUTPUT_STAT *pArg;

            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_GET_OUTPUT_STATUS\n\r")));

            pArg = (SVEARG_FIMD_OUTPUT_STAT *)pBufOut;

            if (pPMCtxt->bPowerOn)
            {
                pArg->dwLineCnt = Disp_get_line_count();
                pArg->dwVerticalStatus = Disp_get_vertical_status();
                pArg->dwHorizontalStatus = 0;
                pArg->bENVID = Disp_get_envid_status();
            }
            else
            {
                // Return Pseudo Values in Power Off State
                pArg->dwLineCnt = 0;
                pArg->dwVerticalStatus = 0;
                pArg->dwHorizontalStatus = 0;
                pArg->bENVID = FALSE;
            }

            *pdwActualOut = sizeof(SVEARG_FIMD_OUTPUT_STAT);

            break;
        }

        case SVE_FIMD_SET_GAMMA:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_GAMMA\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_GAMMA_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            if( (((SVEARG_FIMD_GAMMA_CONTROL *)pBufIn)->bEnable == FALSE) && \
                    (((SVEARG_FIMD_GAMMA_CONTROL *)pBufIn)->uiLUTSize != GAMMA_LUT_SIZE) )
            {
                VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_GAMMA : Failed, LUT size must be 65!\n\r")));
                bRet = FALSE;
                break;
            }

            {
                PBYTE pbMapped;

                pbMapped = (PBYTE)MapCallerPtr(((SVEARG_FIMD_GAMMA_CONTROL *)pBufIn)->pbGammaLUT, GAMMA_LUT_SIZE);

                if(DISP_SUCCESS != Disp_set_gamma(((SVEARG_FIMD_GAMMA_CONTROL *)pBufIn)->bEnable, pbMapped))
                {
                    VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_GAMMA : Disp_set_gamma() failed\n\r")));
                    bRet = FALSE;
                    break;
                }
                else
                {
                    if(((SVEARG_FIMD_GAMMA_CONTROL *)pBufIn)->bEnable)
                    {
                        pPMCtxt->bFIMDOutputGamma = TRUE;
                        memcpy((void*)pPMCtxt->ucFIMDOutputGammaValue,
                                (void*)pbMapped,DISP_GAMMA_VALUE_NUM);
                    }
                    else
                    {
                        pPMCtxt->bFIMDOutputGamma = FALSE;
                    }
                }
            }
            break;

        case SVE_FIMD_SET_COLORGAIN:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_COLORGAIN\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_COLORGAIN_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            if(((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->bEnable == TRUE)
            {
                if(((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wRGain > 1023)
                {
                    VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_COLORGAIN : Color Gain value of R is out of range!\n\r")));
                    bRet = FALSE;
                    break;
                }

                if(((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wGGain > 1023)
                {
                    VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_COLORGAIN : Color Gain value of G is out of range!\n\r")));
                    bRet = FALSE;
                    break;
                }

                if(((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wBGain > 1023)
                {
                    VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_COLORGAIN : Color Gain value of B is out of range!\n\r")));
                    bRet = FALSE;
                    break;
                }
            }

            if(DISP_SUCCESS != Disp_set_colorgain(((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->bEnable, \
                                                    ((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wRGain, \
                                                    ((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wGGain, \
                                                    ((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wBGain))
            {
                VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_COLORGAIN : Disp_set_colorgain() failed\n\r")));
                bRet = FALSE;
            }
            else
            {
                if(((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->bEnable)
                {
                    pPMCtxt->bFIMDOutputColorGain = TRUE;
                    pPMCtxt->usFIMDOutputColorGainR = \
                            ((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wRGain;
                    pPMCtxt->usFIMDOutputColorGainG = \
                            ((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wGGain;
                    pPMCtxt->usFIMDOutputColorGainB = \
                            ((SVEARG_FIMD_COLORGAIN_CONTROL *)pBufIn)->wBGain;
                }
                else
                {
                    pPMCtxt->bFIMDOutputColorGain = FALSE;
                }
            }
            break;

        case SVE_FIMD_SET_HUE:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_HUE\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_HUE_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            if(DISP_SUCCESS != Disp_set_hue(((SVEARG_FIMD_HUE_CONTROL *)pBufIn)->bEnable, \
                                               ((SVEARG_FIMD_HUE_CONTROL *)pBufIn)->hueControl))
            {
                VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_HUE : Disp_set_hue() failed\n\r")));
                bRet = FALSE;
            }
            else
            {
                if(((SVEARG_FIMD_HUE_CONTROL *)pBufIn)->bEnable)
                {
                    pPMCtxt->bFIMDOutputHue = TRUE;
                    memcpy((void*)&(pPMCtxt->tFIMDOutputHueControl), \
                            (void*)&(((SVEARG_FIMD_HUE_CONTROL *)pBufIn)->hueControl), \
                            sizeof(DISP_HUE_CONTROL));
                }
                else
                {
                    pPMCtxt->bFIMDOutputHue = FALSE;
                }
            }
            break;

        case SVE_FIMD_SET_DITHERING:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_DITHERING\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_DITHERING_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            if(DISP_SUCCESS != Disp_set_dithering(((SVEARG_FIMD_DITHERING_CONTROL *)pBufIn)->bEnable, \
                                                    ((SVEARG_FIMD_DITHERING_CONTROL *)pBufIn)->eMode))
            {
                VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_DITHERING : Disp_set_dithering() failed\n\r")));
                bRet = FALSE;
            }
            else
            {
                if(((SVEARG_FIMD_DITHERING_CONTROL *)pBufIn)->bEnable)
                {
                    pPMCtxt->bFIMDOutDithering = TRUE;
                    pPMCtxt->eFIMDOutputDitheringMode = \
                            ((SVEARG_FIMD_DITHERING_CONTROL *)pBufIn)->eMode;
                }
                else
                {
                    pPMCtxt->bFIMDOutDithering = FALSE;
                }
            }
            break;

        case SVE_FIMD_SET_ALPS:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_ALPS\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_ALPS_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            {
                unsigned char* pucMapped_LR;
                unsigned char* pucMapped_LG;
                unsigned char* pucMapped_LB;
                unsigned char* pucMapped_HR;
                unsigned char* pucMapped_HG;
                unsigned char* pucMapped_HB;

                pucMapped_LR = (unsigned char*)MapCallerPtr( \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->pLow_R_LUT, \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize);
                pucMapped_LG = (unsigned char*)MapCallerPtr( \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->pLow_G_LUT, \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize);
                pucMapped_LB = (unsigned char*)MapCallerPtr( \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->pLow_B_LUT, \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize);
                pucMapped_HR = (unsigned char*)MapCallerPtr( \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->pHigh_R_LUT, \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize);
                pucMapped_HG = (unsigned char*)MapCallerPtr( \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->pHigh_G_LUT, \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize);
                pucMapped_HB = (unsigned char*)MapCallerPtr( \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->pHigh_B_LUT, \
                            ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize);

                if(DISP_SUCCESS != Disp_set_alps( \
                             ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->bEnable, \
                             ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->luxControl, \
                             ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiSizeH, \
                             ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiSizeV, \
                             ((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->uiValueSize, \
                             pucMapped_LR, \
                             pucMapped_LG, \
                             pucMapped_LB, \
                             pucMapped_HR, \
                             pucMapped_HG, \
                             pucMapped_HB \
                            ) \
                  )
                {
                    VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_ALPS : Disp_set_dithering() failed\n\r")));
                    bRet = FALSE;
                }
                else
                {
                    if(((SVEARG_FIMD_ALPS_CONTROL *)pBufIn)->bEnable)
                    {
                        pPMCtxt->bFIMDOutALPS = TRUE;
                    }
                    else
                    {
                        pPMCtxt->bFIMDOutALPS = FALSE;
                    }
                }
            }
            break;

        case SVE_FIMD_SET_WONDOW_QOS:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : IOCTL_SVE_FIMD_SET_WONDOW_QOS\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_WINDOW_QOS_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            if(DISP_SUCCESS != Disp_set_window_qos( \
                    ((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->eWinNum, \
                    ((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->bGateEn, \
                    ((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->ucFifoLevel) \
              )
            {
                VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : IOCTL_SVE_FIMD_SET_WONDOW_QOS : Disp_set_window_qos() failed\n\r")));
                bRet = FALSE;
            }
            else
            {
                if(((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->bGateEn)
                {
                    pPMCtxt->bFIMDWinQos[((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->eWinNum] = TRUE;
                    pPMCtxt->ucFIMDWinQosFifoLevel[((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->eWinNum] = \
                            ((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->ucFifoLevel;
                }
                else
                {
                    pPMCtxt->bFIMDWinQos[((SVEARG_FIMD_WINDOW_QOS_CONTROL *)pBufIn)->eWinNum] = FALSE;
                }
            }
            break;

        case SVE_FIMD_SET_WINDOW_COLORMAP:
            VDE_MSG((_T("[VDE] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_COLORMAP\n\r")));

            if(!pBufIn || dwLenIn != sizeof(SVEARG_FIMD_COLORMAP_CONTROL))
            {
                VDE_ERR((TEXT("Invalid parameter : pBufIn=0x%x, dwLenIn=0x%x\n"), pBufIn, dwLenIn));
                bRet = FALSE;
                break;
            }

            if(DISP_SUCCESS != Disp_set_window_color_map( \
                    ((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->eWinNum, \
                    ((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->bOnOff, \
                    ((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->uiColorValue) \
              )
            {
                VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_COLORMAP : Disp_set_window_color_map() failed\n\r")));
                bRet = FALSE;
            }
            else
            {
                if(((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->bOnOff)
                {
                    pPMCtxt->bFIMDWinQos[((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->eWinNum] = TRUE;
                    pPMCtxt->uiFIMDWinColormapColorValue[((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->eWinNum] = \
                            ((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->uiColorValue;
                }
                else
                {
                    pPMCtxt->bFIMDWinQos[((SVEARG_FIMD_COLORMAP_CONTROL *)pBufIn)->eWinNum] = FALSE;
                }
            }
            break;

        case SVE_FIMD_GET_WINDOW_STATUS:
            VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_GET_WINDOW_STATUS : Not Implemented Yet...\n\r")));
            bRet = FALSE;
            break;
        case SVE_FIMD_SET_WINDOW_BLEND_DISABLE:
            VDE_ERR((_T("[VDE:ERR] SVE_DispCon_API_Proc() : SVE_FIMD_SET_WINDOW_BLEND_DISABLE : Not Implemented Yet...\n\r")));
            bRet = FALSE;
            break;
        default:    // This is filtered above switch statement
            bRet = FALSE;
            break;
        }

CleanUp:

    VDE_MSG((_T("[VDE] --SVE_DispCon_API_Proc()\n\r")));

    return bRet;
}


