//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this sample source code is subject to the terms of the Microsoft
// license agreement under which you licensed this sample source code. If
// you did not accept the terms of the license agreement, you are not
// authorized to use this sample source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the LICENSE.RTF on your install media or the root of your tools installation.
// THE SAMPLE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES OR INDEMNITIES.
//
//
// Copyright (c) Samsung Electronics. Co. LTD.  All rights reserved.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    ldi.c

Abstract:       Libray to control LDI

Functions:


Notes:


--*/

#include <windows.h>
#include <bsp_cfg.h>
#include <register_map.h>
#include <mipidsi_reg.h>
#include <pmplatform.h>

#include <bsp.h>
#include <SVEDriverAPI.h>
#include "ldi.h"
#include "display_con.h"
#include "display_con_macro.h"
#include "Dsi_init.h"
#include "LTS222QV_RGB_dataset.h"

//#define LDI_MSG(x)      DBGMSG(DISP_USR1, x)
//#define LDI_INF(x)      DBGMSG(DISP_USR1, x)
#define LDI_MSG(x)      DBGMSG(0, x)
#define LDI_INF(x)      DBGMSG(0, x)
#define LDI_ERR(x)      DBGMSG(DISP_INFO, x)

#define LCD_DELAY_1MS    30000          // Sufficient under 1Ghz
#define SPI_DELAY        100            // Sufficient under 1Ghz


static volatile SPI_REG *g_pSPIReg = NULL;
static volatile GPIO_REG *g_pGPIOReg = NULL;
static volatile DISPLAY_REG *g_pDispConReg = NULL;
static volatile MIPIDSI_REG *g_pMIPIDSIReg = NULL;
static LDI_LCD_MODULE_TYPE g_ModuleType;

LDI_ERROR
LDI_initialize_register_address
(void *pSPIReg,
 void *pDispConReg,
 void *pGPIOReg)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s(0x%08x, 0x%08x, 0x%08x)\n\r"), _T(__FUNCTION__), pSPIReg, pDispConReg, pGPIOReg));

    if (pSPIReg == NULL || pDispConReg == NULL || pGPIOReg == NULL)
    {
        LDI_ERR((_T("[LDI:ERR] %s() : NULL pointer parameter\n\r"), _T(__FUNCTION__)));
        error = LDI_ERROR_NULL_PARAMETER;
    }
    else
    {
        g_pSPIReg = (SPI_REG *)pSPIReg;
        g_pDispConReg = (DISPLAY_REG *)pDispConReg;
        g_pGPIOReg = (GPIO_REG *)pGPIOReg;
        LDI_INF((_T("[LDI:INF] g_pSPIReg = 0x%08x\n\r"), g_pSPIReg));
        LDI_INF((_T("[LDI:INF] g_pDispConReg = 0x%08x\n\r"), g_pDispConReg));
        LDI_INF((_T("[LDI:INF] g_pGPIOReg    = 0x%08x\n\r"), g_pGPIOReg));
    }

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

LDI_ERROR
LDI_DSI_initialize_register_address
(void *pMIPIDSIReg,
 void *pDispConReg,
 void *pGPIOReg)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s(0x%08x, 0x%08x, 0x%08x)\n\r"), _T(__FUNCTION__), pMIPIDSIReg, pDispConReg, pGPIOReg));

    if (pMIPIDSIReg == NULL || pDispConReg == NULL || pGPIOReg == NULL)
    {
        LDI_ERR((_T("[LDI:ERR] %s() : NULL pointer parameter\n\r"), _T(__FUNCTION__)));
        error = LDI_ERROR_NULL_PARAMETER;
    }
    else
    {
        g_pMIPIDSIReg = (MIPIDSI_REG *)pMIPIDSIReg;
        g_pDispConReg = (DISPLAY_REG *)pDispConReg;
        g_pGPIOReg = (GPIO_REG *)pGPIOReg;

        LDI_INF((_T("[LDI:INF] g_pMIPIDSIReg    = 0x%08x\n\r"), g_pMIPIDSIReg));
        LDI_INF((_T("[LDI:INF] g_pDispConReg    = 0x%08x\n\r"), g_pDispConReg));
        LDI_INF((_T("[LDI:INF] g_pGPIOReg       = 0x%08x\n\r"), g_pGPIOReg));
    }

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

LDI_ERROR
LDI_set_LCD_module_type
(LDI_LCD_MODULE_TYPE ModuleType)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s(%d)\n\r"), _T(__FUNCTION__), ModuleType));

    g_ModuleType = ModuleType;

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}


#define BACKLIGHT_ON(gpioreg)  {  Set_PinData(gpioreg, GPH24_Output, 1);      \
								Set_PinFunction(gpioreg, GPH24_Output); \
								Set_PinData(gpioreg, GPF35_Output,1); \
								Set_PinFunction(gpioreg, GPF35_Output); \
								Set_PinData(gpioreg, GPD00_Output, 0);      \
								Set_PinFunction(gpioreg, GPD00_Output); }


// Display use GPIO as output mode 
// After Display initialize, change GPIO to use SPI1 channel
void LDI_SPI_GPIO_Reconfiguration()
{
    Set_PinFunction(g_pGPIOReg, SPI1_CLK);
    Set_PinFunction(g_pGPIOReg, SPI1_nSS);
    Set_PinFunction(g_pGPIOReg, SPI1_MISO);
    Set_PinFunction(g_pGPIOReg, SPI1_MOSI); 
    Set_PinPullUD(g_pGPIOReg, SPI1_CLK, sgip_PULL_DISABLE);
    Set_PinPullUD(g_pGPIOReg, SPI1_nSS, sgip_PULL_DISABLE);
    Set_PinPullUD(g_pGPIOReg, SPI1_MISO, sgip_PULL_DISABLE);
    Set_PinPullUD(g_pGPIOReg, SPI1_MOSI, sgip_PULL_DISABLE);  
}

LDI_ERROR
LDI_initialize_LCD_module
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    switch(g_ModuleType)
    {
    case LDI_LTV350QV_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_LTV350QV_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
// Backlight driver controls the backlight.       
//        LDI_LTE480WV_RGB_port_initialize();
        LDI_LTE480WV_RGB_power_on();
        LDI_LTE480WV_RGB_initialize();
        break;

	
    case AT070TN92_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] AT070TN92_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
// Backlight driver controls the backlight.       
//        LDI_LTE480WV_RGB_port_initialize();
        LDI_LTE480WV_RGB_power_on();
        LDI_LTE480WV_RGB_initialize();
        break;
    case LDI_HT101HD1_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_LTE480WV_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
//        LDI_HT101HD1_RGB_port_initialize();
        LDI_HT101HD1_RGB_power_on();
        LDI_HT101HD1_RGB_initialize();
        break;
    case LDI_DUMMY:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_DUMMY_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_DUMMY_power_on();
        DSI_initialize_LCD_module((void *)g_pMIPIDSIReg);
        break;
    case LDI_TL2796_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_TL2796_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
//        LDI_TL2796_RGB_port_initialize();
        LDI_TL2796_RGB_power_on();
        LDI_TL2796_RGB_initialize();
        break;
    case LDI_LTS222QV_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_LTS222QV_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
//        LDI_LTS222QV_port_initialize();
        LDI_LTS222QV_power_on();
        LDI_LTS222QV_RGB_initialize();
        break;
    case LDI_AUO27_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_AUO27_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
//        LDI_AUO27_port_initialize();
        LDI_AUO27_power_on();
        LDI_AUO27_RGB_initialize();
        break;

case LDI_VGA1280_1024_RGB:
case LDI_VGA1440_900_RGB:

	break;

	
default:
        LDI_ERR((_T("[LDI:ERR] %s() : Unknown LCD Module Type [%d]\n\r"), _T(__FUNCTION__), g_ModuleType));
        error = LDI_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    LDI_SPI_GPIO_Reconfiguration();
    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));



    return error;
}

LDI_ERROR
LDI_deinitialize_LCD_module
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    switch(g_ModuleType)
    {
    case AT070TN92_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] AT070TN92_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_LTE480WV_RGB_power_off();
        break;
    case LDI_HT101HD1_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_LTE480WV_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_HT101HD1_RGB_power_off();
        break;
    case LDI_DUMMY:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_DUMMY_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_DUMMY_power_off();
        break;
    case LDI_TL2796_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_TL2796_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_TL2796_RGB_power_off();
        break;
    case LDI_LTS222QV_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_LTS222QV_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_LTS222QV_power_off();
        break;
    case LDI_AUO27_RGB:
        LDI_INF((_T("[LDI:INF] %s() : Type [%d] LDI_AUO27_RGB\n\r"), _T(__FUNCTION__), g_ModuleType));
        LDI_AUO27_power_off();
        break;

		
		case LDI_VGA1280_1024_RGB:
		case LDI_VGA1440_900_RGB:
			break;
		
    default:
        LDI_ERR((_T("[LDI:ERR] %s() : Unknown LCD Module Type [%d]\n\r"), _T(__FUNCTION__), g_ModuleType));
        error = LDI_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

LDI_ERROR
LDI_fill_output_device_information
(void *pDevInfo)
{
    LDI_ERROR error = LDI_SUCCESS;
    tDevInfo *pDeviceInfo;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    if (pDevInfo == NULL)
    {
        LDI_ERR((_T("[LDI:ERR] %s() : Null Parameter\n\r"), _T(__FUNCTION__)));
        error = DISP_ERROR_NULL_PARAMETER;
        goto CleanUp;
    }

    pDeviceInfo = (tDevInfo *)pDevInfo;

    switch(g_ModuleType)
    {
    case AT070TN92_RGB:
        LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LTE480WV_RGB\n\r"), g_ModuleType));
        pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
        pDeviceInfo->RGBINFO.RGBOutMode = DISP_24BIT_RGB888_P;
        pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
        pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_RGB;
        pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_RGB;
        pDeviceInfo->RGBINFO.uiWidth = 800;
        pDeviceInfo->RGBINFO.uiHeight = 480;
        pDeviceInfo->RGBINFO.VBPD_Value = 23;
        pDeviceInfo->RGBINFO.VFPD_Value = 22;
        pDeviceInfo->RGBINFO.VSPW_Value = 1;
        pDeviceInfo->RGBINFO.HBPD_Value = 46;
        pDeviceInfo->RGBINFO.HFPD_Value = 210;
        pDeviceInfo->RGBINFO.HSPW_Value = 1;
        pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_FALL_EDGE;
        pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE;
        pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
        pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
        pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
        pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
        pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
        pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
        pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE;
        break;
    case LDI_HT101HD1_RGB:
        LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LTE480WV_RGB\n\r"), g_ModuleType));
        pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
        pDeviceInfo->RGBINFO.RGBOutMode = DISP_18BIT_RGB666_P;
        pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
        pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_RGB;
        pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_RGB;
        pDeviceInfo->RGBINFO.uiWidth = 1376;
        pDeviceInfo->RGBINFO.uiHeight = 768;
        pDeviceInfo->RGBINFO.VBPD_Value = 3; //15;
        pDeviceInfo->RGBINFO.VFPD_Value = 5;//4   //4;
        pDeviceInfo->RGBINFO.VSPW_Value = 5; //6;
        pDeviceInfo->RGBINFO.HBPD_Value = 13; //17;
        pDeviceInfo->RGBINFO.HFPD_Value = 8; //49;
        pDeviceInfo->RGBINFO.HSPW_Value = 3; //33;
        pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_FALL_EDGE;
        pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE;
        pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
        pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
        pDeviceInfo->RGBINFO.Frame_Rate = 70;    // VCLK > 24.5 MHz (Max 35.7 MHz)
        pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
        pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
        pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
        pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE;
        break;
    case LDI_DUMMY:
        LDI_INF((_T("[LDI:INF] Output Devce Type [%d] = LDI_(SMRP)LTE480WV_RGB\n\r"), g_ModuleType));
        pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
        pDeviceInfo->RGBINFO.RGBOutMode = DISP_24BIT_RGB888_P;
        pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
        pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_RGB;
        pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_RGB;
        pDeviceInfo->RGBINFO.uiWidth = 320;
        pDeviceInfo->RGBINFO.uiHeight = 480;
        pDeviceInfo->RGBINFO.VBPD_Value = 8;
        pDeviceInfo->RGBINFO.VFPD_Value = 9;
        pDeviceInfo->RGBINFO.VSPW_Value = 5;
        pDeviceInfo->RGBINFO.HBPD_Value = 5;
        pDeviceInfo->RGBINFO.HFPD_Value = 2;
        pDeviceInfo->RGBINFO.HSPW_Value = 3;
        pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_FALL_EDGE;
        pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE;
        pDeviceInfo->RGBINFO.Frame_Rate = 60;
        pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
        pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
        pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
        pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
        pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
        pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE;
        DSI_Get_DeviceInfo((tDevInfo *)pDeviceInfo);
        break;
    case LDI_TL2796_RGB:
        LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_TL2796_RGB\n\r"), g_ModuleType));
        pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
        pDeviceInfo->RGBINFO.RGBOutMode = DISP_24BIT_RGB888_P;
        pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
        pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_RGB;
        pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_RGB;
        pDeviceInfo->RGBINFO.uiWidth = 480;
        pDeviceInfo->RGBINFO.uiHeight = 800;
        pDeviceInfo->RGBINFO.VBPD_Value = 6;
        pDeviceInfo->RGBINFO.VFPD_Value = 5;
        pDeviceInfo->RGBINFO.VSPW_Value = 2;
        pDeviceInfo->RGBINFO.HBPD_Value = 9;
        pDeviceInfo->RGBINFO.HFPD_Value = 9;
        pDeviceInfo->RGBINFO.HSPW_Value = 2;
        pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE;
        pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
        pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
        pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
        pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
        pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
        pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
        pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE;
        break;

    case LDI_AUO27_RGB:
        LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_AUO27_RGB\n\r"), g_ModuleType));
        pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
        pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
        pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
        pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_GBR;
        pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_RGB;
        pDeviceInfo->RGBINFO.uiWidth = 960;
        pDeviceInfo->RGBINFO.uiHeight = 240;
        pDeviceInfo->RGBINFO.VBPD_Value = 12;
        pDeviceInfo->RGBINFO.VFPD_Value = 2;
        pDeviceInfo->RGBINFO.VSPW_Value = 2;
        pDeviceInfo->RGBINFO.HBPD_Value = 110;
        pDeviceInfo->RGBINFO.HFPD_Value = 256;
        pDeviceInfo->RGBINFO.HSPW_Value = 2;
        pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE;
        pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE;
        pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
        pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
        pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
        pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
        pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_ENABLE;
        pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
        pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE;
        break;
     case LDI_VGA8060_RGB:
        LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA8060_RGB\n\r"), g_ModuleType));
		#if 0 
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;  //RGB_ORDER_E_RGB;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;  //RGB_ORDER_O_RGB;
		pDeviceInfo->RGBINFO.uiWidth = 800;
		pDeviceInfo->RGBINFO.uiHeight = 600;
		pDeviceInfo->RGBINFO.VBPD_Value = 23;
		pDeviceInfo->RGBINFO.VFPD_Value = 1;
		pDeviceInfo->RGBINFO.VSPW_Value = 4;
		pDeviceInfo->RGBINFO.HBPD_Value = 210;
		pDeviceInfo->RGBINFO.HFPD_Value = 140;
		pDeviceInfo->RGBINFO.HSPW_Value = 38;
		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE; 
		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
		pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 
		#else //lqm
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;
		pDeviceInfo->RGBINFO.uiWidth = 800;
		pDeviceInfo->RGBINFO.uiHeight = 600;
		pDeviceInfo->RGBINFO.VBPD_Value = 23;
		pDeviceInfo->RGBINFO.VFPD_Value = 1;
		pDeviceInfo->RGBINFO.VSPW_Value = 4;
		pDeviceInfo->RGBINFO.HBPD_Value = 88;
		pDeviceInfo->RGBINFO.HFPD_Value = 40;
		pDeviceInfo->RGBINFO.HSPW_Value = 128;
		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE;;
		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
		pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 
		#endif
        break;
case LDI_VGA800_480_RGB:
        LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA800_480_RGB\n\r"), g_ModuleType));
		
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;
		pDeviceInfo->RGBINFO.uiWidth = 800;
		pDeviceInfo->RGBINFO.uiHeight = 480;
		pDeviceInfo->RGBINFO.VBPD_Value = 23;
		pDeviceInfo->RGBINFO.VFPD_Value = 1;
		pDeviceInfo->RGBINFO.VSPW_Value = 4;
		pDeviceInfo->RGBINFO.HBPD_Value = 88;
		pDeviceInfo->RGBINFO.HFPD_Value = 40;
		pDeviceInfo->RGBINFO.HSPW_Value = 128;
		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE;;
		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
	
		pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 

        break;
   case LDI_VGA1024_RGB:
		LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA1024_RGB\n\r"), g_ModuleType));
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;  //RGB_ORDER_E_RGB;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;  //RGB_ORDER_O_RGB;
		pDeviceInfo->RGBINFO.uiWidth = 1024;
		pDeviceInfo->RGBINFO.uiHeight = 768;
		pDeviceInfo->RGBINFO.VBPD_Value = 29;
		pDeviceInfo->RGBINFO.VFPD_Value = 3;
		pDeviceInfo->RGBINFO.VSPW_Value = 6;
		pDeviceInfo->RGBINFO.HBPD_Value = 160;
		pDeviceInfo->RGBINFO.HFPD_Value = 24;
		pDeviceInfo->RGBINFO.HSPW_Value = 136;
		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE; 
		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
		pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 
		break;



   case LDI_VGA1280_1024_RGB:
		LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA1280_1024_RGB\n\r"), g_ModuleType));
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;  //RGB_ORDER_E_RGB;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;  //RGB_ORDER_O_RGB;
		pDeviceInfo->RGBINFO.uiWidth = 1280;
		pDeviceInfo->RGBINFO.uiHeight = 1024;
		pDeviceInfo->RGBINFO.VBPD_Value = 44;
		pDeviceInfo->RGBINFO.VFPD_Value = 1;
		pDeviceInfo->RGBINFO.VSPW_Value = 3;
		pDeviceInfo->RGBINFO.HBPD_Value = 224;
		pDeviceInfo->RGBINFO.HFPD_Value = 64;
		pDeviceInfo->RGBINFO.HSPW_Value = 160;
		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE; 
		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE; // IHSYNC_HIGH_ACTIVE; //IHSYNC_LOW_ACTIVE
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE; // IVSYNC_HIGH_ACTIVE; //IHSYNC_LOW_ACTIVE
		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
		pDeviceInfo->RGBINFO.Frame_Rate = 85;    // VCLK > 24.5 MHz (Max 35.7 MHz)
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 
		break;	

   case LDI_VGA1440_900_RGB:
   	#if 1 
		LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA1440_900_75Hz \n\r"), g_ModuleType));
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;  //RGB_ORDER_E_RGB;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;  //RGB_ORDER_O_RGB;
		pDeviceInfo->RGBINFO.uiWidth = 1440;
		pDeviceInfo->RGBINFO.uiHeight = 900;
	
		pDeviceInfo->RGBINFO.HFPD_Value = 96;
		pDeviceInfo->RGBINFO.HSPW_Value = 152;
		pDeviceInfo->RGBINFO.HBPD_Value = 248;

		pDeviceInfo->RGBINFO.VFPD_Value = 3;
		pDeviceInfo->RGBINFO.VSPW_Value = 6;
		pDeviceInfo->RGBINFO.VBPD_Value = 33;

		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE; 
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IHSYNC_LOW_ACTIVE; 

		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE; 

		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
		pDeviceInfo->RGBINFO.Frame_Rate = 75;    // VCLK > 24.5 MHz (Max 35.7 MHz)
		
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 
		break;	
		
	#else
		LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA1440_900_60Hz \n\r"), g_ModuleType));
		RETAILMSG(1,(_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_VGA1440_900_60Hz \n\r"), g_ModuleType));
		
		pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
		pDeviceInfo->RGBINFO.RGBOutMode = DISP_16BIT_RGB565_P;
		pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
		pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_MASK;  //RGB_ORDER_E_RGB;
		pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_MASK;  //RGB_ORDER_O_RGB;
		pDeviceInfo->RGBINFO.uiWidth = 1440;
		pDeviceInfo->RGBINFO.uiHeight = 900;
		
		pDeviceInfo->RGBINFO.HFPD_Value = 80;
		pDeviceInfo->RGBINFO.HSPW_Value = 151;
		pDeviceInfo->RGBINFO.HBPD_Value = 232;
		
		pDeviceInfo->RGBINFO.VFPD_Value = 3;
		pDeviceInfo->RGBINFO.VSPW_Value = 6;
		pDeviceInfo->RGBINFO.VBPD_Value = 25;
		
		pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE  ; 
		pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_HIGH_ACTIVE; 
		
		pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_RISE_EDGE; 
		
		pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE; 
		
		pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
		pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
		pDeviceInfo->RGBINFO.Frame_Rate = 60;	 // VCLK > 24.5 MHz (Max 35.7 MHz)
		
		pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
		pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
		pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
		pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE; 
		break;	

	
	#endif
case LDI_LTV350QV_RGB:
	 LDI_INF((_T("[LDI:INF] Output Device Type [%d] = LDI_(SMRP)LDI_LTV350QV_RGB\n\r"), g_ModuleType));
        pDeviceInfo->VideoOutMode = DISP_VIDOUT_RGBIF;
        pDeviceInfo->RGBINFO.RGBOutMode = DISP_24BIT_RGB888_P;
        pDeviceInfo->RGBINFO.MIPIDSIen = MIPI_DSI_DIS;
        pDeviceInfo->RGBINFO.RGBEvenOrder = RGB_ORDER_E_RGB;
        pDeviceInfo->RGBINFO.RGBOddOrder = RGB_ORDER_O_RGB;
        pDeviceInfo->RGBINFO.uiWidth = 480;
        pDeviceInfo->RGBINFO.uiHeight = 272;
        pDeviceInfo->RGBINFO.VBPD_Value = 2;
        pDeviceInfo->RGBINFO.VFPD_Value = 2;
        pDeviceInfo->RGBINFO.VSPW_Value = 10;
        pDeviceInfo->RGBINFO.HBPD_Value = 22;
        pDeviceInfo->RGBINFO.HFPD_Value = 4;
        pDeviceInfo->RGBINFO.HSPW_Value = 1;
        pDeviceInfo->RGBINFO.VCLK_Polarity = IVCLK_FALL_EDGE;
        pDeviceInfo->RGBINFO.HSYNC_Polarity = IHSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VSYNC_Polarity = IVSYNC_LOW_ACTIVE;
        pDeviceInfo->RGBINFO.VDEN_Polarity = IVDEN_HIGH_ACTIVE;
        pDeviceInfo->RGBINFO.VCLK_Source = CLKSEL_F_LCDCLK;
        pDeviceInfo->RGBINFO.VCLK_Direction = CLKDIR_DIVIDED;
        pDeviceInfo->RGBINFO.Frame_Rate = 60;    // VCLK > 24.5 MHz (Max 35.7 MHz)
        pDeviceInfo->RGBINFO.VCLK_Gate = VCLK_NORMAL;
        pDeviceInfo->RGBINFO.RGBSkipEn = RGB_SKIP_DISABLE;
        pDeviceInfo->RGBINFO.RGBDummy = RGB_DUMMY_DISABLE;
        pDeviceInfo->RGBINFO.PixelCompensation = PIXEL_COMP_DISABLE;
	
    default:
        LDI_INF((_T("[LDI:ERR] %s() : Unknown device type [%d]\n\r"), _T(__FUNCTION__), g_ModuleType));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

CleanUp:

    LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));

    return error;
}

static void
LDI_LTE480WV_RGB_port_initialize
(void)
{
    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // by shin....1009
    // Back-light ON
    // Set GPIO direction to output
    Set_PinFunction(g_pGPIOReg, GPD03_Output);
    Set_PinPullUD(g_pGPIOReg, GPD03_Output, sgip_PULL_DISABLE);
    //g_pGPIOReg->GPD0.GP_CON = (g_pGPIOReg->GPD0.GP_CON & ~(0xf<<12)) | (1<<12);
    //g_pGPIOReg->GPD0.GP_PUD = (g_pGPIOReg->GPD0.GP_PUD & ~(0xf<<6)) | (0<<6);

    Set_PinData(g_pGPIOReg, GPD03_Output, sgip_Output_1);
    //g_pGPIOReg->GPD0.GP_DAT |= (1<<3);

    LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));
}

static void
LDI_HT101HD1_RGB_port_initialize
(void)
{
    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // by shin....1009
    // Back-light ON
    // Set GPIO direction to output
    Set_PinFunction(g_pGPIOReg, GPD03_Output);
    Set_PinPullUD(g_pGPIOReg, GPD03_Output, sgip_PULL_DISABLE);
    //g_pGPIOReg->GPD0.GP_CON = (g_pGPIOReg->GPD0.GP_CON & ~(0xf<<12)) | (1<<12);
    //g_pGPIOReg->GPD0.GP_PUD = (g_pGPIOReg->GPD0.GP_PUD & ~(0xf<<6)) | (0<<6);

    Set_PinData(g_pGPIOReg, GPD03_Output, sgip_Output_1);
    //g_pGPIOReg->GPD0.GP_DAT |= (1<<3);


    LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));
}

static LDI_ERROR
LDI_LTE480WV_RGB_power_on
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // PCI                 : GPN[5]
    // LCD_PANNEL_ON    : N/A in SMDK6410

    // Envid Disable
    //g_pDispConReg->VIDCON0 &= ~0x3;    //~(ENVID_ENABLE | ENVID_F_ENABLE);    // Direct Off

    // LCD Pannel Power On
    // Reset
    // CLEAR_TFT_LCD_nRESET(g_pGPIOReg);       // PCI set to Low Required redefinition
    // by shin....1009
    //LDI_MSG((TEXT("g_pGPIOReg->GPH0.GP_CON=0x%x\r\n"),&g_pGPIOReg->GPH0.GP_CON));
    Set_PinFunction(g_pGPIOReg, GPH06_Output);
    Set_PinPullUD(g_pGPIOReg, GPH06_Output, sgip_PULL_DISABLE);
    //Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_1);
    //g_pGPIOReg->GPH0.GP_CON = (g_pGPIOReg->GPH0.GP_CON & ~(0xf<<24)) | (0x1<<24);
    //g_pGPIOReg->GPH0.GP_PUD = (g_pGPIOReg->GPH0.GP_PUD & ~(0x3<<12)) | (0x0<<12);
    //g_pGPIOReg->GPH0.GP_DAT |= (1<<6);


    //DelayLoop_1ms(10);

    Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    //g_pGPIOReg->GPH0.GP_DAT &= ~(1<<6);
    // TODO: LCD Power On Here
    DelayLoop_1ms(10);                    // tp-sig > 10 ms

    // Envid Enable (Start output through RGB I/F)
    //g_pDispConReg->VIDCON0 |= 0x3;        //(ENVID_ENABLE | ENVID_F_ENABLE);
    //DelayLoop_1ms(20);                    // tvsync-don > 1 frame (16.7 ms)

    // Set PCI to High
//    SET_TFT_LCD_nRESET(g_pGPIOReg);         // PCI set to High  // Required redefinition
    //DelayLoop_1ms(70);    // by shin.. 1009
    Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_1);
    //g_pGPIOReg->GPH0.GP_DAT |= (1<<6);          // PCI set to High

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_HT101HD1_RGB_power_on
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // PCI                 : GPN[5]
    // LCD_PANNEL_ON    : N/A in SMDK6410

    // Envid Disable
    //g_pDispConReg->VIDCON0 &= ~0x3;    //~(ENVID_ENABLE | ENVID_F_ENABLE);    // Direct Off

    // LCD Pannel Power On
    // Reset
    // CLEAR_TFT_LCD_nRESET(g_pGPIOReg);       // PCI set to Low Required redefinition
    // by shin....1009
    //LDI_MSG((TEXT("g_pGPIOReg->GPH0.GP_CON=0x%x\r\n"),&g_pGPIOReg->GPH0.GP_CON));
    Set_PinFunction(g_pGPIOReg, GPH06_Output);
    Set_PinPullUD(g_pGPIOReg, GPH06_Output, sgip_PULL_DISABLE);
    //Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_1);
    //g_pGPIOReg->GPH0.GP_CON = (g_pGPIOReg->GPH0.GP_CON & ~(0xf<<24)) | (0x1<<24);
    //g_pGPIOReg->GPH0.GP_PUD = (g_pGPIOReg->GPH0.GP_PUD & ~(0x3<<12)) | (0x0<<12);
    //g_pGPIOReg->GPH0.GP_DAT |= (1<<6);


    //DelayLoop_1ms(10);

    Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    //g_pGPIOReg->GPH0.GP_DAT &= ~(1<<6);
    // TODO: LCD Power On Here
    DelayLoop_1ms(10);                    // tp-sig > 10 ms

    // Envid Enable (Start output through RGB I/F)
    //g_pDispConReg->VIDCON0 |= 0x3;        //(ENVID_ENABLE | ENVID_F_ENABLE);
    //DelayLoop_1ms(20);                    // tvsync-don > 1 frame (16.7 ms)

    // Set PCI to High
//    SET_TFT_LCD_nRESET(g_pGPIOReg);         // PCI set to High  // Required redefinition
    //DelayLoop_1ms(70);    // by shin.. 1009
    Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_1);
    //g_pGPIOReg->GPH0.GP_DAT |= (1<<6);          // PCI set to High

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_DUMMY_power_on
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    Set_PinFunction(g_pGPIOReg, MIPI_DSI_RESET);
    Set_PinPullUD(g_pGPIOReg, MIPI_DSI_RESET, sgip_PULL_DISABLE);
    Set_PinData(g_pGPIOReg, MIPI_DSI_RESET, sgip_Output_0);
    DelayLoop_1ms(10);                    // More than 1ms
    Set_PinData(g_pGPIOReg, MIPI_DSI_RESET, sgip_Output_1);

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_LTE480WV_RGB_power_off
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // PCI                 : GPN[5]
    // LCD_PANNEL_ON    : N/A in SMDK6410

    // Set PCI to Low
//    CLEAR_TFT_LCD_nRESET(g_pGPIOReg);   // PCI set to Low  // Required Redefinition
    Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    //g_pGPIOReg->GPH0.GP_DAT &= ~(1<<6);     // PCI set to Low
    //DelayLoop_1ms(40);                    // twht1 > 2 frame (33.3 ms)
    //DelayLoop_1ms(20);                    // tpoff > 1 frame (16.7 ms)

    // Envid Disable
    //g_pDispConReg->VIDCON0 &= ~(0x1);    //~(ENVID_F_ENABLE);    // Per Frame Off
   // DelayLoop_1ms(20);                    // Wait for frame finished (16.7 ms)
    DelayLoop_1ms(10);                    // tsig0ff-vdd > 10 ms

    // LCD Pannel Power Off
    // TODO: LCD Power Off Here

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_HT101HD1_RGB_power_off
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // PCI                 : GPN[5]
    // LCD_PANNEL_ON    : N/A in SMDK6410

    // Set PCI to Low
//    CLEAR_TFT_LCD_nRESET(g_pGPIOReg);   // PCI set to Low  // Required Redefinition
    Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    //g_pGPIOReg->GPH0.GP_DAT &= ~(1<<6);     // PCI set to Low
    //DelayLoop_1ms(40);                    // twht1 > 2 frame (33.3 ms)
    //DelayLoop_1ms(20);                    // tpoff > 1 frame (16.7 ms)

    // Envid Disable
    //g_pDispConReg->VIDCON0 &= ~(0x1);    //~(ENVID_F_ENABLE);    // Per Frame Off
   // DelayLoop_1ms(20);                    // Wait for frame finished (16.7 ms)
    DelayLoop_1ms(10);                    // tsig0ff-vdd > 10 ms

    // LCD Pannel Power Off
    // TODO: LCD Power Off Here

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_DUMMY_power_off
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    Set_PinData(g_pGPIOReg, MIPI_DSI_RESET, sgip_Output_0);
    DelayLoop_1ms(10);

    LDI_MSG((_T("[LDI]--%s(): %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_LTE480WV_RGB_initialize
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // There is No Power Sequence for LTP480WV

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_HT101HD1_RGB_initialize
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // There is No Power Sequence for LTP480WV

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}


static void
LDI_TL2796_RGB_port_initialize
(void)
{
    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Back-light ON
    // Set GPIO direction to output
    Set_PinFunction(g_pGPIOReg, BACKLIGHT);
    Set_PinPullUD(g_pGPIOReg, BACKLIGHT, sgip_PULL_DISABLE);
    Set_PinData(g_pGPIOReg, BACKLIGHT, sgip_Output_1);

    LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));
}

static LDI_ERROR
LDI_TL2796_RGB_power_on
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Envid Disable
    g_pDispConReg->VIDCON0 &= ~0x3;    //~(ENVID_ENABLE | ENVID_F_ENABLE);    // Direct Off

    Set_PinFunction(g_pGPIOReg, MLCD_RST);
    Set_PinPullUD(g_pGPIOReg, MLCD_RST, sgip_PULL_DISABLE);

    // Reset high
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_1);
    DelayLoop_1ms(10);

    // Reset low
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_0);
    DelayLoop_1ms(10);

    // Reset high
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_1);
    DelayLoop_1ms(10);

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_TL2796_RGB_power_off
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Reset low
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_0);
    //Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    DelayLoop_1ms(40);                    // twht1 > 2 frame (33.3 ms)
    DelayLoop_1ms(20);                    // tpoff > 1 frame (16.7 ms)

    // Envid Disable
    g_pDispConReg->VIDCON0 &= ~(0x1);    //~(ENVID_F_ENABLE);    // Per Frame Off
    DelayLoop_1ms(20);                    // Wait for frame finished (16.7 ms)
    DelayLoop_1ms(10);                    // tsig0ff-vdd > 10 ms

    // LCD Pannel Power Off
    // TODO: LCD Power Off Here

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_TL2796_RGB_initialize
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    DelayLoop_1ms(25); // more than 20ms
    //[1] Panel Condition Set
    Write_LDI_TL2796(0x7031, 0x7208);
    Write_LDI_TL2796(0x7032, 0x7214);
    Write_LDI_TL2796(0x7030, 0x7202);

#if 1
    Write_LDI_TL2796(0x7027, 0x7203);
#else
    Write_LDI_TL2796(0x7027, 0x7201);
#endif

    //[2] Display Condition Set(RGB Data Interface)
    Write_LDI_TL2796(0x7012, 0x7208); //0x7208
    Write_LDI_TL2796(0x7013, 0x7208);
#if 1
    Write_LDI_TL2796(0x7015, 0x7210); //0x10
#else
    Write_LDI_TL2796(0x7015, 0x7200); //0x00
#endif
    Write_LDI_TL2796(0x7016, 0x7200);    //00 24bit 02 16bit RGB interface
    Write_LDI_TL2796(0x70ef, 0x72d0);
    Write_LDI_TL2796(0x0000, 0x72e8);

    // set gamma control 230cd
    Write_LDI_TL2796(0x7039, 0x7244);
    Write_LDI_TL2796(0x7040, 0x7200);
    Write_LDI_TL2796(0x7041, 0x723F);
    Write_LDI_TL2796(0x7042, 0x722B);
    Write_LDI_TL2796(0x7043, 0x721f);
    Write_LDI_TL2796(0x7044, 0x7224);
    Write_LDI_TL2796(0x7045, 0x721b);
    Write_LDI_TL2796(0x7046, 0x7229);
    Write_LDI_TL2796(0x7050, 0x7200);
    Write_LDI_TL2796(0x7051, 0x7200);
    Write_LDI_TL2796(0x7052, 0x7200);
    Write_LDI_TL2796(0x7053, 0x721b);
    Write_LDI_TL2796(0x7054, 0x7222);
    Write_LDI_TL2796(0x7055, 0x721b);
    Write_LDI_TL2796(0x7056, 0x722a);
    Write_LDI_TL2796(0x7060, 0x7200);
    Write_LDI_TL2796(0x7061, 0x723F);
    Write_LDI_TL2796(0x7062, 0x7225);
    Write_LDI_TL2796(0x7063, 0x721c);
    Write_LDI_TL2796(0x7064, 0x7221);
    Write_LDI_TL2796(0x7065, 0x7218);
    Write_LDI_TL2796(0x7066, 0x723e);

    //Analog Power Condition Set
    Write_LDI_TL2796(0x7017, 0x7222);    //Boosting Freq
    Write_LDI_TL2796(0x7018, 0x7233);    //power AMP Medium
    Write_LDI_TL2796(0x7019, 0x7203);    //Gamma Amp Medium
    Write_LDI_TL2796(0x701a, 0x7201);    //Power Boosting
    Write_LDI_TL2796(0x7022, 0x72a4);    //Vinternal = 0.65*VCI
    Write_LDI_TL2796(0x7023, 0x7200);    //VLOUT1 Setting = 0.98*VCI
    Write_LDI_TL2796(0x7026, 0x72a0);    //Display Condition LTPS signal generation : Reference= DOTCLK

    //Stand-by Off Command
    Write_LDI_TL2796(0x701d, 0x72a0);

    DelayLoop_1ms(300); // more than 250ms

    //LCD ON
    Write_LDI_TL2796(0x7014, 0x7203);

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static void
LDI_LTS222QV_port_initialize
(void)
{
    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Back-light ON
    // Set GPIO direction to output
    Set_PinFunction(g_pGPIOReg, BACKLIGHT);
    Set_PinPullUD(g_pGPIOReg, BACKLIGHT, sgip_PULL_DISABLE);

    Set_PinFunction(g_pGPIOReg, MLCD_RST);
    Set_PinPullUD(g_pGPIOReg, MLCD_RST, sgip_PULL_DISABLE);

    Set_PinFunction(g_pGPIOReg, DISPLAY_CS);
    Set_PinPullUD(g_pGPIOReg, DISPLAY_CS, sgip_PULL_DISABLE);

    Set_PinFunction(g_pGPIOReg, DISPLAY_CLK);
    Set_PinPullUD(g_pGPIOReg, DISPLAY_CLK, sgip_PULL_DISABLE);

    Set_PinFunction(g_pGPIOReg, DISPLAY_SI);
    Set_PinPullUD(g_pGPIOReg, DISPLAY_SI, sgip_PULL_DISABLE);

    LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));
}

static LDI_ERROR
LDI_LTS222QV_power_on
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    Set_PinData(g_pGPIOReg, BACKLIGHT, sgip_Output_1);

    Set_PinData(g_pGPIOReg, DISPLAY_CLK, sgip_Output_1);
    Set_PinData(g_pGPIOReg, DISPLAY_SI, sgip_Output_1);

    //g_pDispConReg->SIFCCON0 = 0x01;    // RS:LO nCS:HI nOE:HI nWE:HI, Manual

    // Envid Disable
    g_pDispConReg->VIDCON0 &= ~0x3;    //~(ENVID_ENABLE | ENVID_F_ENABLE);    // Direct Off

    // Reset high
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_1);
    DelayLoop_1ms(10);

    // Reset low
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_0);
    DelayLoop_1ms(10);

    // Reset high
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_1);
    DelayLoop_1ms(10);

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_LTS222QV_power_off
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    Set_PinData(g_pGPIOReg, DISPLAY_CLK, sgip_Output_1);
    Set_PinData(g_pGPIOReg, DISPLAY_SI, sgip_Output_1);

    // Reset low
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_0);
    //Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    DelayLoop_1ms(40);                    // twht1 > 2 frame (33.3 ms)
    DelayLoop_1ms(20);                    // tpoff > 1 frame (16.7 ms)

    // Envid Disable
    g_pDispConReg->VIDCON0 &= ~(0x1);    //~(ENVID_F_ENABLE);    // Per Frame Off
    DelayLoop_1ms(20);                    // Wait for frame finished (16.7 ms)
    DelayLoop_1ms(10);                    // tsig0ff-vdd > 10 ms

    // LCD Pannel Power Off
    // TODO: LCD Power Off Here

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_LTS222QV_RGB_initialize
(void)
{
    LDI_ERROR error = LDI_SUCCESS;
    int i=0;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    while(1)
    {
        LDI_LTS222QV_write(LTS222QV_RGB_initialize[i][0], LTS222QV_RGB_initialize[i][1]);
        if (LTS222QV_RGB_initialize[i][2]) DelayLoop_1ms(LTS222QV_RGB_initialize[i][2]);

        i++;

        if (LTS222QV_RGB_initialize[i][0] == 0 && LTS222QV_RGB_initialize[i][1] == 0) break;
    }

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static void
LDI_AUO27_port_initialize
(void)
{
    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Back-light ON
    // Set GPIO direction to output
    Set_PinFunction(g_pGPIOReg, BACKLIGHT);
    Set_PinPullUD(g_pGPIOReg, BACKLIGHT, sgip_PULL_DISABLE);
    Set_PinData(g_pGPIOReg, BACKLIGHT, sgip_Output_1);

    LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));
}

static LDI_ERROR
LDI_AUO27_power_on
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Envid Disable
    g_pDispConReg->VIDCON0 &= ~0x3;    //~(ENVID_ENABLE | ENVID_F_ENABLE);    // Direct Off

    Set_PinFunction(g_pGPIOReg, MLCD_RST);
    Set_PinPullUD(g_pGPIOReg, MLCD_RST, sgip_PULL_DISABLE);

    // Reset high
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_1);
    DelayLoop_1ms(10);

    // Reset low
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_0);
    DelayLoop_1ms(10);

    // Reset high
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_1);
    DelayLoop_1ms(10);

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_AUO27_power_off
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    // Reset low
    Set_PinData(g_pGPIOReg, MLCD_RST, sgip_Output_0);
    //Set_PinData(g_pGPIOReg, GPH06_Output, sgip_Output_0);
    DelayLoop_1ms(40);                    // twht1 > 2 frame (33.3 ms)
    DelayLoop_1ms(20);                    // tpoff > 1 frame (16.7 ms)

    // Envid Disable
    g_pDispConReg->VIDCON0 &= ~(0x1);    //~(ENVID_F_ENABLE);    // Per Frame Off
    DelayLoop_1ms(20);                    // Wait for frame finished (16.7 ms)
    DelayLoop_1ms(10);                    // tsig0ff-vdd > 10 ms

    // LCD Pannel Power Off
    // TODO: LCD Power Off Here

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static LDI_ERROR
LDI_AUO27_RGB_initialize
(void)
{
    LDI_ERROR error = LDI_SUCCESS;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    DelayLoop_1ms(25); // more than 20ms
/*
    LDI_AUO27_write(0,    0x35);//reset
    LDI_AUO27_write(0,    0x3d);//reset recover
    LDI_AUO27_write(1,    0x0);//ups051 mode ???
    LDI_AUO27_write(9,0x0);//external vcomdc

    LDI_AUO27_write(0, 0x00);
    LDI_AUO27_write(1, 0x00);
    LDI_AUO27_write(2, 0x6c);
//    LDI_AUO27_write(2, 0xff);
    LDI_AUO27_write(3, 0x00);
    LDI_AUO27_write(4, 0x52);
    LDI_AUO27_write(5, 0x0a);
    LDI_AUO27_write(6, 0x38);
    LDI_AUO27_write(8, 0x07);
    LDI_AUO27_write(9, 0xd);
*/

/* from CAM (Original)
    LDI_AUO27_write(0,    0x05);//reset
    LDI_AUO27_write(0,    0x0d);//reset recover
    LDI_AUO27_write(1,    0x00);//ups051 mode ???
//    LDI_AUO27_write(9,0x4b);//internal vcomdc
    LDI_AUO27_write(9,0xb);//external vcomdc

    LDI_AUO27_write(0, 0x00);
    LDI_AUO27_write(1, 0x00);
    LDI_AUO27_write(2, 0x6c);
    LDI_AUO27_write(3, 0x00);
    LDI_AUO27_write(4, 0x52);
    LDI_AUO27_write(5, 0x0a);
    LDI_AUO27_write(6, 0x38);
    LDI_AUO27_write(8, 0x07);
    LDI_AUO27_write(9, 0xd);
*/
/*// HYungYeon
    LDI_AUO27_write(0,0x35);//reset
    LDI_AUO27_write(0,0x3d);//reset recover
    LDI_AUO27_write(1 , 0x11);//ups051 mode ???
    LDI_AUO27_write(4 , 0x57);

    //LDI_AUO27_write(0x9 , 0x4b);//internal vcomdc
    LDI_AUO27_write(9 , 0x0);//external vcomdc

    LDI_AUO27_write(2 , 0x46);
    LDI_AUO27_write(3 , 0x7);
    LDI_AUO27_write(5 , 0x8);
    LDI_AUO27_write(6 , 0x40);
    LDI_AUO27_write(8 , 0x7);
    LDI_AUO27_write(11 , 0x64);
    LDI_AUO27_write(12 , 0x24);
    LDI_AUO27_write(13 , 0x24);
*/

// Woojin
    LDI_AUO27_write(0,    0x5);//reset
    LDI_AUO27_write(0,    0xd);//reset recover
    LDI_AUO27_write(1,    0x0);//ups051 mode 27MHz
//    LDI_AUO27_write(9,0x4b);//internal vcomdc
    LDI_AUO27_write(9,0x0);//external vcomdc

//    LDI_AUO27_write(0, 0x00);
//    LDI_AUO27_write(1, 0x00);
    LDI_AUO27_write(2, 0x6c);
    LDI_AUO27_write(3, 0x00);
//    LDI_AUO27_write(4, 0x57);
    LDI_AUO27_write(4, 0x52);
    LDI_AUO27_write(5, 0x0a);
    LDI_AUO27_write(6, 0x38);
    LDI_AUO27_write(8, 0x07);
//    LDI_AUO27_write(9, 0xd);


    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

#define LCD_DEN_Lo        Set_PinData(g_pGPIOReg, DISPLAY_CS, sgip_Output_0)
#define LCD_DEN_Hi        Set_PinData(g_pGPIOReg, DISPLAY_CS, sgip_Output_1)
#define LCD_DCLK_Lo        Set_PinData(g_pGPIOReg, DISPLAY_CLK, sgip_Output_0)
#define LCD_DCLK_Hi        Set_PinData(g_pGPIOReg, DISPLAY_CLK, sgip_Output_1)
#define LCD_DSERI_Lo    Set_PinData(g_pGPIOReg, DISPLAY_SI, sgip_Output_0)
#define LCD_DSERI_Hi    Set_PinData(g_pGPIOReg, DISPLAY_SI, sgip_Output_1)

static LDI_ERROR Write_LDI_TL2796(UINT16 uiAddr, UINT16 uiData)
{
    LDI_ERROR error = LDI_SUCCESS;
    int j;

    LDI_MSG((_T("[LDI]++%s()\n\r"), _T(__FUNCTION__)));

    if(uiAddr != 0x0000)
    {
        LCD_DEN_Hi;
        LCD_DCLK_Hi;
        LCD_DSERI_Hi;

        LCD_DEN_Lo;
        LCD_DCLK_Lo;

        // Command
        for (j = 15 ; j >= 0; j--)
        {
            LCD_DCLK_Lo;

            if ((uiAddr >> j) & 0x0001)
            {
                LCD_DSERI_Hi;
            }
            else
            {
                LCD_DSERI_Lo;
            }

            LCD_DCLK_Hi;
        }
    }

    if(uiAddr == 0x0000)
    {
        LCD_DSERI_Hi;
    }

    LCD_DEN_Hi;
    LCD_DCLK_Hi;
    LCD_DEN_Lo;
    LCD_DCLK_Lo;

    // parameter
    for (j = 15 ; j >= 0; j--)
    {
        LCD_DCLK_Lo;

        if ((uiData >> j) & 0x0001)
        {
            LCD_DSERI_Hi;
        }
        else
        {
            LCD_DSERI_Lo;
        }

        LCD_DCLK_Hi;
    }

    LDI_MSG((_T("[LDI]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static void LDI_LTS222QV_write(unsigned int address, unsigned int data)
{
    int j;
    //LDI_MSG((_T("[LDI]++%s(0x%08x, 0x%08x)\n\r"), _T(__FUNCTION__), address, data));
#if 1
    //g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    //g_pDispConReg->SIFCCON0 = 0x13;    // RS:LO nCS:LO nOE:HI nWE:LO, Manual
    //DelayLoop(SPI_DELAY);

    LCD_DEN_Hi;
    LCD_DCLK_Hi;
    DelayLoop(SPI_DELAY);

    LCD_DEN_Lo;
    DelayLoop(SPI_DELAY);

    for (j = 7; j >= 0; j--)
    {
        LCD_DCLK_Lo;

        if ((address >> j) & 0x0001)    // DATA HIGH or LOW
        {
            LCD_DSERI_Hi;
        }
        else
        {
            LCD_DSERI_Lo;
        }

        DelayLoop(SPI_DELAY);

        LCD_DCLK_Hi;            // CLOCK = High
        DelayLoop(SPI_DELAY);
    }

    LCD_DSERI_Lo;
    DelayLoop(SPI_DELAY);

    //g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    //g_pDispConReg->SIFCCON0 = 0x01;    // RS:LO nCS:HI nOE:HI nWE:HI, Manual
    //DelayLoop(SPI_DELAY);

    //g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    //g_pDispConReg->SIFCCON0 = 0x13;    // RS:LO nCS:LO nOE:HI nWE:LO, Manual
    //DelayLoop(SPI_DELAY);

    LCD_DEN_Hi;
    LCD_DCLK_Hi;
    DelayLoop(SPI_DELAY);

    LCD_DEN_Lo;
    DelayLoop(SPI_DELAY);

    for (j = 7; j >= 0; j--)
    {
        LCD_DCLK_Lo;                            //    SCL Low

        if ((data >> j) & 0x0001)    // DATA HIGH or LOW
        {
            LCD_DSERI_Hi;
        }
        else
        {
            LCD_DSERI_Lo;
        }

        DelayLoop(SPI_DELAY);

        LCD_DCLK_Hi;            // CLOCK = High
        DelayLoop(SPI_DELAY);
    }

    //g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    //g_pDispConReg->SIFCCON0 = 0x01;    // RS:LO nCS:HI nOE:HI nWE:HI, Manual
    //DelayLoop(SPI_DELAY);

    LCD_DSERI_Lo;
    DelayLoop(SPI_DELAY);
#else
    LCD_DCLK_Hi;
    LCD_DSERI_Lo;
    DelayLoop(SPI_DELAY);

    g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    g_pDispConReg->SIFCCON0 = 0x13;    // RS:LO nCS:LO nOE:HI nWE:LO, Manual
    DelayLoop(SPI_DELAY);

    for (j = 7; j >= 0; j--)
    {
        LCD_DCLK_Lo;

        if ((address >> j) & 0x0001)    // DATA HIGH or LOW
        {
            LCD_DSERI_Hi;
        }
        else
        {
            LCD_DSERI_Lo;
        }

        DelayLoop(SPI_DELAY);

        LCD_DCLK_Hi;            // CLOCK = High
        DelayLoop(SPI_DELAY);
    }

    LCD_DSERI_Lo;
    DelayLoop(SPI_DELAY);

    g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    g_pDispConReg->SIFCCON0 = 0x01;    // RS:LO nCS:HI nOE:HI nWE:HI, Manual
    DelayLoop(SPI_DELAY);

    g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    g_pDispConReg->SIFCCON0 = 0x13;    // RS:LO nCS:LO nOE:HI nWE:LO, Manual
    DelayLoop(SPI_DELAY);

    for (j = 7; j >= 0; j--)
    {
        LCD_DCLK_Lo;                            //    SCL Low

        if ((data >> j) & 0x0001)    // DATA HIGH or LOW
        {
            LCD_DSERI_Hi;
        }
        else
        {
            LCD_DSERI_Lo;
        }

        DelayLoop(SPI_DELAY);

        LCD_DCLK_Hi;            // CLOCK = High
        DelayLoop(SPI_DELAY);
    }

    g_pDispConReg->SIFCCON0 = 0x11;    // RS:LO nCS:LO nOE:HI nWE:HI, Manual
    g_pDispConReg->SIFCCON0 = 0x01;    // RS:LO nCS:HI nOE:HI nWE:HI, Manual
    DelayLoop(SPI_DELAY);

    LCD_DSERI_Lo;
    DelayLoop(SPI_DELAY);
#endif
    //LDI_MSG((_T("[LDI]--%s()\n\r"), _T(__FUNCTION__)));
}

static void
LDI_AUO27_write
(unsigned int address,
 unsigned int data)
{
    unsigned int dv = 1;
    unsigned int i;

    address<<=12;
    address&=0xf000;

    data&=0x7ff;

    data|=address;

    LCD_DEN_Hi;
    LCD_DCLK_Hi;
    DelayLoop(SPI_DELAY);

    LCD_DEN_Lo;
    DelayLoop(SPI_DELAY);

    for( i = 0 ; i < 16 ; i++)
    {
        LCD_DCLK_Lo;
        DelayLoop(SPI_DELAY);

        if(data&(1<<(15-i)))
        {
            LCD_DSERI_Hi;
        }
        else
        {
            LCD_DSERI_Lo;
        }

        DelayLoop(SPI_DELAY);
        LCD_DCLK_Hi;
        DelayLoop(SPI_DELAY);
    }

    DelayLoop(SPI_DELAY);
    LCD_DEN_Hi;
    DelayLoop(SPI_DELAY);
}


static void DelayLoop_1ms(int msec)
{
    volatile int j;
    for(j = 0; j < LCD_DELAY_1MS*msec; j++)  ;
}

static void DelayLoop(int delay)
{
    volatile int j;
    for(j = 0; j < delay; j++)  ;
}

