//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
// Copyright (c) Samsung Electronics. Co. LTD. All rights reserved.

/*++

THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:

    halsurf.cpp

Abstract:

    The Implementation HAL function to support DirectDraw

Functions:

    HalCreateSurface, ...

Notes:

--*/

#include "precomp.h"

DWORD
WINAPI
HalCreateSurface(LPDDHAL_CREATESURFACEDATA lpcsd)
{
    DEBUGENTER( HalCreateSurface );
    /*
    typedef struct _DDHAL_CREATESURFACEDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDSURFACEDESC lpDDSurfaceDesc;
        LPDDRAWI_DDRAWSURFACE_LCL lplpSList;
        DWORD dwSCnt;
        HRESULT ddRVal;
    } DDHAL_CREATESURFACEDATA;
    */

    DWORD dwCaps = lpcsd->lpDDSurfaceDesc->ddsCaps.dwCaps;
    DWORD dwFlags = lpcsd->lpDDSurfaceDesc->dwFlags;

    // Handle Overlay Surface
    if (dwCaps & DDSCAPS_OVERLAY)
    {
        EGPEFormat format;
        EDDGPEPixelFormat pixelFormat;
        SMDKDisp *pDDGPE = (SMDKDisp *)GetDDGPE();

        lpcsd->ddRVal = pDDGPE->DetectPixelFormat(dwCaps, &lpcsd->lpDDSurfaceDesc->ddpfPixelFormat, &format, &pixelFormat);
        RETAIL_DISP_MSG(FALSE,(TEXT("CreateSurface:%d\n"),lpcsd->ddRVal));
        if (FAILED(lpcsd->ddRVal))
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalCreateSurface() : Unsupported format\n\r")));

            return DDHAL_DRIVER_HANDLED;
        }

        // Create Overlay Surface with DDGPECreateSurface() function
        return DDGPECreateSurface(lpcsd);
    }
    else
    {
        // Pass to Non-overlay surface to DDGPECreateSurface() function
        return DDGPECreateSurface(lpcsd);
    }
}

DWORD WINAPI HalCanCreateSurface(LPDDHAL_CANCREATESURFACEDATA lpccsd)
{
    DEBUGENTER( HalCanCreateSurface );
    /*
    typedef struct _DDHAL_CANCREATESURFACEDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDSURFACEDESC lpDDSurfaceDesc;
        DWORD bIsDifferentPixelFormat;
        HRESULT ddRVal;
    } DDHAL_CANCREATESURFACEDATA;

    */

    DDPIXELFORMAT *pddpf = &lpccsd->lpDDSurfaceDesc->ddpfPixelFormat;
    DWORD dwCaps = lpccsd->lpDDSurfaceDesc->ddsCaps.dwCaps;
    DWORD dwWidth = lpccsd->lpDDSurfaceDesc->dwWidth;
    DWORD dwHeight = lpccsd->lpDDSurfaceDesc->dwHeight;

    RETAIL_DISP_MSG(DISP_ZONE_TEMP, (_T("HalCanCreateSurface. dwCaps : 0x%x, dwWidth:%d, dwHeight:%d\n"), dwCaps, dwWidth, dwHeight));

    SMDKDisp *pDDGPE = (SMDKDisp *)GetDDGPE();

    // We do Not allow Primary Surface in System Memory
    if ((dwCaps & DDSCAPS_PRIMARYSURFACE) && (dwCaps & DDSCAPS_SYSTEMMEMORY))
    {
        RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalCanCreateSurface() : Can Not create Primary Surface in System Memory\n\r")));
        goto CannotCreate;
    }

    if ((dwCaps & DDSCAPS_OVERLAY) && (dwCaps & DDSCAPS_SYSTEMMEMORY))
    {
        RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalCanCreateSurface() : Can Not create Overlay Surface in System Memory\n\r")));
        goto CannotCreate;
    }

    if (dwCaps & DDSCAPS_PRIMARYSURFACE)
    {
        RETAIL_DISP_MSG(FALSE, (_T("DiffPixel:%d\n"), lpccsd->bIsDifferentPixelFormat));
        if (lpccsd->bIsDifferentPixelFormat)
        {
            goto CannotCreate;
        }
        else
        {
            goto CanCreate;
        }
    }
    else if (dwCaps & DDSCAPS_OVERLAY)
    {
        EGPEFormat Format;
        EDDGPEPixelFormat PixelFormat;

        lpccsd->ddRVal = pDDGPE->DetectPixelFormat(dwCaps, pddpf, &Format, &PixelFormat);
        if (FAILED(lpccsd->ddRVal))
        {
            goto CannotCreate;
        }
        else
        {
            RETAIL_DISP_MSG(DISP_ZONE_TEMP, (_T("PixelFormat :%d\n"), PixelFormat));
            switch(PixelFormat)
            {
            case ddgpePixelFormat_565:
            // Considering Support
            case ddgpePixelFormat_5551: //Alpha
            case ddgpePixelFormat_5550: //NonAlpha
            //case ddgpePixelFormat_4444:

            //case ddgpePixelFormat_8880:    // FIMD can not support Packed RGB888

            case ddgpePixelFormat_8888:
            case ddgpePixelFormat_I420:
            case ddgpePixelFormat_YV12:
            case ddgpePixelFormat_YUYV:
            case ddgpePixelFormat_YUY2:
            case ddgpePixelFormat_UYVY:
            case ddgpePixelFormat_YVYU:
            case ddgpePixelFormat_VYUY:
            case ddgpePixelFormat_NV12:
                goto CanCreate;
                break;
            default:
                goto CannotCreate;
                break;
            }
        }
    }
    else        // Non Primary, Non Overlay Surface (Maybe Offscreen Surface)
    {
        return DDGPECanCreateSurface(lpccsd);
    }

CanCreate:

    RETAIL_DISP_MSG(DISP_ZONE_ENTER,(_T("[DDHAL] HalCanCreateSurface() OK\n\r")));
    lpccsd->ddRVal = DD_OK;

    return DDHAL_DRIVER_HANDLED;

CannotCreate:

    RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalCanCreateSurface() : Unsupported Surface\n\r")));
    lpccsd->ddRVal = DDERR_UNSUPPORTEDFORMAT;

    return DDHAL_DRIVER_HANDLED;
}

//////////////////////////// DDHAL_DDSURFACECALLBACKS ////////////////////////////

DWORD
WINAPI
HalFlip(LPDDHAL_FLIPDATA lpfd)
{
    DEBUGENTER( HalFlip );
    /*
    typedef struct _DDHAL_FLIPDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDRAWI_DDRAWSURFACE_LCL lpSurfCurr;
        LPDDRAWI_DDRAWSURFACE_LCL lpSurfTarg;
        DWORD dwFlags;
        HRESULT ddRVal;
    } DDHAL_FLIPDATA;
    */

    SMDKDisp *pDDGPE = (SMDKDisp *)GetDDGPE();
    DWORD dwFlags = lpfd->dwFlags;

    if (dwFlags & (DDFLIP_INTERVAL1|DDFLIP_INTERVAL2|DDFLIP_INTERVAL4))
    {
        RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalFlip() : DDFLIP_INTERVAL is not supported\n\r")));
        lpfd->ddRVal = DDERR_UNSUPPORTED;
    }
    else
    {
        if (dwFlags & DDFLIP_WAITNOTBUSY)
        {
            // Our H/W always not busy.. This will be skipped
            while(((SMDKDisp *)GetDDGPE())->IsBusy());
        }

        DDGPESurf* surfTarg = DDGPESurf::GetDDGPESurf(lpfd->lpSurfTarg);

        if (dwFlags & DDFLIP_WAITVSYNC)
        {
            pDDGPE->SetVisibleSurface(surfTarg, TRUE);
        }
        else
        {
            pDDGPE->SetVisibleSurface(surfTarg, FALSE);
        }

        lpfd->ddRVal = DD_OK;
    }

    DEBUGLEAVE( HalFlip );

    return DDHAL_DRIVER_HANDLED;
}

DWORD WINAPI HalGetBltStatus( LPDDHAL_GETBLTSTATUSDATA lpgbsd )
{
    DEBUGENTER( HalGetBltStatus );
    /*
    typedef struct _DDHAL_GETBLTSTATUSDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDRAWI_DDRAWSURFACE_LCL lpDDSurface;
        DWORD dwFlags;
        HRESULT ddRVal;
    } DDHAL_GETBLTSTATUSDATA;
    */

    // Implementation
    lpgbsd->ddRVal = DD_OK;

    return DDHAL_DRIVER_HANDLED;
}

DWORD WINAPI HalGetFlipStatus( LPDDHAL_GETFLIPSTATUSDATA lpgfsd)
{
    DEBUGENTER( HalGetFlipStatus );
    /*
    typedef struct _DDHAL_GETFLIPSTATUSDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDRAWI_DDRAWSURFACE_LCL lpDDSurface;
        DWORD dwFlags;
        HRESULT ddRVal;
    } DDHAL_GETFLIPSTATUSDATA;
    */

    lpgfsd->ddRVal = DD_OK;

    // NOTE: DDGBS_CANFLIP always return DD_OK
    // Actually second flip request in same display frame is blocked to next frame

    return DDHAL_DRIVER_HANDLED;
}

DWORD WINAPI HalUpdateOverlay(LPDDHAL_UPDATEOVERLAYDATA lpuod)
{
    DEBUGENTER( HalUpdateOverlay );
    /*
    typedef struct _DDHAL_UPDATEOVERLAYDATA {
          LPDDRAWI_DIRECTDRAW_GBL lpDD;
          LPDDRAWI_DDRAWSURFACE_LCL lpDDDestSurface;
          RECT rDest;
          LPDDRAWI_DDRAWSURFACE_LCL lpDDSrcSurface;
          RECT rSrc;
          DWORD dwFlags;
          DDOVERLAYFX overlayFX;
          HRESULT ddRVal;
    } DDHAL_UPDATEOVERLAYDATA;
    */

    SMDKDisp    *pDDGPE;
    SMDKSurf    *pSrcSurf;
    SMDKSurf    *pDestSurf;
    LPDDRAWI_DDRAWSURFACE_LCL lpSrcLCL;
    LPDDRAWI_DDRAWSURFACE_LCL lpDestLCL;

    BOOL bEnableOverlay = FALSE;

    /* 'Source' is the overlay surface, 'destination' is the surface to
    * be overlayed:
    */

    lpSrcLCL = lpuod->lpDDSrcSurface;
    lpDestLCL = lpuod->lpDDDestSurface;

    pDDGPE = (SMDKDisp *)GetDDGPE();
    pSrcSurf = (SMDKSurf *)DDGPESurf::GetDDGPESurf(lpSrcLCL);
    pDestSurf = (SMDKSurf *)DDGPESurf::GetDDGPESurf(lpDestLCL);

    if (lpuod->dwFlags & DDOVER_HIDE)
    {
        // If overlay surface is valid, Turn off overlay
        if (pSrcSurf->OffsetInVideoMemory() != NULL)
        {
            if ( (pSrcSurf == pDDGPE->GetCurrentOverlaySurf())
                || (pSrcSurf == pDDGPE->GetPreviousOverlaySurf()) )
            {
                pDDGPE->OverlayDisable();
            }

            lpuod->ddRVal = DD_OK;
        }
        else
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : pSrcSurf->OffsetInVideoMemory() = NULL\n\r")));
            lpuod->ddRVal = DDERR_INVALIDPARAMS;
        }

        return (DDHAL_DRIVER_HANDLED);
    }
    else if (lpuod->dwFlags & DDOVER_SHOW)
    {
        if (pSrcSurf->OffsetInVideoMemory() != NULL)
        {
            if ( (pSrcSurf != pDDGPE->GetCurrentOverlaySurf())
                && (pSrcSurf != pDDGPE->GetPreviousOverlaySurf())
                && (pDDGPE->GetCurrentOverlaySurf() != NULL))
            {
                // Some other overlay surface is already visible:
                RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : Overlay is already in use by another surface\n\r")));

                lpuod->ddRVal = DDERR_OUTOFCAPS;

                return (DDHAL_DRIVER_HANDLED);
            }
            else
            {
                // Initialize Overlay
                if (pDDGPE->OverlayInitialize(pSrcSurf, &lpuod->rSrc, &lpuod->rDest) == FALSE)
                {
                    RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : OverlayInitialize() Failed\n\r")));

                    lpuod->ddRVal = DDERR_OUTOFCAPS;

                    return (DDHAL_DRIVER_HANDLED);
                }

                // Enable Overlay below... after set up blending
                bEnableOverlay = TRUE;
            }
        }
        else
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : pSrcSurf->OffsetInVideoMemory() = NULL\n\r")));

            lpuod->ddRVal = DDERR_INVALIDPARAMS;
            return (DDHAL_DRIVER_HANDLED);
        }
    }
    else
    {
        // If overlay surface is not visiable,  Nothing to do
        lpuod->ddRVal = DD_OK;

        return (DDHAL_DRIVER_HANDLED);
    }

    if ((lpuod->dwFlags & (DDOVER_KEYSRC|DDOVER_KEYSRCOVERRIDE|DDOVER_KEYDEST|DDOVER_KEYDESTOVERRIDE))
        && (lpuod->dwFlags & (DDOVER_ALPHASRC|DDOVER_ALPHADEST)))
    {
        RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : Driver does not support Color Keying & Source/Destination Alpha Blending simultaneously (dwFlags = 0x%08x)\n\r"), lpuod->dwFlags));
        lpuod->ddRVal = DDERR_UNSUPPORTED;
        return (DDHAL_DRIVER_HANDLED);
    }

    // Source Color Key with Constant Alpha Blending
    if ((lpuod->dwFlags & (DDOVER_KEYSRC|DDOVER_KEYSRCOVERRIDE))
                && (lpuod->dwFlags & DDOVER_ALPHACONSTOVERRIDE))
    {
        DWORD dwColorKey;

        if (lpuod->dwFlags & DDOVER_KEYSRCOVERRIDE)
        {
            dwColorKey = lpuod->overlayFX.dckSrcColorkey.dwColorSpaceLowValue;
        }
        else
        {
            dwColorKey = lpSrcLCL->ddckCKSrcOverlay.dwColorSpaceLowValue;
        }

        pDDGPE->OverlaySetColorKeyWithAlpha(TRUE, pSrcSurf->PixelFormat(), dwColorKey, lpuod->overlayFX.dwAlphaConst);
    }
    // Destination Color Key with Constant Alpha Blending
    else if ((lpuod->dwFlags & (DDOVER_KEYDEST|DDOVER_KEYDESTOVERRIDE))
                && (lpuod->dwFlags & DDOVER_ALPHACONSTOVERRIDE))
    {
        DWORD dwColorKey;

        if (lpuod->dwFlags & DDOVER_KEYDESTOVERRIDE)
        {
            dwColorKey = lpuod->overlayFX.dckDestColorkey.dwColorSpaceLowValue;
        }
        else
        {
            dwColorKey = lpDestLCL->ddckCKDestOverlay.dwColorSpaceLowValue;
        }

        pDDGPE->OverlaySetColorKeyWithAlpha(FALSE, pSrcSurf->PixelFormat(), dwColorKey, lpuod->overlayFX.dwAlphaConst);
    }
    // Source Color Key
    else if (lpuod->dwFlags & (DDOVER_KEYSRCOVERRIDE | DDOVER_KEYSRC))
    {
        DWORD dwColorKey;

        if (lpuod->dwFlags & DDOVER_KEYSRCOVERRIDE)
        {
            dwColorKey = lpuod->overlayFX.dckSrcColorkey.dwColorSpaceLowValue;
        }
        else
        {
            dwColorKey = lpSrcLCL->ddckCKSrcOverlay.dwColorSpaceLowValue;
        }

        pDDGPE->OverlaySetColorKey(TRUE, pSrcSurf->PixelFormat(), dwColorKey);
    }
    // Destination Color Key
    else if (lpuod->dwFlags & (DDOVER_KEYDESTOVERRIDE | DDOVER_KEYDEST))
    {
        DWORD dwColorKey;

        if (lpuod->dwFlags & DDOVER_KEYDESTOVERRIDE)
        {
            dwColorKey = lpuod->overlayFX.dckDestColorkey.dwColorSpaceLowValue;
        }
        else
        {
            dwColorKey = lpDestLCL->ddckCKDestOverlay.dwColorSpaceLowValue;
        }

        pDDGPE->OverlaySetColorKey(FALSE, pDestSurf->PixelFormat(), dwColorKey);
    }
    // Source Alpha Blending
    // Alpha information is obtained from the pixels of the source surface
    // The pixel format currently supported is given below:
    // i) Unpacked 32 bpp RGB(ARGB8888)
    else if (lpuod->dwFlags & DDOVER_ALPHASRC)
    {
        if(pSrcSurf->PixelFormat() != ddgpePixelFormat_8888)
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : Source alpha blending is not supported for this pixel format\r\n")));
            lpuod->ddRVal = DDERR_UNSUPPORTED;
            return (DDHAL_DRIVER_HANDLED);
        }

        pDDGPE->OverlaySetAlphaPixel(TRUE);
    }
    // Destination Alpha Blending
    // Alpha information is obtained from the pixels of the destination surface
    // The pixel format currently supported is given below:
    // i) Unpacked 32 bpp RGB(ARGB8888)
    else if (lpuod->dwFlags & DDOVER_ALPHADEST)
    {
        if(pSrcSurf->PixelFormat() != ddgpePixelFormat_8888)
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalUpdateOverlay() : Destination alpha blending is not supported for this pixel format\r\n")));
            lpuod->ddRVal = DDERR_UNSUPPORTED;
            return (DDHAL_DRIVER_HANDLED);
        }

        pDDGPE->OverlaySetAlphaPixel(FALSE);
    }
    // Per Plane Alpha Blending
    // Alpha information is obtained from the DDOVERLAYFX structure
    else if (lpuod->dwFlags & DDOVER_ALPHACONSTOVERRIDE)
    {
        pDDGPE->OverlaySetAlphaPlane(lpuod->overlayFX.dwAlphaConst);
    }
    // No Blending Effect
    else
    {
        pDDGPE->OverlayBlendDisable();
    }

    // Enable Overlay after set up blending
    if (bEnableOverlay) pDDGPE->OverlayEnable();

    lpuod->ddRVal = DD_OK;

    return (DDHAL_DRIVER_HANDLED);
}

DWORD WINAPI HalSetOverlayPosition(LPDDHAL_SETOVERLAYPOSITIONDATA lpsopd)
{
    DEBUGENTER( HalSetOverlayPosition );
    /*
    typedef struct _DDHAL_SETOVERLAYPOSITIONDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDRAWI_DDRAWSURFACE_LCL lpDDSrcSurface;
        LPDDRAWI_DDRAWSURFACE_LCL lpDDDestSurface;
        LONG lXPos;
        LONG lYPos;
        HRESULT ddRVal;
    } DDHAL_SETOVERLAYPOSITIONDATA;
    */

    SMDKDisp    *pDDGPE;
//    SMDKSurf *pSrcSurf;

    pDDGPE = (SMDKDisp *)GetDDGPE();
//    pSrcSurf = (SMDKSurf *)DDGPESurf::GetDDGPESurf(lpsopd->lpDDSrcSurface);

    pDDGPE->OverlaySetPosition((unsigned int)lpsopd->lXPos, (unsigned int)lpsopd->lYPos);
    lpsopd->ddRVal = DD_OK;

    return DDHAL_DRIVER_HANDLED;
}

DWORD WINAPI HalSetColorKey(LPDDHAL_SETCOLORKEYDATA lpdsckd)
{
    DEBUGENTER(HalSetColorKey);
    /*
    typedef struct _DDHAL_SETCOLORKEYDATA
    {
        LPDDRAWI_DIRECTDRAW_GBL lpDD;
        LPDDRAWI_DDRAWSURFACE_LCL lpDDSurface;
        DWORD dwFlags;
        DDCOLORKEY ckNew;
        HRESULT ddRVal;
    } DDHAL_SETCOLORKEYDATA;
    */

    DDGPESurf* pSurf = DDGPESurf::GetDDGPESurf(lpdsckd->lpDDSurface);
    if (pSurf != NULL)
    {
        if (lpdsckd->dwFlags == DDCKEY_COLORSPACE)
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalSetColorKey() : Color Space CKEY is Not Supported\n\r")));
            lpdsckd->ddRVal = DDERR_NOCOLORKEYHW;
        }
        else if ((lpdsckd->dwFlags == DDCKEY_SRCBLT)
            || (lpdsckd->dwFlags == DDCKEY_DESTBLT))
        {
            // NOTE: Currently our driver do not support HW CKEY BLT but CETK DDraw Test #210, #310 trying to use
            RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalSetColorKey() : Color Space CKEY or CKEY BLT is Not Supported\n\r")));

            pSurf->SetColorKeyLow(lpdsckd->ckNew.dwColorSpaceLowValue);
            pSurf->SetColorKeyHigh(lpdsckd->ckNew.dwColorSpaceHighValue);
            lpdsckd->ddRVal = DD_OK;
        }
        else     if ((lpdsckd->dwFlags == DDCKEY_SRCOVERLAY)
            || (lpdsckd->dwFlags == DDCKEY_DESTOVERLAY))
        {
            pSurf->SetColorKeyLow(lpdsckd->ckNew.dwColorSpaceLowValue);
            pSurf->SetColorKeyHigh(lpdsckd->ckNew.dwColorSpaceHighValue);
            lpdsckd->ddRVal = DD_OK;
        }
        else
        {
            RETAIL_DISP_MSG(DISP_ZONE_ERROR,(_T("[DDHAL:ERR] HalSetColorKey() : Invalid dwFlags = 0x%08x\n\r"), lpdsckd->dwFlags));
            lpdsckd->ddRVal = DDERR_INVALIDOBJECT;
        }
    }
    else
    {
        RETAIL_DISP_MSG(DISP_ZONE_ERROR, (_T("[DDHAL:ERR] HalSetColorKey() : Surface Object is Null\n\r")));
        lpdsckd->ddRVal = DDERR_INVALIDOBJECT;
    }

    DEBUGLEAVE(HalSetColorKey);

    return DDHAL_DRIVER_HANDLED;
}
