//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this sample source code is subject to the terms of the Microsoft
// license agreement under which you licensed this sample source code. If
// you did not accept the terms of the license agreement, you are not
// authorized to use this sample source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the LICENSE.RTF on your install media or the root of your tools installation.
// THE SAMPLE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES OR INDEMNITIES.
//
//
// Copyright (c) Samsung Electronics. Co. LTD.  All rights reserved.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    display_con.c

Abstract:       Implementation of Display Controller Control Library
                This module implements Low Level HW control

Functions:


Notes:


--*/

#include <windows.h>
#include <bsp.h>    // for reference HCLK, ECLK
#include <SVEDriverAPI.h>   // This contains Shared type definitions
#include <display_con.h>
#include <display_con_macro.h>
#include <SVE_API.h>       // This contains Private type definitions

//#define DISP_MSG(x)     DBGMSG(DISP_USR1, x)
//#define DISP_INF(x)     DBGMSG(DISP_USR1, x)
#define DISP_MSG(x)     DBGMSG(0, x)
#define DISP_INF(x)     DBGMSG(0, x)
#define DISP_ERR(x)     DBGMSG(DISP_INFO, x)

static volatile DISPLAY_REG *g_pDispConReg = NULL;
static volatile GPIO_REG *g_pGPIOReg = NULL;
static volatile CMU_CLK_REG *g_pCMUCLKReg = NULL;
static volatile BSP_ARGS    *g_pBSPArg = NULL;
static volatile CMU_MISC_REG    *g_pCMUMISCReg = NULL;

static tDevInfo g_DevInfoDisp;
static tDispWindow0Config g_Win0Config;
static tDispWindow12Config g_Win1Config;
static tDispWindow12Config g_Win2Config;
static tDispWindow34Config g_Win3Config;
static tDispWindow34Config g_Win4Config;

static DISP_ERROR Disp_initialize_port_RGBIF(DISP_RGBIFOUT_MODE RGBOutMode);
static DISP_ERROR Disp_initialize_RGBIF(void);
static DISP_ERROR Disp_window0_initialize(DISP_WINDOW_MODE Mode,
                                            DISP_BPP_MODE BPPMode,
                                            unsigned int uiWidth,
                                            unsigned int uiHeight,
                                            unsigned int uiOffsetX,
                                            unsigned int uiOffsetY);
static DISP_ERROR Disp_window1_initialize(DISP_WINDOW_MODE Mode,
                                            DISP_BPP_MODE BPPMode,
                                            unsigned int uiWidth,
                                            unsigned int uiHeight,
                                            unsigned int uiOffsetX,
                                            unsigned int uiOffsetY);
static DISP_ERROR Disp_window2_initialize(DISP_WINDOW_MODE Mode,
                                            DISP_BPP_MODE BPPMode,
                                            unsigned int uiWidth,
                                            unsigned int uiHeight,
                                            unsigned int uiOffsetX,
                                            unsigned int uiOffsetY);
static DISP_ERROR Disp_window3_initialize(DISP_WINDOW_MODE Mode,
                                            DISP_BPP_MODE BPPMode,
                                            unsigned int uiWidth,
                                            unsigned int uiHeight,
                                            unsigned int uiOffsetX,
                                            unsigned int uiOffsetY);
static DISP_ERROR Disp_window4_initialize(DISP_WINDOW_MODE Mode,
                                            DISP_BPP_MODE BPPMode,
                                            unsigned int uiWidth,
                                            unsigned int uiHeight,
                                            unsigned int uiOffsetX,
                                            unsigned int uiOffsetY);
static BOOL Disp_get_vclk_direction_divider(unsigned int CLKSrc,
                                            unsigned int *ClkDir,
                                            unsigned int *ClkDiv);
static DISP_ERROR Disp_window_registers_lock(DISP_WINDOW Win);
static DISP_ERROR Disp_window_registers_unlock(DISP_WINDOW Win);
static BOOL Disp_get_src_clock(unsigned int CLKSrc, unsigned int *pdwVCLKSrc);
static void Disp_set_syscon_clk_fimd(void);

DISP_ERROR
Disp_initialize_register_address
(void *pDispConReg,
 void *pGPIOReg,
 void *pCMUCLKReg,
 void *pBSPArgs,
 void *pCMUMISCReg)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++%s(0x%08x, 0x%08x, 0x%08x, 0x%08x, 0x%08x)\n\r"), _T(__FUNCTION__), pDispConReg, pGPIOReg, pCMUCLKReg,pBSPArgs,pCMUMISCReg));

    if (pDispConReg == NULL || pGPIOReg == NULL || pCMUCLKReg == NULL || \
        pBSPArgs == NULL || pCMUMISCReg == NULL)
    {
        DISP_ERR((_T("[DISP:ERR] Disp_initialize_register_address() : NULL pointer parameter\n\r")));
        error = DISP_ERROR_NULL_PARAMETER;
    }
    else
    {
        g_pDispConReg = (DISPLAY_REG *)pDispConReg;
        g_pGPIOReg = (GPIO_REG *)pGPIOReg;
        g_pCMUCLKReg = (CMU_CLK_REG*)pCMUCLKReg;
        g_pBSPArg = (BSP_ARGS *)pBSPArgs;
        g_pCMUMISCReg = (CMU_MISC_REG *)pCMUMISCReg;
        DISP_INF((_T("[DISP:INF] g_pDispConReg = 0x%08x\n\r"), g_pDispConReg));
        DISP_INF((_T("[DISP:INF] g_pGPIOReg    = 0x%08x\n\r"), g_pGPIOReg));
        DISP_INF((_T("[DISP:INF] g_pCMUCLKReg  = 0x%08x\n\r"), g_pCMUCLKReg));
        DISP_INF((_T("[DISP:INF] g_pBSPArg     = 0x%08x\n\r"), g_pBSPArg));
        DISP_INF((_T("[DISP:INF] g_pCMUMISCReg     = 0x%08x\n\r"), g_pCMUMISCReg));
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

DISP_ERROR Disp_set_output_device_information(void *pInfo)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_output_device_information(%0x08x)\n\r"), pInfo));

    if (pInfo == NULL)
    {
        DISP_ERR((_T("[DISP:ERR] Disp_set_output_device_information() : NULL pointer parameter\n\r")));
        error = DISP_ERROR_NULL_PARAMETER;
    }
    else
    {
        memcpy(&g_DevInfoDisp, pInfo, sizeof(tDevInfo));
    }

    DISP_MSG((_T("[DISP]--Disp_set_output_device_information()\n\r")));

    return error;
}

#if    0    // Depricated
void* Disp_get_output_device_information_buffer(void)
{
    DISP_MSG((_T("[DISP]++Disp_get_output_device_information_buffer()\n\r")));

    // Device Information is Filled by LDI_fill_output_device_information()
    return (void *)&g_DevInfoDisp;
}
#endif

DISP_ERROR Disp_initialize_output_interface(DISP_VIDOUT_MODE VidoutMode)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_initialize_output_interface(%d)\n\r"), VidoutMode));

    if (g_pDispConReg == NULL || g_pGPIOReg == NULL)
    {
        DISP_ERR((_T("[DISP:ERR] Disp_initialize_output_interface() : Register Address Not Initialized\n\r")));
        error = DISP_ERROR_NOT_INITIALIZED;
    }

    g_DevInfoDisp.VideoOutMode = VidoutMode; // need to think whether this line will be deleted or not

    switch(VidoutMode)
    {
    case DISP_VIDOUT_RGBIF:
        if(g_DevInfoDisp.RGBINFO.MIPIDSIen == MIPI_DSI_DIS)
        {
#ifndef USE_DSI
            Disp_initialize_port_RGBIF(g_DevInfoDisp.RGBINFO.RGBOutMode);
#endif
        }
        Disp_initialize_RGBIF();
        break;
    case DISP_VIDOUT_ITU6xx:
    case DISP_VIDOUT_I80IF_LDI0:
    case DISP_VIDOUT_I80IF_LDI1:
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_initialize_output_interface() : Not Implemented Video Output Mode [%d]\n\r"), VidoutMode));
        error = DISP_ERROR_NOT_IMPLEMENTED;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_initialize_output_interface() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_set_window_mode(DISP_WINDOW_MODE Mode, DISP_BPP_MODE BPPMode, unsigned int uiWidth, unsigned int uiHeight, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_window_mode(%d)\n\r"), Mode));

    switch(Mode)
    {
    case DISP_WIN0_DMA:
    case DISP_WIN0_CAMIF0_RGB:
    case DISP_WIN0_CAMIF0_YUV:
        Disp_window0_initialize(Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY);
        break;
    case DISP_WIN1_DMA:
    case DISP_WIN1_CAMIF1_RGB:
    case DISP_WIN1_CAMIF1_YUV:
    case DISP_WIN1_VP_YUV:
        Disp_window1_initialize(Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY);
        break;
    case DISP_WIN2_DMA:
    case DISP_WIN2_CAMIF2_RGB:
    case DISP_WIN2_CAMIF2_YUV:
        Disp_window2_initialize(Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY);
        break;
    case DISP_WIN3_DMA:
        Disp_window3_initialize(Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY);
        break;
    case DISP_WIN4_DMA:
        Disp_window4_initialize(Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY);
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_window_mode() : Unsupported Window Mode [%d]\n\r"), Mode));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_set_window_mode() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_set_window_position(DISP_WINDOW Win, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_window_position(%d, %d, %d)\n\r"), Win, uiOffsetX, uiOffsetY));

    Disp_window_registers_lock(Win);

    switch(Win)
    {
    case DISP_WIN0:
        g_Win0Config.uiOffsetX = uiOffsetX;
        g_Win0Config.uiOffsetY = uiOffsetY;

        g_pDispConReg->VIDOSD0A = OSD_LEFTTOPX_F(g_Win0Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win0Config.uiOffsetY);

        g_pDispConReg->VIDOSD0B = OSD_RIGHTBOTX_F(g_Win0Config.uiWidth+g_Win0Config.uiOffsetX-1) |
                                    OSD_RIGHTBOTY_F(g_Win0Config.uiHeight+g_Win0Config.uiOffsetY-1);
        break;
    case DISP_WIN1:
        g_Win1Config.uiOffsetX = uiOffsetX;
        g_Win1Config.uiOffsetY = uiOffsetY;

        g_pDispConReg->VIDOSD1A = OSD_LEFTTOPX_F(g_Win1Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win1Config.uiOffsetY);

        g_pDispConReg->VIDOSD1B = OSD_RIGHTBOTX_F(g_Win1Config.uiWidth+g_Win1Config.uiOffsetX-1) |
                                    OSD_RIGHTBOTY_F(g_Win1Config.uiHeight+g_Win1Config.uiOffsetY-1);
        break;
    case DISP_WIN2:
        g_Win2Config.uiOffsetX = uiOffsetX;
        g_Win2Config.uiOffsetY = uiOffsetY;

        g_pDispConReg->VIDOSD2A = OSD_LEFTTOPX_F(g_Win2Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win2Config.uiOffsetY);

        g_pDispConReg->VIDOSD2B = OSD_RIGHTBOTX_F(g_Win2Config.uiWidth+g_Win2Config.uiOffsetX-1) |
                                    OSD_RIGHTBOTY_F(g_Win2Config.uiHeight+g_Win2Config.uiOffsetY-1);
        break;
    case DISP_WIN3:
        g_Win3Config.uiOffsetX = uiOffsetX;
        g_Win3Config.uiOffsetY = uiOffsetY;

        g_pDispConReg->VIDOSD3A = OSD_LEFTTOPX_F(g_Win3Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win3Config.uiOffsetY);

        g_pDispConReg->VIDOSD3B = OSD_RIGHTBOTX_F(g_Win3Config.uiWidth+g_Win3Config.uiOffsetX-1) |
                                    OSD_RIGHTBOTY_F(g_Win3Config.uiHeight+g_Win3Config.uiOffsetY-1);
        break;
    case DISP_WIN4:
        g_Win4Config.uiOffsetX = uiOffsetX;
        g_Win4Config.uiOffsetY = uiOffsetY;

        g_pDispConReg->VIDOSD4A = OSD_LEFTTOPX_F(g_Win4Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win4Config.uiOffsetY);

        g_pDispConReg->VIDOSD4B = OSD_RIGHTBOTX_F(g_Win4Config.uiWidth+g_Win4Config.uiOffsetX-1) |
                                    OSD_RIGHTBOTY_F(g_Win4Config.uiHeight+g_Win4Config.uiOffsetY-1);
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_window_position() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    Disp_window_registers_unlock(Win);

    //DISP_MSG((_T("[DISP]--Disp_set_window_position() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_set_framebuffer(DISP_WINDOW Win, unsigned int uiFrameBufferAddress)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_framebuffer(%d, 0x%08x)\n\r"), Win, uiFrameBufferAddress));

    Disp_window_registers_lock(Win);

    switch(Win)
    {
    case DISP_WIN0:
        if (g_Win0Config.LocalPathEnable == LOCAL_PATH_ENABLE)
        {
#if    0
            DISP_ERR((_T("[DISP:ERR] Disp_set_framebuffer() : Window0 Local Path is Enabled\n\r")));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
#else
            // Safe Frame buffer Address for Local Path
            g_pDispConReg->VIDW00ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
            g_pDispConReg->VIDW00ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+0x10);    // 4 Words
            g_pDispConReg->VIDW00ADD2 = PAGEWIDTH_F(0x10);                                    // 4 Words
            g_pDispConReg->WINCON0 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
#endif
        }
        else
        {
            g_pDispConReg->VIDW00ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
            g_pDispConReg->VIDW00ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+g_Win0Config.uiPageWidth*g_Win0Config.uiHeight);
            g_pDispConReg->VIDW00ADD2 = PAGEWIDTH_F(g_Win0Config.uiPageWidth);
            g_pDispConReg->WINCON0 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
        }
        break;
    case DISP_WIN1:
        if (g_Win1Config.LocalPathEnable == LOCAL_PATH_ENABLE)
        {
#if    0
            DISP_ERR((_T("[DISP:ERR] Disp_set_framebuffer() : Window1 Local Path is Enabled\n\r")));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
#else
            // Safe Frame buffer Address for Local Path
            g_pDispConReg->VIDW01ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
            g_pDispConReg->VIDW01ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+0x10);    // 4 Words
            g_pDispConReg->VIDW01ADD2 = PAGEWIDTH_F(0x10);                                    // 4 Words
            g_pDispConReg->WINCON1 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
#endif
        }
        else
        {
            g_pDispConReg->VIDW01ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
            g_pDispConReg->VIDW01ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+g_Win1Config.uiPageWidth*g_Win1Config.uiHeight);
            g_pDispConReg->VIDW01ADD2 = PAGEWIDTH_F(g_Win1Config.uiPageWidth);
            g_pDispConReg->WINCON1 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
        }
        break;
    case DISP_WIN2:
        if (g_Win2Config.LocalPathEnable == LOCAL_PATH_ENABLE)
        {
#if    0
            DISP_ERR((_T("[DISP:ERR] Disp_set_framebuffer() : Window2 Local Path is Enabled\n\r")));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
#else
            // Safe Frame buffer Address for Local Path
            g_pDispConReg->VIDW02ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
            g_pDispConReg->VIDW02ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+0x10);        // 4 Words
            g_pDispConReg->VIDW02ADD2 = PAGEWIDTH_F(0x10);
            g_pDispConReg->WINCON2 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0// 4 Words
#endif
        }
        else
        {
            g_pDispConReg->VIDW02ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
            g_pDispConReg->VIDW02ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+g_Win2Config.uiPageWidth*g_Win2Config.uiHeight);
            g_pDispConReg->VIDW02ADD2 = PAGEWIDTH_F(g_Win2Config.uiPageWidth);
            g_pDispConReg->WINCON2 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
        }
        break;
    case DISP_WIN3:
        g_pDispConReg->VIDW03ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
        g_pDispConReg->VIDW03ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+g_Win3Config.uiPageWidth*g_Win3Config.uiHeight);
        g_pDispConReg->VIDW03ADD2 = PAGEWIDTH_F(g_Win3Config.uiPageWidth);
        g_pDispConReg->WINCON3 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
        break;
    case DISP_WIN4:
        g_pDispConReg->VIDW04ADD0B0 = VBANK_F(uiFrameBufferAddress>>24) | VBASEU_F(uiFrameBufferAddress);
        g_pDispConReg->VIDW04ADD1B0 = VBASEL_F(VBASEU_F(uiFrameBufferAddress)+g_Win4Config.uiPageWidth*g_Win4Config.uiHeight);
        g_pDispConReg->VIDW04ADD2 = PAGEWIDTH_F(g_Win4Config.uiPageWidth);
        g_pDispConReg->WINCON4 &= ~(BUFSEL_BUF1|BUFSEL_BUF2);    // Buffer set to Buf0
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_framebuffer() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    Disp_window_registers_unlock(Win);

    //DISP_MSG((_T("[DISP]--Disp_set_framebuffer() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_envid_onoff(DISP_ENVID_ONOFF EnvidOnOff)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_envid_onoff(%d)\n\r"), EnvidOnOff));

    if (EnvidOnOff == DISP_ENVID_ON)
    {
        g_pDispConReg->VIDCON0 |= (ENVID_ENABLE | ENVID_F_ENABLE);
    }
    else if (EnvidOnOff == DISP_ENVID_OFF)
    {
        g_pDispConReg->VIDCON0 &= ~(ENVID_F_ENABLE);    // Per Frame Off
    }
    else if (EnvidOnOff == DISP_ENVID_DIRECT_OFF)
    {
        g_pDispConReg->VIDCON0 &= ~(ENVID_ENABLE | ENVID_F_ENABLE);    // Direct Off
    }
    else
    {
        DISP_ERR((_T("[DISP:ERR] Disp_envid_onoff() : Unknown Parameter [%d]\n\r"), EnvidOnOff));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
    }
#if 0
    DISP_INF((_T("[DISP:INF] DUALCON = 0x%08x\n\r"), g_pDispConReg->DUALCON));
    DISP_INF((_T("[DISP:INF] VIDCON2    = 0x%08x\n\r"), g_pDispConReg->VIDCON2));
    DISP_INF((_T("[DISP:INF] DISPLAY_CONTROL  = 0x%08x\n\r"), g_pCMUMISCReg->MULTIMEDIA.DISPLAY_CONTROL));
    DISP_INF((_T("[DISP:INF] GPOUTCON0    = 0x%08x\n\r"), g_pDispConReg->GPOUTCON0));
    DISP_INF((_T("[DISP:INF] VIDCON0 = 0x%08x\n\r"), g_pDispConReg->VIDCON0));
    DISP_INF((_T("[DISP:INF] CLK_SRC_MASK0 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK0));
    DISP_INF((_T("[DISP:INF] CLK_SRC1    = 0x%08x\n\r"), g_pCMUCLKReg->CLK_SRC.CLK_SRC1));
    DISP_INF((_T("[DISP:INF] CLK_DIV1  = 0x%08x\n\r"), g_pCMUCLKReg->CLK_DIV.CLK_DIV1));
    DISP_INF((_T("[mDNIe:INF] CLK_SRC_MASK1 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK1));
    DISP_INF((_T("[mDNIe:INF] CLK_SRC3    = 0x%08x\n\r"), g_pCMUCLKReg->CLK_SRC.CLK_SRC3));
    DISP_INF((_T("[mDNIe:INF] CLK_DIV3  = 0x%08x\n\r"), g_pCMUCLKReg->CLK_DIV.CLK_DIV3));

    DISP_INF((_T("[DISP:INF]%s() : VIDCON0 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDCON0));
    DISP_INF((_T("[DISP:INF]%s() : VIDCON1 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDCON1));
    DISP_INF((_T("[DISP:INF]%s() : VIDCON2 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDCON2));
    DISP_INF((_T("[DISP:INF]%s() : VIDCON3 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDCON3));
    DISP_INF((_T("[DISP:INF]%s() : VIDTCON0 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDTCON0));
    DISP_INF((_T("[DISP:INF]%s() : VIDTCON1 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDTCON1));
    DISP_INF((_T("[DISP:INF]%s() : VIDTCON2 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDTCON2));
    DISP_INF((_T("[DISP:INF]%s() : VIDOSD2A = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDOSD2A));
    DISP_INF((_T("[DISP:INF]%s() : VIDOSD2B = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDOSD2B));
    DISP_INF((_T("[DISP:INF]%s() : VIDOSD2C = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDOSD2C));
    DISP_INF((_T("[DISP:INF]%s() : VIDW02ADD0B0 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDW02ADD0B0));
    DISP_INF((_T("[DISP:INF]%s() : VIDW02ADD1B0 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDW02ADD1B0));
    DISP_INF((_T("[DISP:INF]%s() : VIDW02ADD2 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->VIDW02ADD2));
    DISP_INF((_T("[DISP:INF]%s() : WINCON2 = 0x%08x\n\r"), _T(__FUNCTION__),g_pDispConReg->WINCON2));

    DISP_INF((_T("[DISP:INF] CLK_GATE_IP0 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_GATE.CLK_GATE_IP0));
    DISP_INF((_T("[DISP:INF] CLK_GATE_IP1 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_GATE.CLK_GATE_IP1));
    DISP_INF((_T("[DISP:INF] CLK_GATE_IP2 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_GATE.CLK_GATE_IP2));
    DISP_INF((_T("[DISP:INF] CLK_GATE_IP3 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_GATE.CLK_GATE_IP3));
    DISP_INF((_T("[DISP:INF] CLK_GATE_IP4 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_GATE.CLK_GATE_IP4));
    DISP_INF((_T("[DISP:INF] CLK_GATE_BLOCK = 0x%08x\n\r"), g_pCMUCLKReg->CLK_GATE.CLK_GATE_BLOCK));
    DISP_MSG((_T("[DISP]--Disp_envid_onoff() : %d\n\r"), error));
#endif
    return error;
}

BOOL Disp_get_envid_status(void)
{
    BOOL bRet = FALSE;

    DISP_MSG((_T("[DISP]++Disp_get_envid_status()\n\r")));

    if (g_pDispConReg->VIDCON0 & ENVID_ENABLE)
    {
        bRet = TRUE;
    }
    else
    {
        bRet = FALSE;
    }

    DISP_MSG((_T("[DISP]--Disp_envid_onoff() : %d\n\r"), bRet));

    return bRet;
}

// Wait for envid off
BOOL Disp_wait_until_envid_off(unsigned int uiMaxWaitCount)
{
    BOOL bRet = FALSE;
    unsigned int uiLoopCount;

    for (uiLoopCount=0; uiLoopCount < uiMaxWaitCount; uiLoopCount++)
    {
        if (Disp_get_envid_status() == FALSE)
        {
            bRet = TRUE;
            break;
        }
    }

    return bRet;
}

DISP_ERROR Disp_window_onfoff(DISP_WINDOW Win, DISP_WINDOW_ONOFF WinOnOff)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_window_onfoff(%d, %d)\n\r"), Win, WinOnOff));

    if (WinOnOff == DISP_WINDOW_ON || WinOnOff == DISP_WINDOW_OFF)
    {
        switch(Win)
        {
        case DISP_WIN0:
            if (WinOnOff == DISP_WINDOW_ON)
            {
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON |= (1<<0) | ((g_Win0Config.LocalPathEnable>>22)<<5);
#endif
                g_pDispConReg->WINCON0 |= (ENWIN_F_ENABLE | g_Win0Config.LocalPathEnable);
            }
            else
            {
                // ENLOCAL bit must be disabled after ENWIN_F has been disabled
                g_pDispConReg->WINCON0 &= ~(ENWIN_F_ENABLE | g_Win0Config.LocalPathEnable);
                //g_pDispConReg->WINCON0 &= ~ENWIN_F_ENABLE;
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON &= ~((1<<0) | (1<<5));
#endif
            }
            break;
        case DISP_WIN1:
            if (WinOnOff == DISP_WINDOW_ON)
            {
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON |= (1<<1) | ((g_Win1Config.LocalPathEnable>>22)<<6);
#endif
                g_pDispConReg->WINCON1 |= (ENWIN_F_ENABLE | g_Win1Config.LocalPathEnable);
            }
            else
            {
                // ENLOCAL bit must be disabled after ENWIN_F has been disabled
                g_pDispConReg->WINCON1 &= ~(ENWIN_F_ENABLE | g_Win1Config.LocalPathEnable);
                //g_pDispConReg->WINCON1 &= ~ENWIN_F_ENABLE;
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON &= ~((1<<1) | (1<<6));
#endif
            }
            break;
        case DISP_WIN2:
            if (WinOnOff == DISP_WINDOW_ON)
            {
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON |= (1<<2) | ((g_Win2Config.LocalPathEnable>>22)<<7);
#endif
                g_pDispConReg->WINCON2 |= (ENWIN_F_ENABLE | g_Win2Config.LocalPathEnable);
            }
            else
            {
                // ENLOCAL bit must be disabled after ENWIN_F has been disabled
                g_pDispConReg->WINCON2 &= ~(ENWIN_F_ENABLE | g_Win2Config.LocalPathEnable);
                //g_pDispConReg->WINCON2 &= ~ENWIN_F_ENABLE;
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON &= ~((1<<2) | (1<<7));
#endif
            }
            break;
        case DISP_WIN3:
            if (WinOnOff == DISP_WINDOW_ON)
            {
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON |= (1<<3);
#endif
                g_pDispConReg->WINCON3 |= ENWIN_F_ENABLE;
            }
            else
            {
                g_pDispConReg->WINCON3 &= ~ENWIN_F_ENABLE;
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON &= ~(1<<3);
#endif
            }
            break;
        case DISP_WIN4:
            if (WinOnOff == DISP_WINDOW_ON)
            {
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON |= (1<<4);
#endif
                g_pDispConReg->WINCON4 |= ENWIN_F_ENABLE;
            }
            else
            {
                g_pDispConReg->WINCON4 &= ~ENWIN_F_ENABLE;
#if (S5PV210_EVT>0)
                g_pDispConReg->SHADOWCON &= ~(1<<4);
#endif
            }
            break;
        default:
            DISP_ERR((_T("[DISP:ERR] Disp_window_onfoff() : Unknown Window Number [%d]\n\r"), Win));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
            break;
        }
    }
    else
    {
        DISP_ERR((_T("[DISP:ERR] Disp_window_onfoff() : Unknown OnOff Parameter [%d]\n\r"), WinOnOff));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
    }

    DISP_MSG((_T("[DISP]--Disp_window_onfoff() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_local_path_off(DISP_WINDOW Win)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_local_path_off(%d)\n\r"), Win));

    switch(Win)
    {
    case DISP_WIN0:
        // The precondition is that ENLOCAL bit must be disabled after ENWIN_F has been disabled
        g_pDispConReg->WINCON0 &= ~(g_Win0Config.LocalPathEnable);
        break;
    case DISP_WIN1:
        // The precondition is that ENLOCAL bit must be disabled after ENWIN_F has been disabled
        g_pDispConReg->WINCON1 &= ~(g_Win1Config.LocalPathEnable);
        break;
    case DISP_WIN2:
        // The precondition is that ENLOCAL bit must be disabled after ENWIN_F has been disabled
        g_pDispConReg->WINCON2 &= ~(g_Win2Config.LocalPathEnable);
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_local_path_off() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_local_path_off() : %d\n\r"), error));

    return error;
}


DISP_WINDOW_ONOFF Disp_get_window_status(DISP_WINDOW Win)
{
    unsigned int uiWinConReg = 0;

    switch(Win)
    {
    case DISP_WIN0:
        uiWinConReg = g_pDispConReg->WINCON0;
        break;
    case DISP_WIN1:
        uiWinConReg = g_pDispConReg->WINCON1;
        break;
    case DISP_WIN2:
        uiWinConReg = g_pDispConReg->WINCON2;
        break;
    case DISP_WIN3:
        uiWinConReg = g_pDispConReg->WINCON3;
        break;
    case DISP_WIN4:
        uiWinConReg = g_pDispConReg->WINCON4;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_get_window_status() : Unknown Window Number [%d]\n\r"), Win));
        break;
    }

    if (uiWinConReg & ENWIN_F_ENABLE)
    {
        return DISP_WINDOW_ON;
    }
    else
    {
        return DISP_WINDOW_OFF;
    }
}

DISP_ERROR Disp_set_window_color_map(DISP_WINDOW eWinNum, BOOL bOnOff, unsigned int uiColorValue)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_window_color_map(%d, %d, 0x%08x)\n\r"), eWinNum, bOnOff, uiColorValue));

    switch(eWinNum)
    {
    case DISP_WIN0:
        if (bOnOff)
        {
            g_pDispConReg->WIN0MAP = MAPCOLEN_F_ENABLE | MAPCOLOR(uiColorValue);
        }
        else
        {
            g_pDispConReg->WIN0MAP = MAPCOLEN_F_DISABLE;
        }
        break;
    case DISP_WIN1:
        if (bOnOff)
        {
            g_pDispConReg->WIN1MAP = MAPCOLEN_F_ENABLE | MAPCOLOR(uiColorValue);
        }
        else
        {
            g_pDispConReg->WIN1MAP = MAPCOLEN_F_DISABLE;
        }
        break;
    case DISP_WIN2:
        if (bOnOff)
        {
            g_pDispConReg->WIN2MAP = MAPCOLEN_F_ENABLE | MAPCOLOR(uiColorValue);
        }
        else
        {
            g_pDispConReg->WIN2MAP = MAPCOLEN_F_DISABLE;
        }
        break;
    case DISP_WIN3:
        if (bOnOff)
        {
            g_pDispConReg->WIN3MAP = MAPCOLEN_F_ENABLE | MAPCOLOR(uiColorValue);
        }
        else
        {
            g_pDispConReg->WIN3MAP = MAPCOLEN_F_DISABLE;
        }
        break;
    case DISP_WIN4:
        if (bOnOff)
        {
            g_pDispConReg->WIN4MAP = MAPCOLEN_F_ENABLE | MAPCOLOR(uiColorValue);
        }
        else
        {
            g_pDispConReg->WIN4MAP = MAPCOLEN_F_DISABLE;
        }
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_window_color_map() : Unknown Window Number [%d]\n\r"), eWinNum));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_set_window_color_map() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_set_color_key(DISP_WINDOW Win, BOOL bOnOff, BOOL bBlend, DISP_COLOR_KEY_DIRECTION Direction, unsigned int uiColorKey, unsigned int uiComparekey, unsigned int uiKeyAlpha)
{
    DISP_ERROR error = DISP_SUCCESS;

    unsigned int uiKeyDir;

    DISP_MSG((_T("[DISP]++Disp_set_color_key(%d, %d, %d, %d, 0x%08x, 0x%08x)\n\r"), Win, bOnOff, bBlend, Direction, uiColorKey, uiComparekey));

    if (Direction == DISP_FG_MATCH_BG_DISPLAY)
    {
        uiKeyDir = DIRCON_FG_MATCH_BG_DISPLAY;
    }
    else
    {
        uiKeyDir = DIRCON_BG_MATCH_FG_DISPLAY;
    }

    switch(Win)
    {
    case DISP_WIN1:
        if (bOnOff)
        {
            if(bBlend)
            {
                g_pDispConReg->W1KEYCON0 = KEYBLEN_ENABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W1KEYCON1 = COLVAL(uiColorKey);

                g_pDispConReg->W1KEYALPHA = (uiKeyAlpha & 0xFF) | \
                                            ((uiKeyAlpha & 0xFF) << 8) | \
                                            ((uiKeyAlpha & 0xFF) << 16);

                // Enable BLD_PIX
                g_pDispConReg->WINCON1 |= BLEND_PER_PIXEL;
            }
            else
            {
                g_pDispConReg->W1KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W1KEYCON1 = COLVAL(uiColorKey);
            }
        }
        else
        {
            g_pDispConReg->W1KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_DISABLE;
        }
        break;
    case DISP_WIN2:
        if (bOnOff)
        {
            if(bBlend)
            {
                g_pDispConReg->W2KEYCON0 = KEYBLEN_ENABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W2KEYCON1 = COLVAL(uiColorKey);

                g_pDispConReg->W2KEYALPHA = (uiKeyAlpha & 0xFF) | \
                                            ((uiKeyAlpha & 0xFF) << 8) | \
                                            ((uiKeyAlpha & 0xFF) << 16);

                // Enable BLD_PIX
                g_pDispConReg->WINCON2 |= BLEND_PER_PIXEL;
            }
            else
            {
                g_pDispConReg->W2KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W2KEYCON1 = COLVAL(uiColorKey);
            }
        }
        else
        {
            g_pDispConReg->W2KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_DISABLE;
        }
        break;
    case DISP_WIN3:
        if (bOnOff)
        {
            if(bBlend)
            {
                g_pDispConReg->W3KEYCON0 = KEYBLEN_ENABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W3KEYCON1 = COLVAL(uiColorKey);

                g_pDispConReg->W3KEYALPHA = (uiKeyAlpha & 0xFF) | \
                                            ((uiKeyAlpha & 0xFF) << 8) | \
                                            ((uiKeyAlpha & 0xFF) << 16);

                // Enable BLD_PIX
                g_pDispConReg->WINCON3 |= BLEND_PER_PIXEL;
            }
            else
            {
                g_pDispConReg->W3KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W3KEYCON1 = COLVAL(uiColorKey);
            }
        }
        else
        {
            g_pDispConReg->W3KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_DISABLE;
        }
        break;
    case DISP_WIN4:
        if (bOnOff)
        {
            if(bBlend)
            {
                g_pDispConReg->W4KEYCON0 = KEYBLEN_ENABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W4KEYCON1 = COLVAL(uiColorKey);

                g_pDispConReg->W4KEYALPHA = (uiKeyAlpha & 0xFF) | \
                                            ((uiKeyAlpha & 0xFF) << 8) | \
                                            ((uiKeyAlpha & 0xFF) << 16);

                // Enable BLD_PIX
                g_pDispConReg->WINCON4 |= BLEND_PER_PIXEL;
            }
            else
            {
                g_pDispConReg->W4KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_ENABLE | uiKeyDir | COMPKEY(uiComparekey);
                g_pDispConReg->W4KEYCON1 = COLVAL(uiColorKey);
            }
        }
        else
        {
            g_pDispConReg->W4KEYCON0 = KEYBLEN_DISABLE | KEYEN_F_DISABLE;
        }
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_color_key() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_set_color_key() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_set_alpha_blending(DISP_WINDOW Win, DISP_ALPHA_BLEND_METHOD Method, unsigned int uiAlphaSet0, unsigned int uiAlphaSet1)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiMethod;

    DISP_MSG((_T("[DISP]++Disp_set_alpha_blending(%d, %d, 0x%08x, 0x%08x)\n\r"), Win, Method, uiAlphaSet0, uiAlphaSet1));

    if (Method == DISP_ALPHA_PER_PLANE)
    {
        uiMethod = BLEND_PER_PLANE;
    }
    else
    {
        uiMethod = BLEND_PER_PIXEL;
    }

    // Use 4 bit alpha value
    g_pDispConReg->BLENDCON &= ~USE_BLEND_NEW;

    Disp_window_registers_lock(Win);

    switch(Win)
    {
    case DISP_WIN1:
        g_pDispConReg->WINCON1 =  (g_pDispConReg->WINCON1 & ~(BLEND_PER_PIXEL|ALPHASEL_ALPHA1)) | uiMethod;
        g_pDispConReg->VIDOSD1C = ALPHA0_R_4BIT(uiAlphaSet0) | ALPHA0_G_4BIT(uiAlphaSet0) | ALPHA0_B_4BIT(uiAlphaSet0)
                                    | ALPHA1_R_4BIT(uiAlphaSet1) | ALPHA1_G_4BIT(uiAlphaSet1) | ALPHA1_B_4BIT(uiAlphaSet1);
        break;
    case DISP_WIN2:
        g_pDispConReg->WINCON2 =  (g_pDispConReg->WINCON2 & ~(BLEND_PER_PIXEL|ALPHASEL_ALPHA1)) | uiMethod;
        g_pDispConReg->VIDOSD2C = ALPHA0_R_4BIT(uiAlphaSet0) | ALPHA0_G_4BIT(uiAlphaSet0) | ALPHA0_B_4BIT(uiAlphaSet0)
                                    | ALPHA1_R_4BIT(uiAlphaSet1) | ALPHA1_G_4BIT(uiAlphaSet1) | ALPHA1_B_4BIT(uiAlphaSet1);
        break;
    case DISP_WIN3:
        g_pDispConReg->WINCON3 =  (g_pDispConReg->WINCON3 & ~(BLEND_PER_PIXEL|ALPHASEL_ALPHA1)) | uiMethod;
        g_pDispConReg->VIDOSD3C = ALPHA0_R_4BIT(uiAlphaSet0) | ALPHA0_G_4BIT(uiAlphaSet0) | ALPHA0_B_4BIT(uiAlphaSet0)
                                    | ALPHA1_R_4BIT(uiAlphaSet1) | ALPHA1_G_4BIT(uiAlphaSet1) | ALPHA1_B_4BIT(uiAlphaSet1);
        break;
    case DISP_WIN4:
        g_pDispConReg->WINCON4 =  (g_pDispConReg->WINCON4 & ~(BLEND_PER_PIXEL|ALPHASEL_ALPHA1)) | uiMethod;
        g_pDispConReg->VIDOSD4C = ALPHA0_R_4BIT(uiAlphaSet0) | ALPHA0_G_4BIT(uiAlphaSet0) | ALPHA0_B_4BIT(uiAlphaSet0)
                                    | ALPHA1_R_4BIT(uiAlphaSet1) | ALPHA1_G_4BIT(uiAlphaSet1) | ALPHA1_B_4BIT(uiAlphaSet1);
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_alpha_blending() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    Disp_window_registers_unlock(Win);

    // Use 4 bit alpha value
    if(error == DISP_SUCCESS)
    {
        g_pDispConReg->BLENDCON &= ~USE_BLEND_NEW;
    }

    DISP_MSG((_T("[DISP]--Disp_set_alpha_blending() : %d\n\r"), error));

    return error;
}

// This function takes 8bit alphaset value for each color channel
DISP_ERROR Disp_set_alpha_blending_ex(DISP_WINDOW Win, DISP_ALPHA_BLEND_METHOD Method,
                                            DISP_BLEND_EQ A_BlendEq,
                                            DISP_BLEND_EQ B_BlendEq,
                                            DISP_BLEND_EQ P_BlendEq,
                                            DISP_BLEND_EQ Q_BlendEq,
                                            unsigned int uiAlphaSet0, unsigned int uiAlphaSet1)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++%s(%d, %d, 0x%08x, 0x%08x)\n\r"), _T(__FUNCTION__), Win, Method, uiAlphaSet0, uiAlphaSet1));
    DISP_MSG((_T("[DISP]++%s(%d, %d, %d, %d)\n\r"), _T(__FUNCTION__), A_BlendEq, B_BlendEq, P_BlendEq, Q_BlendEq));

    // Use 8 bit alpha value
    g_pDispConReg->BLENDCON |= USE_BLEND_NEW;

    Disp_window_registers_lock(Win);

    // BLD_PIX == 0 --> AEN is ignored, so ALPHA0 is used.
    // BLD_PIX == 1, Pixel Blending
    // ALPHA1 ==> AEN == 1, Transparency is on
    // ALPHA0 ==> AEN == 0, Transparency is off
    switch(Win)
    {
    case DISP_WIN0:
        if(Method == DISP_ALPHA_PER_PIXEL)
        {
            if((g_pDispConReg->WINCON0 & BPPMODE_F_MASK) == \
                                                        BPPMODE_F_32BPP_A8888)
            {
                g_pDispConReg->WINCON0 = (BLEND_PER_PIXEL | ALPHASEL_DATA) | \
                            ((g_pDispConReg->WINCON0)&(~BLEND_MULTIPLIED));
            }
            else
            {
                g_pDispConReg->WINCON0 = BLEND_PER_PIXEL | \
                                    ((g_pDispConReg->WINCON0) & \
                                        (~(BLEND_MULTIPLIED | ALPHASEL_DATA)));
            }
        }
        else
        {
            g_pDispConReg->WINCON0 &= ~(BLEND_MULTIPLIED | ALPHASEL_ALPHA1);
        }

        g_pDispConReg->VIDW0ALPHA0 = ((uiAlphaSet0 & 0xff) << 16) | ((uiAlphaSet0 & 0xff) << 8) | \
                                        (uiAlphaSet0 & 0xff);
        g_pDispConReg->VIDW0ALPHA1 = ((uiAlphaSet1 & 0xff) << 16) | ((uiAlphaSet1 & 0xff) << 8) | \
                                        (uiAlphaSet1 & 0xff);

        //g_pDispConReg->BLENDEQ1 = A_FUNC(A_BlendEq)|B_FUNC(B_BlendEq)|P_FUNC(P_BlendEq)|Q_FUNC(Q_BlendEq);
        DISP_INF((_T("[DISP] %s() : 0x%x,0x%x,0x%x\n\r"), _T(__FUNCTION__),
            g_pDispConReg->WINCON0,
            g_pDispConReg->VIDW0ALPHA0, g_pDispConReg->VIDW0ALPHA1));
        break;
    case DISP_WIN1:
        if(Method == DISP_ALPHA_PER_PIXEL)
        {
            if((g_pDispConReg->WINCON1 & BPPMODE_F_MASK) == \
                                                        BPPMODE_F_32BPP_A8888)
            {
                g_pDispConReg->WINCON1 = (BLEND_PER_PIXEL | ALPHASEL_DATA) | \
                            ((g_pDispConReg->WINCON1)&(~BLEND_MULTIPLIED));
            }
            else
            {
                g_pDispConReg->WINCON1 = BLEND_PER_PIXEL | \
                                    ((g_pDispConReg->WINCON1) & \
                                        (~(BLEND_MULTIPLIED | ALPHASEL_DATA)));
            }
        }
        else
        {
            g_pDispConReg->WINCON1 &= ~(BLEND_MULTIPLIED | ALPHASEL_ALPHA1);
        }

        g_pDispConReg->VIDOSD1C = ALPHA0_R_H(uiAlphaSet0) | ALPHA0_G_H(uiAlphaSet0) | ALPHA0_B_H(uiAlphaSet0) | \
                                    ALPHA1_R_H(uiAlphaSet1) | ALPHA1_G_H(uiAlphaSet1) | ALPHA1_B_H(uiAlphaSet1);
        g_pDispConReg->VIDW1ALPHA0 = ALPHA0_R_L(uiAlphaSet0) | ALPHA0_G_L(uiAlphaSet0) | ALPHA0_B_L(uiAlphaSet0);
        g_pDispConReg->VIDW1ALPHA1 = ALPHA1_R_L(uiAlphaSet1) | ALPHA1_G_L(uiAlphaSet1) | ALPHA1_B_L(uiAlphaSet1);
        g_pDispConReg->BLENDEQ1 = A_FUNC(A_BlendEq)|B_FUNC(B_BlendEq)|P_FUNC(P_BlendEq)|Q_FUNC(Q_BlendEq);
        DISP_INF((_T("[DISP] %s() : 0x%x,0x%x,0x%x,0x%x,0x%x\n\r"), _T(__FUNCTION__),
            g_pDispConReg->WINCON1, g_pDispConReg->VIDOSD1C,
            g_pDispConReg->VIDW1ALPHA0, g_pDispConReg->VIDW1ALPHA1, g_pDispConReg->BLENDEQ1));
        break;
    case DISP_WIN2:
        if(Method == DISP_ALPHA_PER_PIXEL)
        {
            if((g_pDispConReg->WINCON2 & BPPMODE_F_MASK) == \
                                                        BPPMODE_F_32BPP_A8888)
            {
                g_pDispConReg->WINCON2 = (BLEND_PER_PIXEL | ALPHASEL_DATA) | \
                            ((g_pDispConReg->WINCON2)&(~BLEND_MULTIPLIED));
            }
            else
            {
                g_pDispConReg->WINCON2 = BLEND_PER_PIXEL | \
                                    ((g_pDispConReg->WINCON2) & \
                                        (~(BLEND_MULTIPLIED | ALPHASEL_DATA)));
            }
        }
        else
        {
            g_pDispConReg->WINCON2 &= ~(BLEND_MULTIPLIED | ALPHASEL_ALPHA1);
        }

        g_pDispConReg->VIDOSD2C = ALPHA0_R_H(uiAlphaSet0) | ALPHA0_G_H(uiAlphaSet0) | ALPHA0_B_H(uiAlphaSet0) | \
                                    ALPHA1_R_H(uiAlphaSet1) | ALPHA1_G_H(uiAlphaSet1) | ALPHA1_B_H(uiAlphaSet1);
        g_pDispConReg->VIDW2ALPHA0 = ALPHA0_R_L(uiAlphaSet0) | ALPHA0_G_L(uiAlphaSet0) | ALPHA0_B_L(uiAlphaSet0);
        g_pDispConReg->VIDW2ALPHA1 = ALPHA1_R_L(uiAlphaSet1) | ALPHA1_G_L(uiAlphaSet1) | ALPHA1_B_L(uiAlphaSet1);
        g_pDispConReg->BLENDEQ2 = A_FUNC(A_BlendEq)|B_FUNC(B_BlendEq)|P_FUNC(P_BlendEq)|Q_FUNC(Q_BlendEq);
        DISP_INF((_T("[DISP] %s() : 0x%x,0x%x,0x%x,0x%x,0x%x\n\r"), _T(__FUNCTION__),
            g_pDispConReg->WINCON2, g_pDispConReg->VIDOSD2C,
            g_pDispConReg->VIDW2ALPHA0, g_pDispConReg->VIDW2ALPHA1, g_pDispConReg->BLENDEQ2));
        break;
    case DISP_WIN3:
        if(Method == DISP_ALPHA_PER_PIXEL)
        {
            if((g_pDispConReg->WINCON3 & BPPMODE_F_MASK) == \
                                                        BPPMODE_F_32BPP_A8888)
            {
                g_pDispConReg->WINCON3 = (BLEND_PER_PIXEL | ALPHASEL_DATA) | \
                            ((g_pDispConReg->WINCON3)&(~BLEND_MULTIPLIED));
            }
            else
            {
                g_pDispConReg->WINCON3 = BLEND_PER_PIXEL | \
                                    ((g_pDispConReg->WINCON3) & \
                                        (~(BLEND_MULTIPLIED | ALPHASEL_DATA)));
            }
        }
        else
        {
            g_pDispConReg->WINCON3 &= ~(BLEND_MULTIPLIED | ALPHASEL_ALPHA1);
        }

        g_pDispConReg->VIDOSD3C = ALPHA0_R_H(uiAlphaSet0) | ALPHA0_G_H(uiAlphaSet0) | ALPHA0_B_H(uiAlphaSet0) | \
                                    ALPHA1_R_H(uiAlphaSet1) | ALPHA1_G_H(uiAlphaSet1) | ALPHA1_B_H(uiAlphaSet1);
        g_pDispConReg->VIDW3ALPHA0 = ALPHA0_R_L(uiAlphaSet0) | ALPHA0_G_L(uiAlphaSet0) | ALPHA0_B_L(uiAlphaSet0);
        g_pDispConReg->VIDW3ALPHA1 = ALPHA1_R_L(uiAlphaSet1) | ALPHA1_G_L(uiAlphaSet1) | ALPHA1_B_L(uiAlphaSet1);
        g_pDispConReg->BLENDEQ3 = A_FUNC(A_BlendEq)|B_FUNC(B_BlendEq)|P_FUNC(P_BlendEq)|Q_FUNC(Q_BlendEq);
        DISP_INF((_T("[DISP] %s() : 0x%x,0x%x,0x%x,0x%x,0x%x\n\r"), _T(__FUNCTION__),
            g_pDispConReg->WINCON3, g_pDispConReg->VIDOSD3C,
            g_pDispConReg->VIDW3ALPHA0, g_pDispConReg->VIDW3ALPHA1, g_pDispConReg->BLENDEQ3));
        break;
    case DISP_WIN4:
        if(Method == DISP_ALPHA_PER_PIXEL)
        {
            if((g_pDispConReg->WINCON4 & BPPMODE_F_MASK) == \
                                                        BPPMODE_F_32BPP_A8888)
            {
                g_pDispConReg->WINCON4 = (BLEND_PER_PIXEL | ALPHASEL_DATA) | \
                            ((g_pDispConReg->WINCON4)&(~BLEND_MULTIPLIED));
            }
            else
            {
                g_pDispConReg->WINCON4 = BLEND_PER_PIXEL | \
                                    ((g_pDispConReg->WINCON4) & \
                                        (~(BLEND_MULTIPLIED | ALPHASEL_DATA)));
            }
        }
        else
        {
            g_pDispConReg->WINCON4 &= ~(BLEND_MULTIPLIED | ALPHASEL_ALPHA1);
        }

        g_pDispConReg->VIDOSD4C = ALPHA0_R_H(uiAlphaSet0) | ALPHA0_G_H(uiAlphaSet0) | ALPHA0_B_H(uiAlphaSet0) | \
                                    ALPHA1_R_H(uiAlphaSet1) | ALPHA1_G_H(uiAlphaSet1) | ALPHA1_B_H(uiAlphaSet1);
        g_pDispConReg->VIDW4ALPHA0 = ALPHA0_R_L(uiAlphaSet0) | ALPHA0_G_L(uiAlphaSet0) | ALPHA0_B_L(uiAlphaSet0);
        g_pDispConReg->VIDW4ALPHA1 = ALPHA1_R_L(uiAlphaSet1) | ALPHA1_G_L(uiAlphaSet1) | ALPHA1_B_L(uiAlphaSet1);
        g_pDispConReg->BLENDEQ4 = A_FUNC(A_BlendEq)|B_FUNC(B_BlendEq)|P_FUNC(P_BlendEq)|Q_FUNC(Q_BlendEq);
        DISP_INF((_T("[DISP] %s() : 0x%x,0x%x,0x%x,0x%x,0x%x\n\r"), _T(__FUNCTION__),
            g_pDispConReg->WINCON4, g_pDispConReg->VIDOSD4C,
            g_pDispConReg->VIDW4ALPHA0, g_pDispConReg->VIDW4ALPHA1, g_pDispConReg->BLENDEQ4));
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] %s() : Unknown Window Number [%d]\n\r"), _T(__FUNCTION__), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    Disp_window_registers_unlock(Win);

    // Use 8 bit alpha value
    if(error == DISP_SUCCESS)
    {
        g_pDispConReg->BLENDCON = USE_BLEND_NEW;
    }

    DISP_INF((_T("[DISP] %s() : 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->BLENDCON));
    DISP_MSG((_T("[DISP]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}


DISP_ERROR Disp_set_gamma(BOOL bEnable, PBYTE pbGammaLUT)
{
    DISP_ERROR error = DISP_SUCCESS;
    UINT32 *puiStartGammaLUT;
    UINT8 uiCount;

    DISP_MSG((_T("[DISP]++%s() : bEnable = %u\n\r"), _T(__FUNCTION__), (DWORD)bEnable));

    if(FALSE == bEnable)
    {
        // Disable gamma control
        g_pDispConReg->VIDCON3 &= ~GAMMA_ONOFF_MASK;
    }
    else
    {
        // Disable gamma control
        g_pDispConReg->VIDCON3 &= ~GAMMA_ONOFF_MASK;

        // Clear Gamma LUT
        puiStartGammaLUT = (UINT32 *)(&(g_pDispConReg->GAMMALUT_01_00));
        //for(uiCount=0; uiCount<=32; uiCount++)
        for(uiCount=0; uiCount<=(DISP_GAMMA_VALUE_NUM/2-1); uiCount++)
        {
             *puiStartGammaLUT = 0;
             puiStartGammaLUT++;
        }

        // Populate Gamma LUT
        puiStartGammaLUT = (UINT32 *)(&(g_pDispConReg->GAMMALUT_01_00));
        //for(uiCount=0; uiCount<=64; uiCount++)
        for(uiCount=0; uiCount<=(DISP_GAMMA_VALUE_NUM-1); uiCount++)
        {
            if((uiCount%2) == 0)
            {
                *puiStartGammaLUT |= (pbGammaLUT[uiCount]<<2);
            }
            else
            {
                *puiStartGammaLUT |= (pbGammaLUT[uiCount]<<18);
                puiStartGammaLUT++;
            }
        }

        // Enable gamma control
        g_pDispConReg->VIDCON3 |= GAMMA_ENABLE;
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

DISP_ERROR Disp_set_colorgain(BOOL bEnable, WORD wRGain, WORD wGGain, WORD wBGain)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++%s() : bEnable = %u\n\r"), _T(__FUNCTION__), (DWORD)bEnable));

    if(FALSE == bEnable)
    {
        // Disable colorgain control
        g_pDispConReg->VIDCON3 &= ~COLORGAIN_ONOFF_MASK;
        DISP_MSG((_T("[DISP]%s()  : VIDCON3= 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->VIDCON3));
    }
    else
    {
        // Disable colorgain control
        g_pDispConReg->VIDCON3 &= ~COLORGAIN_ONOFF_MASK;

        // Set gain value of R data
        g_pDispConReg->COLORGAINCON &= ~COLORGAIN_R_MASK;
        g_pDispConReg->COLORGAINCON |= ((wRGain & 0x3FF)<<COLORGAIN_R_BITPOS);

        // Set gain value of G data
        g_pDispConReg->COLORGAINCON &= ~COLORGAIN_G_MASK;
        g_pDispConReg->COLORGAINCON |= ((wGGain & 0x3FF)<<COLORGAIN_G_BITPOS);

        // Set gain value of B data
        g_pDispConReg->COLORGAINCON &= ~COLORGAIN_B_MASK;
        g_pDispConReg->COLORGAINCON |= ((wBGain & 0x3FF)<<COLORGAIN_B_BITPOS);

        // Enable colorgain control
        g_pDispConReg->VIDCON3 |= COLORGAIN_ENABLE;
        DISP_INF((_T("[DISP]%s()  : VIDCON3= 0x%x, COLORGAINCON = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->VIDCON3, g_pDispConReg->COLORGAINCON));
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

#define HUE_VALUE_RANGE_CHECK(val)    ((((val>0x100)&&(val<0x300))||(val>0x3FF))?FALSE:TRUE)
#define HUE_OFST_RANGE_CHECK(val)     ((val>0x1FF)?FALSE:TRUE)

DISP_ERROR Disp_set_hue(BOOL bEnable, DISP_HUE_CONTROL hueControl)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++%s() : bEnable = %u\n\r"), _T(__FUNCTION__), (DWORD)bEnable));

    if(FALSE == bEnable)
    {
        // Disable hue control
        g_pDispConReg->VIDCON3 &= ~HUE_ONOFF_MASK;
        g_pDispConReg->VIDCON3 &= ~HUE_CSC_F_ONOFF_MASK;
        g_pDispConReg->VIDCON3 &= ~HUE_CSC_B_ONOFF_MASK;
    }
    else
    {
        // Parameter range check
        if( !HUE_VALUE_RANGE_CHECK(hueControl.wCBG0_P) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCBG0_N) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCBG1_P) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCBG1_N) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCRG0_P) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCRG0_N) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCRG1_P) || \
            !HUE_VALUE_RANGE_CHECK(hueControl.wCRG1_N) )
        {
            // Error condition
            DISP_ERR((_T("[DISP:ERR] %s() : Hue value out of range!\n\r"), _T(__FUNCTION__)));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
            goto CleanUp;
        }

        if( !HUE_OFST_RANGE_CHECK(hueControl.wInOffset) || \
            !HUE_OFST_RANGE_CHECK(hueControl.wOutOffset) )
        {
            // Error condition
            DISP_ERR((_T("[DISP:ERR] %s() : Hue offset out of range!\n\r"), _T(__FUNCTION__)));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
            goto CleanUp;
        }

        // Disable hue control
        g_pDispConReg->VIDCON3 &= ~HUE_ONOFF_MASK;
        g_pDispConReg->VIDCON3 &= ~HUE_CSC_F_ONOFF_MASK;
        g_pDispConReg->VIDCON3 &= ~HUE_CSC_B_ONOFF_MASK;

        // Set CBG0 values
        g_pDispConReg->HUECOEF00 = (g_pDispConReg->HUECOEF00 & ~CBGx_P_MASK) | \
                                    ((hueControl.wCBG0_P & 0x3FF)<<CBGx_P_BITPOS);
        g_pDispConReg->HUECOEF00 = (g_pDispConReg->HUECOEF00 & ~CBGx_N_MASK) | \
                                    ((hueControl.wCBG0_N & 0x3FF)<<CBGx_N_BITPOS);

        // Set CBG1 values
        g_pDispConReg->HUECOEF01 = (g_pDispConReg->HUECOEF01 & ~CBGx_P_MASK) | \
                                    ((hueControl.wCBG1_P & 0x3FF)<<CBGx_P_BITPOS);
        g_pDispConReg->HUECOEF01 = (g_pDispConReg->HUECOEF01 & ~CBGx_N_MASK) | \
                                    ((hueControl.wCBG1_N & 0x3FF)<<CBGx_N_BITPOS);

        // Set CRG0 values
        g_pDispConReg->HUECOEF10 = (g_pDispConReg->HUECOEF10 & ~CBGx_P_MASK) | \
                                    ((hueControl.wCRG0_P & 0x3FF)<<CBGx_P_BITPOS);
        g_pDispConReg->HUECOEF10 = (g_pDispConReg->HUECOEF10 & ~CBGx_N_MASK) | \
                                    ((hueControl.wCRG0_N & 0x3FF)<<CBGx_N_BITPOS);

        // Set CRG1 values
        g_pDispConReg->HUECOEF11 = (g_pDispConReg->HUECOEF11 & ~CBGx_P_MASK) | \
                                    ((hueControl.wCRG1_P & 0x3FF)<<CBGx_P_BITPOS);
        g_pDispConReg->HUECOEF11 = (g_pDispConReg->HUECOEF11 & ~CBGx_N_MASK) | \
                                    ((hueControl.wCRG1_N & 0x3FF)<<CBGx_N_BITPOS);

        // Set offsets
        g_pDispConReg->HUEOFFSET = (g_pDispConReg->HUEOFFSET & ~OFFSET_IN_MASK) | \
                                    ((hueControl.wInOffset & 0x1FF)<<OFFSET_IN_BITPOS);
        g_pDispConReg->HUEOFFSET = (g_pDispConReg->HUEOFFSET & ~OFFSET_OUT_MASK) | \
                                    ((hueControl.wOutOffset & 0x1FF)<<OFFSET_OUT_BITPOS);

        // Set hue color range
        if(DISP_RGB_COLOR_RANGE_WIDE == hueControl.eRange)
        {
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_F_RANGE_MASK) | \
                                        HUE_CSC_F_WIDE;
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_B_RANGE_MASK) | \
                                        HUE_CSC_B_WIDE;
        }
        else if(DISP_RGB_COLOR_RANGE_NARROW == hueControl.eRange)
        {
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_F_RANGE_MASK) | \
                                        HUE_CSC_F_NARROW;
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_B_RANGE_MASK) | \
                                        HUE_CSC_B_NARROW;
        }
        else
        {
            // Error condition
            DISP_ERR((_T("[DISP:ERR] %s() : Unknown color range [%d]\n\r"), _T(__FUNCTION__), hueControl.eRange));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
            goto CleanUp;
        }

        // Set CSC equation
        if(DISP_CSC_EQ_601 == hueControl.eCSCEq)
        {
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_F_EQ_MASK) | \
                                        HUE_CSC_F_EQ601;
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_B_EQ_MASK) | \
                                        HUE_CSC_B_EQ601;
        }
        else if(DISP_CSC_EQ_709 == hueControl.eCSCEq)
        {
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_F_EQ_MASK) | \
                                        HUE_CSC_F_EQ709;
            g_pDispConReg->VIDCON3 = (g_pDispConReg->VIDCON3 & ~HUE_CSC_B_EQ_MASK) | \
                                        HUE_CSC_B_EQ709;
        }
        else
        {
            // Error condition
            DISP_ERR((_T("[DISP:ERR] %s() : Unknown CSC equation [%d]\n\r"), _T(__FUNCTION__), hueControl.eCSCEq));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
            goto CleanUp;
        }

        // Enable hue control
        g_pDispConReg->VIDCON3 |= HUE_CSC_F_ENABLE;
        g_pDispConReg->VIDCON3 |= HUE_CSC_B_ENABLE;
        g_pDispConReg->VIDCON3 |= HUE_ENABLE;
    }

CleanUp:
    DISP_MSG((_T("[DISP]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}


DISP_ERROR Disp_set_dithering(BOOL bEnable, DISP_DITHER_MODE Mode)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++%s(): Mode = %d\n\r"), _T(__FUNCTION__), Mode));

    if(FALSE == bEnable)
    {
        g_pDispConReg->DITHMODE = DITHEN_F_DISABLE;
    }
    else
    {
        g_pDispConReg->DITHMODE = DITHEN_F_DISABLE;

        switch(Mode)
        {
        case DISP_DITHER_565:
            g_pDispConReg->DITHMODE |= (RDITHPOS_5BIT | GDITHPOS_6BIT | BDITHPOS_5BIT | DITHEN_F_ENABLE);
            break;
        case DISP_DITHER_666:
            g_pDispConReg->DITHMODE |= (RDITHPOS_6BIT | GDITHPOS_6BIT | BDITHPOS_6BIT | DITHEN_F_ENABLE);
            break;
        case DISP_DITHER_888:
            g_pDispConReg->DITHMODE |= (RDITHPOS_8BIT | GDITHPOS_8BIT | BDITHPOS_8BIT | DITHEN_F_ENABLE);
            break;
        default:
            DISP_ERR((_T("[DISP:ERR] %s: Unknown Mode [%d]\n\r"), _T(__FUNCTION__), Mode));
            error = DISP_ERROR_ILLEGAL_PARAMETER;
            g_pDispConReg->DITHMODE = DITHEN_F_DISABLE;
            break;
        }
    }
    DISP_INF((_T("[DISP]%s: DITHMODE = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->DITHMODE));

    DISP_MSG((_T("[DISP]--%s: %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

DISP_ERROR Disp_set_alps(BOOL bEnable, DISP_LUX_CONTROL luxControl,
                        unsigned int uiSizeH, unsigned int uiSizeV,
                        unsigned int uiValueSize,
                        unsigned char* pLow_R_LUT,
                        unsigned char* pLow_G_LUT,
                        unsigned char* pLow_B_LUT,
                        unsigned char* pHigh_R_LUT,
                        unsigned char* pHigh_G_LUT,
                        unsigned char* pHigh_B_LUT)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiTempReg0 = 0;
    unsigned int uiTempReg1 = 0;
    unsigned int uiTempReg2 = 0;
    unsigned int* pStartALPSLUT1 = NULL;
    unsigned int* pStartALPSLUT2 = NULL;
    unsigned int i = 0;

    DISP_MSG((_T("[DISP]++%s(): Enable = %d, DISP_LUX_CONTROL = %d\n\r"), _T(__FUNCTION__), bEnable, luxControl));

    g_pDispConReg->VIDCON3 &= ~VE_ENABLE;

    if(!bEnable)
    {
        return error;
    }

    g_pDispConReg->VESFRCON0 = ALPS_SW_RESET;
    //Sleep(100); // I don't know whether ip need to be delayed or not after s/w reset?
    g_pDispConReg->VESFRCON0 &= ~ALPS_SW_RESET;

    switch(luxControl)
    {
    case DISP_LOW_LUX:
        uiTempReg0 |= ALPS_LOW_LUX;
        break;
    case DISP_HIGH_LUX:
        uiTempReg0 |= ALPS_HIGH_LUX;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] %s: Unknown luxControl [%d]\n\r"), _T(__FUNCTION__), luxControl));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        g_pDispConReg->VIDCON3 &= ~VE_ENABLE;
        break;
    }

    if(error == DISP_SUCCESS)
    {
        uiTempReg1 = ALPS_SET_HSIZE_F(uiSizeH) | ALPS_SET_VSIZE_F(uiSizeV);

        if(uiSizeH == 720 && uiSizeV == 288)
        {
            uiTempReg2 = ALPS_ODV1_720_288 | ALPS_ODV2_720_288 | ALPS_ODV_720_288;
        }
        else if(uiSizeH == 720 && uiSizeV == 240)
        {
            uiTempReg2 = ALPS_ODV1_720_240 | ALPS_ODV2_720_240 | ALPS_ODV_720_240;
        }
        else if(uiSizeH == 640 && uiSizeV == 288)
        {
            uiTempReg2 = ALPS_ODV1_640_288 | ALPS_ODV2_640_288 | ALPS_ODV_640_288;
        }
        else if(uiSizeH == 640 && uiSizeV == 240)
        {
            uiTempReg2 = ALPS_ODV1_640_240 | ALPS_ODV2_640_240 | ALPS_ODV_640_240;
        }
        else
        {
            uiTempReg2 = ALPS_ODV1_720_240 | ALPS_ODV2_720_240 | ALPS_ODV_FREE_SIZE;
        }

        if(uiValueSize>0 && uiValueSize<0x400) // 0x400 = VELUT1_00 size
        {
            pStartALPSLUT1 = (unsigned int*)(g_pDispConReg->VELUT1_00);
            pStartALPSLUT2 = (unsigned int*)(g_pDispConReg->VELUT2_000);
            for(i=0;i<uiValueSize;i++)
            {
                *pStartALPSLUT1 = (unsigned int)( (pLow_B_LUT[i]<<16) | (pLow_G_LUT[i]<<8) | (pLow_R_LUT[i]<<0) );
                *pStartALPSLUT2 = (unsigned int)( (pHigh_B_LUT[i]<<16) | (pHigh_G_LUT[i]<<8) | (pHigh_R_LUT[i]<<0) ) ;
                DISP_INF((_T("[DISP] %s() : VELUT1_00(0x%x) = 0x%x, VELUT2_0000(0x%x) = 0x%x\n\rpLow_B_LUT[%d]= %d\r\n"),
                    _T(__FUNCTION__), pStartALPSLUT1, *pStartALPSLUT1,
                    pStartALPSLUT2, *pStartALPSLUT2, i, pLow_B_LUT[i]));
                pStartALPSLUT1++;
                pStartALPSLUT2++;
            }
        }
        else
        {
            DISP_ERR((_T("[DISP:ERR] %s: invalid lut value size = %d\n\r"), _T(__FUNCTION__), uiValueSize));
        }

        g_pDispConReg->VESFRCON0 = uiTempReg0;
        g_pDispConReg->VESFRCON1 = uiTempReg1 | 0x10; // ?
        g_pDispConReg->VESFRCON2 = uiTempReg2;

        g_pDispConReg->VIDCON3 |= VE_ENABLE;

        DISP_INF((_T("[DISP] %s() : 0x%x,0x%x,0x%x,0x%x\n\r"), _T(__FUNCTION__),
            g_pDispConReg->VIDCON3, g_pDispConReg->VESFRCON0,
            g_pDispConReg->VESFRCON1, g_pDispConReg->VESFRCON2));
    }

    DISP_MSG((_T("[DISP]--%s: %d\n\r"), _T(__FUNCTION__), error));

    return error;
}


DISP_ERROR Disp_set_frame_interrupt(DISP_FRAME_INTERRUPT FrameInt)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_frame_interrupt(%d)\n\r"), FrameInt));

    g_pDispConReg->VIDINTCON0 &= ~FRAMEINT_MASK;    // Clear Al Frame Interrupt Bit

    switch(FrameInt)
    {
    case DISP_FRMINT_BACKPORCH:
        g_pDispConReg->VIDINTCON0 |= FRAMESEL0_BACK;
        break;
    case DISP_FRMINT_VSYNC:
        g_pDispConReg->VIDINTCON0 |= FRAMESEL0_VSYNC;
        break;
    case DISP_FRMINT_ACTIVE:
        g_pDispConReg->VIDINTCON0 |= FRAMESEL0_ACTIVE;
        break;
    case DISP_FRMINT_FRONTPORCH:
        g_pDispConReg->VIDINTCON0 |= FRAMESEL0_FRONT;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_frame_interrupt() : Unknown Frame Interrupt [%d]\n\r"), FrameInt));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_set_frame_interrupt() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_enable_frame_interrupt(void)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_enable_frame_interrupt()\n\r")));

    g_pDispConReg->VIDINTCON0 |= (INTFRMEN_ENABLE | INTEN_ENABLE);

    DISP_MSG((_T("[DISP]--Disp_enable_frame_interrupt() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_disable_frame_interrupt(void)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_disable_frame_interrupt()\n\r")));

    if (g_pDispConReg->VIDINTCON0 & INTFIFOEN_ENABLE)
    {
        // If FIFO Interrupt is enabled, Clear only Frame Interrupt Enable Bit
        g_pDispConReg->VIDINTCON0 &= ~INTFRMEN_ENABLE;
    }
    else
    {
        g_pDispConReg->VIDINTCON0 &= ~(INTFRMEN_ENABLE | INTEN_ENABLE);
    }

    DISP_MSG((_T("[DISP]--Disp_disable_frame_interrupt() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_set_fifo_interrupt(DISP_WINDOW Win, DISP_FIFO_INTERRUPT_LEVEL Level)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_set_fifo_interrupt(%d, %d)\n\r"), Win, Level));

    g_pDispConReg->VIDINTCON0 &= ~FIFOINT_MASK;    // Clear Al FIFO Interrupt Bit

    switch(Win)
    {
    case DISP_WIN0:
        g_pDispConReg->VIDINTCON0 |= FIFOSEL_WIN0;
        break;
    case DISP_WIN1:
        g_pDispConReg->VIDINTCON0 |= FIFOSEL_WIN1;
        break;
    case DISP_WIN2:
        g_pDispConReg->VIDINTCON0 |= FIFOSEL_WIN2;
        break;
    case DISP_WIN3:
        g_pDispConReg->VIDINTCON0 |= FIFOSEL_WIN3;
        break;
    case DISP_WIN4:
        g_pDispConReg->VIDINTCON0 |= FIFOSEL_WIN4;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_fifo_interrupt() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    switch(Level)
    {
    case DISP_FIFO_0_25:
        g_pDispConReg->VIDINTCON0 |= FIFOLEVEL_25;
        break;
    case DISP_FIFO_0_50:
        g_pDispConReg->VIDINTCON0 |= FIFOLEVEL_50;
        break;
    case DISP_FIFO_0_75:
        g_pDispConReg->VIDINTCON0 |= FIFOLEVEL_75;
        break;
    case DISP_FIFO_EMPTY:
        g_pDispConReg->VIDINTCON0 |= FIFOLEVEL_EMPTY;
        break;
    case DISP_FIFO_FULL:
        g_pDispConReg->VIDINTCON0 |= FIFOLEVEL_FULL;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_fifo_interrupt() : Unknown FIFO Level [%d]\n\r"), Level));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_set_fifo_interrupt() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_enable_fifo_interrupt(void)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_enable_fifo_interrupt()\n\r")));

    g_pDispConReg->VIDINTCON0 |= (INTFIFOEN_ENABLE | INTEN_ENABLE);

    DISP_MSG((_T("[DISP]--Disp_enable_fifo_interrupt() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_disable_fifo_interrupt(void)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_disable_fifo_interrupt()\n\r")));

    if (g_pDispConReg->VIDINTCON0 & INTFRMEN_ENABLE)
    {
        // If Frame Interrupt is enabled, Clear only FIFO Interrupt Enable Bit
        g_pDispConReg->VIDINTCON0 &= ~INTFIFOEN_ENABLE;
    }
    else
    {
        g_pDispConReg->VIDINTCON0 &= ~(INTFIFOEN_ENABLE | INTEN_ENABLE);
    }

    DISP_MSG((_T("[DISP]--Disp_disable_fifo_interrupt() : %d\n\r"), error));

    return error;
}

DISP_INTERRUPT Disp_clear_interrupt_pending(void)
{
    DISP_INTERRUPT Interrupt = DISP_INT_NONE;

    // Check Interrupt Source and Clear Pending
    if (g_pDispConReg->VIDINTCON1 & INTFIFO_PEND)
    {
        g_pDispConReg->VIDINTCON1 = INTFIFO_PEND;
        Interrupt = DISP_INT_FIFO;
    }
    else if (g_pDispConReg->VIDINTCON1 & INTFRM_PEND)
    {
        g_pDispConReg->VIDINTCON1 = INTFRM_PEND;
        Interrupt = DISP_INT_FRAME;
    }
    else if (g_pDispConReg->VIDINTCON1 & INTSYSIF_PEND)
    {
        g_pDispConReg->VIDINTCON1 = INTSYSIF_PEND;
        Interrupt = DISP_INT_I80;
    }

    return Interrupt;
}

unsigned int Disp_get_line_count(void)
{
    unsigned int uiLineCnt;

    DISP_MSG((_T("[DISP]++Disp_get_line_count()\n\r")));

    uiLineCnt = ((g_pDispConReg->VIDCON1)>>16)&0x7ff;

    DISP_MSG((_T("[DISP]--Disp_get_line_count() : Line Count = %d\n\r"), uiLineCnt));

    return uiLineCnt;
}

DISP_VERTICAL_STATUS Disp_get_vertical_status(void)
{
    unsigned int status;

    DISP_MSG((_T("[DISP]++Disp_get_vertical_status()\n\r")));

    status = ((g_pDispConReg->VIDCON1)>>13)&0x3;

    DISP_MSG((_T("[DISP]--Disp_get_vertical_status() : Status = %d\n\r"), status));

    return (DISP_VERTICAL_STATUS)status;
}

DISP_WINDOW Disp_get_win_num_from_win_mode(DISP_WINDOW_MODE Mode)
{
    DISP_WINDOW win;

    switch(Mode)
    {
    case DISP_WIN0_DMA:
    case DISP_WIN0_CAMIF0_RGB:
    case DISP_WIN0_CAMIF0_YUV:
        win = DISP_WIN0;
        break;
    case DISP_WIN1_DMA:
    case DISP_WIN1_CAMIF1_RGB:
    case DISP_WIN1_CAMIF1_YUV:
    case DISP_WIN1_VP_YUV:
        win = DISP_WIN1;
        break;
    case DISP_WIN2_DMA:
    case DISP_WIN2_CAMIF2_RGB:
    case DISP_WIN2_CAMIF2_YUV:
        win = DISP_WIN2;
        break;
    case DISP_WIN3_DMA:
        win = DISP_WIN3;
        break;
    case DISP_WIN4_DMA:
        win = DISP_WIN4;
        break;
    }

    return win;
}

static DISP_ERROR Disp_initialize_port_RGBIF(DISP_RGBIFOUT_MODE RGBOutMode)
{
	DISP_ERROR error = DISP_SUCCESS;

	// GPB4->SPI1_CLK
	// GPB5->SPI1_nSS
	// GPB6->SPI1_MISO
	// GPB7->SPI1_MOSI
	// But these pins will be configured as ouput to emulate the SPI behavior
	Set_PinFunction(g_pGPIOReg, DISPLAY_CS);
	Set_PinPullUD(g_pGPIOReg, DISPLAY_CS, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, DISPLAY_CLK);
	Set_PinPullUD(g_pGPIOReg, DISPLAY_CLK, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, DISPLAY_SO);
	Set_PinPullUD(g_pGPIOReg, DISPLAY_SO, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, DISPLAY_SI);
	Set_PinPullUD(g_pGPIOReg, DISPLAY_SI, sgip_PULL_DISABLE);


	DISP_MSG((_T("[DISP]++Disp_initialize_port_RGBIF(%d)\n\r"), RGBOutMode));

	// GPF0_4~7->VD0~VD3
	// GPF1_0~7->VD4~VD11
	// GPF2_0~7->VD12~VD19
	// GPF3_0~3->VD20~VD23
	switch(RGBOutMode)
	{
	case DISP_24BIT_RGB888_P:
		Set_PinPullUD(g_pGPIOReg, LCD_D0, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D0);
		Set_PinPullUD(g_pGPIOReg, LCD_D1, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D1);

		Set_PinPullUD(g_pGPIOReg, LCD_D8, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D8);
		Set_PinPullUD(g_pGPIOReg, LCD_D9, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D9);

		Set_PinPullUD(g_pGPIOReg, LCD_D16, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D16);
		Set_PinPullUD(g_pGPIOReg, LCD_D17, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D17);

	case DISP_18BIT_RGB666_P:
		Set_PinPullUD(g_pGPIOReg, LCD_D2, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D2);

		Set_PinPullUD(g_pGPIOReg, LCD_D18, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D18);

	case DISP_16BIT_RGB565_P:
		Set_PinPullUD(g_pGPIOReg, LCD_D3, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D3);
		Set_PinPullUD(g_pGPIOReg, LCD_D4, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D4);
		Set_PinPullUD(g_pGPIOReg, LCD_D5, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D5);
		Set_PinPullUD(g_pGPIOReg, LCD_D6, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D6);
		Set_PinPullUD(g_pGPIOReg, LCD_D7, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D7);

		Set_PinPullUD(g_pGPIOReg, LCD_D10, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D10);
		Set_PinPullUD(g_pGPIOReg, LCD_D11, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D11);
		Set_PinPullUD(g_pGPIOReg, LCD_D12, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D12);
		Set_PinPullUD(g_pGPIOReg, LCD_D13, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D13);
		Set_PinPullUD(g_pGPIOReg, LCD_D14, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D14);
		Set_PinPullUD(g_pGPIOReg, LCD_D15, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D15);

		Set_PinPullUD(g_pGPIOReg, LCD_D19, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D19);
		Set_PinPullUD(g_pGPIOReg, LCD_D20, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D20);
		Set_PinPullUD(g_pGPIOReg, LCD_D21, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D21);
		Set_PinPullUD(g_pGPIOReg, LCD_D22, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D22);
		Set_PinPullUD(g_pGPIOReg, LCD_D23, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D23);
		break;

	case DISP_24BIT_RGB888_S:
		Set_PinPullUD(g_pGPIOReg, LCD_D16, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D16);
		Set_PinPullUD(g_pGPIOReg, LCD_D17, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D17);
	case DISP_18BIT_RGB666_S:
		Set_PinPullUD(g_pGPIOReg, LCD_D18, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D18);
		Set_PinPullUD(g_pGPIOReg, LCD_D19, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D19);
		Set_PinPullUD(g_pGPIOReg, LCD_D20, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D20);
		Set_PinPullUD(g_pGPIOReg, LCD_D21, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D21);
		Set_PinPullUD(g_pGPIOReg, LCD_D22, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D22);
		Set_PinPullUD(g_pGPIOReg, LCD_D23, sgip_PULL_DISABLE);
		Set_PinFunction(g_pGPIOReg, LCD_D23);
		break;

	default:
		DISP_ERR((_T("[DISP:ERR] Disp_initialize_port_RGBIF() : Unsupported RGB I/F Output Mode [%d]\n\r"), RGBOutMode));
		error = DISP_ERROR_NOT_INITIALIZED;
		break;
	}

	// GPF0_0->HSYNC
	// GPF0_1->VSYNC
	// GPF0_2->VDEN
	// GPF0_3->VCLK
	// Control Signal (HSync, VSync, VDEN, VClk)
	Set_PinPullUD(g_pGPIOReg, LCD_HSYNC, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, LCD_HSYNC);
	Set_PinPullUD(g_pGPIOReg, LCD_VSYNC, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, LCD_VSYNC);
	Set_PinPullUD(g_pGPIOReg, LCD_VDEN, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, LCD_VDEN);
	Set_PinPullUD(g_pGPIOReg, LCD_VCLK, sgip_PULL_DISABLE);
	Set_PinFunction(g_pGPIOReg, LCD_VCLK);

	Set_PinDrv(g_pGPIOReg, LCD_HSYNC, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_VSYNC, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_VDEN, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_VCLK, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D0, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D1, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D2, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D3, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D4, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D5, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D6,  sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D7,  sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D8, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D9, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D10, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D11, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D12, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D13, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D14, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D15, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D16, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D17, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D18, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D19, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D20, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D21, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D22, sgip_DRV_4x);
	Set_PinDrv(g_pGPIOReg, LCD_D23, sgip_DRV_4x);

	DISP_MSG((_T("[DISP]--Disp_initialize_port_RGBIF() : %d\n\r"), error));

	return error;

}

static DISP_ERROR
Disp_initialize_RGBIF
(void)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int VCLKDivider;

    DISP_MSG((_T("[DISP]++Disp_initialize_RGBIF()\n\r")));

    if (Disp_get_vclk_direction_divider(\
                                    g_DevInfoDisp.RGBINFO.VCLK_Source,  \
                                    &g_DevInfoDisp.RGBINFO.VCLK_Direction, \
                                    &VCLKDivider))
    {
        g_pDispConReg->VIDCON0 = \
                        g_DevInfoDisp.RGBINFO.MIPIDSIen | \
                        PROGRESSIVE | VIDOUT_RGBIF | \
                        SET_RGBSPSEL(g_DevInfoDisp.RGBINFO.RGBOutMode) | \
                        CLKVALUP_ALWAYS | CLKVAL_F(VCLKDivider) | \
                        g_DevInfoDisp.RGBINFO.VCLK_Gate | \
                        g_DevInfoDisp.RGBINFO.VCLK_Direction |
                        g_DevInfoDisp.RGBINFO.VCLK_Source | \
                        ENVID_DISABLE | ENVID_F_DISABLE;

#if (S5PV210_EVT>0)
        g_pDispConReg->VIDCON1 = g_DevInfoDisp.RGBINFO.VCLK_Polarity | \
                                    g_DevInfoDisp.RGBINFO.HSYNC_Polarity | \
                                    g_DevInfoDisp.RGBINFO.VSYNC_Polarity | \
                                    g_DevInfoDisp.RGBINFO.VDEN_Polarity | \
                                    VCLK_RUNNING_DISABLE_VDEN;
#else
        g_pDispConReg->VIDCON1 = g_DevInfoDisp.RGBINFO.VCLK_Polarity | \
                                    g_DevInfoDisp.RGBINFO.HSYNC_Polarity | \
                                    g_DevInfoDisp.RGBINFO.VSYNC_Polarity | \
                                    g_DevInfoDisp.RGBINFO.VDEN_Polarity;
#endif

        g_pDispConReg->VIDCON2 = \
            (g_pDispConReg->VIDCON2 & ~(RGB_SKIP_MASK|RGB_DUMMY_MASK|\
                                        RGB_ORDER_E_MASK|RGB_ORDER_O_MASK)) | \
            g_DevInfoDisp.RGBINFO.RGBSkipEn | \
            g_DevInfoDisp.RGBINFO.RGBDummy | \
            g_DevInfoDisp.RGBINFO.RGBEvenOrder | \
            g_DevInfoDisp.RGBINFO.RGBOddOrder;

        g_pDispConReg->VIDCON3 = g_DevInfoDisp.RGBINFO.PixelCompensation;

        g_pDispConReg->VIDTCON0 = VBPDE(1) | \
                                VBPD(g_DevInfoDisp.RGBINFO.VBPD_Value) | \
                                VFPD(g_DevInfoDisp.RGBINFO.VFPD_Value) | \
                                VSPW(g_DevInfoDisp.RGBINFO.VSPW_Value);

        g_pDispConReg->VIDTCON1 = VFPDE(1) | \
                                HBPD(g_DevInfoDisp.RGBINFO.HBPD_Value) | \
                                HFPD(g_DevInfoDisp.RGBINFO.HFPD_Value) | \
                                HSPW(g_DevInfoDisp.RGBINFO.HSPW_Value);

        g_pDispConReg->VIDTCON2 =
                                LINEVAL(g_DevInfoDisp.RGBINFO.uiHeight) | \
                                HOZVAL(g_DevInfoDisp.RGBINFO.uiWidth);
    }
    else
    {
        DISP_ERR((_T("[DISP:ERR] Disp_initialize_RGBIF() : Clock Source Decision Failed\n\r")));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
    }

    DISP_MSG((_T("[DISP]--Disp_initialize_RGBIF() : %d\n\r"), error));

    return error;
}

DISP_ERROR Disp_enable_WB_interface(unsigned int uiSkipFrm)
{
    DISP_ERROR error = DISP_SUCCESS;
    UINT32 uiTempVidcon2 = g_pDispConReg->VIDCON2;

    DISP_MSG((_T("[DISP]++%s()\n\r"),_T(__FUNCTION__)));

    uiTempVidcon2 |= WRITEBACK_ENABLE | TVFORMATSEL1_YUV444;
    uiTempVidcon2 = (uiTempVidcon2 & ~(TV_FRAM_SKIP(0x1f))) | \
                    TV_FRAM_SKIP(uiSkipFrm);

    if(error == DISP_SUCCESS)
    {
        g_pDispConReg->VIDCON0 |= VIDOUT_WB_EN;
        g_pDispConReg->VIDCON2 = uiTempVidcon2;
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"),_T(__FUNCTION__) , error));

    return error;
}

DISP_ERROR
Disp_disable_WB_interface
(DISP_VIDOUT_PATH eVideoOutPath)
{
    DISP_ERROR error = DISP_SUCCESS;
    UINT32 uiTempVidcon2 = g_pDispConReg->VIDCON2;

    DISP_MSG((_T("[DISP]++%s()\n\r"),_T(__FUNCTION__)));

    uiTempVidcon2 &= ~(WRITEBACK_ENABLE | TV_FRAM_SKIP(0x1f));

    switch(eVideoOutPath)
    {
    case DISP_OUTPUT_SINGLE_BYPASS:
    case DISP_OUTPUT_DUAL_BYPASS:
        uiTempVidcon2 &= ~TVFORMATSEL1_YUV444;
        break;
    case DISP_OUTPUT_MIE:
        DISP_ERR((_T("[DISP:ERR] %s() : Cannot support[%d]\n\r"),_T(__FUNCTION__), eVideoOutPath));
        error = DISP_ERROR_NOT_IMPLEMENTED;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] %s() : Unknown output mode [%d]\n\r"),_T(__FUNCTION__), eVideoOutPath));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    if(error == DISP_SUCCESS)
    {
        g_pDispConReg->VIDCON0 &= ~VIDOUT_WB_EN;
        g_pDispConReg->VIDCON2 = uiTempVidcon2;
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"),_T(__FUNCTION__) , error));

    return error;
}

DISP_ERROR
Disp_set_window_qos
(DISP_WINDOW eWinNum,
 BOOL bGateEn,
 unsigned char ucFifoLevel)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++%s(%d,%d,%d)\n\r"), _T(__FUNCTION__),eWinNum,bGateEn,ucFifoLevel));

    switch(eWinNum)
    {
    case DISP_WIN0:
        g_pDispConReg->W0RTQOSCON = SET_QOS_FIFOLEVEL(ucFifoLevel) | \
                                        ((bGateEn)? QOS_GATE_EN: QOS_GATE_DIS);
        DISP_INF((_T("[DISP]%s() : W0RTQOSCON = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->W0RTQOSCON));
        break;
    case DISP_WIN1:
        g_pDispConReg->W1RTQOSCON = SET_QOS_FIFOLEVEL(ucFifoLevel) | \
                                        ((bGateEn)? QOS_GATE_EN: QOS_GATE_DIS);
        DISP_INF((_T("[DISP]%s() : W1RTQOSCON = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->W1RTQOSCON));
        break;
    case DISP_WIN2:
        g_pDispConReg->W2RTQOSCON = SET_QOS_FIFOLEVEL(ucFifoLevel) | \
                                        ((bGateEn)? QOS_GATE_EN: QOS_GATE_DIS);
        DISP_INF((_T("[DISP]%s() : W2RTQOSCON = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->W2RTQOSCON));
        break;
    case DISP_WIN3:
        g_pDispConReg->W3RTQOSCON = SET_QOS_FIFOLEVEL(ucFifoLevel) | \
                                        ((bGateEn)? QOS_GATE_EN: QOS_GATE_DIS);
        DISP_INF((_T("[DISP]%s() : W3RTQOSCON = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->W3RTQOSCON));
        break;
    case DISP_WIN4:
        g_pDispConReg->W4RTQOSCON = SET_QOS_FIFOLEVEL(ucFifoLevel) | \
                                        ((bGateEn)? QOS_GATE_EN: QOS_GATE_DIS);
        DISP_INF((_T("[DISP]%s() : W4RTQOSCON = 0x%x\n\r"), _T(__FUNCTION__), g_pDispConReg->W4RTQOSCON));
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_set_window_qos() : Unknown Window Number [%d]\n\r"), eWinNum));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"), _T(__FUNCTION__), error));

    return error;
}

static DISP_ERROR Disp_window0_initialize(DISP_WINDOW_MODE Mode, DISP_BPP_MODE BPPMode, unsigned int uiWidth, unsigned int uiHeight, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiTempReg = g_pDispConReg->WINCON0 & (BLEND_MULTIPLIED | ALPHASEL_ALPHA1);

    DISP_MSG((_T("[DISP]++Disp_window0_initialize(%d, %d, %d, %d, %d, %d)\n\r"), Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY));

    switch(Mode)
    {
    case DISP_WIN0_DMA:
        g_Win0Config.LocalPathEnable = LOCAL_PATH_DISABLE;
        //g_Win0Config.LocaPathSourceFormat;    // Don't care when Local Path is disabled
        break;
    case DISP_WIN0_CAMIF0_RGB:
        g_Win0Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win0Config.LocaPathSourceFormat = LOCAL_IN_RGB888;
        break;
    case DISP_WIN0_CAMIF0_YUV:
        g_Win0Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win0Config.LocaPathSourceFormat = LOCAL_IN_YUV444;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window0_initialize() : Unsupported Window Mode [%d]\n\r"), Mode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_Win0Config.BitSwapping = BITSWP_DISABLE;
    g_Win0Config.ByteSwapping = BYTSWP_DISABLE;
    g_Win0Config.HalfWordSwapping = HAWSWP_DISABLE;
    g_Win0Config.WordSwapping = WSWP_DISABLE;

    switch(BPPMode)
    {
    //case DISP_1BPP:
    //case DISP_2BPP:
    //case DISP_4BPP:
    //case DISP_8BPP_PAL:
    //case DISP_8BPP_NOPAL:
    //case DISP_18BPP_A665:
    //case DISP_19BPP_A666:
    //case DISP_24BPP_A887:
    //case DISP_25BPP_A888:
    case DISP_16BPP_I555:
    case DISP_16BPP_A555:
    case DISP_16BPP_565:
        g_Win0Config.uiPageWidth = uiWidth*2;    // 2 byte per pixel
        g_Win0Config.HalfWordSwapping = HAWSWP_ENABLE;    // 16BPP need Halfword Swapping
        break;
    case DISP_18BPP_666:
    case DISP_24BPP_888:
    case DISP_32BPP_A888:
        g_Win0Config.uiPageWidth = uiWidth*4;    // 4 byte per pixel

        if(g_Win0Config.LocalPathEnable != LOCAL_PATH_ENABLE)
        {
            g_Win0Config.WordSwapping = WSWP_ENABLE;
        }
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window0_initialize() : Unsupported BPP Mode [%d]\n\r"), BPPMode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_Win0Config.BPP_Mode = BPPMode;
    g_Win0Config.BufferSelect = BUFSEL_BUF0;
    g_Win0Config.BufferAutoControl = BUFAUTO_DISABLE;
    g_Win0Config.uiWidth = uiWidth;
    g_Win0Config.uiHeight = uiHeight;
    g_Win0Config.uiOffsetX = uiOffsetX;
    g_Win0Config.uiOffsetY = uiOffsetY;

    if (g_Win0Config.LocalPathEnable == LOCAL_PATH_ENABLE)
    {
        g_Win0Config.BurstLength = BURSTLEN_4WORD;    // 4 Words Burst
    }
    else if ((g_Win0Config.uiPageWidth%64) == 0)        // 16 words burst case
    {
        g_Win0Config.BurstLength = BURSTLEN_16WORD;
    }
    else if ((g_Win0Config.uiPageWidth%32) == 0)    // 8 words burst case
    {
        g_Win0Config.BurstLength = BURSTLEN_8WORD;
    }
    else if ((g_Win0Config.uiPageWidth%16) == 0)    // 4 words burst case
    {
        g_Win0Config.BurstLength = BURSTLEN_4WORD;
    }
    else
    {
        DISP_ERR((_T("[DISP:ERR] Disp_window0_initialize() : uiPageWidth[%d] is not Aligned with Minimum Burst Size (4 Words)\n\r"), g_Win0Config.uiPageWidth));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_pDispConReg->WINCON0 = CSC_WIDE_RANGE | //g_Win0Config.LocalPathEnable |
                            g_Win0Config.BufferSelect | g_Win0Config.BufferAutoControl | g_Win0Config.BitSwapping |
                            g_Win0Config.ByteSwapping | g_Win0Config.HalfWordSwapping |g_Win0Config.WordSwapping |
                            g_Win0Config.LocaPathSourceFormat |g_Win0Config.BurstLength |
                            BPPMODE_F(g_Win0Config.BPP_Mode) | uiTempReg;

    g_pDispConReg->VIDOSD0A = OSD_LEFTTOPX_F(g_Win0Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win0Config.uiOffsetY);

    g_pDispConReg->VIDOSD0B = OSD_RIGHTBOTX_F(g_Win0Config.uiWidth+g_Win0Config.uiOffsetX-1) |
                                OSD_RIGHTBOTY_F(g_Win0Config.uiHeight+g_Win0Config.uiOffsetY-1);

    g_pDispConReg->VIDOSD0C = OSD_SIZE(g_Win0Config.uiWidth*g_Win0Config.uiHeight);

    DISP_MSG((_T("[DISP]--Disp_window0_initialize() : %d\n\r"), error));

    return error;
}

static DISP_ERROR Disp_window1_initialize(DISP_WINDOW_MODE Mode, DISP_BPP_MODE BPPMode, unsigned int uiWidth, unsigned int uiHeight, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiTempReg = g_pDispConReg->WINCON1 & (BLEND_MULTIPLIED | ALPHASEL_ALPHA1);

    DISP_MSG((_T("[DISP]++Disp_window1_initialize(%d, %d, %d, %d, %d, %d)\n\r"), Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY));

    switch(Mode)
    {
    case DISP_WIN1_DMA:
        g_Win1Config.LocalPathEnable = LOCAL_PATH_DISABLE;
        //g_Win1Config.LocalPathSelect;            // Don't care when Local Path is disabled
        //g_Win1Config.LocaPathSourceFormat;    // Don't care when Local Path is disabled
        break;
    case DISP_WIN1_CAMIF1_RGB:
        g_Win1Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win1Config.LocalPathSelect = LOCALSEL_CAMIF1;
        g_Win1Config.LocaPathSourceFormat = LOCAL_IN_RGB888;
        break;
    case DISP_WIN1_CAMIF1_YUV:
        g_Win1Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win1Config.LocalPathSelect = LOCALSEL_CAMIF1;
        g_Win1Config.LocaPathSourceFormat = LOCAL_IN_YUV444;
        break;
    case DISP_WIN1_VP_YUV:
        g_Win1Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win1Config.LocalPathSelect = LOCALSEL_VP;
        g_Win1Config.LocaPathSourceFormat = LOCAL_IN_YUV444;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window1_initialize() : Unsupported Window Mode [%d]\n\r"), Mode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    g_Win1Config.BitSwapping = BITSWP_DISABLE;
    g_Win1Config.ByteSwapping = BYTSWP_DISABLE;
    g_Win1Config.HalfWordSwapping = HAWSWP_DISABLE;
    g_Win1Config.WordSwapping = WSWP_DISABLE;

    switch(BPPMode)
    {
    //case DISP_1BPP:
    //case DISP_2BPP:
    //case DISP_4BPP:
    //case DISP_8BPP_PAL:
    //case DISP_8BPP_NOPAL:
    //case DISP_18BPP_A665:
    //case DISP_19BPP_A666:
    //case DISP_24BPP_A887:
    //case DISP_25BPP_A888:
    case DISP_16BPP_I555:
    case DISP_16BPP_A555:
    case DISP_16BPP_565:
        g_Win1Config.uiPageWidth = uiWidth*2;    // 2 byte per pixel
        g_Win1Config.HalfWordSwapping = HAWSWP_ENABLE;    // 16BPP need Halfword Swapping
        break;
    case DISP_18BPP_666:
    case DISP_24BPP_888:
    case DISP_32BPP_A888:
        g_Win1Config.uiPageWidth = uiWidth*4;    // 4 byte per pixel
        if(g_Win1Config.LocalPathEnable != LOCAL_PATH_ENABLE)
        {
            g_Win1Config.WordSwapping = WSWP_ENABLE;
        }
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window1_initialize() : Unsupported BPP Mode [%d]\n\r"), BPPMode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    g_Win1Config.BPP_Mode = BPPMode;
    g_Win1Config.BufferSelect = BUFSEL_BUF0;
    g_Win1Config.BufferAutoControl = BUFAUTO_DISABLE;
    g_Win1Config.uiWidth = uiWidth;
    g_Win1Config.uiHeight = uiHeight;
    g_Win1Config.uiOffsetX = uiOffsetX;
    g_Win1Config.uiOffsetY = uiOffsetY;

    if (g_Win1Config.LocalPathEnable == LOCAL_PATH_ENABLE)
    {
        g_Win1Config.BurstLength = BURSTLEN_4WORD;    // 4 Words Burst
    }
    else if ((g_Win1Config.uiPageWidth%64) == 0)        // 16 words burst case
    {
        g_Win1Config.BurstLength = BURSTLEN_16WORD;
    }
    else if ((g_Win1Config.uiPageWidth%32) == 0)    // 8 words burst case
    {
        g_Win1Config.BurstLength = BURSTLEN_8WORD;
    }
    else if ((g_Win1Config.uiPageWidth%16) == 0)    // 4 words burst case
    {
        g_Win1Config.BurstLength = BURSTLEN_4WORD;
    }
    else
    {
        DISP_ERR((_T("[DISP:ERR] Disp_window1_initialize() : uiPageWidth is not Word Aligned [%d]\n\r"), g_Win0Config.uiPageWidth));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_pDispConReg->WINCON1 = CSC_WIDE_RANGE | g_Win1Config.LocalPathSelect | //g_Win1Config.LocalPathEnable |
                            g_Win1Config.BufferSelect | g_Win1Config.BufferAutoControl | g_Win1Config.BitSwapping |
                            g_Win1Config.ByteSwapping | g_Win1Config.HalfWordSwapping |g_Win1Config.WordSwapping |
                            g_Win1Config.LocaPathSourceFormat |g_Win1Config.BurstLength | BLEND_PER_PLANE | BPPMODE_F(g_Win1Config.BPP_Mode) |
                            ALPHASEL_ALPHA0 | uiTempReg;

    g_pDispConReg->VIDOSD1A = OSD_LEFTTOPX_F(g_Win1Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win1Config.uiOffsetY);

    g_pDispConReg->VIDOSD1B = OSD_RIGHTBOTX_F(g_Win1Config.uiWidth+g_Win1Config.uiOffsetX-1) |
                                OSD_RIGHTBOTY_F(g_Win1Config.uiHeight+g_Win1Config.uiOffsetY-1);

    g_pDispConReg->VIDOSD1C = 0x0;

    g_pDispConReg->VIDOSD1D = OSD_SIZE(g_Win1Config.uiWidth*g_Win1Config.uiHeight);

    DISP_MSG((_T("[DISP]--Disp_window1_initialize() : %d\n\r"), error));

    return error;
}

static DISP_ERROR Disp_window2_initialize(DISP_WINDOW_MODE Mode, DISP_BPP_MODE BPPMode, unsigned int uiWidth, unsigned int uiHeight, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiTempReg = g_pDispConReg->WINCON2 & (BLEND_MULTIPLIED | ALPHASEL_ALPHA1);

    DISP_MSG((_T("[DISP]++Disp_window2_initialize(%d, %d, %d, %d, %d, %d)\n\r"), Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY));

    switch(Mode)
    {
    case DISP_WIN2_DMA:
        g_Win2Config.LocalPathEnable = LOCAL_PATH_DISABLE;
        //g_Win2Config.LocalPathSelect;            // Don't care when Local Path is disabled
        //g_Win2Config.LocaPathSourceFormat;    // Don't care when Local Path is disabled
        break;
    case DISP_WIN2_CAMIF2_RGB:
        g_Win2Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win2Config.LocaPathSourceFormat = LOCAL_IN_RGB888;
        break;
    case DISP_WIN2_CAMIF2_YUV:
        g_Win2Config.LocalPathEnable = LOCAL_PATH_ENABLE;
        g_Win2Config.LocaPathSourceFormat = LOCAL_IN_YUV444;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window2_initialize() : Unsupported Window Mode [%d]\n\r"), Mode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_Win2Config.BitSwapping = BITSWP_DISABLE;
    g_Win2Config.ByteSwapping = BYTSWP_DISABLE;
    g_Win2Config.HalfWordSwapping = HAWSWP_DISABLE;
    g_Win2Config.WordSwapping = WSWP_DISABLE;

    switch(BPPMode)
    {
    //case DISP_1BPP:
    //case DISP_2BPP:
    //case DISP_4BPP:
    //case DISP_8BPP_PAL:
    //case DISP_8BPP_NOPAL:
    //case DISP_18BPP_A665:
    //case DISP_19BPP_A666:
    //case DISP_24BPP_A887:
    //case DISP_25BPP_A888:
    case DISP_16BPP_I555:
    case DISP_16BPP_A555:
    case DISP_16BPP_565:
        g_Win2Config.uiPageWidth = uiWidth*2;    // 2 byte per pixel
        g_Win2Config.HalfWordSwapping = HAWSWP_ENABLE;    // 16BPP need Halfword Swapping
        break;
    case DISP_18BPP_666:
    case DISP_24BPP_888:
    case DISP_32BPP_A888:
        g_Win2Config.uiPageWidth = uiWidth*4;    // 4 byte per pixel

        if(g_Win2Config.LocalPathEnable != LOCAL_PATH_ENABLE)
        {
            g_Win2Config.WordSwapping = WSWP_ENABLE;
        }
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window2_initialize() : Unsupported BPP Mode [%d]\n\r"), BPPMode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_Win2Config.BPP_Mode = BPPMode;
    g_Win2Config.BufferSelect = BUFSEL_BUF0;
    g_Win2Config.BufferAutoControl = BUFAUTO_DISABLE;
    g_Win2Config.uiWidth = uiWidth;
    g_Win2Config.uiHeight = uiHeight;
    g_Win2Config.uiOffsetX = uiOffsetX;
    g_Win2Config.uiOffsetY = uiOffsetY;

    if (g_Win2Config.LocalPathEnable == LOCAL_PATH_ENABLE)
    {
        g_Win2Config.BurstLength = BURSTLEN_4WORD;    // 4 Words Burst
    }
    else if ((g_Win2Config.uiPageWidth%64) == 0)        // 16 words burst case
    {
        g_Win2Config.BurstLength = BURSTLEN_16WORD;
    }
    else if ((g_Win2Config.uiPageWidth%32) == 0)    // 8 words burst case
    {
        g_Win2Config.BurstLength = BURSTLEN_8WORD;
    }
    else if ((g_Win2Config.uiPageWidth%16) == 0)    // 4 words burst case
    {
        g_Win2Config.BurstLength = BURSTLEN_4WORD;
    }
    else
    {
        DISP_ERR((_T("[DISP:ERR] Disp_window2_initialize() : uiPageWidth is not Word Aligned [%d]\n\r"), g_Win0Config.uiPageWidth));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_pDispConReg->WINCON2 = CSC_WIDE_RANGE | g_Win2Config.LocalPathSelect | //g_Win2Config.LocalPathEnable |
                            g_Win2Config.BufferSelect | g_Win2Config.BufferAutoControl | g_Win2Config.BitSwapping |
                            g_Win2Config.ByteSwapping | g_Win2Config.HalfWordSwapping |g_Win2Config.WordSwapping | g_Win2Config.LocaPathSourceFormat |
                            g_Win2Config.BurstLength | BLEND_PER_PLANE | BPPMODE_F(g_Win2Config.BPP_Mode) |
                            ALPHASEL_ALPHA0 | uiTempReg;

    g_pDispConReg->VIDOSD2A = OSD_LEFTTOPX_F(g_Win2Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win2Config.uiOffsetY);

    g_pDispConReg->VIDOSD2B = OSD_RIGHTBOTX_F(g_Win2Config.uiWidth+g_Win2Config.uiOffsetX-1) |
                                OSD_RIGHTBOTY_F(g_Win2Config.uiHeight+g_Win2Config.uiOffsetY-1);

    g_pDispConReg->VIDOSD2C = 0x0;

    g_pDispConReg->VIDOSD2D = OSD_SIZE(g_Win2Config.uiWidth*g_Win2Config.uiHeight);

    DISP_MSG((_T("[DISP]--Disp_window2_initialize() : %d\n\r"), error));

    return error;
}

static DISP_ERROR Disp_window3_initialize(DISP_WINDOW_MODE Mode, DISP_BPP_MODE BPPMode, unsigned int uiWidth, unsigned int uiHeight, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiTempReg = g_pDispConReg->WINCON3 & (BLEND_MULTIPLIED | ALPHASEL_ALPHA1);

    DISP_MSG((_T("[DISP]++Disp_window3_initialize(%d, %d, %d, %d, %d, %d)\n\r"), Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY));

    g_Win3Config.BitSwapping = BITSWP_DISABLE;
    g_Win3Config.ByteSwapping = BYTSWP_DISABLE;
    g_Win3Config.HalfWordSwapping = HAWSWP_DISABLE;
    g_Win3Config.WordSwapping = WSWP_DISABLE;

    switch(BPPMode)
    {
    //case DISP_1BPP:
    //case DISP_2BPP:
    //case DISP_4BPP:
    //case DISP_8BPP_PAL:
    //case DISP_8BPP_NOPAL:
    //case DISP_18BPP_A665:
    //case DISP_19BPP_A666:
    //case DISP_24BPP_A887:
    //case DISP_25BPP_A888:
    case DISP_16BPP_I555:
    case DISP_16BPP_A555:
    case DISP_16BPP_565:
        g_Win3Config.uiPageWidth = uiWidth*2;    // 2 byte per pixel
        g_Win3Config.HalfWordSwapping = HAWSWP_ENABLE;    // 16BPP need Halfword Swapping
        break;
    case DISP_18BPP_666:
    case DISP_24BPP_888:
    case DISP_32BPP_A888:
        g_Win3Config.uiPageWidth = uiWidth*4;    // 4 byte per pixel
        g_Win3Config.WordSwapping = WSWP_ENABLE;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window3_initialize() : Unsupported BPP Mode [%d]\n\r"), BPPMode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_Win3Config.BPP_Mode = BPPMode;
    g_Win3Config.BufferSelect = BUFSEL_BUF0;
    g_Win3Config.BufferAutoControl = BUFAUTO_DISABLE;
    g_Win3Config.uiWidth = uiWidth;
    g_Win3Config.uiHeight = uiHeight;
    g_Win3Config.uiOffsetX = uiOffsetX;
    g_Win3Config.uiOffsetY = uiOffsetY;

    if (g_Win3Config.uiPageWidth%4)
    {
        DISP_ERR((_T("[DISP:ERR] Disp_window3_initialize() : uiPageWidth is not Word Aligned [%d]\n\r"), g_Win0Config.uiPageWidth));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }
    else if (g_Win3Config.uiPageWidth >= 16*4)
    {
        g_Win3Config.BurstLength = BURSTLEN_16WORD;
    }
    else if (g_Win3Config.uiPageWidth >= 8*4)
    {
        g_Win3Config.BurstLength = BURSTLEN_8WORD;
    }
    else
    {
        g_Win3Config.BurstLength = BURSTLEN_4WORD;
    }

    g_pDispConReg->WINCON3 = g_Win3Config.BufferSelect | g_Win3Config.BufferAutoControl | g_Win3Config.BitSwapping | g_Win3Config.ByteSwapping | g_Win3Config.HalfWordSwapping |
                            g_Win3Config.WordSwapping |g_Win3Config.BurstLength | BLEND_PER_PLANE | BPPMODE_F(g_Win3Config.BPP_Mode) |
                            ALPHASEL_ALPHA0 | uiTempReg;

    g_pDispConReg->VIDOSD3A = OSD_LEFTTOPX_F(g_Win3Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win3Config.uiOffsetY);

    g_pDispConReg->VIDOSD3B = OSD_RIGHTBOTX_F(g_Win3Config.uiWidth+g_Win3Config.uiOffsetX-1) |
                                OSD_RIGHTBOTY_F(g_Win3Config.uiHeight+g_Win3Config.uiOffsetY-1);

    g_pDispConReg->VIDOSD3C = 0x0;

    DISP_MSG((_T("[DISP]--Disp_window3_initialize() : %d\n\r"), error));

    return error;
}

static DISP_ERROR Disp_window4_initialize(DISP_WINDOW_MODE Mode, DISP_BPP_MODE BPPMode, unsigned int uiWidth, unsigned int uiHeight, unsigned int uiOffsetX, unsigned int uiOffsetY)
{
    DISP_ERROR error = DISP_SUCCESS;
    unsigned int uiTempReg = g_pDispConReg->WINCON4 & (BLEND_MULTIPLIED | ALPHASEL_ALPHA1);

    DISP_MSG((_T("[DISP]++Disp_window4_initialize(%d, %d, %d, %d, %d, %d)\n\r"), Mode, BPPMode, uiWidth, uiHeight, uiOffsetX, uiOffsetY));

    g_Win4Config.BitSwapping = BITSWP_DISABLE;
    g_Win4Config.ByteSwapping = BYTSWP_DISABLE;
    g_Win4Config.HalfWordSwapping = HAWSWP_DISABLE;
    g_Win4Config.WordSwapping = WSWP_DISABLE;

    switch(BPPMode)
    {
    //case DISP_1BPP:
    //case DISP_2BPP:
    //case DISP_4BPP:
    //case DISP_8BPP_PAL:
    //case DISP_8BPP_NOPAL:
    //case DISP_18BPP_A665:
    //case DISP_19BPP_A666:
    //case DISP_24BPP_A887:
    //case DISP_25BPP_A888:
    case DISP_16BPP_I555:
    case DISP_16BPP_A555:
    case DISP_16BPP_565:
        g_Win4Config.uiPageWidth = uiWidth*2;    // 2 byte per pixel
        g_Win4Config.HalfWordSwapping = HAWSWP_ENABLE;    // 16BPP need Halfword Swapping
        break;
    case DISP_18BPP_666:
    case DISP_24BPP_888:
    case DISP_32BPP_A888:
        g_Win4Config.uiPageWidth = uiWidth*4;    // 4 byte per pixel
        g_Win4Config.WordSwapping = WSWP_ENABLE;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window4_initialize() : Unsupported BPP Mode [%d]\n\r"), BPPMode));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }

    g_Win4Config.BPP_Mode = BPPMode;
    g_Win4Config.BufferSelect = BUFSEL_BUF0;
    g_Win4Config.BufferAutoControl = BUFAUTO_DISABLE;
    g_Win4Config.uiWidth = uiWidth;
    g_Win4Config.uiHeight = uiHeight;
    g_Win4Config.uiOffsetX = uiOffsetX;
    g_Win4Config.uiOffsetY = uiOffsetY;

    if (g_Win4Config.uiPageWidth%4)
    {
        DISP_ERR((_T("[DISP:ERR] Disp_window4_initialize() : uiPageWidth is not Word Aligned [%d]\n\r"), g_Win0Config.uiPageWidth));
        return DISP_ERROR_ILLEGAL_PARAMETER;
    }
    else if (g_Win4Config.uiPageWidth >= 16*4)
    {
        g_Win4Config.BurstLength = BURSTLEN_16WORD;
    }
    else if (g_Win4Config.uiPageWidth >= 8*4)
    {
        g_Win4Config.BurstLength = BURSTLEN_8WORD;
    }
    else
    {
        g_Win4Config.BurstLength = BURSTLEN_4WORD;
    }

    g_pDispConReg->WINCON4 = g_Win4Config.BufferSelect | g_Win4Config.BufferAutoControl | g_Win4Config.BitSwapping | g_Win4Config.ByteSwapping | g_Win4Config.HalfWordSwapping |
                            g_Win4Config.WordSwapping |g_Win4Config.BurstLength | BLEND_PER_PLANE | BPPMODE_F(g_Win4Config.BPP_Mode) |
                            ALPHASEL_ALPHA0 | uiTempReg;

    g_pDispConReg->VIDOSD4A = OSD_LEFTTOPX_F(g_Win4Config.uiOffsetX) | OSD_LEFTTOPY_F(g_Win4Config.uiOffsetY);

    g_pDispConReg->VIDOSD4B = OSD_RIGHTBOTX_F(g_Win4Config.uiWidth+g_Win4Config.uiOffsetX-1) |
                                OSD_RIGHTBOTY_F(g_Win4Config.uiHeight+g_Win4Config.uiOffsetY-1);

    g_pDispConReg->VIDOSD4C = 0x0;

    DISP_MSG((_T("[DISP]--Disp_window4_initialize() : %d\n\r"), error));

    return error;
}

static BOOL Disp_get_vclk_direction_divider(unsigned int CLKSrc, unsigned int *ClkDir, unsigned int *ClkDiv)
{
    DWORD dwVCLKSrc = 0;
    DWORD dwHozTime = 0;
    DWORD dwVerTime = 0;
    DWORD dwVCLK = 0;
    DWORD dwDiv = 0;
    double dTmpDiv;

    DISP_MSG((_T("[DISP]++Disp_get_vclk_direction_divider(%d)\n\r"), CLKSrc));

    if(!Disp_get_src_clock(CLKSrc, &dwVCLKSrc))
    {
        return FALSE;
    }

    switch(g_DevInfoDisp.RGBINFO.RGBOutMode)
    {
        case DISP_16BIT_RGB565_P:
        case DISP_18BIT_RGB666_P:
        case DISP_24BIT_RGB888_P:
            dwHozTime = g_DevInfoDisp.RGBINFO.uiWidth+ \
                        g_DevInfoDisp.RGBINFO.HBPD_Value+ \
                        g_DevInfoDisp.RGBINFO.HFPD_Value+ \
                        g_DevInfoDisp.RGBINFO.HSPW_Value;
            break;
        case DISP_18BIT_RGB666_S:
        case DISP_24BIT_RGB888_S:
            dwHozTime = g_DevInfoDisp.RGBINFO.uiWidth*3+ \
                        g_DevInfoDisp.RGBINFO.HBPD_Value+ \
                        g_DevInfoDisp.RGBINFO.HFPD_Value+ \
                        g_DevInfoDisp.RGBINFO.HSPW_Value;
            break;
    }
#if 0
    dwVerTime = g_DevInfoRGB.uiHeight+g_DevInfoRGB.VBPD_Value+g_DevInfoRGB.VFPD_Value+g_DevInfoRGB.VSPW_Value;

    dwVCLK = dwHozTime*dwVerTime*g_DevInfoRGB.Frame_Rate;

    dwDiv = dwVCLKSrc/dwVCLK;
#endif
    dwVerTime = g_DevInfoDisp.RGBINFO.uiHeight+ \
                g_DevInfoDisp.RGBINFO.VBPD_Value+ \
                g_DevInfoDisp.RGBINFO.VFPD_Value+ \
                g_DevInfoDisp.RGBINFO.VSPW_Value;

    dTmpDiv = dwVCLKSrc/ \
        (double)(dwHozTime*dwVerTime*g_DevInfoDisp.RGBINFO.Frame_Rate);
    dTmpDiv = (dTmpDiv+0.5)*10;
    dwDiv = (unsigned int)(dTmpDiv/10.0);

    DISP_INF((_T("[DISP:INF] VCLKSrc = %d, VCLK = %d, Div = %d\n\r"), dwVCLKSrc, dwVCLK, dwDiv));

    if (dwDiv < 1)
    {
        DISP_ERR((_T("[DISP:ERR] --Disp_get_vclk_direction_divider() : VCLK Source is Too Slow\n\r")));
        return FALSE;
    }
    else     if (dwDiv == 1)
    {
        // No Divide, Direct Clock
        *ClkDir = CLKDIR_DIRECT;
        *ClkDiv = 1;
        DISP_INF((_T("[DISP:INF] VCLK Direction = Direct, VCLK = %d Hz\n\r"), dwVCLKSrc));
    }
    else
    {
        // Divide by more than 1, Divided Clock
        *ClkDir = CLKDIR_DIVIDED;
        *ClkDiv = dwDiv;
        DISP_INF((_T("[DISP:INF] VCLK Direction = Divided, Divider = %d, VCLK = %d Hz\n\r"), *ClkDiv,  (unsigned int)((double)dwVCLKSrc/(double)(*ClkDiv))));
    }

    DISP_MSG((_T("[DISP] --Disp_get_vclk_direction_divider()\n\r")));

    return TRUE;
}

static DISP_ERROR
Disp_window_registers_lock
(DISP_WINDOW Win)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_window_registers_lock(%d)\n\r"),Win));

    switch(Win)
    {
    case DISP_WIN0:
        g_pDispConReg->SHADOWCON |= W0_SHADOW_PROTECT;
        break;
    case DISP_WIN1:
        g_pDispConReg->SHADOWCON |= W1_SHADOW_PROTECT;
        break;
    case DISP_WIN2:
        g_pDispConReg->SHADOWCON |= W2_SHADOW_PROTECT;
        break;
    case DISP_WIN3:
        g_pDispConReg->SHADOWCON |= W3_SHADOW_PROTECT;
        break;
    case DISP_WIN4:
        g_pDispConReg->SHADOWCON |= W4_SHADOW_PROTECT;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window_registers_lock() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_window_registers_lock() : %d\n\r"), error));

    return error;
}

static DISP_ERROR
Disp_window_registers_unlock
(DISP_WINDOW Win)
{
    DISP_ERROR error = DISP_SUCCESS;

    DISP_MSG((_T("[DISP]++Disp_window_registers_unlock(%d)\n\r"),Win));

    switch(Win)
    {
    case DISP_WIN0:
        g_pDispConReg->SHADOWCON &= ~W0_SHADOW_PROTECT;
        break;
    case DISP_WIN1:
        g_pDispConReg->SHADOWCON &= ~W1_SHADOW_PROTECT;
        break;
    case DISP_WIN2:
        g_pDispConReg->SHADOWCON &= ~W2_SHADOW_PROTECT;
        break;
    case DISP_WIN3:
        g_pDispConReg->SHADOWCON &= ~W3_SHADOW_PROTECT;
        break;
    case DISP_WIN4:
        g_pDispConReg->SHADOWCON &= ~W4_SHADOW_PROTECT;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] Disp_window_registers_unlock() : Unknown Window Number [%d]\n\r"), Win));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    DISP_MSG((_T("[DISP]--Disp_window_registers_unlock() : %d\n\r"), error));

    return error;
}

DISP_ERROR
Disp_set_output_path
(DISP_VIDOUT_PATH eVideoOutPath)
{
    DISP_ERROR error = DISP_SUCCESS;
    UINT32 uiTempVidcon2 = g_pDispConReg->VIDCON2;
    UINT32 uiTempDualcon = g_pDispConReg->DUALCON;
    UINT32 uiTempMmDispcon = g_pCMUMISCReg->MULTIMEDIA.DISPLAY_CONTROL;
    UINT32 uiTempVidcon0 = g_pDispConReg->VIDCON0;

    DISP_MSG((_T("[DISP]++%s(%d)\n\r"),_T(__FUNCTION__), eVideoOutPath));

    uiTempVidcon2 |= TVFORMATSEL0_SW; // Reserved

    switch(eVideoOutPath)
    {
    case DISP_OUTPUT_SINGLE_BYPASS:
        uiTempDualcon &= DUAL_MASK;
        uiTempDualcon |= SINGLE_BYPASS;
        uiTempVidcon2 &= ~TVFORMATSEL1_YUV444;
        uiTempMmDispcon &= ~(BW_DISPLAY_PATH_SEL<<BP_DISPLAY_PATH_SEL);
        uiTempMmDispcon |= (DISPLAY_PATH_FIMD<<BP_DISPLAY_PATH_SEL);
        uiTempVidcon0 &= ~(VIDOUT_WB_EN | VCLK_FREERUN);
        break;
    case DISP_OUTPUT_DUAL_BYPASS:
        uiTempDualcon &= DUAL_MASK;
        uiTempDualcon |= DUAL_BYPASS;
        uiTempVidcon2 &= ~TVFORMATSEL1_YUV444;
        uiTempMmDispcon &= ~(BW_DISPLAY_PATH_SEL<<BP_DISPLAY_PATH_SEL);
        uiTempMmDispcon |= (DISPLAY_PATH_FIMD<<BP_DISPLAY_PATH_SEL);
        uiTempVidcon0 &= ~(VIDOUT_WB_EN | VCLK_FREERUN);
        DISP_ERR((_T("[DISP:ERR] %s() : Cannot support[%d]\n\r"),_T(__FUNCTION__), eVideoOutPath));
        error = DISP_ERROR_NOT_IMPLEMENTED;
        break;
    case DISP_OUTPUT_MIE:
        uiTempDualcon &= DUAL_MASK;
        uiTempDualcon |= DUAL_MIE;
        uiTempVidcon2 |= TVFORMATSEL1_YUV444;
        uiTempMmDispcon &= ~(BW_DISPLAY_PATH_SEL<<BP_DISPLAY_PATH_SEL);
        uiTempMmDispcon |= (DISPLAY_PATH_MIE<<BP_DISPLAY_PATH_SEL);
        uiTempVidcon0 &= ~(VIDOUT_WB_EN | VCLK_FREERUN);
        DISP_ERR((_T("[DISP:ERR] %s() : Cannot support[%d]\n\r"),_T(__FUNCTION__), eVideoOutPath));
        error = DISP_ERROR_NOT_IMPLEMENTED;
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] %s() : Unknown output mode [%d]\n\r"),_T(__FUNCTION__), eVideoOutPath));
        error = DISP_ERROR_ILLEGAL_PARAMETER;
        break;
    }

    if(error == DISP_SUCCESS)
    {
        g_pDispConReg->DUALCON = uiTempDualcon;
        g_pDispConReg->VIDCON2 = uiTempVidcon2;
        g_pCMUMISCReg->MULTIMEDIA.DISPLAY_CONTROL = uiTempMmDispcon;
        g_pDispConReg->VIDCON0 = uiTempVidcon0;

        DISP_INF((_T("[DISP:INF] DUALCON = 0x%08x\n\r"), g_pDispConReg->DUALCON));
        DISP_INF((_T("[DISP:INF] VIDCON2    = 0x%08x\n\r"), g_pDispConReg->VIDCON2));
        DISP_INF((_T("[DISP:INF] DISPLAY_CONTROL  = 0x%08x\n\r"), g_pCMUMISCReg->MULTIMEDIA.DISPLAY_CONTROL));
        DISP_INF((_T("[DISP:INF] GPOUTCON0    = 0x%08x\n\r"), g_pDispConReg->GPOUTCON0));
        DISP_INF((_T("[DISP:INF] VIDCON0 = 0x%08x\n\r"), g_pDispConReg->VIDCON0));
    }

    DISP_MSG((_T("[DISP]--%s() : %d\n\r"),_T(__FUNCTION__), error));

    return error;
}

static BOOL
Disp_get_src_clock
(unsigned int CLKSrc,
 unsigned int *pdwVCLKSrc)
{
    DWORD dwTimeOut = 0;
    volatile DWORD dwReadTemp = 0;
    DWORD dwVCLKSrc = 0;

    switch(CLKSrc)
    {
    case CLKSEL_F_HCLK:
        dwVCLKSrc = g_pBSPArg->SystemClocks.HCLKDSYS_CLK;
        DISP_INF((_T("[DISP:INF] VCLK Source = HCLK (%d Hz)\n\r"), dwVCLKSrc));
        break;
    case CLKSEL_F_LCDCLK:
        Disp_set_syscon_clk_fimd();
        // MPLL
#if MPLL_USE
        // if we use Dout_MPLL then we can calculate Clock from SysClk Register
        dwVCLKSrc = GET_MPLLCLK(g_pCMUCLKReg->PLL_CON.MPLL_CON);
        DISP_INF((_T("[DISP:INF] MPLLout = (%d Hz)\n\r"), dwVCLKSrc));
        dwVCLKSrc /= (((g_pCMUCLKReg->CLK_DIV.CLK_DIV1 & BW_DIV_FIMD_RATIO<<BP_DIV_FIMD_RATIO)>>BP_DIV_FIMD_RATIO) + 1);
#else
        dwVCLKSrc = 54000000;   // 54Mhz
#endif
        DISP_INF((_T("[DISP:INF] VCLK Source = LCDCLK (%d Hz)\n\r"), dwVCLKSrc));
        break;
    case CLKSEL_F_EXT27M:
        dwVCLKSrc = 27000000;    // 27MHz
        DISP_INF((_T("[DISP:INF] VCLK Source = EXT27M (%d Hz)\n\r"), dwVCLKSrc));
        break;
    default:
        DISP_ERR((_T("[DISP:ERR] --Disp_get_vclk_direction_divider() : Unknown CLKSrc = %d\n\r"), CLKSrc));
        return FALSE;
        break;
    }

    *pdwVCLKSrc = dwVCLKSrc;

    return TRUE;
}

static void
Disp_set_syscon_clk_fimd
(void)
{
    DWORD dwTimeOut = 0;
    volatile DWORD dwReadTemp = 0;

#if MPLL_USE
    g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK0 = \
                            (g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK0 & \
                                ~(BW_MUX_FIMD_MASK<<BP_MUX_FIMD_MASK)) | \
                            (MUX_OUT_OFF<<BP_MUX_FIMD_MASK);   //MUX OFF
    g_pCMUCLKReg->CLK_SRC.CLK_SRC1 =
                            (g_pCMUCLKReg->CLK_SRC.CLK_SRC1 & \
                                ~(BW_MUX_FIMD_SEL<<BP_MUX_FIMD_SEL)) | \
                            (CLK_SCLKMPLL<<BP_MUX_FIMD_SEL);
    g_pCMUCLKReg->CLK_DIV.CLK_DIV1 = \
                            (g_pCMUCLKReg->CLK_DIV.CLK_DIV1 & \
                                ~(BW_DIV_FIMD_RATIO<<BP_DIV_FIMD_RATIO)) |
                            ((MPLL_DIV_VALUE-1)<<BP_DIV_FIMD_RATIO);  // MPLL/3

    // Divider Status Check
    dwTimeOut = 10000;
    do
    {
        dwReadTemp = g_pCMUCLKReg->CLK_DIV_STAT.CLK_DIV_STAT0;
        if(dwTimeOut-- == 0)
        {
            break;
        }
    }while(dwReadTemp  & (DIV_STAT_ON_CHANGING<<BP_DIV_FIMD_STAT));

    g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK0 = \
                                (g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK0 & \
                                    ~(BW_MUX_FIMD_MASK<<BP_MUX_FIMD_MASK)) |
                                (MUX_OUT_ON<<BP_MUX_FIMD_MASK);   //MUX ON
    DISP_INF((_T("[DISP:INF] CLK_SRC_MASK0 = 0x%08x\n\r"), g_pCMUCLKReg->CLK_SRC.CLK_SRC_MASK0));
    DISP_INF((_T("[DISP:INF] CLK_SRC1    = 0x%08x\n\r"), g_pCMUCLKReg->CLK_SRC.CLK_SRC1));
    DISP_INF((_T("[DISP:INF] CLK_DIV1  = 0x%08x\n\r"), g_pCMUCLKReg->CLK_DIV.CLK_DIV1));
#else
    g_pCMUCLKReg->CLK_SRC2 = (g_pCMUCLKReg->CLK_SRC2 & ~BM_LCD_SEL) | CLKSEL_VCLK_54(LCD_SEL);
    g_pCMUCLKReg->CLK_DIV3 = (g_pCMUCLKReg->CLK_DIV3 & ~BM_LCD_RATIO) | CLKDIV_LCD_RATIO(LCD_RATIO, 1);
#endif
}

