//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this sample source code is subject to the terms of the Microsoft
// license agreement under which you licensed this sample source code. If
// you did not accept the terms of the license agreement, you are not
// authorized to use this sample source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the LICENSE.RTF on your install media or the root of your tools installation.
// THE SAMPLE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES OR INDEMNITIES.
//
//
// Copyright (c) Samsung Electronics. Co. LTD.  All rights reserved.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    camera_pdd.cpp

Abstract:       Handle Camera device in High level abstract

Functions:


Notes:


--*/

#include <bsp.h>
#include <pm.h>

#include "Cs.h"
#include "Csmedia.h"

#include "CameraPDDProps.h"
#include "dstruct.h"
#include "dbgsettings.h"

#include <pmplatform.h>
#include <CAMDriver.h>
#include "camera_hal.h"
#include "Module.h"

#include <camera.h>
#include "CameraDriver.h"
#include "SensorFormats.h"
#include "SensorProperties.h"
#include "PinDriver.h"

#include "CameraPDD.h"
#include "camera_pdd.h"
#include "wchar.h"
#include <drvmsg.h>


// For Debug
#define CAM_MSG                     (FALSE)
#define CAM_INOUT                   (FALSE)
#define CAM_ERR                     (TRUE)
#define DBG_MSG_HEADER              _T("[CAM:FILTER]")

// Definitions
#define CAMIF_FOR_VIDEO             _T("CAM1:")  // This will be used for STILLCut SHOT
#define CAMIF_FOR_PREVIEW           _T("CAM3:")

static HANDLE hPwrControl=INVALID_HANDLE_VALUE;
static HANDLE g_hVideoCamera=INVALID_HANDLE_VALUE;
static HANDLE g_hPreviewCamera=INVALID_HANDLE_VALUE;
MODULE_SENSOR CAMERA_MODULE_NAME = SYSLSI_OV2655;



#define CAPTURE_FORMATS_COUNT   3
#define STILLSHOT_FORMATS_COUNT 2
#define PREVIEW_FORMATS_COUNT   1
PDDFUNCTBL FuncTbl = {
    sizeof(PDDFUNCTBL),
    PDD_Init,
    PDD_DeInit,
    PDD_GetAdapterInfo,
    PDD_HandleVidProcAmpChanges,
    PDD_HandleCamControlChanges,
    PDD_HandleVideoControlCapsChanges,
    PDD_SetPowerState,
    PDD_HandleAdapterCustomProperties,
    PDD_InitSensorMode,
    PDD_DeInitSensorMode,
    PDD_SetSensorState,
    PDD_TakeStillPicture,
    PDD_GetSensorModeInfo,
    PDD_SetSensorModeFormat,
    PDD_AllocateBuffer,
    PDD_DeAllocateBuffer,
    PDD_RegisterClientBuffer,
    PDD_UnRegisterClientBuffer,
    PDD_FillBuffer,
    PDD_HandleModeCustomProperties,
    PDD_RequestCameraResources,
    PDD_ReleaseCameraResources
};

// Open specified Camera Driver
BOOL CAF_initialize_RAW_camera_driver()
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s ++%s()\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));

    // Open Device Power Control Driver
    g_hVideoCamera = CreateFile( CAMIF_FOR_VIDEO, GENERIC_READ|GENERIC_WRITE, FILE_SHARE_READ|FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, 0);
    if (g_hVideoCamera == INVALID_HANDLE_VALUE)
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s %s() : %s Open Device Failed\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__), CAMIF_FOR_VIDEO));
        goto CleanUp;
    }

    DWORD dwBytes;

    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_RSC_REQUEST, NULL, 0, NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RSC_REQUEST Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        goto CleanUp;
    }

    DWORD Mode;
    Mode = VIDEO_CAPTURE;
    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_SET_OPERATIONMODE, &Mode, sizeof(int), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SET_OPERATIONMODE Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

    g_hPreviewCamera = CreateFile( CAMIF_FOR_PREVIEW, GENERIC_READ|GENERIC_WRITE, FILE_SHARE_READ|FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, 0);
    if (g_hPreviewCamera == INVALID_HANDLE_VALUE)
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s %s() : %s Open Device Failed\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__), CAMIF_FOR_PREVIEW));
        goto CleanUp;
    }

    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_RSC_REQUEST, NULL, 0, NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RSC_REQUEST Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        goto CleanUp;
    }

    Mode = PREVIEW_CAPTURE;
    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_SET_OPERATIONMODE, &Mode, sizeof(int), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SET_OPERATIONMODE Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

    DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s ++%s()\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));

    return TRUE;

CleanUp:

    DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s --%s() : Failed\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));

    return FALSE;
}

// Close Camera Driver
void CAF_deinitialize_RAW_camera_driver()
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s ++%s()\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));

    if (g_hVideoCamera !=INVALID_HANDLE_VALUE)
    {
        CloseHandle(g_hVideoCamera);
        g_hVideoCamera = INVALID_HANDLE_VALUE;
    }
    if (g_hPreviewCamera !=INVALID_HANDLE_VALUE)
    {
        CloseHandle(g_hPreviewCamera);
        g_hPreviewCamera = INVALID_HANDLE_VALUE;
    }


    DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("%s ++%s()\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));
}

#define CAMERA_PORT 0;



// Get MDDContext Handler
int CameraInit(void *pData)
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s ++%s \n"), DBG_MSG_HEADER, _T(__FUNCTION__)));

    // Open Specified Raw Camera Driver
    CAF_initialize_RAW_camera_driver();

    // Call Raw Camera Driver Initialization
    // Only need module initialization and Buffer initialization
    DWORD dwBytes;


    CAMERA_MODULE_DESC CamModuleInfo;

    CamModuleInfo.CamPort = (CAMIF_INPUTPORT)CAMERA_PORT;  // Ķ?ôõֵ?

       //CamModuleInfo.CamPort = (CAMIF_INPUTPORT)0; //modified by terry.
    CamModuleInfo.ModuleName = (MODULE_SENSOR)CAMERA_MODULE_NAME;

    // Request Initialization Camera module and Buffer
    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_INIT, &CamModuleInfo, sizeof(CAMERA_MODULE_DESC), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_INIT Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_INIT, &CamModuleInfo, sizeof(CAMERA_MODULE_DESC), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_INIT Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s ++%s Succeeded\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    return TRUE;
}

void CameraDeinit()
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s ++%s\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    // Request Deinitialization Camera Module and Buffer
    DWORD dwBytes;


    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_DEINIT, NULL, 0, NULL, 0, &dwBytes, NULL))
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : g_hVideoCamera IOCTL_CAM_DEINIT Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_DEINIT, NULL, 0, NULL, 0, &dwBytes, NULL))
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : g_hPreviewCamera IOCTL_CAM_DEINIT Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

    // Close RAW Camera Driver
    CAF_deinitialize_RAW_camera_driver();

    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s --%s\n"), DBG_MSG_HEADER, _T(__FUNCTION__)));

}


const POWER_CAPABILITIES s_PowerCaps =
{
    // DeviceDx:    Supported power states
    DX_MASK(D0 ) | DX_MASK(D4),

    0,              // WakeFromDx:
    0,              // InrushDx:    No inrush of power

    {               // Power: Maximum milliwatts in each state
        0x00000001, //        D0 = 0
        0x00000001, //        D1 = 0
        0x00000001, //        D2 = 0
        0x00000001, //        D3 = 0
        0x00000001  //        D4 = 0 (off)
    },

    {               // Latency
        0x00000000, //        D0 = 0
        0x00000000, //        D1 = 0
        0x00000000, //        D2 = 0
        0x00000000, //        D3 = 0
        0x00000000  //        D4 = 0
    },

    0,                    // Flags: None
};

CCameraPdd::CCameraPdd()
{
    m_ulCTypes = 3;
    m_bStillCapInProgress = false;
    m_hContext = NULL;
    m_pModeVideoFormat = NULL;
    m_pModeVideoCaps = NULL;
    m_ppModeContext = NULL;
    m_pCurrentFormat = NULL;
    m_iPinUseCount     = 0;
    m_PowerState = D0;
    m_bCameraPreviewRunning = FALSE;
    m_bCameraVideoRunning = FALSE;
    m_bCameraPreviewWasRunning = FALSE;
    m_bCameraVideoWasRunning = FALSE;

    InitializeCriticalSection( &m_csPddDevice );

    memset( &m_CsState, 0x0, sizeof(m_CsState));
    memset( &m_SensorModeInfo, 0x0, sizeof(m_SensorModeInfo));
    memset( &m_SensorProps, 0x0, sizeof(m_SensorProps));
    memset( &PowerCaps, 0x0, sizeof(PowerCaps));

}

CCameraPdd::~CCameraPdd()
{

    DeleteCriticalSection( &m_csPddDevice );

    if( NULL != m_pModeVideoCaps )
    {
        delete [] m_pModeVideoCaps;
        m_pModeVideoCaps = NULL;
    }

    if( NULL != m_pCurrentFormat )
    {
        delete [] m_pCurrentFormat;
        m_pCurrentFormat = NULL;
    }

    if( NULL != m_pModeVideoFormat )
    {
        delete [] m_pModeVideoFormat;
        m_pModeVideoFormat = NULL;
    }

    if( NULL != m_ppModeContext )
    {
        delete [] m_ppModeContext;
        m_ppModeContext = NULL;
    }

}

DWORD CCameraPdd::PDDInit( PVOID MDDContext, PPDDFUNCTBL pPDDFuncTbl )
{
    DWORD dwBytes;
    DWORD dwOutVal;
    CAM_CALLBACK_ARGS   CallBackArgs;

    m_hContext = (HANDLE)MDDContext;
    // Real drivers may want to create their context

        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++PDDInit\n")));
    if (!CameraInit(MDDContext))
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, ( _T("PddInit:FAILED to Initialize camera\r\n") ) );
        return ERROR_GEN_FAILURE;
    }


    CallBackArgs.dwCameraDriverContext = reinterpret_cast<DWORD>( this );
    // This callback function is used to signal the application that the video frame is now available.
    CallBackArgs.pfnCameraHandleVideoFrame = CCameraPdd::CameraVideoFrameCallback;
    // This callback function is used to signal the application that the still image is now available.
    CallBackArgs.pfnCameraHandleStillFrame = CCameraPdd::CameraStillFrameCallback;
    // This callback function is used to signal the application that the Preview image is now available.
    CallBackArgs.pfnCameraHandlePreviewFrame = CCameraPdd::CameraPreviewFrameCallback;

    // Set Callback Functions for handle VideoFrame, StillFrame, PrevieFrame


    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_SETCALLBACK, &CallBackArgs, sizeof(CallBackArgs), &dwOutVal, sizeof(DWORD), &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SETCALLBACK Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }
    // Set Callback Functions for handle VideoFrame, StillFrame, PrevieFrame
    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_SETCALLBACK, &CallBackArgs, sizeof(CallBackArgs), &dwOutVal, sizeof(DWORD), &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SETCALLBACK Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

    m_ulCTypes = 3; // This value can be reloaded from Registry.

    // Read registry to override the default number of Sensor Modes.
    ReadMemoryModelFromRegistry();

    if( pPDDFuncTbl->dwSize  > sizeof( PDDFUNCTBL ) )
    {
        return ERROR_INSUFFICIENT_BUFFER;
    }

    memcpy( pPDDFuncTbl, &FuncTbl, sizeof( PDDFUNCTBL ) );

    memset( m_SensorProps, 0x0, sizeof(m_SensorProps) );

    memcpy( &PowerCaps, &s_PowerCaps, sizeof( POWER_CAPABILITIES ) );

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    // Set all VideoProcAmp and CameraControl properties.
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //VideoProcAmp
    m_SensorProps[ENUM_BRIGHTNESS].ulCurrentValue     = BrightnessDefault;
    m_SensorProps[ENUM_BRIGHTNESS].ulDefaultValue     = BrightnessDefault;
    m_SensorProps[ENUM_BRIGHTNESS].pRangeNStep        = &BrightnessRangeAndStep[0];
    m_SensorProps[ENUM_BRIGHTNESS].ulFlags            = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL;
    m_SensorProps[ENUM_BRIGHTNESS].ulCapabilities     = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL|CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_BRIGHTNESS].fSetSupported      = VideoProcAmpProperties[ENUM_BRIGHTNESS].SetSupported;
    m_SensorProps[ENUM_BRIGHTNESS].fGetSupported      = VideoProcAmpProperties[ENUM_BRIGHTNESS].GetSupported;
    m_SensorProps[ENUM_BRIGHTNESS].pCsPropValues      = &BrightnessValues;

    m_SensorProps[ENUM_CONTRAST].ulCurrentValue       = ContrastDefault;
    m_SensorProps[ENUM_CONTRAST].ulDefaultValue       = ContrastDefault;
    m_SensorProps[ENUM_CONTRAST].pRangeNStep          = &ContrastRangeAndStep[0];
    m_SensorProps[ENUM_CONTRAST].ulFlags              = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL;
    m_SensorProps[ENUM_CONTRAST].ulCapabilities       = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL|CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_CONTRAST].fSetSupported        = VideoProcAmpProperties[ENUM_CONTRAST].SetSupported;
    m_SensorProps[ENUM_CONTRAST].fGetSupported        = VideoProcAmpProperties[ENUM_CONTRAST].GetSupported;
    m_SensorProps[ENUM_CONTRAST].pCsPropValues        = &ContrastValues;

    m_SensorProps[ENUM_HUE].ulCurrentValue            = HueDefault;
    m_SensorProps[ENUM_HUE].ulDefaultValue            = HueDefault;
    m_SensorProps[ENUM_HUE].pRangeNStep               = &HueRangeAndStep[0];
    m_SensorProps[ENUM_HUE].ulFlags                   = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_HUE].ulCapabilities            = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_HUE].fSetSupported             = VideoProcAmpProperties[ENUM_HUE].SetSupported;
    m_SensorProps[ENUM_HUE].fGetSupported             = VideoProcAmpProperties[ENUM_HUE].GetSupported;
    m_SensorProps[ENUM_HUE].pCsPropValues             = &HueValues;

    m_SensorProps[ENUM_SATURATION].ulCurrentValue     = SaturationDefault;
    m_SensorProps[ENUM_SATURATION].ulDefaultValue     = SaturationDefault;
    m_SensorProps[ENUM_SATURATION].pRangeNStep        = &SaturationRangeAndStep[0];
    m_SensorProps[ENUM_SATURATION].ulFlags            = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_SATURATION].ulCapabilities     = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL|CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_SATURATION].fSetSupported      = VideoProcAmpProperties[ENUM_SATURATION].SetSupported;
    m_SensorProps[ENUM_SATURATION].fGetSupported      = VideoProcAmpProperties[ENUM_SATURATION].GetSupported;
    m_SensorProps[ENUM_SATURATION].pCsPropValues      = &SaturationValues;

    m_SensorProps[ENUM_SHARPNESS].ulCurrentValue      = SharpnessDefault;
    m_SensorProps[ENUM_SHARPNESS].ulDefaultValue      = SharpnessDefault;
    m_SensorProps[ENUM_SHARPNESS].pRangeNStep         = &SharpnessRangeAndStep[0];
    m_SensorProps[ENUM_SHARPNESS].ulFlags             = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_SHARPNESS].ulCapabilities      = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_SHARPNESS].fSetSupported       = VideoProcAmpProperties[ENUM_SHARPNESS].SetSupported;
    m_SensorProps[ENUM_SHARPNESS].fGetSupported       = VideoProcAmpProperties[ENUM_SHARPNESS].GetSupported;
    m_SensorProps[ENUM_SHARPNESS].pCsPropValues       = &SharpnessValues;

    m_SensorProps[ENUM_GAMMA].ulCurrentValue          = GammaDefault;
    m_SensorProps[ENUM_GAMMA].ulDefaultValue          = GammaDefault;
    m_SensorProps[ENUM_GAMMA].pRangeNStep             = &GammaRangeAndStep[0];
    m_SensorProps[ENUM_GAMMA].ulFlags                 = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_GAMMA].ulCapabilities          = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_GAMMA].fSetSupported           = VideoProcAmpProperties[ENUM_GAMMA].SetSupported;
    m_SensorProps[ENUM_GAMMA].fGetSupported           = VideoProcAmpProperties[ENUM_GAMMA].GetSupported;
    m_SensorProps[ENUM_GAMMA].pCsPropValues           = &GammaValues;

    m_SensorProps[ENUM_COLORENABLE].ulCurrentValue    = ColorEnableDefault;
    m_SensorProps[ENUM_COLORENABLE].ulDefaultValue    = ColorEnableDefault;
    m_SensorProps[ENUM_COLORENABLE].pRangeNStep       = &ColorEnableRangeAndStep[0];
    m_SensorProps[ENUM_COLORENABLE].ulFlags           = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_COLORENABLE].ulCapabilities    = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL|CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_COLORENABLE].fSetSupported     = VideoProcAmpProperties[ENUM_COLORENABLE].SetSupported;
    m_SensorProps[ENUM_COLORENABLE].fGetSupported     = VideoProcAmpProperties[ENUM_COLORENABLE].GetSupported;
    m_SensorProps[ENUM_COLORENABLE].pCsPropValues     = &ColorEnableValues;

    m_SensorProps[ENUM_WHITEBALANCE].ulCurrentValue   = WhiteBalanceDefault;
    m_SensorProps[ENUM_WHITEBALANCE].ulDefaultValue   = WhiteBalanceDefault;
    m_SensorProps[ENUM_WHITEBALANCE].pRangeNStep      = &WhiteBalanceRangeAndStep[0];
    m_SensorProps[ENUM_WHITEBALANCE].ulFlags          = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_WHITEBALANCE].ulCapabilities   = CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL|CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_WHITEBALANCE].fSetSupported    = VideoProcAmpProperties[ENUM_WHITEBALANCE].SetSupported;
    m_SensorProps[ENUM_WHITEBALANCE].fGetSupported    = VideoProcAmpProperties[ENUM_WHITEBALANCE].GetSupported;
    m_SensorProps[ENUM_WHITEBALANCE].pCsPropValues    = &WhiteBalanceValues;

    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].ulCurrentValue = BackLightCompensationDefault;
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].ulDefaultValue = BackLightCompensationDefault;
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].pRangeNStep    = &BackLightCompensationRangeAndStep[0];
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].ulFlags        = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].ulCapabilities = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].fSetSupported  = VideoProcAmpProperties[ENUM_BACKLIGHT_COMPENSATION].SetSupported;
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].fGetSupported  = VideoProcAmpProperties[ENUM_BACKLIGHT_COMPENSATION].GetSupported;
    m_SensorProps[ENUM_BACKLIGHT_COMPENSATION].pCsPropValues  = &BackLightCompensationValues;

    m_SensorProps[ENUM_GAIN].ulCurrentValue           = GainDefault;
    m_SensorProps[ENUM_GAIN].ulDefaultValue           = GainDefault;
    m_SensorProps[ENUM_GAIN].pRangeNStep              = &GainRangeAndStep[0];
    m_SensorProps[ENUM_GAIN].ulFlags                  = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_GAIN].ulCapabilities           = CSPROPERTY_VIDEOPROCAMP_FLAGS_AUTO;
    m_SensorProps[ENUM_GAIN].fSetSupported            = VideoProcAmpProperties[ENUM_GAIN].SetSupported;
    m_SensorProps[ENUM_GAIN].fGetSupported            = VideoProcAmpProperties[ENUM_GAIN].GetSupported;
    m_SensorProps[ENUM_GAIN].pCsPropValues            = &GainValues;

    //CameraControl
    m_SensorProps[ENUM_PAN].ulCurrentValue            = PanDefault;
    m_SensorProps[ENUM_PAN].ulDefaultValue            = PanDefault;
    m_SensorProps[ENUM_PAN].pRangeNStep               = &PanRangeAndStep[0];
    m_SensorProps[ENUM_PAN].ulFlags                   = CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_PAN].ulCapabilities            = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL|CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_PAN].fSetSupported             = VideoProcAmpProperties[ENUM_PAN-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_PAN].fGetSupported             = VideoProcAmpProperties[ENUM_PAN-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_PAN].pCsPropValues             = &PanValues;

    m_SensorProps[ENUM_TILT].ulCurrentValue           = TiltDefault;
    m_SensorProps[ENUM_TILT].ulDefaultValue           = TiltDefault;
    m_SensorProps[ENUM_TILT].pRangeNStep              = &TiltRangeAndStep[0];
    m_SensorProps[ENUM_TILT].ulFlags                  = CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_TILT].ulCapabilities           = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL|CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_TILT].fSetSupported            = VideoProcAmpProperties[ENUM_TILT-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_TILT].fGetSupported            = VideoProcAmpProperties[ENUM_TILT-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_TILT].pCsPropValues            = &TiltValues;

    m_SensorProps[ENUM_ROLL].ulCurrentValue           = RollDefault;
    m_SensorProps[ENUM_ROLL].ulDefaultValue           = RollDefault;
    m_SensorProps[ENUM_ROLL].pRangeNStep              = &RollRangeAndStep[0];
    m_SensorProps[ENUM_ROLL].ulFlags                  = CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_ROLL].ulCapabilities           = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL|CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_ROLL].fSetSupported            = VideoProcAmpProperties[ENUM_ROLL-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_ROLL].fGetSupported            = VideoProcAmpProperties[ENUM_ROLL-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_ROLL].pCsPropValues            = &RollValues;

    m_SensorProps[ENUM_ZOOM].ulCurrentValue           = ZoomDefault;
    m_SensorProps[ENUM_ZOOM].ulDefaultValue           = ZoomDefault;
    m_SensorProps[ENUM_ZOOM].pRangeNStep              = &ZoomRangeAndStep[0];
    m_SensorProps[ENUM_ZOOM].ulFlags                  = CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_ZOOM].ulCapabilities           = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL|CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_ZOOM].fSetSupported            = VideoProcAmpProperties[ENUM_ZOOM-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_ZOOM].fGetSupported            = VideoProcAmpProperties[ENUM_ZOOM-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_ZOOM].pCsPropValues            = &ZoomValues;

    m_SensorProps[ENUM_IRIS].ulCurrentValue           = IrisDefault;
    m_SensorProps[ENUM_IRIS].ulDefaultValue           = IrisDefault;
    m_SensorProps[ENUM_IRIS].pRangeNStep              = &IrisRangeAndStep[0];
    m_SensorProps[ENUM_IRIS].ulFlags                  = CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_IRIS].ulCapabilities           = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL|CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_IRIS].fSetSupported            = VideoProcAmpProperties[ENUM_IRIS-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_IRIS].fGetSupported            = VideoProcAmpProperties[ENUM_IRIS-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_IRIS].pCsPropValues            = &IrisValues;

    m_SensorProps[ENUM_EXPOSURE].ulCurrentValue       = ExposureDefault;
    m_SensorProps[ENUM_EXPOSURE].ulDefaultValue       = ExposureDefault;
    m_SensorProps[ENUM_EXPOSURE].pRangeNStep          = &ExposureRangeAndStep[0];
    m_SensorProps[ENUM_EXPOSURE].ulFlags              = CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_EXPOSURE].ulCapabilities       = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL|CSPROPERTY_CAMERACONTROL_FLAGS_AUTO;
    m_SensorProps[ENUM_EXPOSURE].fSetSupported        = VideoProcAmpProperties[ENUM_EXPOSURE-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_EXPOSURE].fGetSupported        = VideoProcAmpProperties[ENUM_EXPOSURE-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_EXPOSURE].pCsPropValues        = &ExposureValues;

    m_SensorProps[ENUM_FOCUS].ulCurrentValue          = FocusDefault;
    m_SensorProps[ENUM_FOCUS].ulDefaultValue          = FocusDefault;
    m_SensorProps[ENUM_FOCUS].pRangeNStep             = &FocusRangeAndStep[0];
    m_SensorProps[ENUM_FOCUS].ulFlags                 = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL;
    m_SensorProps[ENUM_FOCUS].ulCapabilities          = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL;
    m_SensorProps[ENUM_FOCUS].fSetSupported           = VideoProcAmpProperties[ENUM_FOCUS-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_FOCUS].fGetSupported           = VideoProcAmpProperties[ENUM_FOCUS-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_FOCUS].pCsPropValues           = &FocusValues;

    m_SensorProps[ENUM_FLASH].ulCurrentValue          = FlashDefault;
    m_SensorProps[ENUM_FLASH].ulDefaultValue          = FlashDefault;
    m_SensorProps[ENUM_FLASH].pRangeNStep             = &FlashRangeAndStep[0];
    m_SensorProps[ENUM_FLASH].ulFlags                 = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL;
    m_SensorProps[ENUM_FLASH].ulCapabilities          = CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL;
    m_SensorProps[ENUM_FLASH].fSetSupported           = VideoProcAmpProperties[ENUM_FLASH-NUM_VIDEOPROCAMP_ITEMS].SetSupported;
    m_SensorProps[ENUM_FLASH].fGetSupported           = VideoProcAmpProperties[ENUM_FLASH-NUM_VIDEOPROCAMP_ITEMS].GetSupported;
    m_SensorProps[ENUM_FLASH].pCsPropValues           = &FlashValues;
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    m_pModeVideoFormat = NULL;
    // Allocate Video Format specific array.
    m_pModeVideoFormat = new PINVIDEOFORMAT[m_ulCTypes];
    if( NULL == m_pModeVideoFormat )
    {
        goto error_notenough_memory;
    }
    m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo = NULL;
    m_pModeVideoFormat[STILL].pCsDataRangeVideo = NULL;
    if(3 == m_ulCTypes)
    {
        m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo = NULL;
    }

    // Video Format initialization
    m_pModeVideoFormat[CAPTURE].categoryGUID         = PINNAME_VIDEO_CAPTURE;
    m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo    = new PCS_DATARANGE_VIDEO[MAX_VIDEO_FORMAT];

    if( NULL == m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo )
    {
        goto error_notenough_memory;
    }

    m_pModeVideoFormat[STILL].categoryGUID           = PINNAME_VIDEO_STILL;
    m_pModeVideoFormat[STILL].pCsDataRangeVideo      = new PCS_DATARANGE_VIDEO[MAX_VIDEO_FORMAT];

    if( NULL == m_pModeVideoFormat[STILL].pCsDataRangeVideo )
    {
        goto error_notenough_memory;
    }

    if( 3 == m_ulCTypes )
    {
        m_pModeVideoFormat[PREVIEW].categoryGUID         = PINNAME_VIDEO_PREVIEW;
        m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo    = new PCS_DATARANGE_VIDEO[MAX_VIDEO_FORMAT];

        if( NULL == m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo )
        {
            goto error_notenough_memory;
        }
        }
    // For each Sensor, supportable video format will be changed.
    // TODO: The Selected Camera sensor information may be in registry
    //    GetVideoFormatList(SYSLSI_S5K6AAFX);
    GetVideoFormatList(CAMERA_MODULE_NAME);

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    // Allocate Video Control Caps specific array.
    m_pModeVideoCaps = new VIDCONTROLCAPS[m_ulCTypes];
    if( NULL == m_pModeVideoCaps )
    {
        goto error_notenough_memory;
    }
    // Video Control Caps

    m_pModeVideoCaps[CAPTURE].DefaultVideoControlCaps     = DefaultVideoControlCaps[CAPTURE];
    m_pModeVideoCaps[CAPTURE].CurrentVideoControlCaps     = DefaultVideoControlCaps[CAPTURE];;
    m_pModeVideoCaps[STILL].DefaultVideoControlCaps       = DefaultVideoControlCaps[STILL];
    m_pModeVideoCaps[STILL].CurrentVideoControlCaps       = DefaultVideoControlCaps[STILL];;
    if( 3 == m_ulCTypes )
    {
        // Note PREVIEW control caps are the same, so we don't differentiate
        m_pModeVideoCaps[PREVIEW].DefaultVideoControlCaps     = DefaultVideoControlCaps[PREVIEW];
        m_pModeVideoCaps[PREVIEW].CurrentVideoControlCaps     = DefaultVideoControlCaps[PREVIEW];;
    }

   // m_SensorModeInfo[CAPTURE].MemoryModel = CSPROPERTY_BUFFER_CLIENT_UNLIMITED;
    m_SensorModeInfo[CAPTURE].MaxNumOfBuffers = 1;
    m_SensorModeInfo[CAPTURE].PossibleCount = 1;
  //  m_SensorModeInfo[STILL].MemoryModel = CSPROPERTY_BUFFER_CLIENT_UNLIMITED;
    m_SensorModeInfo[STILL].MaxNumOfBuffers = 1;
    m_SensorModeInfo[STILL].PossibleCount = 1;
    if( 3 == m_ulCTypes )
    {
   //     m_SensorModeInfo[PREVIEW].MemoryModel = CSPROPERTY_BUFFER_CLIENT_UNLIMITED;
        m_SensorModeInfo[PREVIEW].MaxNumOfBuffers = 1;
        m_SensorModeInfo[PREVIEW].PossibleCount = 1;
    }

    m_ppModeContext = new LPVOID[m_ulCTypes];
    if ( NULL == m_ppModeContext )
    {
        goto error_notenough_memory;
    }

    m_pCurrentFormat = new CS_DATARANGE_VIDEO[m_ulCTypes];
    if( NULL == m_pCurrentFormat )
    {
        goto error_notenough_memory;
    }

    return ERROR_SUCCESS;

error_notenough_memory:
    PDD_DeInit();
    return ERROR_INSUFFICIENT_BUFFER;
}

void CCameraPdd::PDD_DeInit()
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++PDD_DeInit\n")));
    CameraDeinit();

    if( NULL != m_ppModeContext )
    {
        delete [] m_ppModeContext;
        m_ppModeContext = NULL;
    }

    if( NULL != m_pModeVideoCaps )
    {
        delete [] m_pModeVideoCaps;
        m_pModeVideoCaps = NULL;
    }

    if( NULL != m_pModeVideoFormat )
    {
        if (NULL != m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo)
        {
            delete [] m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo;
            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo = NULL;
        }

        if (NULL != m_pModeVideoFormat[STILL].pCsDataRangeVideo)
        {
            delete [] m_pModeVideoFormat[STILL].pCsDataRangeVideo;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo = NULL;
        }

        if( 3 == m_ulCTypes)
        {

        if (NULL != m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo)
        {
            delete [] m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo;
            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo = NULL;
        }
        }

        delete [] m_pModeVideoFormat;
        m_pModeVideoFormat = NULL;
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("---------------------PDD_DeInit\n")));
}


DWORD CCameraPdd::GetAdapterInfo( PADAPTERINFO pAdapterInfo )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++GetAdapterInfo\n")));
    pAdapterInfo->ulCTypes = m_ulCTypes;
    pAdapterInfo->PowerCaps = PowerCaps;
    pAdapterInfo->ulVersionID = DRIVER_VERSION_2; //Camera MDD and DShow support DRIVER_VERSION and DRIVER_VERSION_2. Defined in camera.h
    memcpy( &pAdapterInfo->SensorProps, &m_SensorProps, sizeof(m_SensorProps));
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--------------------GetAdapterInfo\n")));

    return ERROR_SUCCESS;
}

void
CCameraPdd::GetVideoFormatList(
    DWORD dwSensorID
    )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(_T("[%s] %s - dwSensorID=%d\r\n"), DBG_MSG_HEADER, _T(__FUNCTION__), dwSensorID));

    DWORD hr = ERROR_SUCCESS;
    int i;

//    BOOL bForCETK=TRUE;

    switch ( dwSensorID )
    {
    case SYSLSI_OV2655:
		// Video Format initialization
        i = 0;
        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_176x144_30;
        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_320x240_30;
//        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_640x480_30;
        m_pModeVideoFormat[CAPTURE].ulAvailFormats          = i;

        // Still Format initialization
        i = 0;
/*
        if( TRUE == bForCETK )
        {
*/
#ifdef OV2655_1600_1200  // added by terry 2012.07.20
            // CETK can test only RGB format.
           m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_1600x1200_30;//&DCAM_StreamMode_RGB565_640x480_30;//&DCAM_StreamMode_RGB565_640x480_30;//&DCAM_StreamMode_RGB565_320x240_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_1600x1200_30;//&DCAM_StreamMode_RGB565_640x480_30;
#else //  800*600  

			m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]	= &DCAM_StreamMode_RGB565_640x480_30;//&DCAM_StreamMode_RGB565_640x480_30;//&DCAM_StreamMode_RGB565_320x240_30;
			m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]	= &DCAM_StreamMode_RGB565_640x480_30;
#endif
			
/*        }
        else // YV format also can be used. actually it shows faster performance than RGB format.
        {
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_320x240_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_640x480_30;

        }
*/
        m_pModeVideoFormat[STILL].ulAvailFormats            = i;

        // if preview pins supports
        if( MAX_SUPPORTED_PINS == m_ulCTypes )
        {
            i = 0;
            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_640x480_30;//&DCAM_StreamMode_RGB565_176x144_30;
            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_640x480_30;
//            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_640x480_30;
            m_pModeVideoFormat[PREVIEW].ulAvailFormats          = i;
        }

        break;
#if 1
    case SYSLSI_OV3640:
        // Video Format initialization
        RETAILMSG(1, (TEXT("SYSLSI_OV3640\r\n")));
        i = 0;
        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_176x144_30;
        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_320x240_30;
//        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_640x480_30;
        m_pModeVideoFormat[CAPTURE].ulAvailFormats          = i;

        // Still Format initialization
        i = 0;
/*
        if( TRUE == bForCETK )
        {
*/
            // CETK can test only RGB format.
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_RGB565_640x480_30;//&DCAM_StreamMode_RGB565_320x240_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_RGB565_640x480_30;
/*        }
        else // YV format also can be used. actually it shows faster performance than RGB format.
        {
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_320x240_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_640x480_30;

        }
*/
        m_pModeVideoFormat[STILL].ulAvailFormats            = i;

        // if preview pins supports
        if( MAX_SUPPORTED_PINS == m_ulCTypes )
        {
            i = 0;
            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_320x240_30;//&DCAM_StreamMode_RGB565_176x144_30;
            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_320x240_30;
//            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_640x480_30;
            m_pModeVideoFormat[PREVIEW].ulAvailFormats          = i;
        }

        break;
#endif
    case SYSLSI_S5K4EAGX:
        // Video Format initialization
        i = 0;
/*
        if(TRUE == bForCETK)
        {
*/
//            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YUY12_320x240_30;
            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_NV12_640x480_30;
/*
        }
        else
        {
            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_176x144_30;
            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_320x240_30;
            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_640x480_30;
            m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_1280x720_30;
        }
*/
        m_pModeVideoFormat[CAPTURE].ulAvailFormats          = i;

        // Still Format initialization
        i = 0;

        // With S5K4EAGX Sensor module, IJPG is used for Still shot, and there is no test kit.
        // The user should dump the camera still output pin buffer to check frame data
/*
        if(TRUE == bForCETK)
        {
*/
            // CETK can test only RGB format.
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_800x480_30;
/*
        }
        else
        {
            // Use Default Format
            // YV format also can be used. actually it shows faster performance than RGB format.
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_640x480_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_800x480_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_1280x960_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_1600x1200_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_2048x1536_30;
            m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_IJPG_2592x1944_30;
        }
*/
        m_pModeVideoFormat[STILL].ulAvailFormats            = i;

        // if preview pins supports
        if( MAX_SUPPORTED_PINS == m_ulCTypes )
        {
            i = 0;
/*
            if(TRUE == bForCETK)
            {
*/
                // CETK can test only RGB format.
                m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_320x240_30;
                m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_640x480_30;
            //    m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_RGB565_800x480_30;
/*
            }
            else
            {
                // YV format also can be used. actually it shows faster performance than RGB format.
                m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_640x480_30;
                m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_1280x720_30;
            }
*/
            m_pModeVideoFormat[PREVIEW].ulAvailFormats          = i;
        }

        break;

    case SYSLSI_S5K6AAFX:
    default:
        // Video Format initialization
        i = 0;

        m_pModeVideoFormat[CAPTURE].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_640x480_30;
        m_pModeVideoFormat[CAPTURE].ulAvailFormats          = i;

        // Still Format initialization
        i = 0;

        m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_640x480_30;
        m_pModeVideoFormat[STILL].pCsDataRangeVideo[i++]    = &DCAM_StreamMode_YV12_1280x1024_30;
        m_pModeVideoFormat[STILL].ulAvailFormats            = i;

        // if preview pins supports
        if( MAX_SUPPORTED_PINS == m_ulCTypes )
        {
            i = 0;
/*
            if(TRUE == bForCETK)
            {
*/
                // But CETK can test only RGB format.
                m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]   = &DCAM_StreamMode_RGB565_640x480_30;
/*
            }
            else
            {
                // YV format also can be used. actually it shows faster performance than RGB format.
            m_pModeVideoFormat[PREVIEW].pCsDataRangeVideo[i++]  = &DCAM_StreamMode_YV12_640x480_30;
            }
*/
            m_pModeVideoFormat[PREVIEW].ulAvailFormats          = i;
        }

        break;
    }
}

DWORD CCameraPdd::HandleVidProcAmpChanges( DWORD dwPropId, LONG lFlags, LONG lValue )
{
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleVidProcAmpChanges\n")));
    PSENSOR_PROPERTY pDevProp = NULL;

    pDevProp = m_SensorProps + dwPropId;

    if( CSPROPERTY_VIDEOPROCAMP_FLAGS_MANUAL == lFlags )
    {
        pDevProp->ulCurrentValue = lValue;
    }

    pDevProp->ulFlags = lFlags;
    return ERROR_SUCCESS;
}

DWORD CCameraPdd::HandleCamControlChanges( DWORD dwPropId, LONG lFlags, LONG lValue )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleCamControlChanges\n")));
    PSENSOR_PROPERTY pDevProp = NULL;
    int    value;
    pDevProp = m_SensorProps + dwPropId;

    if( CSPROPERTY_CAMERACONTROL_FLAGS_MANUAL == lFlags )
    {
        pDevProp->ulCurrentValue = lValue;
        switch(dwPropId)
        {
        case ENUM_ZOOM:
            DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[CAM] Set Zoom value = %d!!!\n"), pDevProp->ulCurrentValue));
            value = (int)((pDevProp->ulCurrentValue-ZoomRangeAndStep[0].Bounds.UnsignedMinimum)/ZoomRangeAndStep[0].SteppingDelta);
            DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[CAM] Zoom value=%ld\n"),value));

            if(!CameraZoom(value))
            {
                DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[CAM_ERROR] Zoom value %ld is not supported\n"),value));
            }
            break;
        default:
            break;
        }
    }

    pDevProp->ulFlags = lFlags;

    return ERROR_SUCCESS;
}

DWORD CCameraPdd::HandleVideoControlCapsChanges( LONG lModeType ,ULONG ulCaps )
{
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleVideoControlCapsChanges\n")));
    m_pModeVideoCaps[lModeType].CurrentVideoControlCaps = ulCaps;
    return ERROR_SUCCESS;
}

DWORD CCameraPdd :: SetPowerState( CEDEVICE_POWER_STATE PowerState )
{
    DWORD dwErr = ERROR_SUCCESS;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++SetPowerState\n")));
    switch (PowerState)
    {
    case D0:
        DBGMSG(CAMF_FUNC && CAMF_USR1, (TEXT("[CAM]: D0\r\n")));

        if (m_PowerState != D0)
            ResumeCamera();

        m_PowerState = D0;

        dwErr = ERROR_SUCCESS;
        break;

    case D4:
        DBGMSG(CAMF_FUNC && CAMF_USR1, (TEXT("[CAM]: D4\r\n")));
        if (m_PowerState != D4)
            SuspendCamera();
           m_PowerState = D4;
        dwErr = ERROR_SUCCESS;
        break;

    default:
        break;
    }
    return dwErr;
}

DWORD CCameraPdd::HandleAdapterCustomProperties( PUCHAR pInBuf, DWORD  InBufLen, PUCHAR pOutBuf, DWORD  OutBufLen, PDWORD pdwBytesTransferred )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleAdapterCustomProperties\n")));
    DBGMSG(CAMF_FUNC && CAMF_USR1, ( _T("IOControl Adapter PDD: Unsupported PropertySet Request\r\n")) );

    return ERROR_NOT_SUPPORTED;
}

DWORD CCameraPdd::InitSensorMode( ULONG ulModeType, LPVOID ModeContext )
{
    DWORD hr = ERROR_SUCCESS;
    DWORD dwBytes;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++InitSensorMode\n")));
    ASSERT( ModeContext );

    EnterCriticalSection( &m_csPddDevice );
    m_iPinUseCount += 1;
    if(m_iPinUseCount > MAX_SUPPORTED_PINS)
        m_iPinUseCount = MAX_SUPPORTED_PINS;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("m_iPinUseCount=%d\r\n"),m_iPinUseCount));
//    if(m_iPinUseCount == 1)
//    {
        // Request Initialization Camera module and Buffer, There is no need to init sensor twice.
        // But for Camera Interface, This init should be called for each cameraIP block
        if(ulModeType == PREVIEW)
        {
        if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_INIT_SENSOR, NULL, 0, NULL, 0, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_INIT_SENSOR Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
            return ERROR_BAD_COMMAND;
        }
        }
        else
        {
        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_INIT_SENSOR, NULL, 0, NULL, 0, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_INIT_SENSOR Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
            return ERROR_BAD_COMMAND;
        }
        }
//    }
    LeaveCriticalSection( &m_csPddDevice );
    m_ppModeContext[ulModeType] = ModeContext;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("----------------InitSensorMode\n")));
    return hr;
}

DWORD CCameraPdd::DeInitSensorMode( ULONG ulModeType )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++DeInitSensorMode\n")));
    EnterCriticalSection( &m_csPddDevice );
    m_iPinUseCount -= 1;
    if(m_iPinUseCount < 0)
        m_iPinUseCount = 0;
    //DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("m_iPinUseCount=%d\r\n"),m_iPinUseCount));
    if(m_iPinUseCount == 0)
    {
        CameraSleep();
    }
    LeaveCriticalSection( &m_csPddDevice );
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--------------------DeInitSensorMode\n")));
    return ERROR_SUCCESS;
}

DWORD CCameraPdd::SetSensorState( ULONG lModeType, CSSTATE csState )
{
    DWORD dwError = ERROR_SUCCESS;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s  %s(%d, %d)\n"), DBG_MSG_HEADER, _T(__FUNCTION__), lModeType, csState));
    CAMIF_OPERATION_MODE format = (lModeType==CAPTURE)?VIDEO_CAPTURE:((lModeType==STILL)?STILL_CAPTURE:PREVIEW_CAPTURE);
    CAM_START_ARGS  StartArgs = {MODE_DIRECT};
    DWORD dwControl;
    HANDLE hTargetCameraIF;

    if(STILL == lModeType || CAPTURE == lModeType)
    {
        hTargetCameraIF = g_hVideoCamera;
    }
    else
    {
        hTargetCameraIF = g_hPreviewCamera;
    }
    if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_SET_OPERATIONMODE, &format, sizeof(DWORD), NULL, 0, &dwError, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_SET_OPERATIONMODE Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

    switch ( csState )
    {
        case CSSTATE_STOP:
            DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[CAM] %d STOP\n"),format));
            m_CsState[lModeType] = CSSTATE_STOP;

            if( STILL == lModeType )    // Still Capture
            {
                m_bStillCapInProgress = false;
                // Possible?
            }
            else if(CAPTURE == lModeType)   // Video Capture
            {
                m_bCameraVideoRunning = false;
                if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_VIDEO_STOP, NULL, NULL, NULL, 0, &dwError, NULL) )
                {
                    DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_VIDEO_STOP Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
                }
            }
            else
            {
                m_bCameraPreviewRunning = false;
                if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_PREVIEW_STOP, NULL, NULL, NULL, 0, &dwError, NULL) )
                {
                    DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_PREVIEW_STOP Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
                }
            }

            dwControl = FALSE;

            break;

        case CSSTATE_PAUSE:
            DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[CAM] %d PAUSE\n"),format));
            if(CAPTURE == lModeType)
            {
                m_bCameraVideoRunning = false;
            }
            else
            {
                m_bCameraPreviewRunning = false;
            }
            m_CsState[lModeType] = CSSTATE_PAUSE;

            if(CAPTURE == lModeType)
            {
                if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_VIDEO_STOP, NULL, NULL, NULL, 0, &dwError, NULL) )
                {
                    DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_VIDEO_STOP Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
                }
            }
            else
            {
                if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_PREVIEW_STOP, NULL, NULL, NULL, 0, &dwError, NULL) )
                {
                    DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_PREVIEW_STOP Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
                }
            }

            break;

        case CSSTATE_RUN:
            DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[CAM] %d RUN\n"),format));
            m_CsState[lModeType] = CSSTATE_RUN;
            // only video and preview
            if(CAPTURE == lModeType)
            {
                m_bCameraVideoRunning = true;
            }
            else
            {
                m_bCameraPreviewRunning = true;
            }
            SetSensorFormat(lModeType);

            if(CAPTURE == lModeType)
            {
                if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_VIDEO_START, &StartArgs, sizeof(CAM_START_ARGS), NULL, 0, &dwError, NULL) )
                {
                    DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_VIDEO_START Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
                }
            }
            else
            {
                if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_PREVIEW_START, &StartArgs, sizeof(CAM_START_ARGS), NULL, 0, &dwError, NULL) )
                {
                    DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_PREVIEW_START Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
                }
            }

            /*
            // CameraSetRegisters(format);
            dwControl = TRUE;
            if ( !DeviceIoControl(hTargetCameraIF, IOCTL_CAM_CAPTURECONTROL, &dwControl, sizeof(DWORD), NULL, 0, &dwError, NULL) )
            {
                DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_CAPTURECONTROL Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
            }
            // CameraCaptureControl(format,TRUE);
            */
            break;

        default:
            DBGMSG(CAMF_FUNC && CAMF_USR1, ( _T("IOControl(%08x): Incorrect State\r\n"), this ) );
            dwError = ERROR_INVALID_PARAMETER;
    }

    return dwError;
}

DWORD CCameraPdd::TakeStillPicture( LPVOID pBurstModeInfo )
{
    DWORD dwError = ERROR_SUCCESS;
    CAM_STILLCUT_ARGS StillCutArgs;

    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++TakeStillPicture\n")));
    // TODO: We should check Video Capturing. Because we use same FIMC hardware for video and still capturing.

    m_bStillCapInProgress = true;
    //Ignore pBurstModeInfo
    m_CsState[STILL] = CSSTATE_RUN;

    SetSensorFormat(STILL);

    StillCutArgs.mode = MODE_DIRECT;
    StillCutArgs.buffer = (PVOID)m_CameraHWStillBuffer.VirtAddr;
    StillCutArgs.directBuffer.virtualAddr = m_CameraHWStillBuffer.VirtAddr;
    StillCutArgs.directBuffer.physicalAddr = (DWORD)m_CameraHWStillBuffer.pY;


    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_STILLCUT, &StillCutArgs, sizeof(CAM_STILLCUT_ARGS), NULL, 0, &dwError, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s %s() : IOCTL_CAM_STILLCUT Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }
    return dwError;
}


DWORD CCameraPdd::GetSensorModeInfo( ULONG ulModeType, PSENSORMODEINFO pSensorModeInfo )
{
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++GetSensorModeInfo\n")));
    pSensorModeInfo->MemoryModel = m_SensorModeInfo[ulModeType].MemoryModel;
    pSensorModeInfo->MaxNumOfBuffers = m_SensorModeInfo[ulModeType].MaxNumOfBuffers;
    pSensorModeInfo->PossibleCount = m_SensorModeInfo[ulModeType].PossibleCount;
    pSensorModeInfo->VideoCaps.DefaultVideoControlCaps = DefaultVideoControlCaps[ulModeType];
    pSensorModeInfo->VideoCaps.CurrentVideoControlCaps = m_pModeVideoCaps[ulModeType].CurrentVideoControlCaps;
    pSensorModeInfo->pVideoFormat = &m_pModeVideoFormat[ulModeType];
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--------------------GetSensorModeInfo\n")));
    return ERROR_SUCCESS;
}

DWORD CCameraPdd::SetSensorModeFormat( ULONG ulModeType, PCS_DATARANGE_VIDEO pCsDataRangeVideo )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++SetSensorModeFormat     %d\n"),ulModeType));
    // MDD will take care of ulModeType. It will never be out of range. MDD will also ask for
    // the format we support. We need not to check it here.

    memcpy( &m_pCurrentFormat[ulModeType], pCsDataRangeVideo, sizeof ( CS_DATARANGE_VIDEO ) );
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("---------------------SetSensorModeFormat     \n")));

    return ERROR_SUCCESS;
}

PVOID CCameraPdd::AllocateBuffer( ULONG ulModeType )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++AllocateBuffer\n")));
    // Real PDD may want to save off this allocated pointer in an array.
    // In this PDD, we don't need the buffer for copy.
    return NULL;
}

DWORD CCameraPdd::DeAllocateBuffer( ULONG ulModeType, PVOID pBuffer )
{
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++DeAllocateBuffer\n")));

    return ERROR_SUCCESS;
}

DWORD CCameraPdd::RegisterClientBuffer( ULONG ulModeType, PVOID pBuffer )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++RegisterClientBuffer(%d, 0x%x)\n"), ulModeType, pBuffer));
    // Real PDD may want to save pBuffer which is a pointer to buffer that DShow created.
    if(ulModeType >= MAX_SUPPORTED_PINS)
    {
        return ERROR_NOT_SUPPORTED;
    }
    m_pClientBuffer[ulModeType] = pBuffer;
    return ERROR_SUCCESS;
}

DWORD CCameraPdd::UnRegisterClientBuffer( ULONG ulModeType, PVOID pBuffer )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++UnRegisterClientBuffer(%d, 0x%x\n"), ulModeType, pBuffer));
    // DShow is not going to use pBuffer (which was originally allocated by DShow) anymore. If the PDD
    // is keeping a cached pBuffer pointer (in RegisterClientBuffer()) then this is the right place to
    // stop using it and maybe set the cached pointer to NULL.
    // Note: PDD must not delete this pointer as it will be deleted by DShow itself
    if(ulModeType >= MAX_SUPPORTED_PINS)
    {
        return ERROR_NOT_SUPPORTED;
    }
    m_pClientBuffer[ulModeType] = NULL;

    return ERROR_SUCCESS;
}

DWORD CCameraPdd::HandleSensorModeCustomProperties( ULONG ulModeType, PUCHAR pInBuf, DWORD  InBufLen, PUCHAR pOutBuf, DWORD  OutBufLen, PDWORD pdwBytesTransferred )
{
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleSensorModeCustomProperties\n")));
    DBGMSG(CAMF_FUNC && CAMF_USR1, ( _T("IOControl: Unsupported PropertySet Request\r\n")) );
    return ERROR_NOT_SUPPORTED;
}

// This external function comes from NULL_PDD adapter
//extern "C" { WINGDIAPI HBITMAP WINAPI CreateBitmapFromPointer( CONST BITMAPINFO *pbmi, int iStride, PVOID pvBits); }

DWORD CCameraPdd::FillBuffer( ULONG ulModeType, PUCHAR pImage )
{
    DWORD dwRet = 0;
    DWORD dwBytes;
    DWORD CurrentFrame;
    DWORD dwJPEGSize = 0;
    DWORD dwCapturedSize = 0;

    PCS_VIDEOINFOHEADER pCsVideoInfoHdr = &m_pCurrentFormat[ulModeType].VideoInfoHeader;

    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++FillBuffer\n")));
        ASSERT(pCsVideoInfoHdr->bmiHeader.biSizeImage != 0);
    if(pImage == NULL)
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("pImage is null!\n")));
        return 0;
    }
    // MDD will make sure that the buffer is sufficient for the image.

    if (ulModeType == CAPTURE)
    {
        // Use Proper Device Handle
        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_GETCURRENTFRAMENUM, &dwCapturedSize, sizeof(DWORD), &CurrentFrame, sizeof(DWORD), &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : CAPTURE IOCTL_CAM_GETCURRENTFRAMENUM Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

        //dwRet = pCsVideoInfoHdr->bmiHeader.biSizeImage;
        //memcpy(pImage, (void *)m_CameraHWVideoBuffers[CurrentFrame].VirtAddr, dwRet);
        dwRet = dwCapturedSize;
        memcpy(pImage, (void *)m_CameraHWVideoBuffers[CurrentFrame].VirtAddr, dwRet);
    }
    else if (ulModeType == STILL)
    {
        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_GETCURRENTFRAMENUM, &dwJPEGSize, sizeof(DWORD), &CurrentFrame, sizeof(DWORD), &dwBytes, NULL ))
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : STILL IOCTL_CAM_GETCURRENTFRAMENUM Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
        RETAILMSG(1,(TEXT("STILL Size = %d\r\n"), dwJPEGSize));
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("m_CameraHWStillBuffer.VirtAddr=0x%x\n"),m_CameraHWStillBuffer.VirtAddr));
        //dwRet = pCsVideoInfoHdr->bmiHeader.biSizeImage;
        //memcpy(pImage, (void *)m_CameraHWStillBuffer.VirtAddr, dwRet);
        dwRet = dwJPEGSize;
        memcpy(pImage, (void *)m_CameraHWStillBuffer.VirtAddr, dwJPEGSize);
        SetSensorFreeBuffer( STILL);
    }
    else if(ulModeType == PREVIEW)
    {
        if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_GETCURRENTFRAMENUM, &dwCapturedSize, sizeof(DWORD), &CurrentFrame, sizeof(DWORD), &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : PREVIEW IOCTL_CAM_GETCURRENTFRAMENUM Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

        //dwRet = pCsVideoInfoHdr->bmiHeader.biSizeImage;
        //memcpy(pImage, (void *)m_CameraHWVideoBuffers[CurrentFrame].VirtAddr, dwRet);
        dwRet = dwCapturedSize;
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("m_CameraHWPreviewBuffers[%d].VirtAddr=0x%x, Dest:0x%x, size:%d\n"),CurrentFrame,m_CameraHWPreviewBuffers[CurrentFrame].VirtAddr, pImage, dwRet));
//	memset(pImage, 0, dwRet);
//	RETAILMSG(1, (TEXT("0x%x,0x%x "),*(pImage + 10000),*(pImage + 100000)));
	memcpy(pImage, (void *)m_CameraHWPreviewBuffers[CurrentFrame].VirtAddr, dwRet);
//	RETAILMSG(1, (TEXT("0x%x,0x%x\r\n"),*(pImage + 10000),*(pImage + 100000)));
//	memset((void *)m_CameraHWPreviewBuffers[CurrentFrame].VirtAddr, 0, dwRet);
	//RETAILMSG(1, (TEXT("%d,"),dwRet));
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--------------------FillBuffer\n")));
    // return the size of the image filled
        return dwRet;
}


void CCameraPdd :: HandleCaptureInterrupt( ULONG ulModeTypeIn )
{
    ULONG ulModeType;
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleCaptureInterrupt\n")));
    if( m_bStillCapInProgress )
    {
        return;
    }

    if( ulModeTypeIn == CAPTURE)
    {
        ulModeType = CAPTURE;
    }
    else if ( m_ulCTypes == 3 && ulModeTypeIn == PREVIEW )
    {
        ulModeType = PREVIEW;
    }
    else
    {
        ASSERT(false);
        return;
    }

    MDD_HandleIO( m_ppModeContext[ulModeType], ulModeType );
     DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("------------------------HandleCaptureInterrupt  %d\n"),ulModeType));

}


void CCameraPdd :: HandleStillInterrupt( )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++HandleStillInterrupt\n")));
    MDD_HandleIO( m_ppModeContext[STILL], STILL );
    m_bStillCapInProgress = false;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--------------------HandleStillInterrupt\n")));
}

// Read registry to override the default number of Sensor Modes.
//  MemoryModel, PinCount
bool CCameraPdd::ReadMemoryModelFromRegistry()
{
    HKEY  hKey = 0;
    DWORD dwType  = 0;
    DWORD dwSize  = sizeof ( DWORD );
    DWORD dwValue = -1;
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++++++++++++++++++++ReadMemoryModelFromRegistry\n")));

    if( ERROR_SUCCESS != RegOpenKeyEx( HKEY_LOCAL_MACHINE, L"Drivers\\Capture\\Camera", 0, 0, &hKey ))
    {
        //mio
        return false;
    }

    if( ERROR_SUCCESS == RegQueryValueEx( hKey, L"MemoryModel", 0, &dwType, (BYTE *)&dwValue, &dwSize ) )
    {
        if(   ( REG_DWORD == dwType )
           && ( sizeof( DWORD ) == dwSize )
           && (( dwValue == CSPROPERTY_BUFFER_DRIVER ) || ( dwValue == CSPROPERTY_BUFFER_CLIENT_LIMITED ) || ( dwValue == CSPROPERTY_BUFFER_CLIENT_UNLIMITED )))
        {
            for( int i=0; i<MAX_SUPPORTED_PINS ; i++ )
            {
                m_SensorModeInfo[i].MemoryModel = (CSPROPERTY_BUFFER_MODE) dwValue;
            }
        }
    }

    // Find out if we should be using some other number of supported modes. The only
    // valid options are 2 or 3. Default to 2.
    if ( ERROR_SUCCESS == RegQueryValueEx( hKey, L"PinCount", 0, &dwType, (BYTE *)&dwValue, &dwSize ) )
    {
        if ( REG_DWORD == dwType
             && sizeof ( DWORD ) == dwSize
             && 3 == dwValue )
        {
            m_ulCTypes = 3;
        }
    }

    RegCloseKey( hKey );
    return true;
}

bool CCameraPdd::SetSensorFreeBuffer( ULONG ulModeType)
{
    DWORD dwBytes;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s  %s(%d)\n"), DBG_MSG_HEADER, _T(__FUNCTION__), ulModeType));

    if (ulModeType == CAPTURE)
    {
        BUFFER_DESC VideoBuffer;

        VideoBuffer.Type = VIDEO_CAPTURE;

        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_RELEASEBUFFER, &(VideoBuffer.Type), sizeof(int),NULL,NULL, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RELEASEBUFFER Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
    }
    else if (ulModeType == STILL)
    {
        BUFFER_DESC StillBuffer;
        StillBuffer.Type = STILL_CAPTURE;

        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_RELEASEBUFFER, &(StillBuffer.Type), sizeof(int),NULL,NULL, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RELEASEBUFFER Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
    }
    else if(ulModeType == PREVIEW)
    {
        BUFFER_DESC PreviewBuffer;
        PreviewBuffer.Type = PREVIEW_CAPTURE;

        if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_RELEASEBUFFER, &(PreviewBuffer.Type), sizeof(int),NULL,NULL, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RELEASEBUFFER Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

    }
    return true;
}


bool CCameraPdd::SetSensorFormat( ULONG ulModeType)
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s  %s(%d)\n"), DBG_MSG_HEADER, _T(__FUNCTION__), ulModeType));
    CAM_IMG_FORMAT format = CAM_FORMAT_NONE;
    DWORD dwBytes;
    // Target Format
    PCS_VIDEOINFOHEADER pCsVideoInfoHdr = &m_pCurrentFormat[ulModeType].VideoInfoHeader;
//    MODULE_DESCRIPTOR ModuleDesc;
    // Camera Sensor Source Format
//    ModuleGetFormat(&ModuleDesc);

    UINT biWidth        = pCsVideoInfoHdr->bmiHeader.biWidth;
    UINT biHeight       = abs(pCsVideoInfoHdr->bmiHeader.biHeight);
    DWORD biBitCount    = pCsVideoInfoHdr->bmiHeader.biBitCount;
    DWORD biCompression = pCsVideoInfoHdr->bmiHeader.biCompression;
    DWORD dwAvgTimePerFrame = (DWORD)pCsVideoInfoHdr->AvgTimePerFrame / 10000;


    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s  VideoInfoHeader:%d,%d,%d,0x%x) dwAvgTimePerFrame = %d\n"), DBG_MSG_HEADER,
                            biWidth, biHeight, biBitCount, biCompression, dwAvgTimePerFrame));
    // Prepare buffers here for Preview and Still mode.
    if ( (FOURCC_YUY2 == (biCompression & ~BI_SRCPREROTATE)) ||
         (FOURCC_YUYV == (biCompression & ~BI_SRCPREROTATE)))
    {
        format = CAM_FORMAT_YCBYCR422_1PLANE;
    }
    else if((FOURCC_YVYU == (biCompression & ~BI_SRCPREROTATE)))
    {
        format = CAM_FORMAT_YCRYCB422_1PLANE;
    }
    else if((FOURCC_UYVY == (biCompression & ~BI_SRCPREROTATE)))
    {
        format = CAM_FORMAT_CBYCRY422_1PLANE;
    }

    else if((FOURCC_YV12 == (biCompression & ~BI_SRCPREROTATE)))
    {
        format = CAM_FORMAT_YV12;
    }
    else if((FOURCC_NV12 == (biCompression & ~BI_SRCPREROTATE)))
    {
        format = CAM_FORMAT_NV12;
    }
    else if((CS_BI_BITFIELDS == (biCompression & ~BI_SRCPREROTATE)))
    {
        if(biBitCount == 24)
        {
            // BPP 3
            format = CAM_FORMAT_RGB24;
        }
        else
        {
            // BPP 2
            format = CAM_FORMAT_RGB16;
        }
    }
    else if((FOURCC_IJPG == (biCompression & ~BI_SRCPREROTATE)))
    {
        format = CAM_FORMAT_IJPG;
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("%s  format:%d\n"), DBG_MSG_HEADER,
                            format));

    // Set Camera Source Size?

    if (ulModeType == CAPTURE)
    {
        BUFFER_DESC VideoBuffer;
        IMG_TYPE_ITEM				ImgDesc;

		ImgDesc.width = biWidth;
		ImgDesc.height = biHeight;
		ImgDesc.outputFormat = format;

        // Set Target Video Size
        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_SET_VIDEOSIZE, &ImgDesc, sizeof(IMG_TYPE_ITEM), NULL, 0, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SET_VIDEOSIZE Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

        VideoBuffer.Height = biHeight;
        VideoBuffer.Width = biWidth;
        VideoBuffer.Format = format;
        VideoBuffer.Type = VIDEO_CAPTURE;

        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_PREPAREBUFFER, &VideoBuffer, sizeof(BUFFER_DESC), &m_CameraHWVideoBuffers, sizeof(CAMERA_DMA_BUFFER_INFO), &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_PREPAREBUFFER Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
    }
    else if (ulModeType == STILL)
    {
        BUFFER_DESC StillBuffer;
        StillBuffer.Height = biHeight;
        StillBuffer.Width = biWidth;
        StillBuffer.Format = format;
        StillBuffer.Type = STILL_CAPTURE;

        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_SET_STILLSIZE, &StillBuffer, sizeof(BUFFER_DESC), &m_CameraHWStillBuffer, sizeof(CAMERA_DMA_BUFFER_INFO), &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SET_STILLSIZE Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_PREPAREBUFFER, &StillBuffer, sizeof(BUFFER_DESC), &m_CameraHWStillBuffer, sizeof(CAMERA_DMA_BUFFER_INFO), &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_PREPAREBUFFER Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
    }
    else if(ulModeType == PREVIEW)
    {
        BUFFER_DESC PreviewBuffer;
        IMG_TYPE_ITEM				ImgDesc;

		ImgDesc.width = biWidth;
		ImgDesc.height = biHeight;
		ImgDesc.outputFormat = format;

        if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_SET_PREVIEWSIZE, &ImgDesc, sizeof(IMG_TYPE_ITEM),NULL, 0 , &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_SET_PREVIEWSIZE Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

        PreviewBuffer.Height = biHeight;
        PreviewBuffer.Width = biWidth;
        PreviewBuffer.Format = format;
        PreviewBuffer.Type = PREVIEW_CAPTURE;

        if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_PREPAREBUFFER, &PreviewBuffer, sizeof(BUFFER_DESC), &m_CameraHWPreviewBuffers, sizeof(CAMERA_DMA_BUFFER_INFO), &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_PREPAREBUFFER Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[FILTER]m_CameraHWPreviewBuffers[0].pY = 0x%x\r\n"),m_CameraHWPreviewBuffers[0].pY));
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[FILTER]m_CameraHWPreviewBuffers[1].pY = 0x%x\r\n"),m_CameraHWPreviewBuffers[1].pY));
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[FILTER]m_CameraHWPreviewBuffers[2].pY = 0x%x\r\n"),m_CameraHWPreviewBuffers[2].pY));
        DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("[FILTER]m_CameraHWPreviewBuffers[3].pY = 0x%x\r\n"),m_CameraHWPreviewBuffers[3].pY));
    }
    else
    {
        return FALSE;
    }

    return TRUE;
}

void CCameraPdd::SuspendCamera( )
{
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++%s()\n"), _T(__FUNCTION__)));

    if(m_bCameraVideoRunning)
    {
        m_bCameraVideoWasRunning = TRUE;
        if (m_CsState[CAPTURE]== CSSTATE_RUN)
        {
            SetSensorState(CAPTURE, CSSTATE_PAUSE);
        }
    }

    if(m_bCameraPreviewRunning)
    {
        m_bCameraPreviewWasRunning = TRUE;
        if (m_CsState[PREVIEW]== CSSTATE_RUN)
        {
            SetSensorState(PREVIEW, CSSTATE_PAUSE);
        }
    }

    if(m_iPinUseCount > 0)
    {
        CameraSleep();
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--%s()\n"), _T(__FUNCTION__)));

}

void CCameraPdd::ResumeCamera( )
{
    // Restart camera sensor if it was running before
    DWORD dwBytes;
    DWORD Setting = TRUE;

    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++%s()\n"), _T(__FUNCTION__)));

    if(m_iPinUseCount > 0)
    {
        // Request Initialization Camera module and Buffer
        if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_INIT_SENSOR, NULL, 0, NULL, 0, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_INIT_SENSOR Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }
        if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_INIT_SENSOR, NULL, 0, NULL, 0, &dwBytes, NULL) )
        {
            DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_INIT_SENSOR Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        }

    }

    if(m_bCameraVideoWasRunning)
    {
        m_bCameraVideoWasRunning = FALSE;
        SetSensorState(CAPTURE, CSSTATE_RUN);
    }
    if ( m_bCameraPreviewWasRunning )
    {
        m_bCameraPreviewWasRunning = FALSE;
        SetSensorState(PREVIEW, CSSTATE_RUN);
    }
}

void CCameraPdd::CameraVideoFrameCallback( DWORD dwContext )
{
    CCameraPdd * pCamDevice = reinterpret_cast<CCameraPdd *>( dwContext );

    // Video frame is ready - put it into stream

    if (NULL != pCamDevice)
    {
        pCamDevice->HandleCaptureInterrupt(CAPTURE);
    }
}

void CCameraPdd::CameraStillFrameCallback( DWORD dwContext )
{
    CCameraPdd * pCamDevice = reinterpret_cast<CCameraPdd *>( dwContext );

    // Still image frame is ready - put it into stream

    if (NULL != pCamDevice)
    {
        pCamDevice->HandleStillInterrupt();
    }
}

void CCameraPdd::CameraPreviewFrameCallback( DWORD dwContext )
{
    CCameraPdd * pCamDevice = reinterpret_cast<CCameraPdd *>( dwContext );

    // Video frame is ready - put it into stream

    if (NULL != pCamDevice)
    {
        pCamDevice->HandleCaptureInterrupt(PREVIEW);
    }
}

// These Methods comes from Old Hal function
int CCameraPdd::CameraZoom(int value)
{
    DWORD dwBytes;
    DWORD ZoomRatio = value;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++%s()\n"), _T(__FUNCTION__)));

    // Zoom Range Check
    // x1 zoom value = 9
    // x2 zoom value =19
    // ...
    // x6 zoom value =59
    if(value<0 || value > 60)
    {
        return FALSE;
    }
    // Set Device Zoom
    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_ZOOM, &ZoomRatio, sizeof(DWORD), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_ZOOM Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_ZOOM, &ZoomRatio, sizeof(DWORD), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_ZOOM Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--%s()\n"), _T(__FUNCTION__)));

    return TRUE;
}

// Standby mode to save power
void CCameraPdd::CameraSleep()
{
    DWORD dwBytes;
    DWORD Setting = FALSE;

    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++%s()\n"), _T(__FUNCTION__)));

    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_DEINIT_SENSOR, &Setting, sizeof(DWORD), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_DEINIT_SENSOR Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }
    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_DEINIT_SENSOR, &Setting, sizeof(DWORD), NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_DEINIT_SENSOR Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
    }

}

BOOL CCameraPdd::RequestCameraResources()
{
    DWORD dwBytes;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++%s()\n"), _T(__FUNCTION__)));

    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_RSC_REQUEST, NULL, 0, NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RSC_REQUEST for Video Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_RSC_REQUEST, NULL, 0, NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RSC_REQUEST for Preview Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--%s()\n"), _T(__FUNCTION__)));

    return TRUE;
}


BOOL CCameraPdd::ReleaseCameraResources()
{
    DWORD dwBytes;
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("++%s()\n"), _T(__FUNCTION__)));

    if ( !DeviceIoControl(g_hVideoCamera, IOCTL_CAM_RSC_RELEASE, NULL, 0, NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RSC_RELEASE for Video Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    if ( !DeviceIoControl(g_hPreviewCamera, IOCTL_CAM_RSC_RELEASE, NULL, 0, NULL, 0, &dwBytes, NULL) )
    {
        DBGMSG(CAMF_FUNC && CAMF_USR1, (_T("%s --%s() : IOCTL_CAM_RSC_RELEASE for Preview Failed\n\r"), DBG_MSG_HEADER, _T(__FUNCTION__)));
        return FALSE;
    }
    DBGMSG(CAMF_FUNC && CAMF_USR1,(TEXT("--%s()\n"), _T(__FUNCTION__)));

    return TRUE;
}

