//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  Header:  CAMIF_reg.h
//
//  Defines the Camera Interface control registers and definitions.
//
#ifndef __CAMIF_REG_H
#define __CAMIF_REG_H

#define LITTLE_ENDIAN

#if __cplusplus
extern "C" {
#endif

// Register Bitfield Definition
#if (CAMIF_IP_VER == 0320)
#elif (CAMIF_IP_VER == 0410)        // CI is prefix that means "Camera Interface"

typedef struct _CISRCFMTBITFIELD {
#ifdef  LITTLE_ENDIAN
    UINT32  SrcVsize_CAM    :14;    // Source Vertical Pixel Number (if WinOfsEn is 0, when V scale down must be multiple of PreVerRatio)        
    UINT32  Order422_CAM    : 2;    // See Comment #1        
    UINT32  SrcHsize_CAM    :13;    // Source Horizontal Pixel Number (should 16's multiple)(if WinOfsEn is 0, 4's multiple of PreHorRatio)        
    UINT32  In16bit         : 1;    // 1: ITU601 CbCr 16Bit Enable, 0: Disable        
    UINT32  UVOffset        : 1;    // 1: Cb=Cb+128, Cr=Cr+128        
    UINT32  ITU601_656n     : 1;    // 1: ITU601, 0:ITU656        
#else
    UINT32  ITU601_656n     : 1;    // 1: ITU601, 0:ITU656
    UINT32  UVOffset        : 1;    // 1: Cb=Cb+128, Cr=Cr+128
    UINT32  In16bit         : 1;    // 1: ITU601 CbCr 16Bit Enable, 0: Disable
    UINT32  SrcHsize_CAM    :13;    // Source Horizontal Pixel Number (should 16's multiple)(if WinOfsEn is 0, 4's multiple of PreHorRatio)
    UINT32  Order422_CAM    : 2;    // See Comment #1
    UINT32  SrcVsize_CAM    :14;    // Source Vertical Pixel Number (if WinOfsEn is 0, when V scale down must be multiple of PreVerRatio)
#endif
} BITFIELD_CISRCFMT;

#define BP_SELECT_ITU_INTERFACE     (31)
#define BP_UVOFFSET                 (30)
#define BP_IN16BIT                  (29)
#define BP_SRCHSIZE_CAM             (16)
#define BP_ORDER422_CAM             (14)
#define BP_SRCVSIZE_CAM             (0)

#define BS_SELECT_ITU_INTERFACE     (1<<BP_SELECT_ITU_INTERFACE)
#define BS_UVOFFSET                 (1<<BP_UVOFFSET)
#define BS_IN16BIT                  (1<<BP_IN16BIT)
#define BS_SRCHSIZE_CAM             (1<<BP_SRCHSIZE_CAM)
#define BS_ORDER422_CAM             (1<<BP_ORDER422_CAM)
#define BS_SRCVSIZE_CAM             (1<<BP_SRCVSIZE_CAM)

#define BC_SELECT_ITU_INTERFACE     (1)
#define BC_UVOFFSET                 (1)
#define BC_IN16BIT                  (1)
#define BC_SRCHSIZE_CAM             (13)
#define BC_ORDER422_CAM             (2)
#define BC_SRCVSIZE_CAM             (14)

#define BW_SELECT_ITU_INTERFACE     (0x1)
#define BW_UVOFFSET                 (0x1)
#define BW_IN16BIT                  (0x1)
#define BW_SRCHSIZE_CAM             (0x1FFF)
#define BW_ORDER422_CAM             (0x3)
#define BW_SRCVSIZE_CAM             (0x3FFF)

#define BM_SELECT_ITU_INTERFACE     (BW_SELECT_ITU_INTERFACE<<BP_SELECT_ITU_INTERFACE)
#define BM_UVOFFSET                 (BW_UVOFFSET<<BP_UVOFFSET)
#define BM_IN16BIT                  (BW_IN16BIT<<BP_IN16BIT)
#define BM_SRCHSIZE_CAM             (BW_SRCHSIZE_CAM<<BP_SRCHSIZE_CAM)
#define BM_ORDER422_CAM             (BW_ORDER422_CAM<<BP_ORDER422_CAM)
#define BM_SRCVSIZE_CAM             (BW_SRCVSIZE_CAM<<BP_SRCVSIZE_CAM)

typedef struct _CIWDOFSTBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  WinVerOfst      :12;        
    UINT32  Reserved2       : 2;        
    UINT32  ClrOvFiCr       : 1;        
    UINT32  ClrOvFiCb       : 1;        
    UINT32  WinHorOfst      :11;        
    UINT32  Reserved1       : 2;        
    UINT32  ClrOvRLB        : 1;        
    UINT32  ClrOvFiY        : 1;        
    UINT32  WinOfsEn        : 1;        
#else
    UINT32  WinOfsEn        : 1;
    UINT32  ClrOvFiY        : 1;
    UINT32  ClrOvRLB        : 1;
    UINT32  Reserved1       : 2;
    UINT32  WinHorOfst      :11;
    UINT32  ClrOvFiCb       : 1;
    UINT32  ClrOvFiCr       : 1;
    UINT32  Reserved2       : 2;
    UINT32  WinVerOfst      :12;
#endif
} BITFIELD_CIWDOFST;
#define BP_WINOFSEN                 (31)
#define BP_CLROVFIY                 (30)
#define BP_CLROVRLB                 (29)
#define BP_WINHOROFST               (16)
#define BP_CLROVFICB                (15)
#define BP_CLROVFICR                (14)
#define BP_WINVEROFST               (0)

#define BS_WINOFSEN                 (1<<BP_WINOFSEN)
#define BS_CLROVFIY                 (1<<BP_CLROVFIY)
#define BS_CLROVRLB                 (1<<BP_CLROVRLB)
#define BS_WINHOROFST               (1<<BP_WINHOROFST)
#define BS_CLROVFICB                (1<<BP_CLROVFICB)
#define BS_CLROVFICR                (1<<BP_CLROVFICR)
#define BS_WINVEROFST               (1<<BP_WINVEROFST)

#define BC_WINOFSEN                 (1)
#define BC_CLROVFIY                 (1)
#define BC_CLROVRLB                 (1)
#define BC_WINHOROFST               (11)
#define BC_CLROVFICB                (1)
#define BC_CLROVFICR                (1)
#define BC_WINVEROFST               (12)        

#define BW_WINOFSEN                 (0x1)
#define BW_CLROVFIY                 (0x1)
#define BW_CLROVRLB                 (0x1)
#define BW_WINHOROFST               (0x7FF)
#define BW_CLROVFICB                (0x1)
#define BW_CLROVFICR                (0x1)
#define BW_WINVEROFST               (0xFFF)        

#define BM_WINOFSEN                 (BW_WINOFSEN<<BP_WINOFSEN)
#define BM_CLROVFIY                 (BW_CLROVFIY<<BP_CLROVFIY)
#define BM_CLROVRLB                 (BW_CLROVRLB<<BP_CLROVRLB)
#define BM_WINHOROFST               (BW_WINHOROFST<<BP_WINHOROFST)
#define BM_CLROVFICB                (BW_CLROVFICB<<BP_CLROVFICB)
#define BM_CLROVFICR                (BW_CLROVFICR<<BP_CLROVFICR)
#define BM_WINVEROFST               (BW_WINVEROFST<<BP_WINVEROFST)

typedef struct _CIGCTRLBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  Cam_Interface   : 1;        
    UINT32  InvPolFIELD     : 1;        
    UINT32  FIELDMODE       : 1;        
    UINT32  SelCam_FIMC     : 1;        
    UINT32  InvPolHSYNC     : 1;        
    UINT32  Reserved5       : 2;        
    UINT32  Reserved4       : 1;        
    UINT32  Reserved3       : 8;        
    UINT32  IRQ_Enable      : 1;        
    UINT32  Reserved2       : 2;        
    UINT32  IRQ_CLR         : 1;        
    UINT32  IRQ_LEVEL       : 1;        
    UINT32  Href_mask       : 1;        
    UINT32  IRQ_Ovfen       : 1;        
    UINT32  Reserved1       : 1;        
    UINT32  InvPolHREF      : 1;        
    UINT32  InvPolVSYNC     : 1;        
    UINT32  InvPolPCLK      : 1;        
    UINT32  TestPattern     : 2;        
    UINT32  SelCam_ITU      : 1;        
    UINT32  CamRst_A        : 1;        
    UINT32  SwRst           : 1;        
#else
    UINT32  SwRst           : 1;
    UINT32  CamRst_A        : 1;
    UINT32  SelCam_ITU      : 1;
    UINT32  TestPattern     : 2;
    UINT32  InvPolPCLK      : 1;
    UINT32  InvPolVSYNC     : 1;
    UINT32  InvPolHREF      : 1;
    UINT32  Reserved1       : 1;
    UINT32  IRQ_Ovfen       : 1;
    UINT32  Href_mask       : 1;
    UINT32  IRQ_LEVEL       : 1;
    UINT32  IRQ_CLR         : 1;
    UINT32  Reserved2       : 2;
    UINT32  IRQ_Enable      : 1;
    UINT32  Reserved3       : 8;
    UINT32  Reserved4       : 1;
    UINT32  Reserved5       : 2;
    UINT32  InvPolHSYNC     : 1;
    UINT32  SelCam_FIMC     : 1;
    UINT32  FIELDMODE       : 1;
    UINT32  InvPolFIELD     : 1;
    UINT32  Cam_Interface   : 1;
#endif
} BITFIELD_CIGCTRL;
#define BP_CAM_SWRESET              (31)
#define BP_CAM_CAMRST_A             (30)
#define BP_CAM_SELCAM_ITU           (29)
#define BP_CAM_TESTPATTERN          (27)
#define BP_CAM_INVPOLPCLK           (26)
#define BP_CAM_INVPOLVSYNC          (25)
#define BP_CAM_INVPOLHREF           (24)
#define BP_CAM_IRQ_OVFEN            (22)
#define BP_CAM_HREF_MASK            (21)
#define BP_CAM_IRQ_LEVEL            (20)
#define BP_CAM_IRQ_CLR              (19)
#define BP_CAM_IRQ_ENABLE           (16)
#define BP_CAM_INVPOLHSYNC          (4)
#define BP_CAM_SELCAM_FIMC          (3)
#define BP_CAM_FIELDMODE            (2)
#define BP_CAM_INVPOLFIELD          (1)
#define BP_CAM_INTERFACE            (0)

#define BS_CAM_SWRESET              (1<<BP_CAM_SWRESET)
#define BS_CAM_CAMRST_A             (1<<BP_CAM_CAMRST_A)
#define BS_CAM_SELCAM_ITU           (1<<BP_CAM_SELCAM_ITU)
#define BS_CAM_TESTPATTERN          (1<<BP_CAM_TESTPATTERN)
#define BS_CAM_INVPOLPCLK           (1<<BP_CAM_INVPOLPCLK)
#define BS_CAM_INVPOLVSYNC          (1<<BP_CAM_INVPOLVSYNC)
#define BS_CAM_INVPOLHREF           (1<<BP_CAM_INVPOLHREF)
#define BS_CAM_IRQ_OVFEN            (1<<BP_CAM_IRQ_OVFEN)
#define BS_CAM_HREF_MASK            (1<<BP_CAM_HREF_MASK)
#define BS_CAM_IRQ_LEVEL            (1<<BP_CAM_IRQ_LEVEL)
#define BS_CAM_IRQ_CLR              (1<<BP_CAM_IRQ_CLR)
#define BS_CAM_IRQ_ENABLE           (1<<BP_CAM_IRQ_ENABLE)
#define BS_CAM_INVPOLHSYNC          (1<<BP_CAM_INVPOLHSYNC)
#define BS_CAM_SELCAM_FIMC          (1<<BP_CAM_SELCAM_FIMC)
#define BS_CAM_FIELDMODE            (1<<BP_CAM_FIELDMODE)
#define BS_CAM_INVPOLFIELD          (1<<BP_CAM_INVPOLFIELD)
#define BS_CAM_INTERFACE            (1<<BP_CAM_INTERFACE)        

#define BC_CAM_SWRESET              (1)
#define BC_CAM_CAMRST_A             (1)
#define BC_CAM_SELCAM_ITU           (1)
#define BC_CAM_TESTPATTERN          (2)
#define BC_CAM_INVPOLPCLK           (1)
#define BC_CAM_INVPOLVSYNC          (1)
#define BC_CAM_INVPOLHREF           (1)
#define BC_CAM_IRQ_OVFEN            (1)
#define BC_CAM_HREF_MASK            (1)
#define BC_CAM_IRQ_LEVEL            (1)
#define BC_CAM_IRQ_CLR              (1)
#define BC_CAM_IRQ_ENABLE           (1)
#define BC_CAM_INVPOLHSYNC          (1)
#define BC_CAM_SELCAM_FIMC          (1)
#define BC_CAM_FIELDMODE            (1)
#define BC_CAM_INVPOLFIELD          (1)
#define BC_CAM_INTERFACE            (1)        

#define BW_CAM_SWRESET              (0x1)
#define BW_CAM_CAMRST_A             (0x1)
#define BW_CAM_SELCAM_ITU           (0x1)
#define BW_CAM_TESTPATTERN          (0x3)
#define BW_CAM_INVPOLPCLK           (0x1)
#define BW_CAM_INVPOLVSYNC          (0x1)
#define BW_CAM_INVPOLHREF           (0x1)
#define BW_CAM_IRQ_OVFEN            (0x1)
#define BW_CAM_HREF_MASK            (0x1)
#define BW_CAM_IRQ_LEVEL            (0x1)
#define BW_CAM_IRQ_CLR              (0x1)
#define BW_CAM_IRQ_ENABLE           (0x1)
#define BW_CAM_INVPOLHSYNC          (0x1)
#define BW_CAM_SELCAM_FIMC          (0x1)
#define BW_CAM_FIELDMODE            (0x1)
#define BW_CAM_INVPOLFIELD          (0x1)
#define BW_CAM_INTERFACE            (0x1)        

#define BM_CAM_SWRESET              (BW_CAM_SWRESET<<BP_CAM_SWRESET)
#define BM_CAM_CAMRST_A             (BW_CAM_CAMRST_A<<BP_CAM_CAMRST_A)
#define BM_CAM_SELCAM_ITU           (BW_CAM_SELCAM_ITU<<BP_CAM_SELCAM_ITU)
#define BM_CAM_TESTPATTERN          (BW_CAM_TESTPATTERN<<BP_CAM_TESTPATTERN)
#define BM_CAM_INVPOLPCLK           (BW_CAM_INVPOLPCLK<<BP_CAM_INVPOLPCLK)
#define BM_CAM_INVPOLVSYNC          (BW_CAM_INVPOLVSYNC<<BP_CAM_INVPOLVSYNC)
#define BM_CAM_INVPOLHREF           (BW_CAM_INVPOLHREF<<BP_CAM_INVPOLHREF)
#define BM_CAM_IRQ_OVFEN            (BW_CAM_IRQ_OVFEN<<BP_CAM_IRQ_OVFEN)
#define BM_CAM_HREF_MASK            (BW_CAM_HREF_MASK<<BP_CAM_HREF_MASK)
#define BM_CAM_IRQ_LEVEL            (BW_CAM_IRQ_LEVEL<<BP_CAM_IRQ_LEVEL)
#define BM_CAM_IRQ_CLR              (BW_CAM_IRQ_CLR<<BP_CAM_IRQ_CLR)
#define BM_CAM_IRQ_ENABLE           (BW_CAM_IRQ_ENABLE<<BP_CAM_IRQ_ENABLE)
#define BM_CAM_INVPOLHSYNC          (BW_CAM_INVPOLHSYNC<<BP_CAM_INVPOLHSYNC)
#define BM_CAM_SELCAM_FIMC          (BW_CAM_SELCAM_FIMC<<BP_CAM_SELCAM_FIMC)
#define BM_CAM_FIELDMODE            (BW_CAM_FIELDMODE<<BP_CAM_FIELDMODE)
#define BM_CAM_INVPOLFIELD          (BW_CAM_INVPOLFIELD<<BP_CAM_INVPOLFIELD)
#define BM_CAM_INTERFACE            (BW_CAM_INTERFACE<<BP_CAM_INTERFACE)        

typedef struct _CIWDOFST2BITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  WinVerOfst2     :12;    // Pixel Unit        
    UINT32  Reserved2       : 4;        
    UINT32  WinHorOfst2     :12;    // Pixel Unit(2's multiple), SourceHSize-WinHorOfst-WinHorOfst2 should be 16's multiple and minimum 16        
    UINT32  Reserved1       : 4;        
#else
    UINT32  Reserved1       : 4;
    UINT32  WinHorOfst2     :12;    // Pixel Unit(2's multiple), SourceHSize-WinHorOfst-WinHorOfst2 should be 16's multiple and minimum 16
    UINT32  Reserved2       : 4;
    UINT32  WinVerOfst2     :12;    // Pixel Unit
#endif
} BITFIELD_CIWDOFST2;

#define BP_WINHOROFST2              (16)
#define BP_WINVEROFST2              (0)

#define BS_WINHOROFST2              (1<<BP_WINHOROFST2)
#define BS_WINVEROFST2              (1<<BP_WINVEROFST2)

#define BC_WINHOROFST2              (12)
#define BC_WINVEROFST2              (12)

#define BW_WINHOROFST2              (0xFFF)
#define BW_WINVEROFST2              (0xFFF)

#define BM_WINHOROFST2              (BW_WINHOROFST2<<BP_WINHOROFST2)
#define BM_WINVEROFST2              (BW_WINVEROFST2<<BP_WINVEROFST2)

typedef struct _CITRGFMTBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  TargetVsize     : 13;        
    UINT32  OutRot90        : 1;        
    UINT32  OutFlipMd       : 2;        
    UINT32  TargetHsize     : 13;        
    UINT32  OutFormat       : 2;        
    UINT32  InRot90         : 1;        
#else
    UINT32  InRot90         : 1;
    UINT32  OutFormat       : 2;
    UINT32  TargetHsize     : 13;
    UINT32  OutFlipMd       : 2;
    UINT32  OutRot90        : 1;
    UINT32  TargetVsize     : 13;
#endif
} BITFIELD_CITRGFMT;
#define BP_INROT90                  (31)
#define BP_OUTFORMAT                (29)
#define BP_TARGETHSIZE              (16)
#define BP_OUTFLIPMD                (14)
#define BP_OUTROT90                 (13)
#define BP_TARGETVSIZE              (0)

#define BS_INROT90                  (1<<BP_INROT90)
#define BS_OUTFORMAT                (1<<BP_OUTFORMAT)
#define BS_TARGETHSIZE              (1<<BP_TARGETHSIZE)
#define BS_OUTFLIPMD                (1<<BP_OUTFLIPMD)
#define BS_OUTROT90                 (1<<BP_OUTROT90)
#define BS_TARGETVSIZE              (1<<BP_TARGETVSIZE)

#define BC_INROT90                  (1)
#define BC_OUTFORMAT                (2)
#define BC_TARGETHSIZE              (13)
#define BC_OUTFLIPMD                (2)
#define BC_OUTROT90                 (1)
#define BC_TARGETVSIZE              (13)

#define BW_INROT90                  (0x1)
#define BW_OUTFORMAT                (0x3)
#define BW_TARGETHSIZE              (0x1FFF)
#define BW_OUTFLIPMD                (0x3)
#define BW_OUTROT90                 (0x1)
#define BW_TARGETVSIZE              (0x1FFF)

#define BM_INROT90                  (BW_INROT90<<BP_INROT90)
#define BM_OUTFORMAT                (BW_OUTFORMAT<<BP_OUTFORMAT)
#define BM_TARGETHSIZE              (BW_TARGETHSIZE<<BP_TARGETHSIZE)
#define BM_OUTFLIPMD                (BW_OUTFLIPMD<<BP_OUTFLIPMD)
#define BM_OUTROT90                 (BW_OUTROT90<<BP_OUTROT90)
#define BM_TARGETVSIZE              (BW_TARGETVSIZE<<BP_TARGETVSIZE)

typedef struct _CIOCTRLBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  Order422_out    : 2;    //         
    UINT32  LastIRQEn       : 1;        
    UINT32  C_INT_OUT       : 1;        
    UINT32  Reserved2       :20;        
    UINT32  Order2p_out     : 2;        
    UINT32  Reserved1       : 6;        
#else
    UINT32  Reserved1       : 6;
    UINT32  Order2p_out     : 2;
    UINT32  Reserved2       :20;
    UINT32  C_INT_OUT       : 1;
    UINT32  LastIRQEn       : 1;
    UINT32  Order422_out    : 2;    // 
#endif
} BITFIELD_CIOCTRL;

#define BP_ORDER2P_OUT              (24)
#define BP_C_INT_OUT                (3)
#define BP_LASTIRQEN                (2)
#define BP_ORDER422_OUT             (0)

#define BS_ORDER2P_OUT              (1<<BP_ORDER2P_OUT)
#define BS_C_INT_OUT                (1<<BP_C_INT_OUT)
#define BS_LASTIRQEN                (1<<BP_LASTIRQEN)
#define BS_ORDER422_OUT             (1<<BP_ORDER422_OUT)

#define BC_ORDER2P_OUT              (2)
#define BC_C_INT_OUT                (1)
#define BC_LASTIRQEN                (1)
#define BC_ORDER422_OUT             (1)

#define BW_ORDER2P_OUT              (0x3)
#define BW_C_INT_OUT                (0x1)
#define BW_LASTIRQEN                (0x1)
#define BW_ORDER422_OUT             (0x1)

#define BM_ORDER2P_OUT              (BW_ORDER2P_OUT<<BP_ORDER2P_OUT)
#define BM_C_INT_OUT                (BW_C_INT_OUT<<BP_C_INT_OUT)
#define BM_LASTIRQEN                (BW_LASTIRQEN<<BP_LASTIRQEN)
#define BM_ORDER422_OUT             (BW_ORDER422_OUT<<BP_ORDER422_OUT)

typedef struct _CISCPRERATIOBITFIELD{
#ifdef LITTLE_ENDIAN
    UINT32  PreVerRatio     : 7;
    UINT32  RESERVED1       : 9;
    UINT32  PreHorRatio     : 7;
    UINT32  RESERVED2       : 5;
    UINT32  SHfactor        : 4;
#else
    UINT32  SHfactor        : 4;        
    UINT32  RESERVED2       : 5;        
    UINT32  PreHorRatio     : 7;            
    UINT32  RESERVED1       : 9;            
    UINT32  PreVerRatio     : 7;            
#endif
} BITFIELD_CISCPRERATIO;

#define BP_SHFACTOR                 (28)
#define BP_PREHORRATIO              (16)
#define BP_PREVERRATIO              (0)

#define BS_SHFACTOR                 (1<<BP_SHFACTOR)
#define BS_PREHORRATIO              (1<<BP_PREHORRATIO)
#define BS_PREVERRATIO              (1<<BP_PREVERRATIO)

#define BC_SHFACTOR                 (4)
#define BC_PREHORRATIO              (7)
#define BC_PREVERRATIO              (7)

#define BW_SHFACTOR                 (0xF)
#define BW_PREHORRATIO              (0x7F)
#define BW_PREVERRATIO              (0x7F)

#define BM_SHFACTOR                 (BW_SHFACTOR<<BP_SHFACTOR)
#define BM_PREHORRATIO              (BW_PREHORRATIO<<BP_PREHORRATIO)
#define BM_PREVERRATIO              (BW_PREVERRATIO<<BP_PREVERRATIO)

typedef struct _CISCPREDSTBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  PreDstHeight    :14;
    UINT32  RESERVED1       : 2;
    UINT32  PreDstWidth     :14;
    UINT32  RESERVED2       : 2;
#else
    UINT32  RESERVED2       : 2;        
    UINT32  PreDstWidth     :14;        
    UINT32  RESERVED1       : 2;        
    UINT32  PreDstHeight    :14;        
#endif
} BITFIELD_CISCPREDST;

#define BP_PREDSTWIDTH              (16)
#define BP_PREDSTHEIGHT             (0)

#define BS_PREDSTWIDTH              (1<<BP_PREDSTWIDTH)
#define BS_PREDSTHEIGHT             (1<<BS_PREDSTHEIGHT)

#define BC_PREDSTWIDTH              (14)
#define BC_PREDSTHEIGHT             (14)

#define BW_PREDSTWIDTH              (0x3FFF)
#define BW_PREDSTHEIGHT             (0x3FFF)

#define BM_PREDSTWIDTH              (BW_PREDSTWIDTH<<BP_PREDSTWIDTH)
#define BM_PREDSTHEIGHT             (BW_PREDSTHEIGHT<<BP_PREDSTHEIGHT)

typedef struct _CISCCTRLBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  MainVerRatio    : 9;        
    UINT32  One2One         : 1;        
    UINT32  Ext_RGB         : 1;        
    UINT32  OutRGB_FMT      : 2;        
    UINT32  InRGB_FMT       : 2;        
    UINT32  ScalerStart     : 1;        
    UINT32  MainHorRatio    : 9;        
    UINT32  Interlace       : 1;        
    UINT32  LCDPathEn       : 1;        
    UINT32  CSCY2R          : 1;        
    UINT32  CSCR2Y          : 1;        
    UINT32  ScaleUp_V       : 1;        
    UINT32  ScaleUp_H       : 1;        
    UINT32  ScalerBypass    : 1;
#else
    UINT32  ScalerBypass    : 1;
    UINT32  ScaleUp_H       : 1;
    UINT32  ScaleUp_V       : 1;
    UINT32  CSCR2Y          : 1;
    UINT32  CSCY2R          : 1;
    UINT32  LCDPathEn       : 1;
    UINT32  Interlace       : 1;
    UINT32  MainHorRatio    : 9;
    UINT32  ScalerStart     : 1;
    UINT32  InRGB_FMT       : 2;
    UINT32  OutRGB_FMT      : 2;
    UINT32  Ext_RGB         : 1;
    UINT32  One2One         : 1;
    UINT32  MainVerRatio    : 9;
#endif
} BITFIELD_CISCCTRL;

#define BP_SCALERBYPASS             (31)
#define BP_SCALEUP_H                (30)
#define BP_SCALEUP_V                (29)
#define BP_CSCR2Y                   (28)
#define BP_CSCY2R                   (27)
#define BP_LCDPATHEN                (26)
#define BP_INTERLACE                (25)
#define BP_MAINHORRATIO             (16)
#define BP_SCALERSTART              (15)
#define BP_INRGB_FMT                (13)
#define BP_OUTRGB_FMT               (11)
#define BP_EXT_RGB                  (10)
#define BP_ONE2ONE                  (9)
#define BP_MAINVERRATIO             (0)

#define BS_SCALERBYPASS             (1<<BP_SCALERBYPASS)
#define BS_SCALEUP_H                (1<<BP_SCALEUP_H)
#define BS_SCALEUP_V                (1<<BP_SCALEUP_V)
#define BS_CSCR2Y                   (1<<BP_CSCR2Y)
#define BS_CSCY2R                   (1<<BP_CSCY2R)
#define BS_LCDPATHEN                (1<<BP_LCDPATHEN)
#define BS_INTERLACE                (1<<BP_INTERLACE)
#define BS_MAINHORRATIO             (1<<BP_MAINHORRATIO)
#define BS_SCALERSTART              (1<<BP_SCALERSTART)
#define BS_INRGB_FMT                (1<<BP_INRGB_FMT)
#define BS_OUTRGB_FMT               (1<<BP_OUTRGB_FMT)
#define BS_EXT_RGB                  (1<<BP_EXT_RGB)
#define BS_ONE2ONE                  (1<<BP_ONE2ONE)
#define BS_MAINVERRATIO             (1<<BP_MAINVERRATIO)

#define BC_SCALERBYPASS             (1)
#define BC_SCALEUP_H                (1)
#define BC_SCALEUP_V                (1)
#define BC_CSCR2Y                   (1)
#define BC_CSCY2R                   (1)
#define BC_LCDPATHEN                (1)
#define BC_INTERLACE                (1)
#define BC_MAINHORRATIO             (9)
#define BC_SCALERSTART              (1)
#define BC_INRGB_FMT                (2)
#define BC_OUTRGB_FMT               (2)
#define BC_EXT_RGB                  (1)
#define BC_ONE2ONE                  (1)
#define BC_MAINVERRATIO             (9)

#define BW_SCALERBYPASS             (0x1)
#define BW_SCALEUP_H                (0x1)
#define BW_SCALEUP_V                (0x1)
#define BW_CSCR2Y                   (0x1)
#define BW_CSCY2R                   (0x1)
#define BW_LCDPATHEN                (0x1)
#define BW_INTERLACE                (0x1)
#define BW_MAINHORRATIO             (0x1FF)
#define BW_SCALERSTART              (0x1)
#define BW_INRGB_FMT                (0x3)
#define BW_OUTRGB_FMT               (0x3)
#define BW_EXT_RGB                  (0x1)
#define BW_ONE2ONE                  (0x1)
#define BW_MAINVERRATIO             (0x1FF)

#define BM_SCALERBYPASS             (BW_SCALERBYPASS<<BP_SCALERBYPASS)
#define BM_SCALEUP_H                (BW_SCALEUP_H<<BP_SCALEUP_H)
#define BM_SCALEUP_V                (BW_SCALEUP_V<<BP_SCALEUP_V)
#define BM_CSCR2Y                   (BW_CSCR2Y<<BP_CSCR2Y)
#define BM_CSCY2R                   (BW_CSCY2R<<BP_CSCY2R)
#define BM_LCDPATHEN                (BW_LCDPATHEN<<BP_LCDPATHEN)
#define BM_INTERLACE                (BW_INTERLACE<<BP_INTERLACE)
#define BM_MAINHORRATIO             (BW_MAINHORRATIO<<BP_MAINHORRATIO)
#define BM_SCALERSTART              (BW_SCALERSTART<<BP_SCALERSTART)
#define BM_INRGB_FMT                (BW_INRGB_FMT<<BP_INRGB_FMT)
#define BM_OUTRGB_FMT               (BW_OUTRGB_FMT<<BP_OUTRGB_FMT)
#define BM_EXT_RGB                  (BW_EXT_RGB<<BP_EXT_RGB)
#define BM_ONE2ONE                  (BW_ONE2ONE<<BP_ONE2ONE)
#define BM_MAINVERRATIO             (BW_MAINVERRATIO<<BP_MAINVERRATIO)

typedef struct _CISTATUSBITFIELD{
#ifdef LITTLE_ENDIAN
    UINT32  Reserved1       :10;
    UINT32  FrameFieldStatus: 1;
    UINT32  FrameCptStatus  : 1;
    UINT32  IRQ_DMAend      : 1;
    UINT32  IRQ_CAM         : 1;
    UINT32  VVALID_B        : 1;
    UINT32  VVALID_A        : 1;
    UINT32  LastCaptureEnd  : 1;
    UINT32  FrameEnd        : 1;
    UINT32  OvRLB           : 1;
    UINT32  VSYNC_B         : 1;        // Read Only
    UINT32  VSYNC_A         : 1;        // Read Only
    UINT32  ImgCptEn_SC     : 1;
    UINT32  ImgCptEn        : 1;
    UINT32  FlipMd          : 2;
    UINT32  WinOfstEn       : 1;
    UINT32  FrameCnt        : 2;        // This counter value measn the next frame number
    UINT32  VSYNC           : 1;
    UINT32  OvFiCr          : 1;
    UINT32  OvFiCb          : 1;
    UINT32  OvFiY           : 1;
#else
    UINT32  OvFiY           : 1;
    UINT32  OvFiCb          : 1;
    UINT32  OvFiCr          : 1;
    UINT32  VSYNC           : 1;
    UINT32  FrameCnt        : 2;        // This counter value measn the next frame number
    UINT32  WinOfstEn       : 1;
    UINT32  FlipMd          : 2;
    UINT32  ImgCptEn        : 1;
    UINT32  ImgCptEn_SC     : 1;
    UINT32  VSYNC_A         : 1;        // Read Only
    UINT32  VSYNC_B         : 1;        // Read Only
    UINT32  OvRLB           : 1;
    UINT32  FrameEnd        : 1;
    UINT32  LastCaptureEnd  : 1;
    UINT32  VVALID_A        : 1;
    UINT32  VVALID_B        : 1;
    UINT32  IRQ_CAM         : 1;
    UINT32  IRQ_DMAend      : 1;
    UINT32  FrameCptStatus  : 1;
    UINT32  FrameFieldStatus: 1;
    UINT32  Reserved1       :10;
#endif
} BITFIELD_CISTATUS;

#define BP_CAM_OVERFLOW_FIFOY                   (31)
#define BP_CAM_OVERFLOW_FIFOCB                  (30)
#define BP_CAM_OVERFLOW_FIFOCR                  (29)
#define BP_CAM_VSYNC                            (28)
#define BP_CAM_FRAMECOUNT                       (26)
#define BP_CAM_WINDOWOFFSET_ENABLE              (25)
#define BP_CAM_FLIPMODE_OUTPUTDMA               (23) // 2bit
#define BP_CAM_IMAGECAPTURE_ENABLE              (22)
#define BP_CAM_IMAGECAPTURE_SCALER_ENABLE       (21)
#define BP_CAM_EXTCAM_A_VSYNC                   (20)
#define BP_CAM_EXTCAM_B_VSYNC                   (19)
#define BP_CAM_OVERFLOW_LINEBUFFER              (18)
#define BP_CAM_FRAME_END                        (17)
#define BP_CAM_LASTCAPTURE_END                  (16)
#define BP_CAM_EXTCAM_A_VVALID                  (15)
#define BP_CAM_EXTCAM_B_VVALID                  (14)
#define BP_CAM_IRQ_CAM                          (13)
#define BP_CAM_IRQ_DMA_END                      (12)
#define BP_CAM_FRAMECAPTRUE                     (11)
#define BP_CAM_FRAMEFIELD                       (10)

#define BS_CAM_OVERFLOW_FIFOY                   (1<<BP_CAM_OVERFLOW_FIFOY)
#define BS_CAM_OVERFLOW_FIFOCB                  (1<<BP_CAM_OVERFLOW_FIFOCB)
#define BS_CAM_OVERFLOW_FIFOCR                  (1<<BP_CAM_OVERFLOW_FIFOCR)
#define BS_CAM_VSYNC                            (1<<BP_CAM_VSYNC)
#define BS_CAM_FRAMECOUNT                       (1<<BP_CAM_FRAMECOUNT) // 2bit
#define BS_CAM_WINDOWOFFSET_ENABLE              (1<<BP_CAM_WINDOWOFFSET_ENABLE)
#define BS_CAM_FLIPMODE_OUTPUTDMA               (1<<BP_CAM_FLIPMODE_OUTPUTDMA) // 2bit
#define BS_CAM_IMAGECAPTURE_ENABLE              (1<<BP_CAM_IMAGECAPTURE_ENABLE)
#define BS_CAM_IMAGECAPTURE_SCALER_ENABLE       (1<<BP_CAM_IMAGECAPTURE_SCALER_ENABLE)
#define BS_CAM_EXTCAM_A_VSYNC                   (1<<BP_CAM_EXTCAM_A_VSYNC)
#define BS_CAM_EXTCAM_B_VSYNC                   (1<<BP_CAM_EXTCAM_B_VSYNC)
#define BS_CAM_OVERFLOW_LINEBUFFER              (1<<BP_CAM_OVERFLOW_LINEBUFFER)
#define BS_CAM_FRAME_END                        (1<<BP_CAM_FRAME_END)
#define BS_CAM_LASTCAPTURE_END                  (1<<BP_CAM_LASTCAPTURE_END)
#define BS_CAM_EXTCAM_A_VVALID                  (1<<BP_CAM_EXTCAM_A_VVALID)
#define BS_CAM_EXTCAM_B_VVALID                  (1<<BP_CAM_EXTCAM_B_VVALID)
#define BS_CAM_IRQ_CAM                          (1<<BP_CAM_IRQ_CAM)
#define BS_CAM_IRQ_DMA_END                      (1<<BP_CAM_IRQ_DMA_END)
#define BS_CAM_FRAMECAPTRUE                     (1<<BP_CAM_FRAMECAPTRUE)
#define BS_CAM_FRAMEFIELD                       (1<<BP_CAM_FRAMEFIELD)

#define BC_CAM_OVERFLOW_FIFOY                   (1)
#define BC_CAM_OVERFLOW_FIFOCB                  (1)
#define BC_CAM_OVERFLOW_FIFOCR                  (1)
#define BC_CAM_VSYNC                            (1)
#define BC_CAM_FRAMECOUNT                       (2)
#define BC_CAM_WINDOWOFFSET_ENABLE              (1)
#define BC_CAM_FLIPMODE_OUTPUTDMA               (2)
#define BC_CAM_IMAGECAPTURE_ENABLE              (1)
#define BC_CAM_IMAGECAPTURE_SCALER_ENABLE       (1)
#define BC_CAM_EXTCAM_A_VSYNC                   (1)
#define BC_CAM_EXTCAM_B_VSYNC                   (1)
#define BC_CAM_OVERFLOW_LINEBUFFER              (1)
#define BC_CAM_FRAME_END                        (1)
#define BC_CAM_LASTCAPTURE_END                  (1)
#define BC_CAM_EXTCAM_A_VVALID                  (1)
#define BC_CAM_EXTCAM_B_VVALID                  (1)
#define BC_CAM_IRQ_CAM                          (1)
#define BC_CAM_IRQ_DMA_END                      (1)
#define BC_CAM_FRAMECAPTRUE                     (1)
#define BC_CAM_FRAMEFIELD                       (1)        

#define BW_CAM_OVERFLOW_FIFOY                   (0x1)
#define BW_CAM_OVERFLOW_FIFOCB                  (0x1)
#define BW_CAM_OVERFLOW_FIFOCR                  (0x1)
#define BW_CAM_VSYNC                            (0x1)
#define BW_CAM_FRAMECOUNT                       (0x3)
#define BW_CAM_WINDOWOFFSET_ENABLE              (0x1)
#define BW_CAM_FLIPMODE_OUTPUTDMA               (0x3)
#define BW_CAM_IMAGECAPTURE_ENABLE              (0x1)
#define BW_CAM_IMAGECAPTURE_SCALER_ENABLE       (0x1)
#define BW_CAM_EXTCAM_A_VSYNC                   (0x1)
#define BW_CAM_EXTCAM_B_VSYNC                   (0x1)
#define BW_CAM_OVERFLOW_LINEBUFFER              (0x1)
#define BW_CAM_FRAME_END                        (0x1)
#define BW_CAM_LASTCAPTURE_END                  (0x1)
#define BW_CAM_EXTCAM_A_VVALID                  (0x1)
#define BW_CAM_EXTCAM_B_VVALID                  (0x1)
#define BW_CAM_IRQ_CAM                          (0x1)
#define BW_CAM_IRQ_DMA_END                      (0x1)
#define BW_CAM_FRAMECAPTRUE                     (0x1)
#define BW_CAM_FRAMEFIELD                       (0x1)        

#define BM_CAM_OVERFLOW_FIFOY                   (BW_CAM_OVERFLOW_FIFOY<<BP_CAM_OVERFLOW_FIFOY)
#define BM_CAM_OVERFLOW_FIFOCB                  (BW_CAM_OVERFLOW_FIFOCB<<BP_CAM_OVERFLOW_FIFOCB)
#define BM_CAM_OVERFLOW_FIFOCR                  (BW_CAM_OVERFLOW_FIFOCR<<BP_CAM_OVERFLOW_FIFOCR)
#define BM_CAM_VSYNC                            (BW_CAM_VSYNC<<BP_CAM_VSYNC)
#define BM_CAM_FRAMECOUNT                       (BW_CAM_FRAMECOUNT<<BP_CAM_FRAMECOUNT)
#define BM_CAM_WINDOWOFFSET_ENABLE              (BW_CAM_WINDOWOFFSET_ENABLE<<BP_CAM_WINDOWOFFSET_ENABLE)
#define BM_CAM_FLIPMODE_OUTPUTDMA               (BW_CAM_FLIPMODE_OUTPUTDMA<<BP_CAM_FLIPMODE_OUTPUTDMA)
#define BM_CAM_IMAGECAPTURE_ENABLE              (BW_CAM_IMAGECAPTURE_ENABLE<<BP_CAM_IMAGECAPTURE_ENABLE)
#define BM_CAM_IMAGECAPTURE_SCALER_ENABLE       (BW_CAM_IMAGECAPTURE_SCALER_ENABLE<<BP_CAM_IMAGECAPTURE_SCALER_ENABLE)
#define BM_CAM_EXTCAM_A_VSYNC                   (BW_CAM_EXTCAM_A_VSYNC<<BP_CAM_EXTCAM_A_VSYNC)
#define BM_CAM_EXTCAM_B_VSYNC                   (BW_CAM_EXTCAM_B_VSYNC<<BP_CAM_EXTCAM_B_VSYNC)
#define BM_CAM_OVERFLOW_LINEBUFFER              (BW_CAM_OVERFLOW_LINEBUFFER<<BP_CAM_OVERFLOW_LINEBUFFER)
#define BM_CAM_FRAME_END                        (BW_CAM_FRAME_END<<BP_CAM_FRAME_END)
#define BM_CAM_LASTCAPTURE_END                  (BW_CAM_LASTCAPTURE_END<<BP_CAM_LASTCAPTURE_END)
#define BM_CAM_EXTCAM_A_VVALID                  (BW_CAM_EXTCAM_A_VVALID<<BP_CAM_EXTCAM_A_VVALID)
#define BM_CAM_EXTCAM_B_VVALID                  (BW_CAM_EXTCAM_B_VVALID<<BP_CAM_EXTCAM_B_VVALID)
#define BM_CAM_IRQ_CAM                          (BW_CAM_IRQ_CAM<<BP_CAM_IRQ_CAM)
#define BM_CAM_IRQ_DMA_END                      (BW_CAM_IRQ_DMA_END<<BP_CAM_IRQ_DMA_END)
#define BM_CAM_FRAMECAPTRUE                     (BW_CAM_FRAMECAPTRUE<<BP_CAM_FRAMECAPTRUE)
#define BM_CAM_FRAMEFIELD                       (BW_CAM_FRAMEFIELD<<BP_CAM_FRAMEFIELD)

typedef struct _CIIMGCPTBITFIELD{
#ifdef  LITTLE_ENDIAN
    UINT32  Reserved3       :10;        
    UINT32  Cpt_FrCnt       : 8;        
    UINT32  Cpt_FrMod       : 1;        
    UINT32  Cpt_FrPtr       : 5;        
    UINT32  Reserved2       : 1;        
    UINT32  Cpt_FrEn        : 1;
    UINT32  Reserved1       : 4;        
    UINT32  ImgCptEn_Sc     : 1;        
    UINT32  ImgCptEn        : 1;        
#else
    UINT32  ImgCptEn        : 1;
    UINT32  ImgCptEn_Sc     : 1;
    UINT32  Reserved1       : 4;
    UINT32  Cpt_FrEn        : 1;
    UINT32  Reserved2       : 1;
    UINT32  Cpt_FrPtr       : 5;
    UINT32  Cpt_FrMod       : 1;
    UINT32  Cpt_FrCnt       : 8;
    UINT32  Reserved3       :10;
#endif
} BITFIELD_CIIMGCPT;

#define BP_IMGCPTEN             (31)
#define BP_IMGCPTEN_SC          (30)
#define BP_CPT_FREN             (25)
#define BP_CPT_FRPTR            (19)
#define BP_CPT_FRMOD            (18)
#define BP_CPT_FRCNT            (10)

#define BS_IMGCPTEN             (1<<BP_IMGCPTEN)
#define BS_IMGCPTEN_SC          (1<<BP_IMGCPTEN_SC)
#define BS_CPT_FREN             (1<<BP_CPT_FREN)
#define BS_CPT_FRPTR            (1<<BP_CPT_FRPTR)
#define BS_CPT_FRMOD            (1<<BP_CPT_FRMOD)
#define BS_CPT_FRCNT            (1<<BP_CPT_FRCNT)

#define BC_IMGCPTEN             (1)
#define BC_IMGCPTEN_SC          (1)
#define BC_CPT_FREN             (1)
#define BC_CPT_FRPTR            (5)
#define BC_CPT_FRMOD            (1)
#define BC_CPT_FRCNT            (8)

#define BW_IMGCPTEN             (0x1)
#define BW_IMGCPTEN_SC          (0x1)
#define BW_CPT_FREN             (0x1)
#define BW_CPT_FRPTR            (0x1F)
#define BW_CPT_FRMOD            (0x1)
#define BW_CPT_FRCNT            (0xFF)

#define BM_IMGCPTEN             (BW_IMGCPTEN<<BP_IMGCPTEN)
#define BM_IMGCPTEN_SC          (BW_IMGCPTEN_SC<<BP_IMGCPTEN_SC)
#define BM_CPT_FREN             (BW_CPT_FREN<<BP_CPT_FREN)
#define BM_CPT_FRPTR            (BW_CPT_FRPTR<<BP_CPT_FRPTR)
#define BM_CPT_FRMOD            (BW_CPT_FRMOD<<BP_CPT_FRMOD)
#define BM_CPT_FRCNT            (BW_CPT_FRCNT<<BP_CPT_FRCNT)

typedef struct _CIREAL_ISIZEBITFIELD{
#ifdef LITTLE_ENDIAN
    UINT32  REAL_WIDTH          :14;        
    UINT32  RESERVED1           : 2;        
    UINT32  REAL_HEIGHT         :14;        
    UINT32  ADDR_CH_DIS         : 1;        
    UINT32  AutoLoadEnable      : 1;        
#else
    UINT32  AutoLoadEnable      : 1;
    UINT32  ADDR_CH_DIS         : 1;
    UINT32  REAL_HEIGHT         :14;
    UINT32  RESERVED1           : 2;
    UINT32  REAL_WIDTH          :14;
#endif
} BITFIELD_CIREAL_ISIZE;

#define BP_AUTOLOADENABLE       (31)
#define BP_ADDR_CH_DIS          (30)
#define BP_REAL_HEIGHT          (16)
#define BP_REAL_WIDTH           (0)

#define BS_AUTOLOADENABLE       (1<<BP_AUTOLOADENABLE)
#define BS_ADDR_CH_DIS          (1<<BP_ADDR_CH_DIS)
#define BS_REAL_HEIGHT          (1<<BP_REAL_HEIGHT)
#define BS_REAL_WIDTH           (1<<BP_REAL_WIDTH)

#define BC_AUTOLOADENABLE       (1)
#define BC_ADDR_CH_DIS          (1)
#define BC_REAL_HEIGHT          (14)
#define BC_REAL_WIDTH           (14)

#define BW_AUTOLOADENABLE       (0x1)
#define BW_ADDR_CH_DIS          (0x1)
#define BW_REAL_HEIGHT          (0x3FFF)
#define BW_REAL_WIDTH           (0x3FFF)

#define BM_AUTOLOADENABLE       (BW_AUTOLOADENABLE<<BP_AUTOLOADENABLE)
#define BM_ADDR_CH_DIS          (BW_ADDR_CH_DIS<<BP_ADDR_CH_DIS)
#define BM_REAL_HEIGHT          (BW_REAL_HEIGHT<<BP_REAL_HEIGHT)
#define BM_REAL_WIDTH           (BW_REAL_WIDTH<<BP_REAL_WIDTH)

typedef struct _CIMSCTRLBITFIELD{
#ifdef LITTLE_ENDIAN
    UINT32  ENVID_M             : 1;        
    UINT32  InFormat_M          : 2;        
    UINT32  SEL_DMA_CAM         : 1;        
    UINT32  Order422_M          : 2;        
    UINT32  EOF_M               : 1;        
    UINT32  RESERVED3           : 6;        
    UINT32  InFlipMd            : 2;        
    UINT32  C_INT_IN            : 1;        
    UINT32  Order2p_in          : 2;        
    UINT32  RESERVED2           : 6;        
    UINT32  Successive_cnt      : 4;        
    UINT32  RESERVED1           : 4;        
#else
    UINT32  RESERVED1           : 4;
    UINT32  Successive_cnt      : 4;
    UINT32  RESERVED2           : 6;
    UINT32  Order2p_in          : 2;
    UINT32  C_INT_IN            : 1;
    UINT32  InFlipMd            : 2;
    UINT32  RESERVED3           : 6;
    UINT32  EOF_M               : 1;
    UINT32  Order422_M          : 2;
    UINT32  SEL_DMA_CAM         : 1;
    UINT32  InFormat_M          : 2;
    UINT32  ENVID_M             : 1;
#endif
} BITFIELD_CIMSCTRL;
#define BP_SUCCESSIVE_CNT       (24)
#define BP_ORDER2P_IN           (16)
#define BP_C_INT_IN             (15)
#define BP_INFLIPMD             (13)
#define BP_EOF_M                (6)
#define BP_ORDER422_M           (4)
#define BP_SEL_DMA_CAM          (3)
#define BP_INFORMAT_M           (1)
#define BP_ENVID_M              (0)

#define BS_SUCCESSIVE_CNT       (1<<BP_SUCCESSIVE_CNT)
#define BS_ORDER2P_IN           (1<<BP_ORDER2P_IN)
#define BS_C_INT_IN             (1<<BP_C_INT_IN)
#define BS_INFLIPMD             (1<<BP_INFLIPMD)
#define BS_EOF_M                (1<<BP_EOF_M)
#define BS_ORDER422_M           (1<<BP_ORDER422_M)
#define BS_SEL_DMA_CAM          (1<<BP_SEL_DMA_CAM)
#define BS_INFORMAT_M           (1<<BP_INFORMAT_M)
#define BS_ENVID_M              (1<<BP_ENVID_M)

#define BC_SUCCESSIVE_CNT       (4)
#define BC_ORDER2P_IN           (2)
#define BC_C_INT_IN             (1)
#define BC_INFLIPMD             (2)
#define BC_EOF_M                (1)
#define BC_ORDER422_M           (2)
#define BC_SEL_DMA_CAM          (1)
#define BC_INFORMAT_M           (2)
#define BC_ENVID_M              (1)

#define BW_SUCCESSIVE_CNT       (0xF)
#define BW_ORDER2P_IN           (0x3)
#define BW_C_INT_IN             (0x1)
#define BW_INFLIPMD             (0x3)
#define BW_EOF_M                (0x1)
#define BW_ORDER422_M           (0x3)
#define BW_SEL_DMA_CAM          (0x1)
#define BW_INFORMAT_M           (0x3)
#define BW_ENVID_M              (0x1)

#define BM_SUCCESSIVE_CNT       (BW_SUCCESSIVE_CNT<<BP_SUCCESSIVE_CNT)
#define BM_ORDER2P_IN           (BW_ORDER2P_IN<<BP_ORDER2P_IN)
#define BM_C_INT_IN             (BW_C_INT_IN<<BP_C_INT_IN)
#define BM_INFLIPMD             (BW_INFLIPMD<<BP_INFLIPMD)
#define BM_EOF_M                (BW_EOF_M<<BP_EOF_M)
#define BM_ORDER422_M           (BW_ORDER422_M<<BP_ORDER422_M)
#define BM_SEL_DMA_CAM          (BW_SEL_DMA_CAM<<BP_SEL_DMA_CAM)
#define BM_INFORMAT_M           (BW_INFORMAT_M<<BP_INFORMAT_M)
#define BM_ENVID_M              (BW_ENVID_M<<BP_ENVID_M)

typedef struct _CIORGISIZEBITFIELD{
#ifdef LITTLE_ENDIAN
    UINT32  ORG_IN_H            :14;
    UINT32  RESERVED2           : 2;        
    UINT32  ORG_IN_V            :14;
    UINT32  RESERVED1           : 2;        
#else
    UINT32  RESERVED1           : 2;
    UINT32  ORG_IN_V            :14;
    UINT32  RESERVED2           : 2;
    UINT32  ORG_IN_H            :14;
#endif
} BITFIELD_CIORGISIZE;
#define BP_ORG_IN_V             (16)
#define BP_ORG_IN_H             (0)

#define BS_ORG_IN_V             (1<<BP_ORG_IN_V)
#define BS_ORG_IN_H             (1<<BP_ORG_IN_H)

#define BC_ORG_IN_V             (14)
#define BC_ORG_IN_H             (14)

#define BW_ORG_IN_V             (0x3FFF)
#define BW_ORG_IN_H             (0x3FFF)

#define BM_ORG_IN_V             (BW_ORG_IN_V<<BP_ORG_IN_V)
#define BM_ORG_IN_H             (BW_ORG_IN_H<<BP_ORG_IN_H)

typedef struct _CIORGOSIZEBITFIELD{
#ifdef LITTLE_ENDIAN
    UINT32  ORG_OUT_H           :14;
    UINT32  RESERVED2           : 2;        
    UINT32  ORG_OUT_V           :14;
    UINT32  RESERVED1           : 2;        
#else
    UINT32  RESERVED1           : 2;
    UINT32  ORG_OUT_V           :14;
    UINT32  RESERVED2           : 2;
    UINT32  ORG_OUT_H           :14;
#endif
} BITFIELD_CIORGOSIZE;
#define BP_ORG_OUT_V            (16)
#define BP_ORG_OUT_H            (0)

#define BS_ORG_OUT_V            (1<<BP_ORG_OUT_V)
#define BS_ORG_OUT_H            (1<<BP_ORG_OUT_H)

#define BC_ORG_OUT_V            (14)
#define BC_ORG_OUT_H            (14)

#define BW_ORG_OUT_V            (0x3FFF)
#define BW_ORG_OUT_H            (0x3FFF)

#define BM_ORG_OUT_V            (BW_ORG_OUT_V<<BP_ORG_OUT_V)
#define BM_ORG_OUT_H            (BW_ORG_OUT_H<<BP_ORG_OUT_H)

#endif




//------------------------------------------------------------------------------
// Camera SFR Registers' Definition
#if (CAMIF_IP_VER == 0320)
typedef struct {
    UINT32      CISRCFMT;        // 00
    UINT32      CIWDOFST;        // 04
    UINT32      CIGCTRL;         // 08
    UINT32      CIFCTRL1;        // 0c
    UINT32      CIFCTRL2;        // 10
    UINT32      CIDOWSFT2;       // 14    
    UINT32      CICOYSA1;        // 18
    UINT32      CICOYSA2;        // 1c
    UINT32      CICOYSA3;        // 20
    UINT32      CICOYSA4;        // 24
    UINT32      CICOCBSA1;       // 28
    UINT32      CICOCBSA2;       // 2c
    UINT32      CICOCBSA3;       // 30
    UINT32      CICOCBSA4;       // 34
    UINT32      CICOCRSA1;       // 38
    UINT32      CICOCRSA2;       // 3c
    UINT32      CICOCRSA3;       // 40
    UINT32      CICOCRSA4;       // 44
    UINT32      CICOTRGFMT;      // 48
    UINT32      CICOCTRL;        // 4c
    UINT32      CICOSCPRERATIO;  // 50
    UINT32      CICOSCPREDST;    // 54
    UINT32      CICOSCCTRL;      // 58
    UINT32      CICOTAREA;       // 5c
    UINT32      PAD4;            // 60
    UINT32      CICOSTATUS;      // 64
    UINT32      PAD5;            // 68
    UINT32      CIPRYSA1;        // 6c
    UINT32      CIPRYSA2;        // 70
    UINT32      CIPRYSA3;        // 74
    UINT32      CIPRYSA4;        // 78
    UINT32      CIPRCBSA1;       // 7C
    UINT32      CIPRCBSA2;       // 80
    UINT32      CIPRCBSA3;       // 84
    UINT32      CIPRCBSA4;       // 88
    UINT32      CIPRCRSA1;       // 8C
    UINT32      CIPRCRSA2;       // 90
    UINT32      CIPRCRSA3;       // 94
    UINT32      CIPRCRSA4;       // 98        
    UINT32      CIPRTRGFMT;      // 9c
    UINT32      CIPRCTRL;        // A0
    UINT32      CIPRSCPRERATIO;  // A4
    UINT32      CIPRSCPREDST;    // A8
    UINT32      CIPRSCCTRL;      // AC
    UINT32      CIPRTAREA;       // B0
    UINT32      PAD6;            // B4
    UINT32      CIPRSTATUS;      // B8
    UINT32      PAD7;            // BC
    UINT32      CIIMGCPT;        // C0
    UINT32      CICPTSEQ;        // C4
    UINT32      PAD8;            // C8
    UINT32      PAD9;            // CC
    UINT32      CIIMGEFF;        // D0
    UINT32      MSCOY0SA;        // D4
    UINT32      MSCOCB0SA;       // D8
    UINT32      MSCOCR0SA;       // DC
    UINT32      MSCOY0END;       // E0
    UINT32      MSCOCB0END;      // E4
    UINT32      MSCOCR0END;      // E8
    UINT32      MSCOYOFF;        // EC
    UINT32      MSCOCBOFF;       // F0
    UINT32      MSCOCROFF;       // F4
    UINT32      MSCOWIDTH;       // F8
    UINT32      MSCOCTRL;        // FC
    UINT32      MSPRY0SA;        // 100
    UINT32      MSPRCB0SA;       // 104
    UINT32      MSPRCR0SA;       // 108
    UINT32      MSPRY0END;       // 10C
    UINT32      MSPRCB0END;      // 110
    UINT32      MSPRCR0END;      // 114
    UINT32      MSPRYOFF;        // 118
    UINT32      MSPRCBOFF;       // 11C
    UINT32      MSPRCROFF;       // 120
    UINT32      MSPRWIDTH;       // 124
    UINT32      CIMSCTRL;        // 128
    UINT32      CICOSCOSY;       // 12C
    UINT32      CICOSCOSCB;      // 130
    UINT32      CICOSCOSCR;      // 134
    UINT32      CIPRSCOSY;       // 138
    UINT32      CIPRSCOSCB;      // 13C
    UINT32      CIPRSCOSCR;      // 140
}CAMIF_REG,*PCAMIF_REG;    
#elif (CAMIF_IP_VER == 0410)        // CI is prefix that means "Camera Interface"
typedef struct {
    union{
        UINT32      CISRCFMT;       // 00       // Source Format
        BITFIELD_CISRCFMT       BFS_CISRCFMT;

    };
    union{
        UINT32      CIWDOFST;       // 04       // Window Offset
        BITFIELD_CIWDOFST       BFS_CIWDOFST;
    };
    union{
        UINT32      CIGCTRL;        // 08       // Global Control
        BITFIELD_CIGCTRL        BFS_CIGCTRL;
    };
    UINT8       RESERVED1[8];   // 0c~13    // RESERVED
    
    union{
        UINT32      CIWDOFST2;      // 14       // Window Offset 2 
        BITFIELD_CIWDOFST2      BFS_CIWDOFST2;
    };
    UINT32      CIOYSA1;        // 18       // Y Channel 1st Frame Start Address for Output DMA, YCbCr 2/3Plane -> Y 1st Frame, YCbCr 1Plane -> YCbCr 1st Frame, RGB -> RGB 1st Frame
    UINT32      CIOYSA2;        // 1c       // Y Channel 2nd Frame Start Address for Output DMA, YCbCr 2/3Plane -> Y 2nd Frame, YCbCr 1Plane -> YCbCr 2nd Frame, RGB -> RGB 2nd Frame
    UINT32      CIOYSA3;        // 20       // Y Channel 3rd Frame Start Address for Output DMA, YCbCr 2/3Plane -> Y 3rd Frame, YCbCr 1Plane -> YCbCr 3rd Frame, RGB -> RGB 3rd Frame
    UINT32      CIOYSA4;        // 24       // Y Channel 4th Frame Start Address for Output DMA, YCbCr 2/3Plane -> Y 4th Frame, YCbCr 1Plane -> YCbCr 4th Frame, RGB -> RGB 4th Frame
    UINT32      CIOCBSA1;       // 28       // Cb Channel 1st Frame Start Address for Output DMA
    UINT32      CIOCBSA2;       // 2c       // Cb Channel 2nd Frame Start Address for Output DMA
    UINT32      CIOCBSA3;       // 30       // Cb Channel 3rd Frame Start Address for Output DMA
    UINT32      CIOCBSA4;       // 34       // Cb Channel 4th Frame Start Address for Output DMA
    UINT32      CIOCRSA1;       // 38       // Cr Channel 1st Frame Start Address for Output DMA
    UINT32      CIOCRSA2;       // 3c       // Cr Channel 2nd Frame Start Address for Output DMA
    UINT32      CIOCRSA3;       // 40       // Cr Channel 3rd Frame Start Address for Output DMA
    UINT32      CIOCRSA4;       // 44       // Cr Channel 4th Frame Start Address for Output DMA
    union{
        UINT32      CITRGFMT;       // 48       // Target Image Format
        BITFIELD_CITRGFMT       BFS_CITRGFMT;
    };
    union {
        UINT32      CIOCTRL;        // 4c       // Output DMA Control Register
        BITFIELD_CIOCTRL        BFS_CIOCTRL;
    };
    union {
        UINT32      CISCPRERATIO;   // 50       // Pre-scaler Ratio 
        BITFIELD_CISCPRERATIO   BFS_CISCPRERATIO;
    };
    union {
        UINT32      CISCPREDST;     // 54       // Pre-scaler Destination Size
        BITFIELD_CISCPREDST     BFS_CISCPREDST;
    };
    union {
        UINT32      CISCCTRL;       // 58       // Main-scaler Control
        BITFIELD_CISCCTRL       BFS_CISCCTRL;
       
    };
    UINT32      CITAREA;        // 5c       // Target Area
    UINT8       RESERVED2[4];   // 60~63    // RESERVED

    union {
        UINT32      CISTATUS;       // 64       // Status        
        BITFIELD_CISTATUS       BFS_CISTATUS;
    };
    UINT8       RESERVED3[0x58];// 68~BF    // RESERVED
    union {
        UINT32      CIIMGCPT;       // C0       // Image Capture Enable Command
        BITFIELD_CIIMGCPT       BFS_CIIMGCPT;
    };
    
    UINT32      CICPTSEQ;       // C4       // Image Capture Sequence
    UINT8       RESERVED4[8];   // C8~CF    // RESERVED
    UINT32      CIIMGEFF;       // D0       // Image Effect
    UINT32      CIIYSA0;        // D4       // Y Frame Start Address for Input DMA
    UINT32      CIICBSA0;       // D8       // Cb Frame Start Address for Input DMA
    UINT32      CIICRSA0;       // DC       // Cr Frame Start Address for Input DMA
    UINT8       RESERVED5[0x18];// E0~F7    // RESERVED
    union {
        UINT32      CIREAL_ISIZE;   // F8       // Real Input DMA Image Size
        BITFIELD_CIREAL_ISIZE   BFS_CIREAL_ISIZE;
    };
    union {
        UINT32      CIMSCTRL;       // FC       // Input DMA Control Register
        BITFIELD_CIMSCTRL       BFS_CIMSCTRL;
    };
    
    UINT8       RESERVED6[0x68];// 100~167  // RESERVED
    UINT32      CIOYOFF;        // 168      // Output DMA Y Offset
    UINT32      CIOCBOFF;       // 16C      // Output DMA Cb Offset
    UINT32      CIOCROFF;       // 170      // Output DMA Cr Offset
    UINT32      CIIYOFF;        // 174      // Input DMA Y Offset
    UINT32      CIICBOFF;       // 178      // Input DMA Cb Offset
    UINT32      CIICROFF;       // 17c      // Input DMA Cr Offset
    union {
        UINT32      CIORGISIZE;       // 180      // Input DMA Original Image Size
        BITFIELD_CIORGISIZE     BFS_CIORGISIZE;
    };
    union {
        UINT32      CIORGOSIZE;       // 184      // Output DMA Original Image Size
        BITFIELD_CIORGOSIZE     BFS_CIORGOSIZE;
    };
    UINT32      CIEXTEN;        // 188      // Real Output DMA Image Size
    UINT32      CIDMAPARAM;     // 18c      // DMA Parameter Register
    UINT8       RESERVED7[4];   // 190~193  // RESERVED
    UINT32      CSIIMGFMT;      // 194      // MIPI CSI Image Format Register
}CAMIF_REG,*PCAMIF_REG;
#else
#error  NOT_DEFINED_IP_VER
#endif


//------------------------------------------------------------------------------

#if __cplusplus
}
#endif

/*  Additional Comments

  = Comment #1
        8Bit mode(In16bit == 0)
            00 : Y0 Cb0 Y1 Cr0
            01 : Y0 Cr0 Y1 Cb0
            10 : Cb0 Y0 Cr0 Y1
            11 : Cr0 Y0 Cb0 Y1
        16Bit mode(In16bit == 1)
            00 : Y0 Y1 Y2 Y3 Cb0 Cr0 Cb1 Cr1
            01 : Y0 Y1 Y2 Y3 Cr0 Cb0 Cr1 Cb1
            
*/

#endif // __CAMIF_REG_H
