//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  File:  profiler.c
//
//  This file contains implementation of profiler module suitable for the
//  Samsung S5PV210 CPU/SoC with count/compare timer.  The routines
//  use match register 2 (M2) for the profiling timer.
//
#include <windows.h>
#include <nkintr.h>
#include <oal.h>
#include "soc_cfg.h"
#include "base_regs.h"
#include "register_map.h"
#include "bsp_gpio.h"
#include "clkinfo.h"
#include "bsp_args.h"
#include "image_cfg.h"

//------------------------------------------------------------------------------
// Definitions
#define TIMER2_DIVIDER      2
#define TIMER2_PRESCALER    10

//------------------------------------------------------------------------------
// Local Variables
static volatile PWM_REG *g_pPWMReg = NULL;

static struct
{
    BOOL enabled;        // is profiler active?
    UINT32 countsPerHit;    // counts per profiler interrupt
} g_profiler;

//------------------------------------------------------------------------------
// External Variables
extern PFN_PROFILER_ISR g_pProfilerISR;

//------------------------------------------------------------------------------
// Local Functions
UINT32 OALProfileIntrHandler(UINT32 ra);

//------------------------------------------------------------------------------
//
//  Function:  ConfigureNextProfilerCount
//
//  Updates the profiler count (prepares the timer for the next timer event).
//
static void ConfigureNextProfilerCount(DWORD dwCountInterval)
{
    UINT32 tcon;
    
    if (!g_pProfilerISR) return;

    // Change number of timer ticks in the period.
    //
    OUTREG32(&g_pPWMReg->TCFG1, INREG32(&g_pPWMReg->TCFG1) & ~(DIV_MUX2_MASK));
    OUTREG32(&g_pPWMReg->TCFG1, INREG32(&g_pPWMReg->TCFG1) | (D1_2 << DIV_MUX2_START)); //PCLK 66MHz / 33/ 2 = 1MHz = 1us
    // Set timer register to interval
    OUTREG32(&g_pPWMReg->TCNTB2, dwCountInterval);

    // Start timer in auto-reload mode
    tcon = INREG32(&g_pPWMReg->TCON) & ~(TIMER2_MASK);
    OUTREG32(&g_pPWMReg->TCON, tcon);

    // Timer2 Clear Interrupt Pending
    CLR_PWMINT(g_pPWMReg->TINT_CSTAT, 2);
    // Timer2 Enable Interrupt
    ENABLE_PWMINT(g_pPWMReg->TINT_CSTAT, 2);    

    OUTREG32(&g_pPWMReg->TCON, tcon | (TIMER2_MANUAL_UPDATE) );
    OUTREG32(&g_pPWMReg->TCON, tcon | (TIMER2_START) );							//One-Shot 1
    

}

//------------------------------------------------------------------------------
//
//  Function:  OEMProfileTimerEnable
//
//  This function is called by kernel to start kernel profiling timer.
//
VOID OEMProfileTimerEnable(DWORD interval)
{
    BOOL enabled;
    UINT32 Irq;
    SYSTEM_CLOCK *SystemClocks;
    UINT32 uTimerClockFreq;
    UINT32 uTimerCount1ms;
    
    OALMSG(TRUE, (L"+OEMProfileTimerEnable(%d)\r\n", interval));

    SystemClocks = (SYSTEM_CLOCK *)OALArgsQuery(OAL_ARGS_QUERY_SYSTEMCLOCKS);

    uTimerClockFreq = (SystemClocks->PCLKPSYS_CLK/(TIMER2_PRESCALER+1)/TIMER2_DIVIDER);
    uTimerCount1ms = ((uTimerClockFreq/1000)-1);

    // We can't enable timer second time
    if (g_profiler.enabled) return;

    // Obtain a pointer to the PWM registers.
    if (!g_pPWMReg)
    {
        g_pPWMReg = (PWM_REG *)OALPAtoVA(BASE_REG_PA_PWMTIMER, FALSE);
    }

    // How many hi-res ticks per profiler hit
    g_profiler.countsPerHit = (uTimerCount1ms*interval)/1000;

    // Following code should not be interrupted
    enabled = INTERRUPTS_ENABLE(FALSE);

    // Configure profiling ISR callback function.
    g_pProfilerISR = OALProfileIntrHandler;

    // Update the compare register for the next profile hit.
    ConfigureNextProfilerCount(g_profiler.countsPerHit);

    // Enable interrupts
    INTERRUPTS_ENABLE(enabled);
    Irq = IRQ_TIMER2;
    OALIntrDoneIrqs(1, &Irq);

    // Set flag
    g_profiler.enabled = TRUE;

    OALMSG(TRUE, (L"-OEMProfileTimerEnable\r\n"));
}


//------------------------------------------------------------------------------
//
//  Function:  OEMProfileTimerDisable
//
//  This function is called by kernel to stop kernel profiling timer.
//

VOID OEMProfileTimerDisable()
{
    BOOL enabled;
    UINT32 irq;

    OALMSG(TRUE, (L"+OEMProfileTimerDisable()\r\n"));

    // No disable without enable
    if (!g_profiler.enabled) goto cleanUp;

    // Following code should not be interrupted
    enabled = INTERRUPTS_ENABLE(FALSE);

    // Timer2 Clear Interrupt Pending
    CLR_PWMINT(g_pPWMReg->TINT_CSTAT, 2);
    // Timer2 Enable Interrupt
    DISABLE_PWMINT(g_pPWMReg->TINT_CSTAT, 2); 

    // Disable the profile timer interrupt
    irq = IRQ_TIMER2;
    OALIntrDisableIrqs(1, &irq);

    // Deconfigure profiling ISR callback function.
    g_pProfilerISR = NULL;

    // Reset flag
    g_profiler.enabled = FALSE;

    // Enable interrupts
    INTERRUPTS_ENABLE(enabled);

cleanUp:
    OALMSG(TRUE, (L"-OEMProfileTimerDisable\r\n"));
}


//------------------------------------------------------------------------------
//
//  Function:  OALProfileIntrHandler
//
//  This is timer interrupt handler which replace default handler in time when
//  kernel profiling is active. It calls original interrupt handler in
//  appropriate times.
//
UINT32 OALProfileIntrHandler(UINT32 ra)
{
    UINT32 Irq;

    // Update the compare register for the next profile hit.
    ConfigureNextProfilerCount(g_profiler.countsPerHit);

    // First call profiler
    ProfilerHit(ra);

    // Enable interrupts
    Irq = IRQ_TIMER2;
    OALIntrDoneIrqs(1, &Irq);

    return(SYSINTR_NOP);
}

//------------------------------------------------------------------------------

