//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  Header: hdmi_reg.h
//
//  Defines the HDMI register layout and definitions.
//
#ifndef __HDMI_REG_H__
#define __HDMI_REG_H__

#if __cplusplus
	extern "C"
	{
#endif

// Base Address
// HDMI Subsystem
#define HDMI_SS_BASE_ADDRESS            0xFA100000

// HDMI Core
#define HDMI_CORE_BASE_ADDRESS          0xFA110000
#define HDMI_HDCP_BASE_ADDRESS          0xFA110600

// SPDIF
#define HDMI_SPDIF_BASE_ADDRESS         0xFA130000

// I2S
#define HDMI_I2S_BASE_ADDRESS           0xFA140000

// EFUSE
#define HDMI_EFUSE_BASE_ADDRESS         0xFA150000

// TG
#define HDMI_TG_BASE_ADDRESS            0xFA160000

// register structure
// HDMI Subsystem
typedef struct 
{
	unsigned int   HDMI_INTC_CON;				// 0x0000
	unsigned int   HDMI_INTC_FLAG;			// 0x0004
	unsigned int   HDMI_HDCP_KEY_LOAD;		 // 0x0008
	unsigned int   HDMI_HPD;					// 0x000C
	unsigned int   HDMI_AUDIO_CLKSEL;		// 0x0010
	unsigned int   HDMI_PHY_RSTOUT;			// 0x0014
	unsigned int   HDMI_PHY_VPLL;				// 0x0018
	unsigned int   HDMI_PHY_CMU;				// 0x001C
	unsigned int   HDMI_CORE_RSTOUT;			// 0x0020
}HDMI_SS_REG,*PHDMI_SS_REG;              

//@{
/**
 * @name HDMI IRQ bit values
 */
#define HDMI_IRQ_PIN_POLAR_CTL			(7)
#define HDMI_IRQ_GLOBAL					(6)
#define HDMI_IRQ_I2S						(5)
#define HDMI_IRQ_CEC						(4)
#define HDMI_IRQ_HPD_PLUG					(3)
#define HDMI_IRQ_HPD_UNPLUG				(2)
#define HDMI_IRQ_SPDIF						(1)
#define HDMI_IRQ_HDCP						(0)
//@}

#define HPD_HI              (1)
#define HPD_LO              (0)
                                            
// HDMI Core (Video + Audio + Packet)       
typedef struct                              
{                                           
	// Control related regisers             
	unsigned int   HDMI_CON0;                      // 0x0000
	unsigned int   HDMI_CON1;                      // 0x0004
	unsigned int   HDMI_CON2;                      // 0x0008
	unsigned int   PAD0;                           // 0x000C
	                                        
	unsigned int   HDMI_STATUS;                    // 0x0010
	unsigned int   HDMI_PHY_STATUS;                // 0x0014
	unsigned int   PAD1[2];                        // 0x0018, 0x001C
	                                        
	unsigned int   HDMI_STATUS_EN;                 // 0x0020
	unsigned int   PAD2[3];                        // 0x0024, 0x0028, 0x002C    
	                                        
	unsigned int   HDMI_HPD;                       // 0x0030
	unsigned int   PAD3[3];                        // 0x0034, 0x0038, 0x003C
	                                        
	unsigned int   HDMI_MODE_SEL;                  // 0x0040
	unsigned int   HDMI_ENC_EN;                    // 0x0044
	unsigned int   PAD4[2];                        // 0x0048, 0x004C
	                                        
	// Video related registers              
	unsigned int HDMI_BLUE_SCREEN0;                // 0x0050
	unsigned int HDMI_BLUE_SCREEN1;                // 0x0054
	unsigned int HDMI_BLUE_SCREEN2;                // 0x0058
	unsigned int PAD5;                             // 0x005C
	                                        
	unsigned int HDMI_YMAX;                        // 0x0060
	unsigned int HDMI_YMIN;                        // 0x0064
	unsigned int HDMI_CMAX;                        // 0x0068
	unsigned int HDMI_CMIN;                        // 0x006C
	                                        
	unsigned int PAD6[12];                         // 0x0070, 0x0074, 0x0078, 0x007C
	                                        // 0x0080, 0x0084, 0x0088, 0x008C
	                                        // 0x0090, 0x0094, 0x0098, 0x009C
	                                        
	unsigned int HDMI_H_BLANK0;                    // 0x00A0
	unsigned int HDMI_H_BLANK1;                    // 0x00A4
	unsigned int PAD7[2];                          // 0x00A8, 0x00AC
	                                        
	unsigned int HDMI_V_BLANK0;                    // 0x00B0
	unsigned int HDMI_V_BLANK1;                    // 0x00B4
	unsigned int HDMI_V_BLANK2;                    // 0x00B8
	unsigned int PAD8;                             // 0x00BC
	                                        
	unsigned int HDMI_H_V_LINE0;                   // 0x00C0
	unsigned int HDMI_H_V_LINE1;                   // 0x00C4
	unsigned int HDMI_H_V_LINE2;                   // 0x00C8
	unsigned int PAD9[5];                          // 0x00CC 
	                                        // 0x00D0 
	                                        // 0x00D4, 0x00D8, 0x00DC
	                                        
	unsigned int PAD10;                            // 0x00E0
	unsigned int HDMI_VSYNC_POL;                   // 0x00E4
	unsigned int HDMI_INT_PRO_MODE;                // 0x00E8
	unsigned int PAD11;                            // 0x00EC
	                                        
	unsigned int PAD12[4];                         // 0x00F0, 0x00F4, 0x00F8, 0x00FC
	                                        
	unsigned int PAS13[4];                         // 0x0100, 0x0104, 0x0108, 0x010C
	                                        
	unsigned int HDMI_V_BLANK_F0;                  // 0x0110
	unsigned int HDMI_V_BLANK_F1;                  // 0x0114
	unsigned int HDMI_V_BLANK_F2;                  // 0x0118
	unsigned int PAD14;                            // 0x011C
	                                        
	unsigned int HDMI_H_SYNC_GEN0;                 // 0x0120
	unsigned int HDMI_H_SYNC_GEN1;                 // 0x0124
	unsigned int HDMI_H_SYNC_GEN2;                 // 0x0128
	unsigned int PAD15;                            // 0x012C
	                                        
	unsigned int HDMI_V_SYNC_GEN1_0;               // 0x0130
	unsigned int HDMI_V_SYNC_GEN1_1;               // 0x0134
	unsigned int HDMI_V_SYNC_GEN1_2;               // 0x0138
	unsigned int PAD16;                            // 0x013C
	                                        
	unsigned int HDMI_V_SYNC_GEN2_0;               // 0x0140
	unsigned int HDMI_V_SYNC_GEN2_1;               // 0x0144
	unsigned int HDMI_V_SYNC_GEN2_2;               // 0x0148
	unsigned int PAD17;                            // 0x014C
	                                        
	unsigned int HDMI_V_SYNC_GEN3_0;               // 0x0150
	unsigned int HDMI_V_SYNC_GEN3_1;               // 0x0154
	unsigned int HDMI_V_SYNC_GEN3_2;               // 0x0158
	unsigned int PAD18;                            // 0x015C
	                                        
	// Audio releated packet register       
	unsigned int HDMI_ASP_CON;                     // 0x0160
	unsigned int HDMI_ASP_SP_FLAT;                 // 0x0164
	unsigned int PAD19[2];                         // 0x0168, 0x016C
	                                        
	unsigned int HDMI_ASP_CHCFG0;                  // 0x0170
	unsigned int HDMI_ASP_CHCFG1;                  // 0x0174
	unsigned int HDMI_ASP_CHCFG2;                  // 0x0178
	unsigned int HDMI_ASP_CHCFG3;                  // 0x017C
	                                        
	unsigned int HDMI_ACR_CON;                     // 0x0180
	unsigned int HDMI_ACR_MCTS0;                   // 0x0184
	unsigned int HDMI_ACR_MCTS1;                   // 0x0188
	unsigned int HDMI_ACR_MCTS2;                   // 0x018C
	                                        
	unsigned int HDMI_ACR_CTS0;                    // 0x0190
	unsigned int HDMI_ACR_CTS1;                    // 0x0194
	unsigned int HDMI_ACR_CTS2;                    // 0x0198
	unsigned int PAD20;                            // 0x019C
	                                        
	unsigned int HDMI_ACR_N0;                      // 0x01A0
	unsigned int HDMI_ACR_N1;                      // 0x01A4
	unsigned int HDMI_ACR_N2;                      // 0x01A8
	unsigned int PAD21;                            // 0x01AC
	                                        
	unsigned int HDMI_ACR_LSB2;                    // 0x01B0
	unsigned int HDMI_ACR_TXCNT;                   // 0x01B4
	unsigned int HDMI_ACR_TXINTERNAL;              // 0x01B8
	unsigned int HDMI_ACR_CTS_OFFSET;              // 0x01BC
	                                        
	unsigned int HDMI_GCP_CON;                     // 0x01C0
	unsigned int PAD22[3];                         // 0x01C4, 0x01C8, 0x01CC
	                                        
	unsigned int HDMI_GCP_BYTE1;                   // 0x01D0
	unsigned int HDMI_GCP_BYTE2;                   // 0x01D4
	unsigned int HDMI_GCP_BYTE3;                   // 0x01D8
	unsigned int PAD23;                            // 0x01DC
	                                        
	unsigned int HDMI_ACP_CON;                     // 0x01E0
	unsigned int PAD24[3];                         // 0X01E4, 0x01E8, 0x01EC
	                                        
	unsigned int HDMI_ACP_TYPE;                    // 0x01F0
	unsigned int PAD25[3];                         // 0x01F4, 0x01F8, 0x01FC
	                                        
	unsigned int HDMI_ACP_DATA0;                   // 0x0200
	unsigned int HDMI_ACP_DATA1;                   // 0x0204
	unsigned int HDMI_ACP_DATA2;                   // 0x0208
	unsigned int HDMI_ACP_DATA3;                   // 0x020C
	unsigned int HDMI_ACP_DATA4;                   // 0x0210
	unsigned int HDMI_ACP_DATA5;                   // 0x0214
	unsigned int HDMI_ACP_DATA6;                   // 0x0218
	unsigned int HDMI_ACP_DATA7;                   // 0x021C
	unsigned int HDMI_ACP_DATA8;                   // 0x0220
	unsigned int HDMI_ACP_DATA9;                   // 0x0224
	unsigned int HDMI_ACP_DATA10;                  // 0x0228
	unsigned int HDMI_ACP_DATA11;                  // 0x022C
	unsigned int HDMI_ACP_DATA12;                  // 0x0230
	unsigned int HDMI_ACP_DATA13;                  // 0x0234
	unsigned int HDMI_ACP_DATA14;                  // 0x0238
	unsigned int HDMI_ACP_DATA15;                  // 0x023C
	unsigned int HDMI_ACP_DATA16;                  // 0x0240
	unsigned int PAD26[3];                         // 0x0244, 0x0248, 0x024C
	                                        
	unsigned int HDMI_ISRC_CON;                    // 0x0250
	unsigned int PAD27[4];                         // 0x0254, 0x0258, 0x025C
	                                        // 0x0260
	                                        
	unsigned int HDMI_ISRC1_HEADER1;               // 0x0264,
	unsigned int PAD28[2];                         // 0x0268, 0x026C
	                                        
	unsigned int HDMI_ISRC1_DATA0;                 // 0x0270
	unsigned int HDMI_ISRC1_DATA1;                 // 0x0274
	unsigned int HDMI_ISRC1_DATA2;                 // 0x0278
	unsigned int HDMI_ISRC1_DATA3;                 // 0x027C
	unsigned int HDMI_ISRC1_DATA4;                 // 0x0280
	unsigned int HDMI_ISRC1_DATA5;                 // 0x0284
	unsigned int HDMI_ISRC1_DATA6;                 // 0x0288
	unsigned int HDMI_ISRC1_DATA7;                 // 0x028C
	unsigned int HDMI_ISRC1_DATA8;                 // 0x0290
	unsigned int HDMI_ISRC1_DATA9;                 // 0x0294
	unsigned int HDMI_ISRC1_DATA10;                // 0x0298
	unsigned int HDMI_ISRC1_DATA11;                // 0x029C
	unsigned int HDMI_ISRC1_DATA12;                // 0x02A0
	unsigned int HDMI_ISRC1_DATA13;                // 0x02A4
	unsigned int HDMI_ISRC1_DATA14;                // 0x02A8
	unsigned int HDMI_ISRC1_DATA15;                // 0x02AC
	                                        
	unsigned int HDMI_ISRC2_DATA0;                 // 0x02B0
	unsigned int HDMI_ISRC2_DATA1;                 // 0x02B4
	unsigned int HDMI_ISRC2_DATA2;                 // 0x02B8
	unsigned int HDMI_ISRC2_DATA3;                 // 0x02BC
	unsigned int HDMI_ISRC2_DATA4;                 // 0x02C0
	unsigned int HDMI_ISRC2_DATA5;                 // 0x02C4
	unsigned int HDMI_ISRC2_DATA6;                 // 0x02C8
	unsigned int HDMI_ISRC2_DATA7;                 // 0x02CC
	unsigned int HDMI_ISRC2_DATA8;                 // 0x02D0
	unsigned int HDMI_ISRC2_DATA9;                 // 0x02D4
	unsigned int HDMI_ISRC2_DATA10;                // 0x02D8
	unsigned int HDMI_ISRC2_DATA11;                // 0x02DC
	unsigned int HDMI_ISRC2_DATA12;                // 0x02E0
	unsigned int HDMI_ISRC2_DATA13;                // 0x02E4
	unsigned int HDMI_ISRC2_DATA14;                // 0x02E8
	unsigned int HDMI_ISRC2_DATA15;                // 0x02EC
	unsigned int PAD29[4];                         // 0x02F0, 0x02F4, 0x02F8, 0x02FC
	                                        
	unsigned int HDMI_AVI_CON;                     // 0x0300
	unsigned int PAD30[3];                         // 0x0304, 0x0308, 0x030C
	                                        
	unsigned int HDMI_AVI_CHECK_SUM;               // 0x0310
	unsigned int PAD31[3];                         // 0x0314, 0x0318, 0x031C
	                                        
	unsigned int HDMI_AVI_BYTE1;                   // 0x0320
	unsigned int HDMI_AVI_BYTE2;                   // 0x0324
	unsigned int HDMI_AVI_BYTE3;                   // 0x0328
	unsigned int HDMI_AVI_BYTE4;                   // 0x032C
	unsigned int HDMI_AVI_BYTE5;                   // 0x0330
	unsigned int HDMI_AVI_BYTE6;                   // 0x0334
	unsigned int HDMI_AVI_BYTE7;                   // 0x0338
	unsigned int HDMI_AVI_BYTE8;                   // 0x033C
	unsigned int HDMI_AVI_BYTE9;                   // 0x0340
	unsigned int HDMI_AVI_BYTE10;                  // 0x0344
	unsigned int HDMI_AVI_BYTE11;                  // 0x0348
	unsigned int HDMI_AVI_BYTE12;                  // 0x034C
	unsigned int HDMI_AVI_BYTE13;                  // 0x0350
	unsigned int PAD32[3];                         // 0x0354, 0x0358, 0x035C
	                                        
	unsigned int HDMI_AUI_CON ;                    // 0x0360
	unsigned int PAD33[3];                         // 0x0364, 0x0368, 0x036C
	                                        
	unsigned int HDMI_AUI_CHECK_SUM;               // 0x0370
	unsigned int PAD34[3];                         // 0x0374, 0x0378, 0x037C
	                                        
	unsigned int HDMI_AUI_BYTE1;                   // 0x0380
	unsigned int HDMI_AUI_BYTE2;                   // 0x0384
	unsigned int HDMI_AUI_BYTE3;                   // 0x0388
	unsigned int HDMI_AUI_BYTE4;                   // 0x038C
	unsigned int HDMI_AUI_BYTE5;                   // 0x0390
	unsigned int PAD35[3];                         // 0x0394, 0x0398, 0x039C
	                                        
	                                        
	unsigned int HDMI_MPG_CON;                     // 0x03A0,
	unsigned int PAD36[3];                         // 0x03A4, 0x03A8, 0x03AC
	                                        
	unsigned int HDMI_MPG_CHECK_SUM;               // 0x03B0,
	unsigned int PAD37[3];                         // 0x03B4, 0x03B8, 0x03BC
	                                        
	unsigned int HDMI_MPG_BYTE1;                   // 0x03C0
	unsigned int HDMI_MPG_BYTE2;                   // 0x03C4
	unsigned int HDMI_MPG_BYTE3;                   // 0x03C8
	unsigned int HDMI_MPG_BYTE4;                   // 0x03CC
	unsigned int HDMI_MPG_BYTE5;                   // 0x03D0
	unsigned int PAD38[11];                        // 0x03D4, 0x03D8, 0x03DC
	                                        // 0x03E0, 0x03E4, 0x03E8, 0x03EC 
	                                        // 0x03F0, 0x03F4, 0x03F8, 0x03FC
	                                        
	unsigned int HDMI_SPD_CON;                     // 0x0400
	unsigned int PAD39[3];                         // 0x0404, 0x0408, 0x040C
	                                        
	unsigned int HDMI_SPD_HEADER0;                 // 0x0410
	unsigned int HDMI_SPD_HEADER1;                 // 0x0414
	unsigned int HDMI_SPD_HEADER2;                 // 0x0418
	unsigned int PAD40;                            // 0x041C
	                                        
	unsigned int HDMI_SPD_DATA0;                   // 0x0420
	unsigned int HDMI_SPD_DATA1;                   // 0x0424
	unsigned int HDMI_SPD_DATA2;                   // 0x0428
	unsigned int HDMI_SPD_DATA3;                   // 0x042C
	unsigned int HDMI_SPD_DATA4;                   // 0x0430
	unsigned int HDMI_SPD_DATA5;                   // 0x0434
	unsigned int HDMI_SPD_DATA6;                   // 0x0438
	unsigned int HDMI_SPD_DATA7;                   // 0x043C
	unsigned int HDMI_SPD_DATA8;                   // 0x0440
	unsigned int HDMI_SPD_DATA9;                   // 0x0444
	unsigned int HDMI_SPD_DATA10;                  // 0x0448
	unsigned int HDMI_SPD_DATA11;                  // 0x044C
	unsigned int HDMI_SPD_DATA12;                  // 0x0450
	unsigned int HDMI_SPD_DATA13;                  // 0x0454
	unsigned int HDMI_SPD_DATA14;                  // 0x0458
	unsigned int HDMI_SPD_DATA15;                  // 0x045C
	unsigned int HDMI_SPD_DATA16;                  // 0x0460
	unsigned int HDMI_SPD_DATA17;                  // 0x0464
	unsigned int HDMI_SPD_DATA18;                  // 0x0468
	unsigned int HDMI_SPD_DATA19;                  // 0x046C
	unsigned int HDMI_SPD_DATA20;                  // 0x0470
	unsigned int HDMI_SPD_DATA21;                  // 0x0474
	unsigned int HDMI_SPD_DATA22;                  // 0x0478
	unsigned int HDMI_SPD_DATA23;                  // 0x047C
	unsigned int HDMI_SPD_DATA24;                  // 0x0480
	unsigned int HDMI_SPD_DATA25;                  // 0x0484
	unsigned int HDMI_SPD_DATA26;                  // 0x0488
	unsigned int HDMI_SPD_DATA27;                  // 0x048C
	unsigned int PAD41[4];                         // 0x0490, 0x0494, 0x0498, 0x049C
	                                        
	unsigned int HDMI_GAMUT_CON;                   // 0x0500
	unsigned int HDMI_GAMUT_HEADER0;               // 0x0504
	unsigned int HDMI_GAMUT_HEADER1;               // 0x0508
	unsigned int HDMI_GAMUT_HEADER2;               // 0x050C
	                                        
	unsigned int HDMI_GAMUT_DATA00;                // 0x0510
	unsigned int HDMI_GAMUT_DATA01;                // 0x0514
	unsigned int HDMI_GAMUT_DATA02;                // 0x0518
	unsigned int HDMI_GAMUT_DATA03;                // 0x051C
	unsigned int HDMI_GAMUT_DATA04;                // 0x0520
	unsigned int HDMI_GAMUT_DATA05;                // 0x0524
	unsigned int HDMI_GAMUT_DATA06;                // 0x0528
	unsigned int HDMI_GAMUT_DATA07;                // 0x052C
	unsigned int HDMI_GAMUT_DATA08;                // 0x0530
	unsigned int HDMI_GAMUT_DATA09;                // 0x0534
	unsigned int HDMI_GAMUT_DATA10;                // 0x0538
	unsigned int HDMI_GAMUT_DATA11;                // 0x053C
	unsigned int HDMI_GAMUT_DATA12;                // 0x0540
	unsigned int HDMI_GAMUT_DATA13;                // 0x0544
	unsigned int HDMI_GAMUT_DATA14;                // 0x0548
	unsigned int HDMI_GAMUT_DATA15;                // 0x054C
	unsigned int HDMI_GAMUT_DATA16;                // 0x0550
	unsigned int HDMI_GAMUT_DATA17;                // 0x0554
	unsigned int HDMI_GAMUT_DATA18;                // 0x0558
	unsigned int HDMI_GAMUT_DATA19;                // 0x055C
	unsigned int HDMI_GAMUT_DATA20;                // 0x0560
	unsigned int HDMI_GAMUT_DATA21;                // 0x0564
	unsigned int HDMI_GAMUT_DATA22;                // 0x0568
	unsigned int HDMI_GAMUT_DATA23;                // 0x056C
	unsigned int HDMI_GAMUT_DATA24;                // 0x0570
	unsigned int HDMI_GAMUT_DATA25;                // 0x0574
	unsigned int HDMI_GAMUT_DATA26;                // 0x0578
	unsigned int HDMI_GAMUT_DATA27;                // 0x057C
	unsigned int PAD42[16];                        // 0x0580, 0x0584, 0x0588, 0x058C
	                                        // 0x0590, 0x0594, 0x0598, 0x059C
	                                        // 0x05A0, 0x05A4, 0x05A8, 0x05AC
	                                        // 0x05B0, 0x05B4, 0x05B8, 0x05BC    
	                                        
	unsigned int HDMI_DC_CONTROL;                  // 0x05C0
	unsigned int HDMI_VIDEO_PATTERN_GEN;           // 0x05C4
	unsigned int HDMI_HPD_GEN;                     // 0x05C8
}HDMI_CORE_REG, *PHDMI_CORE_REG;           
                                            
// HDCP Part                                
typedef struct                              
{                                           
	unsigned int HDCP_SHA1_00;                     // 0X0600
	unsigned int HDCP_SHA1_01;                     // 0X0604
	unsigned int HDCP_SHA1_02;                     // 0X0608
	unsigned int HDCP_SHA1_03;                     // 0X060C
	unsigned int HDCP_SHA1_10;                     // 0X0610
	unsigned int HDCP_SHA1_11;                     // 0X0614
	unsigned int HDCP_SHA1_12;                     // 0X0618
	unsigned int HDCP_SHA1_13;                     // 0X061C
	unsigned int HDCP_SHA1_20;                     // 0X0620
	unsigned int HDCP_SHA1_21;                     // 0X0624
	unsigned int HDCP_SHA1_22;                     // 0X0628
	unsigned int HDCP_SHA1_23;                     // 0X062C
	unsigned int HDCP_SHA1_30;                     // 0X0630
	unsigned int HDCP_SHA1_31;                     // 0X0634
	unsigned int HDCP_SHA1_32;                     // 0X0638
	unsigned int HDCP_SHA1_33;                     // 0X063C
	unsigned int HDCP_SHA1_40;                     // 0X0640
	unsigned int HDCP_SHA1_41;                     // 0X0644
	unsigned int HDCP_SHA1_42;                     // 0X0648
	unsigned int HDCP_SHA1_43;                     // 0X064C
	                                        
	unsigned int HDCP_KSV_LIST_0;                  // 0X0650
	unsigned int HDCP_KSV_LIST_1;                  // 0X0654
	unsigned int HDCP_KSV_LIST_2;                  // 0X0658
	unsigned int HDCP_KSV_LIST_3;                  // 0X065C
	unsigned int HDCP_KSV_LIST_4;                  // 0X0660
	unsigned int HDCP_KSV_LIST_CON;                // 0X0664
	unsigned int PAD0[2];                          // 0X0668, 0X066C
	                                        
	unsigned int HDCP_SHA_RESULT;                  // 0X0670
	unsigned int PAD1[3];                          // 0X0674, 0X0678, 0X067C
	                                        
	unsigned int HDCP_CTRL1;                       // 0X0680
	unsigned int HDCP_CTRL2;                       // 0X0684
	unsigned int PAD2[2];                          // 0X0688, 0X068C
	                                        
	unsigned int HDCP_CHECK_RESULT;                // 0X0690,
	unsigned int PAD3[3];                          // 0X0694, 0X0698, 0X069C
	                                        
	unsigned int HDCP_BKSV_0;                      // 0X06A0
	unsigned int HDCP_BKSV_1;                      // 0X06A4
	unsigned int HDCP_BKSV_2;                      // 0X06A8
	unsigned int HDCP_BKSV_3;                      // 0X06AC
	unsigned int HDCP_BKSV_4;                      // 0X06B0
	unsigned int PAD4[3];                          // 0X06B4, 0X06B8, 0X06BC
	                                        
	unsigned int HDCP_AKSV_0;                      // 0X06C0
	unsigned int HDCP_AKSV_1;                      // 0X06C4
	unsigned int HDCP_AKSV_2;                      // 0X06C8
	unsigned int HDCP_AKSV_3;                      // 0X06CC
	unsigned int HDCP_AKSV_4;                      // 0X06D0
	unsigned int PAD5[3];                          // 0X06D4, 0X06D8, 0X06DC
	                                        
	unsigned int HDCP_AN_0;                        // 0X06E0
	unsigned int HDCP_AN_1;                        // 0X06E4
	unsigned int HDCP_AN_2;                        // 0X06E8
	unsigned int HDCP_AN_3;                        // 0X06EC
	unsigned int HDCP_AN_4;                        // 0X06F0
	unsigned int HDCP_AN_5;                        // 0X06F4
	unsigned int HDCP_AN_6;                        // 0X06F8
	unsigned int HDCP_AN_7;                        // 0X06FC
	                                        
	unsigned int HDCP_BCAPS;                       // 0X0700
	unsigned int PAD6[3];                          // 0X0704, 0X0708, 0X070C
	                                        
	unsigned int HDCP_BSTATUS_0;                   // 0X0710
	unsigned int HDCP_BSTATUS_1;                   // 0X0714
	unsigned int PAD7[10];                         // 0X0718, 0X071C
	                                        // 0X0720, 0X0724, 0X0728, 0X072C
	                                        // 0X0730, 0X0734, 0X0738, 0X073C
	                                        
	unsigned int HDCP_RI_0;                        // 0X0740
	unsigned int HDCP_RI_1;                        // 0X0744
	unsigned int PAD8[14];                         // 0X0748, 0X074C
	                                        // 0X0750, 0X0754, 0X0758, 0X075C
	                                        // 0X0760, 0X0764, 0X0768, 0X076C
	                                        // 0X0770, 0X0774, 0X0778, 0X077C
	                                        
	unsigned int HDCP_I2C_INT;                     // 0X0780
	unsigned int PAD9[3];                          // 0X0784, 0X0788, 0X078C
	                                        
	unsigned int HDCP_AN_INT;                      // 0X0790
	unsigned int PAD10[3];                         // 0X0794, 0X0798, 0X079C
	                                        
	unsigned int HDCP_WATCHDOG_INT;                // 0X07A0
	unsigned int PAD11[3];                         // 0X07A4, 0X07A8, 0X07AC
	                                        
	unsigned int HDCP_RI_INT;                      // 0X07B0
	unsigned int PAD12[7];                         // 0X07B4, 0X07B8, 0X07BC
	                                        // 0X07C0, 0x07C4, 0x07C8, 0x07CC
	                                        
	unsigned int HDCP_RI_COMPARE_0;                // 0x07D0
	unsigned int HDCP_RI_COMPARE_1;                // 0x07D4 
	unsigned int PAD13[2];                         // 0x07D8, 0x07DC
	                                        
	unsigned int HDCP_FRAME_COUNT;                 // 0x07E0
}HDMI_HDCP_REG, *PHDMI_HDCP_REG;           
                                            
// bit mask
//@{
/**
 * @name HDCP IRQ bit values
 */
#define HDCP_I2C_INT_NUM            (0)
#define HDCP_WATCHDOG_INT_NUM       (1)
#define HDCP_AN_WRITE_INT_NUM       (2)
#define HDCP_UPDATE_PJ_INT_NUM      (3)
#define HDCP_UPDATE_RI_INT_NUM      (4)
#define HDCP_AUD_FIFO_OVF_EN_NUM    (5)
#define HDCP_AUTHEN_ACK_NUM         (7)
#define HDCP_INT_MASK               (0x17)
//@}

//@{
/**
 * @name HDCP bit values
 */
 #define HDCP_ENABLE                 (1<<1)
#define HDCP_DISABLE                (0)
#define HDCP_TIMEOUT                (1<<2)
#define HDCP_REVOCATION_SET         (1<<0)

#define HDCP_RI_NOT_MATCH           (1<<1)
#define HDCP_RI_MATCH               (1<<0|1<<1)

#define HDCP_SHA1_VALID_READY       (1<<1)
#define HDCP_SHA1_VALID             (1<<0)

#define HDCP_KSV_LIST_READ_DONE     (1<<0)
#define HDCP_KSV_LIST_END           (1<<1)
#define HDCP_KSV_LIST_EMPTY         (1<<2)
#define HDCP_KSV_WRITE_DONE         (1<<3)

#define HDCP_COMPARE_FRAME_COUNT0_ENABLE    (1<<7 | 0)
#define HDCP_COMPARE_FRAME_COUNT1_ENABLE    (1<<7 | 0x7F)
//@}

//@{
/**
 * @name HDMI bit values
 */
#define HPD_SW_ENABLE               (1<<0)
#define HPD_SW_DISABLE              (0)
#define HPD_ON                      (1<<1)
#define HPD_OFF                     (0)

#define HDMI_EXTERNAL_VIDEO         (1<<1)

#define DO_NOT_TRANSMIT             (0)
#define TRANSMIT_ONCE               (1)
#define TRANSMIT_EVERY_VSYNC        (1<<1)
#define GCP_TRANSMIT_EVERY_VSYNC    (1<<3|1<<2|1<<1)

#define HDMI_SYS_ENABLE             (1<<0)
#define HDMI_PWDN_ENABLE            (1<<1)
#define HDMI_ASP_ENABLE             (1<<2)
#define HDMI_ENCODING_OPTION_ENABLE (1<<4)
#define HDMI_BLUE_SCR_ENABLE        (1<<5)

#define HDMI_MODE_SEL_DVI           (1<<0)
#define HDMI_MODE_SEL_HDMI          (1<<1)

#define HDMI_PHY_READY              (1<<0)

#define HDMI_ENC_ENABLE             (1<<0)
#define HDMI_ENC_DISABLE            (0)

#define HDMICON1_DOUBLE_PIXEL_REPETITION    (1<<0)

#define HDMICON1_LIMIT_MASK         (1<<5|1<<6)
#define HDMICON1_RGB_LIMIT          (1<<5)
#define HDMICON1_YCBCR_LIMIT        (1<<6)

#define HDMICON2_DVI                (1<<5|1<<1)
#define HDMICON2_HDMI               (0)

#define ACR_MEASURED_CTS_MODE       (1<<2)

#define AVI_HEADER                  (0x82 + 0x02 + 0x0D)
#define AVI_PACKET_BYTE_LENGTH      (13)
#define AVI_CS_RGB                  (0)
#define AVI_CS_Y422                 (1<<5)
#define AVI_CS_Y444                 (1<<6)
#define AVI_QUANTIZATION_MASK       (1<<2|1<<3)
#define AVI_QUANTIZATION_DEFAULT    (0)
#define AVI_QUANTIZATION_LIMITED    (1<<2)
#define AVI_QUANTIZATION_FULL       (1<<3)
#define AVI_PIXEL_REPETITION_DOUBLE (1<<0)
#define AVI_FORMAT_MASK             (0x0F)
#define AVI_FORMAT_ASPECT_AS_PICTURE    (1<<3)
#define AVI_PICTURE_ASPECT_RATIO_4_3    (1<<4)
#define AVI_PICTURE_ASPECT_RATIO_16_9   (1<<5)
#define AVI_COLORIMETRY_MASK                        0xC0 // 0b11000000
#define AVI_COLORIMETRY_NO_DATA                     0
#define AVI_COLORIMETRY_ITU601                      0x40 // SMPTE170M, ITU601
#define AVI_COLORIMETRY_ITU709                      0x80 // ITU709
#define AVI_COLORIMETRY_EXTENDED                    0xC0 // Extended Colorimetry
#define AVI_COLORIMETRY_EXT_MASK                    0x70 // 0b01110000
#define AVI_COLORIMETRY_EXT_xvYCC601                (0<<4)
#define AVI_COLORIMETRY_EXT_xvYCC709                (1<<4)

#define AUI_HEADER                  (0x84 + 0x01 + 0x0A)
#define AUI_PACKET_BYTE_LENGTH      (10)
#define AUI_CC_2CH                  (1)
#define AUI_CC_3CH                  (2)
#define AUI_CC_4CH                  (3)
#define AUI_CC_5CH                  (4)
#define AUI_CC_6CH                  (5)
#define AUI_CC_7CH                  (6)
#define AUI_CC_8CH                  (7)
#define HDMI_AUI_SF_MASK            (1<<4|1<<3|1<<2)
#define HDMI_AUI_SF_SF_32KHZ        (1<<2)
#define HDMI_AUI_SF_SF_44KHZ        (1<<3)
#define HDMI_AUI_SF_SF_88KHZ        (1<<4)
#define HDMI_AUI_SF_SF_176KHZ       (1<<4|1<<3)
#define HDMI_AUI_SF_SF_48KHZ        (1<<3|1<<2)
#define HDMI_AUI_SF_SF_96KHZ        (1<<4|1<<2)
#define HDMI_AUI_SF_SF_192KHZ       (1<<4|1<<3|1<<2)

#define GCP_AVMUTE_ON               (1<<0)
#define GCP_AVMUTE_OFF              (1<<4)
#define GCP_CD_NOT_INDICATED        (0)
#define GCP_CD_24BPP                (1<<2)
#define GCP_CD_30BPP                (1<<0|1<<2)
#define GCP_CD_36BPP                (1<<1|1<<2)

#define HDMI_DC_CTL_8               (0)
#define HDMI_DC_CTL_10              (1<<0)
#define HDMI_DC_CTL_12              (1<<1)

#define ASP_TYPE_MASK               (1<<5|1<<6)
#define ASP_LPCM_TYPE               (0)
#define ASP_DSD_TYPE                (1<<5)
#define ASP_HBR_TYPE                (1<<6)
#define ASP_DST_TYPE                (1<<5|1<<6)
#define ASP_MODE_MASK               (1<<4)
#define ASP_LAYOUT_0                (0)
#define ASP_LAYOUT_1                (1<<4)
#define ASP_SP_MASK                 (0x0F)
#define ASP_SP_0                    (1<<0)
#define ASP_SP_1                    (1<<1)
#define ASP_SP_2                    (1<<2)
#define ASP_SP_3                    (1<<3)

#define MPEG_PACKET_BYTE_LENGTH     (5)

#define SPD_INFO_PACKET_BYTE_LENGTH     (25)
#define SPD_INFO_PACKET_TYPE            (0x80 + 0x03)
#define SPD_INFO_PACKET_VERSION         (1)

#define GAMUT_PACKET_TYPE               (0x0A)
#define GAMUT_PACKET_BODY_MAX_SIZE      (28)

#define HDMI_PACKET_HEADER_LENGTH       (3)

#define ACR_N_32KHZ                     4096
#define ACR_N_44KHZ                     6272
#define ACR_N_48KHZ                     6144
#define ACR_N_88KHZ                     12544
#define ACR_N_96KHZ                     12288
#define ACR_N_176KHZ                    25088
#define ACR_N_192KHZ                    24876
//@}
                                            
// SPDIF                                    
typedef struct
{                                           
	unsigned int SPDIFIN_CLK_CTRL;                 // 0x0000
	unsigned int SPDIFIN_OP_CTRL;                  // 0x0004
	unsigned int SPDIFIN_IRQ_MASK;                 // 0x0008
	unsigned int SPDIFIN_IRQ_STATUS;               // 0x000C

	unsigned int SPDIFIN_CONFIG_1;                 // 0x0010
	unsigned int SPDIFIN_CONFIG_2;                 // 0x0014
	unsigned int PAD0[2];                          // 0x0018, 0x0001C

	unsigned int SPDIFIN_USER_VALUE_1;             // 0x0020
	unsigned int SPDIFIN_USER_VALUE_2;             // 0x0024
	unsigned int SPDIFIN_USER_VALUE_3;             // 0x0028
	unsigned int SPDIFIN_USER_VALUE_4;             // 0x002C

	unsigned int SPDIFIN_CH_STATUS_0_1;            // 0x0030
	unsigned int SPDIFIN_CH_STATUS_0_2;            // 0x0034
	unsigned int SPDIFIN_CH_STATUS_0_3;            // 0x0038
	unsigned int SPDIFIN_CH_STATUS_0_4;            // 0x003C

	unsigned int SPDIFIN_CH_STATUS_1;              // 0x0040
	unsigned int PAD1;                             // 0x0044
	unsigned int SPDIFIN_FRAME_PERIOD_1;           // 0x0048
	unsigned int SPDIFIN_FRAME_PERIOD_2;           // 0x004C

	unsigned int SPDIFIN_PC_INFO_1;                // 0x0050
	unsigned int SPDIFIN_PC_INFO_2;                // 0x0054
	unsigned int SPDIFIN_PD_INFO_1;                // 0x0058
	unsigned int SPDIFIN_PD_INFO_2;                // 0x005C

	unsigned int SPDIFIN_DATA_BUF_0_1;             // 0x0060
	unsigned int SPDIFIN_DATA_BUF_0_2;             // 0x0064
	unsigned int SPDIFIN_DATA_BUF_0_3;             // 0x0068
	unsigned int SPDIFIN_USER_BUF_0;               // 0x006C

	unsigned int SPDIFIN_USER_BUF_1_1;             // 0x0070
	unsigned int SPDIFIN_USER_BUF_1_2;             // 0x0074
	unsigned int SPDIFIN_USER_BUF_1_3;             // 0x0078
	unsigned int SPDIFIN_USER_BUF_1;               // 0x007C
}HDMI_SPDIF_REG, *PHDMI_SPDIF_REG;

//@{
/**
 * @name SPDIF bit values
 */
#define SPDIF_CLK_CTRL_ENABLE           (1)
#define SPDIF_CLK_CTRL_DISABLE          (0)

#define SPDIF_IRQ_ENABLE_ALL            (0xFF)
#define SPDIF_IRQ_DISABLE_ALL           (0)

#define SPDIF_CH_STATUS_0_1_LPCM        (0)
#define SPDIF_CH_STATUS_0_1_NPCM        (1<<1)

#define SPDIF_CONFIG_1_NOISE_FILTER_2_SAMPLES   (1<<6)
#define SPDIF_CONFIG_1_NOISE_FILTER_3_SAMPLES   (0)
#define SPDIF_CONFIG_1_LPCM                     (0)
#define SPDIF_CONFIG_1_NPCM                     (1<<5)
#define SPDIF_CONFIG_1_PCPD_MANUAL              (1<<4)
#define SPDIF_CONFIG_1_WORD_LENGTH_MANUAL       (1<<3)
#define SPDIF_CONFIG_1_UVCP_ENABLE              (1<<2)
#define SPDIF_CONFIG_1_HDMI_1_BURST             (0)
#define SPDIF_CONFIG_1_HDMI_2_BURST             (1<<1)
#define SPDIF_CONFIG_1_ALIGN_16BIT              (0)
#define SPDIF_CONFIG_1_ALIGN_32BIT              (1<<0)

#define SPDIF_USER_VALUE_WORD_24BIT             (0x0B)

#define SPDIF_CLK_RECOVERY_FAIL_MASK            (1<<0)
#define SPDIF_STATUS_RECOVERED_MASK             (1<<1)
#define SPDIF_PREAMBLE_DETECTED_MASK            (1<<2)
#define SPDIF_HEADER_NOT_DETECTED_MASK          (1<<3)
#define SPDIF_HEADER_DETECTED_MASK              (1<<4)
#define SPDIF_PAPD_NOT_DETECTED_MASK            (1<<5)
#define SPDIF_ABNORMAL_PD_MASK                  (1<<6)
#define SPDIF_BUFFER_OVERFLOW_MASK              (1<<7)

#define SPDIF_SIGNAL_RESET              (0)
#define SPDIF_SIGNAL_DETECT             (1<<0)
#define SPDIF_RUNNING                   (1<<1 | 1<<0)
//@}

// I2S
typedef struct 
{
	unsigned int I2S_CLK_CON;                      // 0x0000

	unsigned int I2S_CON_1;                        // 0x0004
	unsigned int I2S_CON_2;                        // 0x0008

	unsigned int I2S_PIN_SEL_0;                    // 0x000C
	unsigned int I2S_PIN_SEL_1;                    // 0x0010
	unsigned int I2S_PIN_SEL_2;                    // 0x0014
	unsigned int I2S_PIN_SEL_3;                    // 0x0018

	unsigned int I2S_DSD_CON;                      // 0x001C
	unsigned int I2S_MUX_CON;                      // 0x0020

	unsigned int I2S_CH_ST_CON;                    // 0x0024

	unsigned int I2S_CH_ST_0;                      // 0x0028
	unsigned int I2S_CH_ST_1;                      // 0x002C
	unsigned int I2S_CH_ST_2;                      // 0x0030
	unsigned int I2S_CH_ST_3;                      // 0x0034
	unsigned int I2S_CH_ST_4;                      // 0x0038

	unsigned int I2S_CH_ST_SH_0;                   // 0x003C
	unsigned int I2S_CH_ST_SH_1;                   // 0x0040
	unsigned int I2S_CH_ST_SH_2;                   // 0x0044
	unsigned int I2S_CH_ST_SH_3;                   // 0x0048
	unsigned int I2S_CH_ST_SH_4;                   // 0x004C

	unsigned int I2S_VD_DATA;                      // 0x0050
	unsigned int I2S_MUX_CH;                       // 0x0054
	unsigned int I2S_MUX_CUV;                      // 0x0058

	unsigned int I2S_IRQ_MASK;                     // 0x005C
	unsigned int I2S_IRQ_STATUS;                   // 0x0060

	unsigned int I2S_CH0_L0;                       // 0x0064
	unsigned int I2S_CH0_L1;                       // 0x0068
	unsigned int I2S_CH0_L2;                       // 0x006C
	unsigned int I2S_CH0_L3;                       // 0x0070
	unsigned int I2S_CH0_R0;                       // 0x0074
	unsigned int I2S_CH0_R1;                       // 0x0078
	unsigned int I2S_CH0_R2;                       // 0x007C
	unsigned int I2S_CH0_R3;                       // 0x0080

	unsigned int I2S_CH1_L0;                       // 0x0084
	unsigned int I2S_CH1_L1;                       // 0x0088
	unsigned int I2S_CH1_L2;                       // 0x008C
	unsigned int I2S_CH1_L3;                       // 0x0090
	unsigned int I2S_CH1_R0;                       // 0x0094
	unsigned int I2S_CH1_R1;                       // 0x0098
	unsigned int I2S_CH1_R2;                       // 0x009C
	unsigned int I2S_CH1_R3;                       // 0x00A0

	unsigned int I2S_CH2_L0;                       // 0x00A4
	unsigned int I2S_CH2_L1;                       // 0x00A8
	unsigned int I2S_CH2_L2;                       // 0x00AC
	unsigned int I2S_CH2_L3;                       // 0x00B0
	unsigned int I2S_CH2_R0;                       // 0x00B4
	unsigned int I2S_CH2_R1;                       // 0x00B8
	unsigned int I2S_CH2_R2;                       // 0x00BC
	unsigned int I2S_CH2_R3;                       // 0x00C0

	unsigned int I2S_CH3_L0;                       // 0x00C4
	unsigned int I2S_CH3_L1;                       // 0x00C8
	unsigned int I2S_CH3_L2;                       // 0x00CC
	unsigned int I2S_CH3_R0;                       // 0x00D0
	unsigned int I2S_CH3_R1;                       // 0x00D4
	unsigned int I2S_CH3_R2;                       // 0x00D8
	                      
	unsigned int I2S_CUV_L_R;                      // 0x00DC
}HDMI_I2S_REG, *PHDMI_I2S_REG;

//@{
/**
 * @name I2S bit values
 */
#define I2S_PIN_SEL_AUDIO_0             (0)
#define I2S_PIN_SEL_AUDIO_1             (1)
#define I2S_PIN_SEL_AUDIO_2             (2)
#define I2S_PIN_SEL_AUDIO_3             (3)
#define I2S_PIN_SEL_AUDIO_4             (4)
#define I2S_PIN_SEL_AUDIO_5             (5)
#define I2S_PIN_SEL_AUDIO_6             (6)

#define I2S_CLK_CON_ENABLE              (1)
#define I2S_CLK_CON_DISABLE             (0)

#define I2S_CON_SC_POL_FALLING          (0<<1)
#define I2S_CON_SC_POL_RISING           (1<<1)

#define I2S_CON_CH_POL_LOW              (0)
#define I2S_CON_CH_POL_HIGH             (1)

#define I2S_CON_MSB                     (0<<6)
#define I2S_CON_LSB                     (1<<6)

#define I2S_CON_BIT_CH_MASK             (1<<4 | 1<<5)
#define I2S_CON_BIT_CH_32               (0<<4)
#define I2S_CON_BIT_CH_48               (1<<4)
#define I2S_CON_BIT_CH_64               (2<<4)

#define I2S_CON_DATA_NUM_MASK           (1<<3|1<<2)
#define I2S_CON_DATA_NUM_16             (1<<2)
#define I2S_CON_DATA_NUM_20             (2<<2)
#define I2S_CON_DATA_NUM_24             (3<<2)

#define I2S_CON_I2S_MODE_MASK               (1<<0|1<<1)
#define I2S_CON_I2S_MODE_BASIC              (0)
#define I2S_CON_I2S_MODE_LEFT_JUSTIFIED     (2)
#define I2S_CON_I2S_MODE_RIGHT_JUSTIFIED    (3)

#define I2S_DSD_CON_POL_RISING          (1<<1)
#define I2S_DSD_CON_POL_FALLING         (0<<1)

#define I2S_DSD_CON_ENABLE              (1)
#define I2S_DSD_CON_DISABLE             (0)

#define I2S_IN_MUX_IN_ENABLE            (1<<4)
#define I2S_IN_MUX_SELECT_DSD           (2<<2)
#define I2S_IN_MUX_SELECT_I2S           (1<<2)
#define I2S_IN_MUX_SELECT_SPDIF         (0<<1)
#define I2S_IN_MUX_CUV_ENABLE           (1<<1)
#define I2S_IN_MUX_ENABLE               (1)
#define I2S_IN_MUX_DISABLE              (0)

#define I2S_MUX_CH_0_LEFT_ENABLE        (1<<0)
#define I2S_MUX_CH_0_RIGHT_ENABLE       (1<<1)
#define I2S_MUX_CH_1_LEFT_ENABLE        (1<<2)
#define I2S_MUX_CH_1_RIGHT_ENABLE       (1<<3)
#define I2S_MUX_CH_2_LEFT_ENABLE        (1<<4)
#define I2S_MUX_CH_2_RIGHT_ENABLE       (1<<5)
#define I2S_MUX_CH_3_LEFT_ENABLE        (1<<6)
#define I2S_MUX_CH_3_RIGHT_ENABLE       (1<<7)
#define I2S_MUX_CH_ALL_ENABLE           (0xFF)

#define I2S_MUX_CUV_LEFT_ENABLE         (1<<0)
#define I2S_MUX_CUV_RIGHT_ENABLE        (1<<1)

#define I2S_CH_ST_0_TYPE_MASK           (1<<1)
#define I2S_CH_ST_0_TYPE_LPCM           (0)
#define I2S_CH_ST_0_TYPE_NLPCM          (1<<1)

#define I2S_CH_ST_0_COPY_MASK           (1<<2)
#define I2S_CH_ST_0_NOT_COPYRIGHTED     (1<<2)

#define I2S_CH_ST_3_SF_MASK             (0x3F)
#define I2S_CH_ST_3_SF_44KHZ            (0)
#define I2S_CH_ST_3_SF_88KHZ            (1<<3)
#define I2S_CH_ST_3_SF_176KHZ           (1<<3|1<<2)
#define I2S_CH_ST_3_SF_48KHZ            (1<<1)
#define I2S_CH_ST_3_SF_96KHZ            (1<<3|1<<1)
#define I2S_CH_ST_3_SF_192KHZ           (1<<3|1<<2|1<<1)
#define I2S_CH_ST_3_SF_32KHZ            (1<<1|1<<0)
#define I2S_CH_ST_3_SF_LEV1              (1<<4)
#define I2S_CH_ST_3_SF_LEV2              (0<<4)
#define I2S_CH_ST_3_SF_LEV3              (1<<5)

#define I2S_CH_ST_4_WL_MASK             (0x0F)
#define I2S_CH_ST_4_WL_20_NOT_DEFINED   (0)
#define I2S_CH_ST_4_WL_20_16            (1<<1)
#define I2S_CH_ST_4_WL_20_18            (1<<2)
#define I2S_CH_ST_4_WL_20_19            (1<<3)
#define I2S_CH_ST_4_WL_20_20            (1<<3|1<<1)
#define I2S_CH_ST_4_WL_20_17            (1<<3|1<<2)
#define I2S_CH_ST_4_WL_24_NOT_DEFINED   (1<<0)
#define I2S_CH_ST_4_WL_24_20            (1<<1|1<<0)
#define I2S_CH_ST_4_WL_24_22            (1<<2|1<<0)
#define I2S_CH_ST_4_WL_24_23            (1<<3|1<<0)
#define I2S_CH_ST_4_WL_24_24            (1<<3|1<<1|1<<0)
#define I2S_CH_ST_4_WL_24_21            (1<<3|1<<2|1<<0)

#define I2S_CH_ST_RELOAD                (1<<0)
//@}

// EFUSE
typedef struct 
{
	unsigned int HDCP_E_FUSE_CTRL;                 // 0x0000
	unsigned int HDCP_E_FUSE_STATUS;               // 0x0004
	unsigned int EFUSE_ADDR_WIDTH;                 // 0x0008
	unsigned int EFUSE_SIGDEV_ASSERT;              // 0x000C
	unsigned int EFUSE_SIGDEV_DEASSERT;            // 0x0010
	unsigned int EFUSE_PRCHG_ASSERT;               // 0x0014
	unsigned int EFUSE_PRCHG_DEASSERT;             // 0x0018
	unsigned int EFUSE_FSET_ASSERT;                // 0x001C
	unsigned int EFUSE_FSET_DEASSERT;              // 0x0020
	unsigned int EFUSE_SENSING;                    // 0x0024
	unsigned int EFUSE_SCK_ASSERT;                 // 0x0028
	unsigned int EFUSE_SCK_DEASSERT;               // 0x002C
	unsigned int EFUSE_SDOUT_OFFSET;               // 0x0030
	unsigned int EFUSE_READ_OFFSET;                // 0x0034
} HDMI_EFUSE_REG, *PHDMI_EFUSE_REG;
  
// TG (timing generator for BT656)
typedef struct
{
	// TG SFR
	unsigned int TG_CMD;                           // 0x0000
	unsigned int TG_CFG;                           // 0x0004
	unsigned int PAD0;                             // 0x0008
	                                        
	unsigned int TG_INDELAY_L;                     // 0x000C
	unsigned int TG_INDELAY_H;                     // 0x0010
	unsigned int TG_POL_CTRL;                      // 0x0014
	unsigned int TG_H_FSZ_L;                       // 0x0018
	unsigned int TG_H_FSZ_H;                       // 0x001C
	                                        
	unsigned int TG_HACT_ST_L;                     // 0x0020
	unsigned int TG_HACT_ST_H;                     // 0x0024
	unsigned int TG_HACT_SZ_L;                     // 0x0028
	unsigned int TG_HACT_SZ_H;                     // 0x002C
	                                        
	unsigned int TG_V_FSZ_L;                       // 0x0030
	unsigned int TG_V_FSZ_H;                       // 0x0034
	                                        
	unsigned int TG_VSYNC_L;                       // 0x0038
	unsigned int TG_VSYNC_H;                       // 0x003C
	unsigned int TG_VSYNC2_L;                      // 0x0040
	unsigned int TG_VSYNC2_H;                      // 0x0044
	                                        
	unsigned int TG_VACT_ST_L;                     // 0x0048
	unsigned int TG_VACT_ST_H;                     // 0x004C
	unsigned int TG_VACT_SZ_L;                     // 0x0050
	unsigned int TG_VACT_SZ_H;                     // 0x0054
	                                        
	unsigned int TG_FIELD_CHG_L;                   // 0x0058
	unsigned int TG_FIELD_CHG_H;                   // 0x005C
	unsigned int TG_VACT_ST2_L;                    // 0x0060
	unsigned int TG_VACT_ST2_H;                    // 0x0064
	unsigned int PAD1[4];                          // 0x0068, 0x006C
	                                        // 0x0070, 0x0074
	                                        
	unsigned int TG_VSYNC_TOP_HDMI_L;              // 0x0078
	unsigned int TG_VSYNC_TOP_HDMI_H;              // 0x007C
	unsigned int TG_VSYNC_BOT_HDMI_L;              // 0x0080
	unsigned int TG_VSYNC_BOT_HDMI_H;              // 0x0084
	unsigned int TG_FIELD_TOP_HDMI_L;              // 0x0088
	unsigned int TG_FIELD_TOP_HDMI_H;              // 0x008C
	unsigned int TG_FIELD_BOT_HDMI_L;              // 0x0090
	unsigned int TG_FIELD_BOT_HDMI_H;              // 0x0094
	unsigned int MHL_HSYNC_WIDTH;                  // 0x0098
	unsigned int MHL_VSYNC_WIDTH;                  // 0x009C
	unsigned int MHL_CLK_INV;                      // 0x00A0
}HDMI_TG_REG, *PHDMI_TG_REG;

#if __cplusplus
    }
#endif

#endif    // __S5PV210_HDMI_REG_H
