//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  Header: InitSystem.h
//
//  Define the Clock, DRAM and LCD attributes.
//
#ifndef __S5PV210_INITSYSTEM_H
#define __S5PV210_INITSYSTEM_H

#if __cplusplus
extern "C" {
#endif

#define CLK_12MHz            12000000
#define CLK_24MHz            24000000

#define CLK_45_16MHz         45158000
#define CLK_84MHz            84000000
#define CLK_96MHz            96000000

#define CLK_54MHz            54000000

#define CLK_66_5MHz          66000000
#define CLK_83MHz            83000000


#define CLK_100MHz          100000000
#define CLK_133MHz          133000000
#define CLK_166MHz          166000000
#define CLK_200MHz          200000000
#define CLK_667MHz          667000000

#define CLK_400MHz          400000000
#define CLK_500MHz          500000000
#define CLK_800MHz          800000000
#define CLK_1GHz           1000000000

//----------------------------------------------------------------------
// Clock Setting for BL1
//
//----------------------------------------------------------------------
#define FIN                CLK_24MHz

#define ARM_CLK_BL        (CLK_800MHz)
#define HCLKMSYS_CLK_BL   (CLK_200MHz)
#define HCLKDSYS_CLK_BL   (CLK_166MHz)
#define HCLKPSYS_CLK_BL   (CLK_133MHz)
#define PCLKMSYS_CLK_BL   (CLK_100MHz)
#define PCLKDSYS_CLK_BL   (CLK_83MHz)
#define PCLKPSYS_CLK_BL   (CLK_66_5MHz)
#define SCLK_A2M_BL       (CLK_200MHz)

#define EPLL_CLK_BL       (CLK_96MHz)
#define VPLL_CLK_BL       (CLK_54MHz)

#if (ARM_CLK_BL == CLK_400MHz)
	#define APLL_RATIO			2	// APLLout/2-400
#else
	#define APLL_RATIO          1    // APLLout/1-800
#endif

#if (ARM_CLK_BL == CLK_1GHz)
	#define A2M_RATIO			5    // APLLout/5-200
#else
	#define A2M_RATIO           4    // APLLout/4-200
#endif

#define HCLK_DSYS_RATIO     4    // MPLLout/4-166
#define HCLK_PSYS_RATIO     5    // MPLLout/5-133
#define ONENAND_RATIO       0    // HCLK_DSYS/2-83
#define DMC0_RATIO          1

#define HCLK_MSYS_RATIO     ARM_CLK_BL/HCLKMSYS_CLK_BL
#define PCLK_MSYS_RATIO     HCLKMSYS_CLK_BL/PCLKMSYS_CLK_BL
#define PCLK_DSYS_RATIO     HCLKDSYS_CLK_BL/PCLKDSYS_CLK_BL
#define PCLK_PSYS_RATIO     HCLKPSYS_CLK_BL/PCLKPSYS_CLK_BL

#if (ARM_CLK_BL == CLK_800MHz) || (ARM_CLK_BL == CLK_400MHz)
#define APLL_MDIV           100
#define APLL_PDIV           3
#define APLL_SDIV           1
#if (S5PV210_EVT>0)
#define APLL_AFC_ENB        0
#define APLL_AFC            0
#endif
#elif (ARM_CLK_BL == CLK_1GHz)
#define APLL_MDIV			125
#define APLL_PDIV			3
#define APLL_SDIV			1
#if (S5PV210_EVT>0)
#define APLL_AFC_ENB        0
#define APLL_AFC            0
#endif
#else
#error
#endif

#if (HCLKDSYS_CLK_BL == CLK_166MHz)
#define MPLL_MDIV           667
#define MPLL_PDIV           12
#define MPLL_SDIV           1
#else
#error
#endif

#if (EPLL_CLK_BL == CLK_45_16MHz)
#if (S5PV210_EVT==0)
#define EPLL_MDIV           271
#define EPLL_PDIV           18
#define EPLL_SDIV           3
#else
#define EPLL_MDIV           45
#define EPLL_PDIV           3
#define EPLL_SDIV           3
#define EPLL_DSM_K          10355
#endif
#elif (EPLL_CLK_BL == CLK_96MHz)
#if (S5PV210_EVT==0)
#define EPLL_MDIV           96
#define EPLL_PDIV           6
#define EPLL_SDIV           2
#else
#define EPLL_MDIV           48
#define EPLL_PDIV           3
#define EPLL_SDIV           2
#define EPLL_DSM_K          0
#endif
#else
#error
#endif
#if (VPLL_CLK_BL == CLK_54MHz)
#define VPLL_MDIV           108
#define VPLL_PDIV           6
#define VPLL_SDIV           3
#else
#error
#endif


#define APLL_MDIV_1GHz      125
#define APLL_PDIV_1GHz      3
#define APLL_SDIV_1GHz      1
#if (S5PV210_EVT>0)
#define APLL_AFC_ENB_1GHz   0
#define APLL_AFC_1GHz       0
#endif

//----------------------------------------------------------------------
// Macro funtions
// : Read Clock registers.
//----------------------------------------------------------------------
//Read M,P,S value
#define vAPLL_MDIV(pll_reg)     ((pll_reg>>16)&0x3ff)
#define vAPLL_PDIV(pll_reg)     ((pll_reg>>8)&0x3f)
#define vAPLL_SDIV(pll_reg)     ((pll_reg>>0)&0x7)

#define vMPLL_MDIV(pll_reg)     ((pll_reg>>16)&0x3ff)
#define vMPLL_PDIV(pll_reg)     ((pll_reg>>8)&0x3f)
#define vMPLL_SDIV(pll_reg)     ((pll_reg>>0)&0x7)

#define vEPLL_MDIV(pll_reg)     ((pll_reg>>16)&0x1ff)
#define vEPLL_PDIV(pll_reg)     ((pll_reg>>8)&0x3f)
#define vEPLL_SDIV(pll_reg)     ((pll_reg>>0)&0x7)

#define vVPLL_MDIV(pll_reg)     ((pll_reg>>16)&0x1ff)
#define vVPLL_PDIV(pll_reg)     ((pll_reg>>8)&0x3f)
#define vVPLL_SDIV(pll_reg)     ((pll_reg>>0)&0x7)

//Read Divider
#define vDIV_APLL(div_reg)      (((div_reg>>0)&0x7)+1)
#define vDIV_A2M(div_reg)       (((div_reg>>4)&0x7)+1)
#define vDIV_HCLKMSYS(div_reg)  (((div_reg>>8)&0x7)+1)
#define vDIV_PCLKMSYS(div_reg)  (((div_reg>>12)&0x7)+1)
#define vDIV_HCLKDSYS(div_reg)  (((div_reg>>16)&0xf)+1)
#define vDIV_PCLKDSYS(div_reg)  (((div_reg>>20)&0x7)+1)
#define vDIV_HCLKPSYS(div_reg)  (((div_reg>>24)&0xf)+1)
#define vDIV_PCLKPSYS(div_reg)  (((div_reg>>28)&0x7)+1)

#define GET_APLLCLK(pll_reg)    (vAPLL_MDIV(pll_reg) * (FIN / vAPLL_PDIV(pll_reg) / (1<<(vAPLL_SDIV(pll_reg)-1))))
#define GET_MPLLCLK(pll_reg)    (vMPLL_MDIV(pll_reg) * (FIN / vMPLL_PDIV(pll_reg) / (1<<vMPLL_SDIV(pll_reg))))
#define GET_EPLLCLK(pll_reg)    (vEPLL_MDIV(pll_reg) * (FIN / vEPLL_PDIV(pll_reg) / (1<<vEPLL_SDIV(pll_reg))))
#define GET_VPLLCLK(pll_reg)    (vVPLL_MDIV(pll_reg) * (FIN / vVPLL_PDIV(pll_reg) / (1<<vVPLL_SDIV(pll_reg))))

#define GET_ARMCLK(pll_reg, div_reg)        (GET_APLLCLK(pll_reg) / vDIV_APLL(div_reg))
#define GET_HCLKMSYS(pll_reg, div_reg)      (GET_ARMCLK(pll_reg, div_reg) / vDIV_HCLKMSYS(div_reg))
#define GET_PCLKMSYS(pll_reg, div_reg)      (GET_HCLKMSYS(pll_reg, div_reg) / vDIV_PCLKMSYS(div_reg))
#define GET_HCLKDSYS(pll_reg, div_reg)      (GET_MPLLCLK(pll_reg) / vDIV_HCLKDSYS(div_reg))
#define GET_PCLKDSYS(pll_reg, div_reg)      (GET_HCLKDSYS(pll_reg, div_reg) / vDIV_PCLKDSYS(div_reg))
#define GET_HCLKPSYS(pll_reg, div_reg)      (GET_MPLLCLK(pll_reg) / vDIV_HCLKPSYS(div_reg))
#define GET_PCLKPSYS(pll_reg, div_reg)      (GET_HCLKPSYS(pll_reg, div_reg) / vDIV_PCLKPSYS(div_reg))



//======================================================================
//
// DRAM Controller Definitions (LPCON)
//
//======================================================================

//--------------------------------------------------------------------
#define MEM_TYPE            4         // DDR2
#define CAS_DELAY           3
#define ENABLE_PHY_DLL      1
#define QoS_Enable          1    // Priority: VP-Mixer-FIMD-FIMC-MFC-3D
#define Fast_QoS_Enable     0    // Fast QoS/FIMD/FIMC/VP/MIXER
//----------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
#define DRAM0_KHZ            (HCLKMSYS_CLK_BL/1000)   // 200MHz
#define DRAM0_tCK            (((10*1000000/DRAM0_KHZ)+9)/10)
#define DRAM1_KHZ            (SCLK_A2M_BL/1000)      // 200MHz
#define DRAM1_tCK            (((10*1000000/DRAM1_KHZ)+9)/10)
//--------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------------------------------------------------------
// DMC0 Initialization Parameter
// DMC0_TIMING_ROW
#define DRAM0_tRFC           (10*(DRAM0_KHZ*128)/1000000+9)/10  //	auto refresh to active - max.0xff
#define DRAM0_tRRD           (10*(DRAM0_KHZ*8)/1000000+9)/10  //	active bank A to B delay - max.0xf
#define DRAM0_tRP            (10*(DRAM0_KHZ*15)/1000000+9)/10  //	precharge comm. period - max.0xf
#define DRAM0_tRCD           (10*(DRAM0_KHZ*15)/1000000+9)/10  //	active to read or write delay - max.oxf
#define DRAM0_tRC            (10*(DRAM0_KHZ*60)/1000000+9)/10  //	active to active period - max.0x3f
#define DRAM0_tRAS           (10*(DRAM0_KHZ*45)/1000000+9)/10  //	active to precharge comm. period - max.3f
// DMC0_TIMING_DATA
#define DRAM0_tWTR           (10*(DRAM0_KHZ*8)/1000000+9)/10  //	internal write to read comm. delay - max.0xf
#define DRAM0_tWR            (10*(DRAM0_KHZ*15)/1000000+9)/10          //	write recovery time - max.0xf
#define DRAM0_tRTP           (10*(DRAM0_KHZ*8)/1000000+9)/10          //	internal read to precharge comm. delay - max.0xf
#define DRAM0_tCL            (10*(DRAM0_KHZ*3*DRAM0_tCK)/1000000+5)/10  //	CAS latency - max.0xf
#define DRAM0_tWL            (10*(DRAM0_KHZ*3*DRAM0_tCK)/1000000+5)/10  //	write data latency - max.0xf
#define DRAM0_tRL            (10*(DRAM0_KHZ*3*DRAM0_tCK)/1000000+5)/10  //	read data latency - max.0xf
// DMC0_TIMING_POWER
#define DRAM0_tFAW           (10*(DRAM0_KHZ*38)/1000000+9)/10          //	four activate window - max.0x3f
#define DRAM0_tXSR           (10*(DRAM0_KHZ*200*DRAM0_tCK)/1000000+9)/10         //	self refresh exit power down to next valid comm. delay - max.0xff
#define DRAM0_tXP            (10*(DRAM0_KHZ*2*DRAM0_tCK)/1000000+9)/10	     //	exit power down to next valid comm. delay - max.0xff
#define DRAM0_tCKE           (10*(DRAM0_KHZ*3*DRAM0_tCK)/1000000+9)/10  //	CKE min. pulse width - max.0xf
#define DRAM0_tMRD           (10*(DRAM0_KHZ*2*DRAM0_tCK)/1000000+9)/10  //	mode register set comm. period - max.0xf	
#define DRAM0_tREF           (10*(DRAM0_KHZ/10000)*78+9)/10
//----------------------------------------------------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------------------------------------------------------------------------
// DMC1 Initialization Parameter
// DMC1_TIMING_ROW
#define DRAM1_tRFC           (10*(DRAM1_KHZ*128)/1000000+9)/10  //	auto refresh to active - max.0xff
#define DRAM1_tRRD           (10*(DRAM1_KHZ*8)/1000000+9)/10  //	active bank A to B delay - max.0xf
#define DRAM1_tRP            (10*(DRAM1_KHZ*15)/1000000+9)/10  //	precharge comm. period - max.0xf
#define DRAM1_tRCD           (10*(DRAM1_KHZ*15)/1000000+9)/10  //	active to read or write delay - max.oxf
#define DRAM1_tRC            (10*(DRAM1_KHZ*60)/1000000+9)/10  //	active to active period - max.0x3f
#define DRAM1_tRAS           (10*(DRAM1_KHZ*45)/1000000+9)/10  //	active to precharge comm. period - max.3f
// DMC1_TIMING_DATA
#define DRAM1_tWTR           (10*(DRAM1_KHZ*8)/1000000+9)/10  //	internal write to read comm. delay - max.0xf
#define DRAM1_tWR            (10*(DRAM1_KHZ*15)/1000000+9)/10          //	write recovery time - max.0xf
#define DRAM1_tRTP           (10*(DRAM1_KHZ*8)/1000000+9)/10          //	internal read to precharge comm. delay - max.0xf
#define DRAM1_tCL            (10*(DRAM1_KHZ*3*DRAM1_tCK)/1000000+5)/10  //	CAS latency - max.0xf
#define DRAM1_tWL            (10*(DRAM1_KHZ*3*DRAM1_tCK)/1000000+5)/10  //	write data latency - max.0xf
#define DRAM1_tRL            (10*(DRAM1_KHZ*3*DRAM1_tCK)/1000000+5)/10  //	read data latency - max.0xf
// DMC1_TIMING_POWER
#define DRAM1_tFAW           (10*(DRAM1_KHZ*38)/1000000+9)/10          //	four activate window - max.0x3f
#define DRAM1_tXSR           (10*(DRAM1_KHZ*200*DRAM1_tCK)/1000000+9)/10         //	self refresh exit power down to next valid comm. delay - max.0xff
#define DRAM1_tXP            (10*(DRAM1_KHZ*2*DRAM1_tCK)/1000000+9)/10	     //	exit power down to next valid comm. delay - max.0xff
#define DRAM1_tCKE           (10*(DRAM1_KHZ*3*DRAM1_tCK)/1000000+9)/10  //	CKE min. pulse width - max.0xf
#define DRAM1_tMRD           (10*(DRAM1_KHZ*2*DRAM1_tCK)/1000000+9)/10  //	mode register set comm. period - max.0xf	
#define DRAM1_tREF           (10*(DRAM1_KHZ/10000)*78+9)/10
//----------------------------------------------------------------------------------------------------------------------------------------------

//--------------------------------------------------------------------
// DMC_PHY_CONTROL0 register : PHY control 0 register
//--------------------------------------------------------------------
#define CTRL_FORCE			0x82		//  DLL force - max.0xff
#define CTRL_INC			0x10		//	Increase amount - max.0xff
#define CTRL_START_POINT	0x10		//	DLL lock start point - max.0xff
#define CTRL_FNC_FB			0			//  Function Feedback Test
#define CTRL_DDR			1		    // 	reset value of control output
#define CTRL_DFDQS			1           //    single DQS setting
#define CTRL_HALF			0		    // 	DLL low speed
#define CTRL_DLL_ON			0		    // 	DLL on
#define CTRL_START			0		    // 	DLL start

//--------------------------------------------------------------------
// DMC_PHY_CONTROL1 register : PHY control 1 register
//-------------------------------------------------------------------
#define DQS_DELAY			0		   //   shiftc + dqs delay clock <- DDR2, LPDDR2 upper 333MHz
#define CTRL_OFFSETD		0		   //	offset for debug - max.0x7f
#define	DRV_TYPE			0		   //   Driving type of bidirectional pins in ildel state
#define CTRL_OFFSETC		4		   //	gate offset - max.0x7f
#define CTRL_REF			8		   //   0~0xF, reference count for DLL lock confirmation
#define FD_RESYNC			0		   //   force dll resync.
#define CTRL_SHIFTC			6		   //	gateout signal delay - 0:T/128, 1:T/64, 2:T/32, 3:T/16, 4:T/8, 5:T/4, 6:T/2, 7:T

//--------------------------------------------------------------------
// DMC_PHY_CONTROL2 register : PHY control 2 register
//--------------------------------------------------------------------
#define CTRL_OFFSET3		0		  //    offset for debug - max.0x7f
#define CTRL_OFFSET2		0		  //	offset for debug - max.0x7f
#define CTRL_OFFSET1		0		  //	offset for debug - max.0x7f
#define CTRL_OFFSET0		0		  //	offset for debug - max.0x7f

//===================================================================


//======================================================================
//
// LCD Module Type Definitions
//
//======================================================================
#define LCD_MODULE_LTS222       (0)    // Portrait 2.2" QVGA RGB16
#define LCD_MODULE_LTV350       (1)    // Landscape 3.5" QVGA RGB16  // modefied by terry  qimei 4.3 inch 240*320
#define LCD_MODULE_AT070TN92    (2)    // Landscape 4.8" WVGA RGB16
#define LCD_MODULE_EMUL48_D1    (3)    // Landscape 4.8" WVGA RGB16 as D1 (720x480)
#define LCD_MODULE_EMUL48_QV    (4)    // Landscape 4.8" WVGA RGB16 as QVGA (320x240)
#define LCD_MODULE_EMUL48_PQV   (5)    // Landscape 4.8" WVGA RGB16 as PQVGA (240x320)
#define LCD_MODULE_EMUL48_ML    (6)    // Landscape 4.8" WVGA RGB16 as 480x320
#define LCD_MODULE_EMUL48_MP    (7)    // Landscape 4.8" WVGA RGB16 as 320x480
#define LCD_MODULE_LTP700       (8)    // Landscape 7" WVGA RGB24
#define LCD_MODULE_TL2796       (9)    // Portrait 480x800
#define LCD_MODULE_DSI_DUMMY	(10)   // Portrait 3.3" MIPI Type 320x480
#define LCD_MODULE_HT101HD1     (11)   // Landscape 1376x768


#define LCD_MODULE_VGA8060   	(12)
#define LCD_MODULE_VGA1024   	(13)

#define LCD_MODULE_VGA1280	(14)
#define LCD_MODULE_VAG8048      (15)

#define LCD_MODULE_VGA1440    (16)

#define LCD_TYPE_PORTRAIT       (0)
#define LCD_TYPE_LANDSCAPE      (1)

//----------------------------------------------------------------------
// Define SMDKV210 LCD Default Module Type
//----------------------------------------------------------------------

//#define LCD_DEFAULT_MODULE    (LCD_MODULE_LTE480)
//#define LCD_DEFAULT_MODULE    (LCD_MODULE_VGA8060)
//#define LCD_DEFAULT_MODULE    (LCD_MODULE_VGA1024)

/* //lqm masked.
#if (LCD_DEFAULT_MODULE == LCD_MODULE_LTS222)
#define LCD_WIDTH               240
#define LCD_HEIGHT              320
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_PORTRAIT
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_LTV350)
#define LCD_WIDTH               320
#define LCD_HEIGHT              240
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_AT070TN92)
#define LCD_WIDTH               800
#define LCD_HEIGHT              480
#define LCD_BPP                 16  //32
#define LCD_TYPE                LCD_TYPE_PORTRAIT //LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_HT101HD1)
#define LCD_WIDTH               1376
#define LCD_HEIGHT              768
#define LCD_BPP                 16  //32
#define LCD_TYPE                LCD_TYPE_PORTRAIT //LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_EMUL48_D1)
#define LCD_WIDTH               720
#define LCD_HEIGHT              480
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_EMUL48_QV)
#define LCD_WIDTH               320
#define LCD_HEIGHT              240
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_EMUL48_PQV)
#define LCD_WIDTH               240
#define LCD_HEIGHT              320
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_PORTRAIT
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_EMUL48_ML)
#define LCD_WIDTH               480
#define LCD_HEIGHT              320
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_EMUL48_MP)
#define LCD_WIDTH               320
#define LCD_HEIGHT              480
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_PORTRAIT
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_LTP700)
#define LCD_WIDTH               800
#define LCD_HEIGHT              480
#define LCD_BPP                 32    // rgb888 XRGB
#define LCD_TYPE                LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_TL2796)
#define LCD_WIDTH               480
#define LCD_HEIGHT              800
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_PORTRAIT
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_DSI_DUMMY)
#define LCD_WIDTH               320
#define LCD_HEIGHT              480
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_PORTRAIT
#define USE_DSI
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_VGA8060)
#define LCD_WIDTH               800
#define LCD_HEIGHT              600
#define LCD_BPP                 16 
#define LCD_TYPE              LCD_TYPE_PORTRAIT//  LCD_TYPE_LANDSCAPE
#elif (LCD_DEFAULT_MODULE == LCD_MODULE_VGA1024)
#define LCD_WIDTH               1024
#define LCD_HEIGHT              768
#define LCD_BPP                 16
#define LCD_TYPE                LCD_TYPE_PORTRAIT//LCD_TYPE_LANDSCAPE
#else
#error LCD_MODULE_UNDEFINED_ERROR
#endif
*/


#if __cplusplus
}
#endif

#endif  //__INITSYSTEM_H

