//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this sample source code is subject to the terms of the Microsoft
// license agreement under which you licensed this sample source code. If
// you did not accept the terms of the license agreement, you are not
// authorized to use this sample source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the LICENSE.RTF on your install media or the root of your tools installation.
// THE SAMPLE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES OR INDEMNITIES.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:  
    system.c
    
Abstract:  
    Device dependant part of the USB Universal Host Controller Driver (EHCD).

Notes: 
--*/
#include <windows.h>
#include <nkintr.h>
#include <ceddk.h>
#include <ddkreg.h>
#include <devload.h>
#include <giisr.h>
#include <hcdddsi.h>
#include <cebuscfg.h>

#include <bsp_cfg.h>
#include <register_map.h>
#include <DrvLib_wait.h>
#include <DrvLib_mem.h>


#define USBH_MSG(x)
#define USBH_INF(x)    DEBUGMSG(ZONE_FUNCTION, x)
#define USBH_ERR(x)    REATILMSG(1, x)

/*
#ifdef DEBUGMSG
#undef DEBUGMSG
#endif

#define DEBUGMSG(x,p)    RETAILMSG(1,p)
*/

#define REG_PHYSICAL_PAGE_SIZE TEXT("PhysicalPageSize")

// Amount of memory to use for HCD buffer
#define gcTotalAvailablePhysicalMemory (0x20000) // 128K
#define gcHighPriorityPhysicalMemory (gcTotalAvailablePhysicalMemory/4)   // 1/4 as high priority

typedef struct {
    DWORD Addr_64Bit:1;
    DWORD Frame_Prog:1;
    DWORD Async_Park:1;
    DWORD Reserved1:1;
    DWORD Isoch_Sched_Threshold:4;
    DWORD EHCI_Ext_Cap_Pointer:8;
    DWORD Reserved2:16;
} HCCP_CAP_Bit;
typedef union {
    volatile HCCP_CAP_Bit   bit;
    volatile DWORD          ul;
} HCCP_CAP;
typedef struct {
    DWORD Cap_ID:8;
    DWORD Next_Cap_Pointer:8;
    DWORD HC_BIOS_Owned:1;
    DWORD Reserved1:7;
    DWORD HC_OS_Owned:1;
    DWORD Reserved2:7;
} USBLEGSUP_Bit;
typedef union {
    volatile USBLEGSUP_Bit  bit;
    volatile DWORD          ul;
} USBLEGSUP;

static volatile CMU_CLK_REG        *g_pCMUCLKregs;
static volatile PMU_MISC_REG       *g_pPMUMISCregs;
static volatile OTG_PHY_REG        *g_pOTGPHYregs;
static volatile PMU_OSC_REG           *g_pPMUOSCregs;
static volatile USBEHCIHOST_REG            *g_pUSBEHCIHostregs;    

typedef struct _SEHCDPdd
{
    LPVOID lpvMemoryObject;
    LPVOID lpvEHCDMddObject;
    PVOID pvVirtualAddress;                        // DMA buffers as seen by the CPU
    DWORD dwPhysicalMemSize;
    PHYSICAL_ADDRESS LogicalAddress;        // DMA buffers as seen by the DMA controller and bus interfaces
    DMA_ADAPTER_OBJECT AdapterObject;
    TCHAR szDriverRegKey[MAX_PATH];
    PUCHAR ioPortBase;
    DWORD dwSysIntr;
    CRITICAL_SECTION csPdd;                     // serializes access to the PDD object
    HANDLE          IsrHandle;
    HANDLE hParentBusHandle;    
} SEHCDPdd;

#define UnusedParameter(x)  x = x

/* HcdPdd_DllMain
 * 
 *  DLL Entry point.
 *
 * Return Value:
 */

void EnableUSBPHYClock( UINT32 bFlag );
void SetUSBHostCLKPWR(UINT32 bFlag);

extern BOOL HcdPdd_DllMain(HANDLE hinstDLL, DWORD dwReason, LPVOID lpvReserved)
{
    UnusedParameter(hinstDLL);
    UnusedParameter(dwReason);
    UnusedParameter(lpvReserved);

    return TRUE;
}
static BOOL
GetRegistryPhysicalMemSize(
    LPCWSTR RegKeyPath,         // IN - driver registry key path
    DWORD * lpdwPhyscialMemSize)       // OUT - base address
{
    HKEY hKey;
    DWORD dwData;
    DWORD dwSize;
    DWORD dwType;
    BOOL  fRet=FALSE;
    DWORD dwRet;

	RETAILMSG(1,(_T("++%s\r\n"),__FUNCTION__));
    // Open key
    dwRet = RegOpenKeyEx(HKEY_LOCAL_MACHINE,RegKeyPath,0,0,&hKey);
    if (dwRet != ERROR_SUCCESS) {
        DEBUGMSG(ZONE_ERROR,(TEXT("!EHCD:GetRegistryConfig RegOpenKeyEx(%s) failed %d\r\n"),
                             RegKeyPath, dwRet));
        return FALSE;
    }

    // Read base address, range from registry and determine IOSpace
    dwSize = sizeof(dwData);
    dwRet = RegQueryValueEx(hKey, REG_PHYSICAL_PAGE_SIZE, 0, &dwType, (PUCHAR)&dwData, &dwSize);
    if (dwRet == ERROR_SUCCESS) {
        if (lpdwPhyscialMemSize)
            *lpdwPhyscialMemSize = dwData;
        fRet=TRUE;
    }
    RegCloseKey(hKey);
    return fRet;
}


/* ConfigureEHCICard
 * 
 */
BOOL
ConfigureEHCICard(
    SEHCDPdd * pPddObject, // IN - contains PDD reference pointer.
    PUCHAR *pioPortBase,   // IN - contains physical address of register base
                           // OUT- contains virtual address of register base
    DWORD dwAddrLen,
    DWORD dwIOSpace,
    INTERFACE_TYPE IfcType,
    DWORD dwBusNumber
    )
{
    ULONG               inIoSpace = dwIOSpace;
    ULONG               portBase;
    PHYSICAL_ADDRESS    ioPhysicalBase = {0, 0};

	//RETAILMSG(1,(_T("++%s\r\n"),__FUNCTION__));
    
    portBase = (ULONG)*pioPortBase;
    ioPhysicalBase.LowPart = portBase;

    if (!BusTransBusAddrToVirtual(pPddObject->hParentBusHandle, IfcType, dwBusNumber, ioPhysicalBase, dwAddrLen, &inIoSpace, (PPVOID)pioPortBase)) {
        DEBUGMSG(ZONE_ERROR, (L"EHCD: Failed TransBusAddrToVirtual\r\n"));
        return FALSE;
    }

    DEBUGMSG(ZONE_INIT, 
             (TEXT("EHCD: ioPhysicalBase 0x%X, IoSpace 0x%X\r\n"),
              ioPhysicalBase.LowPart, inIoSpace));
    DEBUGMSG(ZONE_INIT, 
             (TEXT("EHCD: ioPortBase 0x%X, portBase 0x%X\r\n"),
              *pioPortBase, portBase));

    return TRUE;
}


/* InitializeEHCI
 *
 *  Configure and initialize EHCI card
 *
 * Return Value:
 *  Return TRUE if card could be located and configured, otherwise FALSE
 */
static BOOL 
InitializeEHCI(
    SEHCDPdd * pPddObject,    // IN - Pointer to PDD structure
    LPCWSTR szDriverRegKey)   // IN - Pointer to active registry key string
{
    PUCHAR ioPortBase = NULL;
    DWORD dwAddrLen;
    DWORD dwIOSpace;
    BOOL InstallIsr = FALSE;
    BOOL fResult = FALSE;
    LPVOID pobMem = NULL;
    LPVOID pobEHCD = NULL;
    DWORD PhysAddr;
    DWORD dwHPPhysicalMemSize;
    HKEY    hKey;
    HANDLE hBus;
    DWORD dwOffset, dwLength, dwCount;
    PCI_SLOT_NUMBER Slot;
    HCCP_CAP Hccparam;
    USBLEGSUP Usblegsup;

    DDKWINDOWINFO dwi;
    DDKISRINFO dii;
    DDKPCIINFO dpi;

    PHYSICAL_ADDRESS ioPhysicalBase = {0, 0};
    //RETAILMSG(1,(TEXT("[EHCI] InitailizeEHCI\r\n")));

    g_pCMUCLKregs = (volatile CMU_CLK_REG *)DrvLib_MapIoSpace(BASE_REG_PA_CMU_CLK, sizeof(CMU_CLK_REG), FALSE);
    if (g_pCMUCLKregs == NULL)
    {
        RETAILMSG(TRUE, (TEXT("[EHCI] CMU Clock Register is *NOT* mapped.\n")));
        return FALSE;
    }

    g_pPMUMISCregs= (volatile PMU_MISC_REG *)DrvLib_MapIoSpace(BASE_REG_PA_PMU_MISC, sizeof(PMU_MISC_REG), FALSE);
    if (g_pPMUMISCregs == NULL)
    {
        RETAILMSG(TRUE, (TEXT("[EHCI] PMU Power Register is *NOT* mapped.\n")));
        return FALSE;
    }

    g_pOTGPHYregs = (volatile OTG_PHY_REG *)DrvLib_MapIoSpace(BASE_REG_PA_USBOTG_PHY, sizeof(OTG_PHY_REG), FALSE);
    if (g_pOTGPHYregs == NULL)
    {
        RETAILMSG(TRUE, (TEXT("[EHCI] OTG PHY Register is *NOT* mapped.\n")));
        return FALSE;
    }

    g_pPMUOSCregs = (volatile PMU_OSC_REG *)DrvLib_MapIoSpace(BASE_REG_PA_PMU_OSC , sizeof(PMU_OSC_REG), FALSE);
    if (g_pPMUOSCregs == NULL)
    {
        RETAILMSG(TRUE, (TEXT("[EHCI] PMU OSC Register is *NOT* mapped.\n")));
        return FALSE;
    }

    g_pUSBEHCIHostregs = (volatile USBEHCIHOST_REG *)DrvLib_MapIoSpace(BASE_REG_PA_USBHOST_EHCI , sizeof(USBEHCIHOST_REG), FALSE);
    if (g_pUSBEHCIHostregs == NULL)
    {
        RETAILMSG(TRUE, (TEXT("[EHCI] USB HOST EHCI Register is *NOT* mapped.\n")));
        return FALSE;
    }


    SetUSBHostCLKPWR(TRUE);

    g_pUSBEHCIHostregs->INSNREG00 = 0xe0000; // enable burst trans
    g_pUSBEHCIHostregs->INSNREG01 = 0x300010; // set threshold as 512 byte.
    
    if (RegOpenKeyEx(HKEY_LOCAL_MACHINE,szDriverRegKey,0,0,&hKey)!= ERROR_SUCCESS) {
        DEBUGMSG(ZONE_ERROR,(TEXT("InitializeEHCI:GetRegistryConfig RegOpenKeyEx(%s) failed\r\n"),
                             szDriverRegKey));
        return FALSE;
    }
    
    dwi.cbSize=sizeof(dwi);
    dii.cbSize=sizeof(dii);
    dpi.cbSize=sizeof(dpi);
    if ( DDKReg_GetWindowInfo(hKey, &dwi ) != ERROR_SUCCESS ) {
        DEBUGMSG(ZONE_ERROR,(TEXT("InitializeEHCI:DDKReg_GetWindowInfo failed\r\n")));
        goto InitializeEHCI_Error;
    }
    
    if (dwi.dwNumMemWindows!=0) {
        PhysAddr = dwi.memWindows[0].dwBase;
        dwAddrLen= dwi.memWindows[0].dwLen;
        dwIOSpace = 0;
    }
    else if (dwi.dwNumIoWindows!=0) {
        PhysAddr= dwi.ioWindows[0].dwBase;
        dwAddrLen = dwi.ioWindows[0].dwLen;
        dwIOSpace = 1;
    }
    else
        goto InitializeEHCI_Error;
    
       DEBUGMSG(ZONE_INIT,(TEXT("EHCD: Read config from registry: Base Address: 0x%X, Length: 0x%X, I/O Port: %s,Interface Type: %u, Bus Number: %u\r\n"),
                        PhysAddr, dwAddrLen, dwIOSpace ? L"YES" : L"NO",dwi.dwInterfaceType, dwi.dwBusNumber));
    
    
    ioPortBase = (PUCHAR)PhysAddr;
                        
    if (!(fResult = ConfigureEHCICard(pPddObject, &ioPortBase, dwAddrLen, dwIOSpace, dwi.dwInterfaceType, dwi.dwBusNumber))) {
        goto InitializeEHCI_Error;
    }
    
    if ( DDKReg_GetIsrInfo (hKey, &dii ) == ERROR_SUCCESS  &&
            dii.szIsrDll[0] != 0 && dii.szIsrHandler[0]!=0 && dii.dwIrq<0xff && dii.dwIrq>0 ) {
        // Install ISR handler
        pPddObject->IsrHandle = LoadIntChainHandler(dii.szIsrDll, dii.szIsrHandler, (BYTE)dii.dwIrq);

        if (!pPddObject->IsrHandle) {
            DEBUGMSG(ZONE_ERROR, (L"EHCD: Couldn't install ISR handler\r\n"));
        } else {
            GIISR_INFO Info;
            PHYSICAL_ADDRESS PortAddress = {PhysAddr, 0};
            
            DEBUGMSG(ZONE_INIT, (L"EHCD: Installed ISR handler, Dll = '%s', Handler = '%s', Irq = %d\r\n",
                dii.szIsrDll, dii.szIsrHandler, dii.dwIrq));
            
            if (!BusTransBusAddrToStatic(pPddObject->hParentBusHandle,dwi.dwInterfaceType, dwi.dwBusNumber, PortAddress, dwAddrLen, &dwIOSpace, &(PVOID)PhysAddr)) {
                DEBUGMSG(ZONE_ERROR, (L"EHCD: Failed TransBusAddrToStatic\r\n"));
                return FALSE;
            }
        
            // Set up ISR handler
            Info.SysIntr = dii.dwSysintr;
            Info.CheckPort = TRUE;
            Info.PortIsIO = (dwIOSpace) ? TRUE : FALSE;
            Info.UseMaskReg = TRUE;
            Info.PortAddr = PhysAddr + *ioPortBase + 0x04;
            Info.PortSize = sizeof(DWORD);
            Info.MaskAddr = PhysAddr + *ioPortBase + 0x08;
            
            if (!KernelLibIoControl(pPddObject->IsrHandle, IOCTL_GIISR_INFO, &Info, sizeof(Info), NULL, 0, NULL)) {
                DEBUGMSG(ZONE_ERROR, (L"EHCD: KernelLibIoControl call failed.\r\n"));
            }
        }
    }

    DEBUGMSG(ZONE_INIT,(TEXT("EHCD: Read config from registry: Base Address: 0x%X, Length: 0x%X, I/O Port: %s, SysIntr: 0x%X, Interface Type: %u, Bus Number: %u\r\n"),
                        PhysAddr, dwAddrLen, dwIOSpace ? L"YES" : L"NO", dii.dwSysintr,dwi.dwInterfaceType, dwi.dwBusNumber));

    // Ask pre-OS software to release the control of the EHCI hardware.
    if (dwi.dwInterfaceType == PCIBus) {
        if ( DDKReg_GetPciInfo(hKey, &dpi ) == ERROR_SUCCESS) {            
            Slot.u.AsULONG = 0;
            Slot.u.bits.DeviceNumber = dpi.dwDeviceNumber;
            Slot.u.bits.FunctionNumber = dpi.dwFunctionNumber;
            Hccparam.ul = READ_REGISTER_ULONG((PULONG)(ioPortBase + 8));
            dwOffset = Hccparam.bit.EHCI_Ext_Cap_Pointer;

            if (dwOffset) {
                ASSERT(dwOffset>=0x40); // EHCI spec 2.2.4
                hBus = CreateBusAccessHandle(szDriverRegKey);
                if (hBus) {
                    while (dwOffset) {
                        dwLength = GetDeviceConfigurationData(hBus, PCI_WHICHSPACE_CONFIG, dwi.dwBusNumber, Slot.u.AsULONG, dwOffset, sizeof(Usblegsup), &Usblegsup);
                        if (dwLength == sizeof(Usblegsup)) {
                            if (Usblegsup.bit.Cap_ID == 1) {
                                Usblegsup.bit.HC_OS_Owned = 1;
                                dwLength = SetDeviceConfigurationData(hBus, PCI_WHICHSPACE_CONFIG, dwi.dwBusNumber, Slot.u.AsULONG, dwOffset, sizeof(Usblegsup), &Usblegsup);
                                break;
                            } else {
                                dwOffset = Usblegsup.bit.Next_Cap_Pointer;
                                continue;
                            }
                        } else {
                            dwOffset = 0;
                        }
                    }

                    if (dwOffset && dwLength == sizeof(Usblegsup)) {
                        // Wait up to 1 sec for pre-OS software to release the EHCI hardware.
                        dwCount = 0;
                        while (dwLength == sizeof(Usblegsup) && dwCount < 1000) {
                            dwLength = GetDeviceConfigurationData(hBus, PCI_WHICHSPACE_CONFIG, dwi.dwBusNumber, Slot.u.AsULONG, dwOffset, sizeof(Usblegsup), &Usblegsup);
                            if (Usblegsup.bit.HC_OS_Owned == 1 && Usblegsup.bit.HC_BIOS_Owned == 0) {
                                break;
                            }
                            dwCount++;
                            Sleep(1);
                        }
                    }
                }

                if (hBus) {
                    CloseBusAccessHandle(hBus);
                }
            }
        }
    }

    // The PDD can supply a buffer of contiguous physical memory here, or can let the 
    // MDD try to allocate the memory from system RAM.  We will use the HalAllocateCommonBuffer()
    // API to allocate the memory and bus controller physical addresses and pass this information
    // into the MDD.
    if (GetRegistryPhysicalMemSize(szDriverRegKey,&pPddObject->dwPhysicalMemSize)) {
        // A quarter for High priority Memory.
        dwHPPhysicalMemSize = pPddObject->dwPhysicalMemSize/4;
        // Align with page size.        
        pPddObject->dwPhysicalMemSize = (pPddObject->dwPhysicalMemSize + PAGE_SIZE -1) & ~(PAGE_SIZE -1);
        dwHPPhysicalMemSize = ((dwHPPhysicalMemSize +  PAGE_SIZE -1) & ~(PAGE_SIZE -1));
    }
    else 
        pPddObject->dwPhysicalMemSize=0;
    
    if (pPddObject->dwPhysicalMemSize<gcTotalAvailablePhysicalMemory) { // Setup Minimun requirement.
        pPddObject->dwPhysicalMemSize = gcTotalAvailablePhysicalMemory;
        dwHPPhysicalMemSize = gcHighPriorityPhysicalMemory;
    }

    pPddObject->AdapterObject.ObjectSize = sizeof(DMA_ADAPTER_OBJECT);
    pPddObject->AdapterObject.InterfaceType = dwi.dwInterfaceType;
    pPddObject->AdapterObject.BusNumber = dwi.dwBusNumber;
    if ((pPddObject->pvVirtualAddress = HalAllocateCommonBuffer(&pPddObject->AdapterObject, pPddObject->dwPhysicalMemSize, &pPddObject->LogicalAddress, FALSE)) == NULL) {
        goto InitializeEHCI_Error;
    }
    if (!(pobMem = HcdMdd_CreateMemoryObject(pPddObject->dwPhysicalMemSize, dwHPPhysicalMemSize, (PUCHAR) pPddObject->pvVirtualAddress, (PUCHAR) pPddObject->LogicalAddress.LowPart))) {
        goto InitializeEHCI_Error;
    }

    if (!KernelIoControl(IOCTL_HAL_REQUEST_SYSINTR, &dii.dwIrq, sizeof(UINT32), &dii.dwSysintr, sizeof(UINT32), NULL))
    {
        RETAILMSG(1,(TEXT("[EHCI:ERR] ++%s(): Failed to request sysintr value for EHCI interrupt.\r\n"), _T(__FUNCTION__)));
        goto InitializeEHCI_Error;
    }
    RETAILMSG(1,(TEXT("[EHCI:INF] ++%s(): IRQ %d sysintr %d for EHCI interrupt.\r\n"), _T(__FUNCTION__),dii.dwIrq, dii.dwSysintr));

    if (!(pobEHCD = HcdMdd_CreateHcdObject(pPddObject, pobMem, szDriverRegKey, ioPortBase, dii.dwSysintr))) {
        goto InitializeEHCI_Error;
    }

    pPddObject->lpvMemoryObject = pobMem;
    pPddObject->lpvEHCDMddObject = pobEHCD;
    _tcsncpy(pPddObject->szDriverRegKey, szDriverRegKey, MAX_PATH);
    pPddObject->ioPortBase = ioPortBase;
    pPddObject->dwSysIntr =  dii.dwSysintr;

   
    if ( hKey!=NULL)  {
        DWORD dwCapability;
        DWORD dwType;
        DWORD dwLength = sizeof(DWORD);
        if (RegQueryValueEx(hKey, HCD_CAPABILITY_VALNAME, 0, &dwType, (PUCHAR)&dwCapability, &dwLength) == ERROR_SUCCESS)
            HcdMdd_SetCapability (pobEHCD,dwCapability);
        RegCloseKey(hKey);
    }

    return TRUE;

InitializeEHCI_Error:

    if ( g_pCMUCLKregs )
        DrvLib_UnmapIoSpace((PVOID)g_pCMUCLKregs);

    if ( g_pPMUMISCregs )
        DrvLib_UnmapIoSpace((PVOID)g_pPMUMISCregs);

    if ( g_pOTGPHYregs )
        DrvLib_UnmapIoSpace((PVOID)g_pOTGPHYregs);

    
    if (pPddObject->IsrHandle) {
        FreeIntChainHandler(pPddObject->IsrHandle);
        pPddObject->IsrHandle = NULL;
    }
    
    if (pobEHCD)
        HcdMdd_DestroyHcdObject(pobEHCD);
    if (pobMem)
        HcdMdd_DestroyMemoryObject(pobMem);
    if(pPddObject->pvVirtualAddress)
        HalFreeCommonBuffer(&pPddObject->AdapterObject, pPddObject->dwPhysicalMemSize, pPddObject->LogicalAddress, pPddObject->pvVirtualAddress, FALSE);

    pPddObject->lpvMemoryObject = NULL;
    pPddObject->lpvEHCDMddObject = NULL;
    pPddObject->pvVirtualAddress = NULL;
    if ( hKey!=NULL) 
        RegCloseKey(hKey);

    return FALSE;
}

/* HcdPdd_Init
 *
 *   PDD Entry point - called at system init to detect and configure EHCI card.
 *
 * Return Value:
 *   Return pointer to PDD specific data structure, or NULL if error.
 */
extern DWORD 
HcdPdd_Init(
    DWORD dwContext)  // IN - Pointer to context value. For device.exe, this is a string 
                      //      indicating our active registry key.
{
    SEHCDPdd *  pPddObject = malloc(sizeof(SEHCDPdd));
    BOOL        fRet = FALSE;

	//RETAILMSG(1,(_T("++%s\r\n"),__FUNCTION__));

    if (pPddObject) {
        pPddObject->pvVirtualAddress = NULL;
        InitializeCriticalSection(&pPddObject->csPdd);
        pPddObject->IsrHandle = NULL;
        pPddObject->hParentBusHandle = CreateBusAccessHandle((LPCWSTR)g_dwContext); 
        
        fRet = InitializeEHCI(pPddObject, (LPCWSTR)dwContext);

        if(!fRet)
        {
            if (pPddObject->hParentBusHandle)
                CloseBusAccessHandle(pPddObject->hParentBusHandle);
            
            DeleteCriticalSection(&pPddObject->csPdd);
            free(pPddObject);
            pPddObject = NULL;
        }
    }

    return (DWORD)pPddObject;
}

/* HcdPdd_CheckConfigPower
 *
 *    Check power required by specific device configuration and return whether it
 *    can be supported on this platform.  For CEPC, this is trivial, just limit to
 *    the 500mA requirement of USB.  For battery powered devices, this could be 
 *    more sophisticated, taking into account current battery status or other info.
 *
 * Return Value:
 *    Return TRUE if configuration can be supported, FALSE if not.
 */
extern BOOL HcdPdd_CheckConfigPower(
    UCHAR bPort,         // IN - Port number
    DWORD dwCfgPower,    // IN - Power required by configuration
    DWORD dwTotalPower)  // IN - Total power currently in use on port
{
	RETAILMSG(1,(_T("++HcdPdd_CheckConfigPower\r\n")));
    return ((dwCfgPower + dwTotalPower) > 500) ? FALSE : TRUE;
}

extern void HcdPdd_PowerUp(DWORD hDeviceContext)
{
    SEHCDPdd * pPddObject = (SEHCDPdd *)hDeviceContext;

	RETAILMSG(1,(_T("++HcdPdd_PowerUp\r\n")));

    SetUSBHostCLKPWR(TRUE);
    g_pUSBEHCIHostregs->INSNREG00 = 0xe0000; // enable burst trans
    g_pUSBEHCIHostregs->INSNREG01 = 0x300010; // set threshold as 512 byte.
    
    HcdMdd_PowerUp(pPddObject->lpvEHCDMddObject);

    return;
}

extern void HcdPdd_PowerDown(DWORD hDeviceContext)
{
    SEHCDPdd * pPddObject = (SEHCDPdd *)hDeviceContext;

	RETAILMSG(1,(_T("++HcdPdd_PowerDown\r\n")));

    HcdMdd_PowerDown(pPddObject->lpvEHCDMddObject);

    return;
}


extern BOOL HcdPdd_Deinit(DWORD hDeviceContext)
{
    SEHCDPdd * pPddObject = (SEHCDPdd *)hDeviceContext;

	//RETAILMSG(1,(_T("++%s\r\n"),__FUNCTION__));

    if(pPddObject->lpvEHCDMddObject)
        HcdMdd_DestroyHcdObject(pPddObject->lpvEHCDMddObject);
    if(pPddObject->lpvMemoryObject)
        HcdMdd_DestroyMemoryObject(pPddObject->lpvMemoryObject);
    if(pPddObject->pvVirtualAddress)
        HalFreeCommonBuffer(&pPddObject->AdapterObject, pPddObject->dwPhysicalMemSize, pPddObject->LogicalAddress, pPddObject->pvVirtualAddress, FALSE);

    if (pPddObject->IsrHandle) {
        FreeIntChainHandler(pPddObject->IsrHandle);
        pPddObject->IsrHandle = NULL;
    }
    
    if (pPddObject->hParentBusHandle)
        CloseBusAccessHandle(pPddObject->hParentBusHandle);

    if (g_pCMUCLKregs && g_pPMUMISCregs)
        SetUSBHostCLKPWR(FALSE);

    if ( g_pCMUCLKregs )
        DrvLib_UnmapIoSpace((PVOID)g_pCMUCLKregs);

    if ( g_pPMUMISCregs )
        DrvLib_UnmapIoSpace((PVOID)g_pPMUMISCregs);

    if ( g_pOTGPHYregs )
        DrvLib_UnmapIoSpace((PVOID)g_pOTGPHYregs);    
        

    
    free(pPddObject);
    return TRUE;
    
}


extern DWORD HcdPdd_Open(DWORD hDeviceContext, DWORD AccessCode,
        DWORD ShareMode)
{
	//RETAILMSG(1,(_T("++%s\r\n"),__FUNCTION__));
    UnusedParameter(hDeviceContext);
    UnusedParameter(AccessCode);
    UnusedParameter(ShareMode);

    return 1; // we can be opened, but only once!
}


extern BOOL HcdPdd_Close(DWORD hOpenContext)
{
	RETAILMSG(1,(_T("++HcdPdd_Close\r\n")));
    UnusedParameter(hOpenContext);

    return TRUE;
}


extern DWORD HcdPdd_Read(DWORD hOpenContext, LPVOID pBuffer, DWORD Count)
{
    UnusedParameter(hOpenContext);
    UnusedParameter(pBuffer);
    UnusedParameter(Count);

	RETAILMSG(1,(_T("++HcdPdd_Read\r\n")));

    return (DWORD)-1; // an error occured
}


extern DWORD HcdPdd_Write(DWORD hOpenContext, LPCVOID pSourceBytes,
        DWORD NumberOfBytes)
{
    UnusedParameter(hOpenContext);
    UnusedParameter(pSourceBytes);
    UnusedParameter(NumberOfBytes);

	RETAILMSG(1,(_T("++HcdPdd_Write\r\n")));

    return (DWORD)-1;
}


extern DWORD HcdPdd_Seek(DWORD hOpenContext, LONG Amount, DWORD Type)
{
	RETAILMSG(1,(_T("++HcdPdd_Seek\r\n")));
    UnusedParameter(hOpenContext);
    UnusedParameter(Amount);
    UnusedParameter(Type);

    return (DWORD)-1;
}


extern BOOL HcdPdd_IOControl(DWORD hOpenContext, DWORD dwCode, PBYTE pBufIn,
        DWORD dwLenIn, PBYTE pBufOut, DWORD dwLenOut, PDWORD pdwActualOut)
{
	RETAILMSG(1,(_T("++HcdPdd_IOControl\r\n")));
    UnusedParameter(hOpenContext);
    UnusedParameter(dwCode);
    UnusedParameter(pBufIn);
    UnusedParameter(dwLenIn);
    UnusedParameter(pBufOut);
    UnusedParameter(dwLenOut);
    UnusedParameter(pdwActualOut);

    return FALSE;
}


// Manage WinCE suspend/resume events

// This gets called by the MDD's IST when it detects a power resume.
// By default it has nothing to do.
extern void HcdPdd_InitiatePowerUp (DWORD hDeviceContext)
{
    return;
}

void SetUSBHostCLKPWR(UINT32 bFlag)
{
	RETAILMSG(1,(_T("++SetUSBHostCLKPWR\r\n")));
return ;
    if (bFlag)
    {
        if (g_pCMUCLKregs)
            g_pCMUCLKregs->CLK_GATE.CLK_GATE_IP1 |= 0x3<<BP_CLK_IP_USBOTG;    //OTG CLK enable 
            //g_pCMUCLKregs->CLK_GATE.CLK_GATE_IP1 |= CLK_ON<<BP_CLK_IP_USBHOST;    //OTG CLK enable 
        EnableUSBPHYClock(TRUE);
    }
    else
    {
        if (g_pCMUCLKregs)
            g_pCMUCLKregs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON<<BP_CLK_IP_USBHOST);    //OTG CLK enable 
        EnableUSBPHYClock(FALSE);
    }
        
    

}


void EnableUSBPHYClock( UINT32 bFlag )
{
    volatile DWORD dwRegValue;

    DWORD dwOSCCON;

	RETAILMSG(1,(_T("++EnableUSBPHYClock\r\n")));
    
    if ( g_pPMUOSCregs )
    {
        dwOSCCON = g_pPMUOSCregs->OSC_CON;
        dwOSCCON &= ~(1<<1); // OSC for USB enable
        g_pPMUOSCregs->OSC_CON |= (dwOSCCON|(1<<1));        
    }
    

    if ( g_pPMUMISCregs )
    {
        dwRegValue = g_pPMUMISCregs->IP_CON_REG.USB_PHY_CONTROL;
        dwRegValue |= ((dwRegValue & ~(0x3<<0)) | 0x3);
        g_pPMUMISCregs->IP_CON_REG.USB_PHY_CONTROL = dwRegValue;

        if ( bFlag )
        {
            if (g_pOTGPHYregs)
            {
                //dwRegValue = g_pOTGPHYregs->OPHYPWR;
                //g_pOTGPHYregs->OPHYPWR = (dwRegValue|0x100);
                g_pOTGPHYregs->OPHYPWR = 0x130;
                g_pOTGPHYregs->OPHYCLK = 0x03;
                //g_pOTGPHYregs->ORSTCON = (0x1)<<3;
                g_pOTGPHYregs->ORSTCON = 0x9;
                DrvLib_WaitUsec(40);

                g_pOTGPHYregs->ORSTCON = 0x0;
                DrvLib_WaitUsec(40);

            }
        }
    }
}
