//
// Copyright  2009 Samsung Electronics Co; Ltd. All Rights Reserved.
//
//
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    sdout.c

Abstract:       Implementation of sdout Library
                This module implements Low Level HW control

Functions:


Notes:


--*/

#include <windows.h>
#include <bsp_cfg.h>    // for reference HCLK, ECLK
#include <register_map.h>
#include "tvout_global.h"
#include "sdo_reg.h"
#include "tvout_message.h"
#include "sdout.h"
#include "sdout_internal.h"


// initialization
//  - iniization functions are only called under stopping sdout
SDOUT_ERROR
Sdout_initialize_register_address
(void *pSdoutReg)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_register_address(0x%08x)\n\r"), pSdoutReg));

	if (pSdoutReg == NULL)
	{
		DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_register_address() : NULL pointer parameter\n\r")));
		return SDOUT_ERROR_INVALID_PARAM;
	}
	else
	{
		g_pSdoutConfig = (PSDOUT_REG)pSdoutReg;
		DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_register_address(0x%08x)\n\r"), g_pSdoutConfig));
	}

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_display_mode
(TVOUT_DISPLAY_MODE dispMode,
TVOUT_OUTPUT_MODE outMode)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_display_mode(%d,%d)\n\r"),dispMode,outMode));

	switch(dispMode)
	{
		case TVOUT_NTSC_M:
			uiTempReg |= SDO_NTSC_M;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_75IRE, SDOUT_VTOS_RATIO_10_4);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_75IRE,
										SDOUT_VTOS_RATIO_10_4,
										outMode);
			break;
		case TVOUT_NTSC_J:
			uiTempReg |= SDO_NTSC_M;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_10_4);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_0IRE,
										SDOUT_VTOS_RATIO_10_4,
										outMode);
			break;
		case TVOUT_PAL_BDGHI:
			uiTempReg |= SDO_PAL_BGHID;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_0IRE,
										SDOUT_VTOS_RATIO_7_3,
										outMode);
			break;
		case TVOUT_PAL_M:
			uiTempReg |= SDO_PAL_M;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_0IRE,
										SDOUT_VTOS_RATIO_7_3,
									outMode);
			break;
		case TVOUT_PAL_N:
			uiTempReg |= SDO_PAL_N;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_75IRE,
										SDOUT_VTOS_RATIO_10_4,
										outMode);
			break;
		case TVOUT_PAL_NC:
			uiTempReg |= SDO_PAL_NC;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_0IRE,
										SDOUT_VTOS_RATIO_7_3,
										outMode);
			break;
		case TVOUT_PAL_60:
			uiTempReg |= SDO_PAL_60;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_0IRE,
										SDOUT_VTOS_RATIO_7_3,
										outMode);
			break;
		case TVOUT_NTSC_443:
			uiTempReg |= SDO_NTSC_443;
			Sdout_initialize_video_scale_cfg(SDOUT_LEVEL_0IRE, SDOUT_VTOS_RATIO_7_3,
										SDOUT_LEVEL_75IRE, SDOUT_VTOS_RATIO_10_4);
			Sdout_initialize_antialias_filter_coeff_default(SDOUT_LEVEL_75IRE,
										SDOUT_VTOS_RATIO_10_4,
										outMode);
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_display_mode() : invalid dispMode parameter(%d)\n\r"),dispMode));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(outMode)
	{
		case TVOUT_OUTPUT_COMPOSITE:
			uiTempReg |= SDO_COMPOSITE | SDO_INTERLACED;

//			case SDOUT_OUTPUT_ORDER_COMPOSITE_Y_C_CVBS:
					uiTempReg |= SDO_DAC2_Y | SDO_DAC1_C | SDO_DAC0_CVBS;

					break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_display_mode() : invalid outMode parameter(%d)\n\r"),outMode));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	Sdout_initialize_oversampling_filter_coeff_default(outMode);

	g_pSdoutConfig->SDO_CONFIG = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_display_mode(0x%08x)\n\r"), g_pSdoutConfig->SDO_CONFIG));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR Sdout_initialize_video_scale_cfg
(SDOUT_LEVEL componentLevel,
SDOUT_VIDEO_SYNC_RATIO componentRatio,
SDOUT_LEVEL compositeLevel,
SDOUT_VIDEO_SYNC_RATIO compositeRatio)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_video_scale_cfg(%d,%d,%d,%d)\n\r"),componentLevel,componentRatio,compositeLevel,compositeRatio));

	switch(componentLevel)
	{
		case SDOUT_LEVEL_0IRE:
			uiTempReg = SDO_COMPONENT_LEVEL_SEL_0IRE;
			break;
		case SDOUT_LEVEL_75IRE:
			uiTempReg = SDO_COMPONENT_LEVEL_SEL_75IRE;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_video_scale_cfg() : invalid componentLevel parameter(%d)\n\r"),componentLevel));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(compositeLevel)
	{
		case SDOUT_VTOS_RATIO_10_4:
			uiTempReg |= SDO_COMPONENT_VTOS_RATIO_10_4;
			break;
		case SDOUT_VTOS_RATIO_7_3:
			uiTempReg |= SDO_COMPONENT_VTOS_RATIO_7_3;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_video_scale_cfg() : invalid compositeLevel parameter(%d)\n\r"),compositeLevel));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(compositeLevel)
	{
		case SDOUT_LEVEL_0IRE:
			uiTempReg |= SDO_COMPOSITE_LEVEL_SEL_0IRE;
			break;
		case SDOUT_LEVEL_75IRE:
			uiTempReg |= SDO_COMPOSITE_LEVEL_SEL_75IRE;
			break;
//		default:
//			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_video_scale_cfg() : invalid compositeRatio parameter(%d)\n\r"),compositeRatio));
//			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(compositeRatio)
	{
		case SDOUT_VTOS_RATIO_10_4:
			uiTempReg |= SDO_COMPOSITE_VTOS_RATIO_10_4;
			break;
		case SDOUT_VTOS_RATIO_7_3:
			uiTempReg |= SDO_COMPOSITE_VTOS_RATIO_7_3;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_video_scale_cfg() : invalid componentRatio parameter(%d)\n\r"),componentRatio));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_SCALE = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_video_scale_cfg(0x%08x)\n\r"), g_pSdoutConfig->SDO_SCALE));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_sync_signal_pin
(SDOUT_SYNC_SIG_PIN pin)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_sync_signal_pin(%d)\n\r"),pin));

	switch(pin)
	{
		case SDOUT_SYNC_SIG_NO:
			g_pSdoutConfig->SDO_SYNC = SDO_COMPONENT_SYNC_ABSENT;
			break;
		case SDOUT_SYNC_SIG_YG:
			g_pSdoutConfig->SDO_SYNC = SDO_COMPONENT_SYNC_YG;
			break;
		case SDOUT_SYNC_SIG_ALL:
			g_pSdoutConfig->SDO_SYNC = SDO_COMPONENT_SYNC_ALL;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_sync_signal_pin() : invalid pin parameter(%d)\n\r"),pin));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_sync_signal_pin(0x%08x)\n\r"), g_pSdoutConfig->SDO_SYNC));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_vbi
(BOOL wssCVBS,
SDOUT_CLOSED_CAPTION_INS captionCVBS,
BOOL wssYSideo,
SDOUT_CLOSED_CAPTION_INS captionYSideo,
BOOL cgmsaRGB,
BOOL wssRGB,
SDOUT_CLOSED_CAPTION_INS captionRGB,
BOOL cgmsaYPbPr,
BOOL wssYPbPr,
SDOUT_CLOSED_CAPTION_INS captionYPbPr)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_vbi(%d,%d,%d,%d,%d,%d,%d,%d,%d,%d)\n\r"),
	wssCVBS,captionCVBS,wssYSideo,captionYSideo,cgmsaRGB,wssRGB,captionRGB,cgmsaYPbPr,wssYPbPr,captionYPbPr));

	if(wssCVBS)
	{
		uiTempReg = SDO_CVBS_WSS_INS;
	}
	else
	{
		uiTempReg = SDO_CVBS_NO_WSS;
	}

	switch(captionCVBS)
	{
		case SDOUT_NO_INS:
			uiTempReg |= SDO_CVBS_NO_CLOSED_CAPTION;
			break;
		case SDOUT_INS_1:
			uiTempReg |= SDO_CVBS_21H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_2:
			uiTempReg |= SDO_CVBS_21H_284H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_OTHERS:
			uiTempReg |= SDO_CVBS_USE_OTHERS;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vbi() : invalid captionCVBS parameter(%d)\n\r"),captionCVBS));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(wssYSideo)
	{
		uiTempReg |= SDO_SVIDEO_WSS_INS;
	}
	else
	{
		uiTempReg |= SDO_SVIDEO_NO_WSS;
	}

	switch(captionYSideo)
	{
		case SDOUT_NO_INS:
			uiTempReg |= SDO_SVIDEO_NO_CLOSED_CAPTION;
			break;
		case SDOUT_INS_1:
			uiTempReg |= SDO_SVIDEO_21H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_2:
			uiTempReg |= SDO_SVIDEO_21H_284H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_OTHERS:
			uiTempReg |= SDO_SVIDEO_USE_OTHERS;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vbi() : invalid captionYSideo parameter(%d)\n\r"),captionYSideo));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(cgmsaRGB)
	{
		uiTempReg |= SDO_RGB_CGMSA_INS;
	}
	else
	{
		uiTempReg |= SDO_RGB_NO_CGMSA;
	}

	if(wssRGB)
	{
		uiTempReg |= SDO_RGB_WSS_INS;
	}
	else
	{
		uiTempReg |= SDO_RGB_NO_WSS;
	}

	switch(captionRGB)
	{
		case SDOUT_NO_INS:
			uiTempReg |= SDO_RGB_NO_CLOSED_CAPTION;
			break;
		case SDOUT_INS_1:
			uiTempReg |= SDO_RGB_21H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_2:
			uiTempReg |= SDO_RGB_21H_284H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_OTHERS:
			uiTempReg |= SDO_RGB_USE_OTHERS;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vbi() : invalid captionRGB parameter(%d)\n\r"),captionRGB));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(cgmsaYPbPr)
	{
		uiTempReg |= SDO_YPBPR_CGMSA_INS;
	}
	else
	{
		uiTempReg |= SDO_YPBPR_NO_CGMSA;
	}

	if(wssYPbPr)
	{
		uiTempReg |= SDO_YPBPR_WSS_INS;
	}
	else
	{
		uiTempReg |= SDO_YPBPR_NO_WSS;
	}

	switch(captionYPbPr)
	{
		case SDOUT_NO_INS:
			uiTempReg |= SDO_YPBPR_NO_CLOSED_CAPTION;
			break;
		case SDOUT_INS_1:
			uiTempReg |= SDO_YPBPR_21H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_2:
			uiTempReg |= SDO_YPBPR_21H_284H_CLOSED_CAPTION;
			break;
		case SDOUT_INS_OTHERS:
			uiTempReg |= SDO_YPBPR_USE_OTHERS;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vbi() : invalid captionYPbPr parameter(%d)\n\r"),captionYPbPr));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_VBI = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_vbi(0x%08x)\n\r"), g_pSdoutConfig->SDO_VBI));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_offset_gain
(SDOUT_CHANNEL_SEL channel,
unsigned int uiOffset,
unsigned int uiGain)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_offset_gain(%d,%d,%d)\n\r"),channel,uiOffset,uiGain));

	switch(channel)
	{
		case SDOUT_CHANNEL_0:
			g_pSdoutConfig->SDO_SCALE_CH0 = SDO_SCALE_CONV_OFFSET(uiOffset) |
			SDO_SCALE_CONV_GAIN(uiGain);
			DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_offset_gain(0x%08x)\n\r"), g_pSdoutConfig->SDO_SCALE_CH0));
			break;
		case SDOUT_CHANNEL_1:
			g_pSdoutConfig->SDO_SCALE_CH1 = SDO_SCALE_CONV_OFFSET(uiOffset) |
			SDO_SCALE_CONV_GAIN(uiGain);
			DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_offset_gain(0x%08x)\n\r"), g_pSdoutConfig->SDO_SCALE_CH1));
			break;
		case SDOUT_CHANNEL_2:
			g_pSdoutConfig->SDO_SCALE_CH2 = SDO_SCALE_CONV_OFFSET(uiOffset) |
			SDO_SCALE_CONV_GAIN(uiGain);
			DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_offset_gain(0x%08x)\n\r"), g_pSdoutConfig->SDO_SCALE_CH2));
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_offset_gain() : invalid channel parameter(%d)\n\r"),channel));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	return SDOUT_NO_ERROR;
}

void
Sdout_initialize_delay
(unsigned int uiDelayY,
unsigned int uiOffsetVideoStart,
unsigned int uiOffsetVideoEnd)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_delay(%d,%d,%d)\n\r"),uiDelayY,uiOffsetVideoStart,uiOffsetVideoEnd));

	g_pSdoutConfig->SDO_YCDELAY = SDO_DELAY_YTOC(uiDelayY) |
	SDO_ACTIVE_START_OFFSET(uiOffsetVideoStart) |
	SDO_ACTIVE_END_OFFSET(uiOffsetVideoEnd);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_delay(0x%08x)\n\r"), g_pSdoutConfig->SDO_YCDELAY));
}

void
Sdout_initialize_schlock
(BOOL bColorSubCarrierPhaseAdjustment)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_schlock(%d)\n\r"),bColorSubCarrierPhaseAdjustment));

	if(bColorSubCarrierPhaseAdjustment)
	{
		g_pSdoutConfig->SDO_SCHLOCK = SDO_COLOR_SC_PHASE_ADJ;
	}
	else
	{
		g_pSdoutConfig->SDO_SCHLOCK = SDO_COLOR_SC_PHASE_NOADJ;
	}

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_schlock(0x%08x)\n\r"), g_pSdoutConfig->SDO_SCHLOCK));
}

SDOUT_ERROR
Sdout_initialize_dac_power_onoff
(SDOUT_CHANNEL_SEL channel,
BOOL bDacOn)
{
	unsigned int uiTempOnOff;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_dac_power_onoff(%d,%d)\n\r"),channel,bDacOn));

	switch(channel)
	{
		case SDOUT_CHANNEL_0:
			uiTempOnOff = SDO_POWER_ON_DAC0;
			break;
		case SDOUT_CHANNEL_1:
			uiTempOnOff = SDO_POWER_ON_DAC1;
			break;
		case SDOUT_CHANNEL_2:
			uiTempOnOff = SDO_POWER_ON_DAC2;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_dac_power_onoff() : invalid channel parameter(%d)\n\r"),channel));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(bDacOn)
	{
		g_pSdoutConfig->SDO_DAC |= uiTempOnOff;
	}
	else
	{
		g_pSdoutConfig->SDO_DAC &= ~uiTempOnOff;
	}

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_dac_power_onoff(0x%08x)\n\r"), g_pSdoutConfig->SDO_DAC));

	return SDOUT_NO_ERROR;
}
/*
SDOUT_ERROR
Sdout_initialize_antialias_filter_coeff
(SDOUT_ANTIALIAS_FILTER_COEFFICIENTS coeff)
{
}
*/

void
Sdout_initialize_color_compensaton_onoff
(BOOL bBrightHueSaturationAdj,
BOOL bYPbPrColorCompensation,
BOOL bRGBcolorCompensation,
BOOL bYCcolorCompensaton,
BOOL bCVBSCcolorCompensaton)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_color_compensaton_onoff(%d,%d,%d,%d,%d)\n\r"),
	bBrightHueSaturationAdj,bYPbPrColorCompensation,bRGBcolorCompensation,bYCcolorCompensaton,bCVBSCcolorCompensaton));

	if(bBrightHueSaturationAdj)
	{
		uiTempReg &= ~SDO_COMPONENT_BHS_ADJ_OFF;
	}
	else
	{
		uiTempReg |= SDO_COMPONENT_BHS_ADJ_OFF;
	}

	if(bYPbPrColorCompensation)
	{
		uiTempReg &= ~SDO_COMPONENT_YPBPR_COMP_OFF;
	}
	else
	{
		uiTempReg |= SDO_COMPONENT_YPBPR_COMP_OFF;
	}
	if(bRGBcolorCompensation)
	{
		uiTempReg &= ~SDO_COMPONENT_RGB_COMP_OFF;
	}
	else
	{
		uiTempReg |= SDO_COMPONENT_RGB_COMP_OFF;
	}
	if(bYCcolorCompensaton)
	{
		uiTempReg &= ~SDO_COMPONENT_YC_COMP_OFF;
	}
	else
	{
		uiTempReg |= SDO_COMPONENT_YC_COMP_OFF;
	}
	if(bCVBSCcolorCompensaton)
	{
		uiTempReg &= ~SDO_COMPONENT_CVBS_COMP_OFF;
	}
	else
	{
		uiTempReg |= SDO_COMPONENT_CVBS_COMP_OFF;
	}

	g_pSdoutConfig->SDO_CCCON = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_color_compensaton_onoff(0x%08x)\n\r"), g_pSdoutConfig->SDO_CCCON));
}

void
Sdout_initialize_brightness_hue_saturation
(unsigned int uiGainBrightness,
unsigned int uiOffsetBrightness,
unsigned int uiGain0CbHueSaturation,
unsigned int uiGain1CbHueSaturation,
unsigned int uiGain0CrHueSaturation,
unsigned int uiGain1CrHueSaturation,
unsigned int uiOffsetCbHueSaturation,
unsigned int uiOffsetCrHueSaturation)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_brightness_hue_saturation(%d,%d,%d,%d,%d,%d,%d,%d)\n\r"),
	uiGainBrightness,uiOffsetBrightness,uiGain0CbHueSaturation,uiGain1CbHueSaturation,uiGain0CrHueSaturation,
	uiGain1CrHueSaturation,uiOffsetCbHueSaturation,uiOffsetCrHueSaturation));

	g_pSdoutConfig->SDO_YSCALE = SDO_BRIGHTNESS_GAIN(uiGainBrightness) |
							SDO_BRIGHTNESS_OFFSET(uiOffsetBrightness);
	g_pSdoutConfig->SDO_CBSCALE = SDO_HS_CB_GAIN0(uiGain0CbHueSaturation) |
							SDO_HS_CB_GAIN1(uiGain1CbHueSaturation);
	g_pSdoutConfig->SDO_CRSCALE = SDO_HS_CR_GAIN0(uiGain0CrHueSaturation) |
							SDO_HS_CR_GAIN1(uiGain1CrHueSaturation);
	g_pSdoutConfig->SDO_CB_CR_OFFSET = SDO_HS_CR_OFFSET(uiOffsetCrHueSaturation) |
							SDO_HS_CB_OFFSET(uiOffsetCbHueSaturation);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_brightness_hue_saturation(0x%08x,0x%08x,0x%08x,0x%08x)\n\r"),
							g_pSdoutConfig->SDO_YSCALE,g_pSdoutConfig->SDO_CBSCALE,g_pSdoutConfig->SDO_CRSCALE,g_pSdoutConfig->SDO_CB_CR_OFFSET));
}

void
Sdout_initialize_rgb_color_compensation
(unsigned int uiMaxRGBCube,
unsigned int uiMinRGBCube)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_rgb_color_compensaton(0x%08x,0x%08x)\n\r"),uiMaxRGBCube,uiMinRGBCube));

	g_pSdoutConfig->SDO_RGB_CC = SDO_MAX_RGB_CUBE(uiMaxRGBCube) |
								SDO_MIN_RGB_CUBE(uiMinRGBCube);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_rgb_color_compensaton(0x%08x)\n\r"), g_pSdoutConfig->SDO_RGB_CC));
}

void
Sdout_initialize_cvbs_color_compensation
(unsigned int uiYLowerMid,
unsigned int uiYBottom,
unsigned int uiYTop,
unsigned int uiYUpperMid,
unsigned int uiRadius)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_cvbs_color_compensaton(%d,%d,%d,%d,%d)\n\r"),
							uiYLowerMid,uiYBottom,uiYTop,uiYUpperMid,uiRadius));

	g_pSdoutConfig->SDO_CVBS_CC_Y1 = SDO_Y_LOWER_MID_CVBS_CORN(uiYLowerMid) |
								SDO_Y_BOTTOM_CVBS_CORN(uiYBottom);
	g_pSdoutConfig->SDO_CVBS_CC_Y2 = SDO_Y_TOP_CVBS_CORN(uiYTop) |
								SDO_Y_UPPER_MID_CVBS_CORN(uiYUpperMid);
	g_pSdoutConfig->SDO_CVBS_CC_C = SDO_RADIUS_CVBS_CORN(uiRadius);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_cvbs_color_compensaton(0x%08x,0x%08x,0x%08x)\n\r"),
							g_pSdoutConfig->SDO_CVBS_CC_Y1,g_pSdoutConfig->SDO_CVBS_CC_Y2,g_pSdoutConfig->SDO_CVBS_CC_C));
}

void
Sdout_initialize_svideo_color_compensation
(unsigned int uiYTop,
unsigned int uiYBottom,
unsigned int uiYCCylinder)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_svideo_color_compensaton(%d,%d,%d)\n\r"),
	uiYTop,uiYBottom,uiYCCylinder));

	g_pSdoutConfig->SDO_YC_CC_Y = SDO_Y_TOP_YC_CYLINDER(uiYTop) |
								SDO_Y_BOTOM_YC_CYLINDER(uiYBottom);
	g_pSdoutConfig->SDO_YC_CC_C = SDO_RADIUS_YC_CYLINDER(uiYCCylinder);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_svideo_color_compensaton(0x%08x,0x%08x)\n\r"),
							g_pSdoutConfig->SDO_YC_CC_Y,g_pSdoutConfig->SDO_YC_CC_C));
}

void
Sdout_initialize_component_porch
(unsigned int ui525Back,
unsigned int ui525Front,
unsigned int ui625Back,
unsigned int ui625Front)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_component_porch(%d,%d,%d,%d)\n\r"),
							ui525Back,ui525Front,ui625Back,ui625Front));

	g_pSdoutConfig->SDO_CSC_525_PORCH = SDO_COMPONENT_525_BP(ui525Back) |
									SDO_COMPONENT_525_FP(ui525Front);
	g_pSdoutConfig->SDO_CSC_625_PORCH = SDO_COMPONENT_625_BP(ui625Back) |
									SDO_COMPONENT_625_FP(ui625Front);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_component_porch(0x%08x,0x%08x)\n\r"),
							g_pSdoutConfig->SDO_CSC_525_PORCH,g_pSdoutConfig->SDO_CSC_625_PORCH));
}

SDOUT_ERROR
Sdout_initialize_vesa_rgb_sync
(SDOUT_VESA_RGB_SYNC_TYPE syncType,
TVOUT_ACTIVE_POLARITY vsyncActive,
TVOUT_ACTIVE_POLARITY hsyncActive)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_vesa_rgb_sync(%d,%d,%d)\n\r"),syncType,vsyncActive,hsyncActive));

	switch(syncType)
	{
		case SDOUT_VESA_RGB_SYNC_COMPOSITE:
			uiTempReg |= SDO_RGB_SYNC_COMPOSITE;
			break;
		case SDOUT_VESA_RGB_SYNC_SEPARATE:
			uiTempReg |= SDO_RGB_SYNC_SEPERATE;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vesa_rgb_sync() : invalid syncType parameter(%d)\n\r"),syncType));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(vsyncActive)
	{
		case TVOUT_POL_ACTIVE_LOW:
			uiTempReg |= SDO_RGB_VSYNC_LOW_ACT;
			break;
		case TVOUT_POL_ACTIVE_HIGH:
			uiTempReg |= SDO_RGB_VSYNC_HIGH_ACT;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vesa_rgb_sync() : invalid vsyncActive parameter(%d)\n\r"),vsyncActive));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(hsyncActive)
	{
		case TVOUT_POL_ACTIVE_LOW:
			uiTempReg |= SDO_RGB_HSYNC_LOW_ACT;
			break;
		case TVOUT_POL_ACTIVE_HIGH:
			uiTempReg |= SDO_RGB_HSYNC_HIGH_ACT;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_vesa_rgb_sync() : invalid hsyncActive parameter(%d)\n\r"),hsyncActive));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_RGBSYNC = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_vesa_rgb_sync(0x%08x)\n\r"), g_pSdoutConfig->SDO_RGBSYNC));

	return SDOUT_NO_ERROR;
}

void
Sdout_initialize_oversampling_filter_coeff
(unsigned int uiSize,
unsigned int* puiCoeff0,
unsigned int* puiCoeff1,
unsigned int* puiCoeff2)
{
	unsigned int* puiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_oversampling_filter_coeff(%d,0x%x,0x%x,0x%x)\n\r"),uiSize,puiCoeff0,puiCoeff1,puiCoeff2));

	if(puiCoeff0 != NULL)
	{
		puiTempReg = (unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_0);
		memcpy((void *)puiTempReg, (const void *)puiCoeff0, uiSize*4);
	}
	if(puiCoeff1 != NULL)
	{
		puiTempReg = (unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_1);
		memcpy((void *)puiTempReg, (const void *)puiCoeff1, uiSize*4);
	}
	if(puiCoeff2 != NULL)
	{
		puiTempReg = (unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_2);
		memcpy((void *)puiTempReg, (const void *)puiCoeff2, uiSize*4);
	}

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_oversampling_filter_coeff()\n\r")));
}

SDOUT_ERROR
Sdout_initialize_channel_xtalk_cancellation_coeff
(SDOUT_CHANNEL_SEL channel,
unsigned int uiCoeff2,
unsigned int uiCoeff1)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_channel_xtalk_cancellation_coeff(%d,%d,%d)\n\r"),channel,uiCoeff2,uiCoeff1));

	switch(channel)
	{
		case SDOUT_CHANNEL_0:
			g_pSdoutConfig->SDO_XTALK0 = SDO_XTALK_COEF02(uiCoeff2) |
									SDO_XTALK_COEF01(uiCoeff1);
			DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_channel_xtalk_cancellation_coeff(0x%08x)\n\r"), g_pSdoutConfig->SDO_XTALK0));
			break;
		case SDOUT_CHANNEL_1:
			g_pSdoutConfig->SDO_XTALK1 = SDO_XTALK_COEF02(uiCoeff2) |
									SDO_XTALK_COEF01(uiCoeff1);
			DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_channel_xtalk_cancellation_coeff(0x%08x)\n\r"), g_pSdoutConfig->SDO_XTALK1));
			break;
		case SDOUT_CHANNEL_2:
			g_pSdoutConfig->SDO_XTALK2 = SDO_XTALK_COEF02(uiCoeff2) |
									SDO_XTALK_COEF01(uiCoeff1);
			DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_channel_xtalk_cancellation_coeff(0x%08x)\n\r"), g_pSdoutConfig->SDO_XTALK2));
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_channel_xtalk_cancellation_coeff() : invalid channel parameter(%d)\n\r"),channel));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	return SDOUT_NO_ERROR;
}

void
Sdout_initialize_closed_caption
(unsigned int uiDisplayCC,
unsigned int uiNonDisplayCC)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_closed_caption(%d,%d)\n\r"),uiDisplayCC,uiNonDisplayCC));

	g_pSdoutConfig->SDO_ARMCC = SDO_DISPLAY_CC_CAPTION(uiDisplayCC)|
	SDO_NON_DISPLAY_CC_CAPTION(uiNonDisplayCC);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_closed_caption(0x%x)\n\r"),g_pSdoutConfig->SDO_ARMCC));
}

SDOUT_ERROR
Sdout_initialize_wss525_data
(SDOUT_525_DATA_COPY_PERMIT_MODE copyPermit,
SDOUT_525_DATA_MV_PSP_MODE mvPsp,
SDOUT_525_DATA_COPY_INFO_MODE copyInfo,
BOOL bAnalogOn,
SDOUT_525_DATA_DISPLAY_ASPECT_RATIO displayRatio)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_wss525_data(%d,%d,%d,%d)\n\r"),copyPermit,mvPsp,copyInfo,displayRatio));

	switch(copyPermit)
	{
		case SDO_525_COPY_PERMIT:
			uiTempReg = SDO_WORD2_WSS525_COPY_PERMIT;
			break;
		case SDO_525_ONECOPY_PERMIT:
			uiTempReg = SDO_WORD2_WSS525_ONECOPY_PERMIT;
			break;
		case SDO_525_NOCOPY_PERMIT:
			uiTempReg = SDO_WORD2_WSS525_NOCOPY_PERMIT;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss525_data() : invalid copyPermit parameter(%d)\n\r"),copyPermit));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(mvPsp)
	{
		case SDO_525_MV_PSP_OFF:
			uiTempReg |= SDO_WORD2_WSS525_MV_PSP_OFF;
			break;
		case SDO_525_MV_PSP_ON_2LINE_BURST:
			uiTempReg |= SDO_WORD2_WSS525_MV_PSP_ON_2LINE_BURST;
			break;
		case SDO_525_MV_PSP_ON_BURST_OFF:
			uiTempReg |= SDO_WORD2_WSS525_MV_PSP_ON_BURST_OFF;
			break;
		case SDO_525_MV_PSP_ON_4LINE_BURST:
			uiTempReg |= SDO_WORD2_WSS525_MV_PSP_ON_4LINE_BURST;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss525_data() : invalid mvPsp parameter(%d)\n\r"),mvPsp));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(copyInfo)
	{
		case SDO_525_COPY_INFO:
			uiTempReg |= SDO_WORD1_WSS525_COPY_INFO;
			break;
		case SDO_525_DEFAULT:
			uiTempReg |= SDO_WORD1_WSS525_DEFAULT;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss525_data() : invalid copyInfo parameter(%d)\n\r"),copyInfo));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(bAnalogOn)
	{
		uiTempReg |= SDO_WORD2_WSS525_ANALOG_ON;
	}
	else
	{
		uiTempReg |= SDO_WORD2_WSS525_ANALOG_OFF;
	}

	switch(displayRatio)
	{
		case SDO_525_COPY_PERMIT:
			uiTempReg |= SDO_WORD0_WSS525_4_3_NORMAL;
			break;
		case SDO_525_ONECOPY_PERMIT:
			uiTempReg |= SDO_WORD0_WSS525_16_9_ANAMORPIC;
			break;
		case SDO_525_NOCOPY_PERMIT:
			uiTempReg |= SDO_WORD0_WSS525_4_3_LETTERBOX;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss525_data() : invalid displayRatio parameter(%d)\n\r"),displayRatio));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_ARMWSS525 = uiTempReg |
	SDO_CRC_WSS525(Sdout_initialize_wss_cgms_crc(uiTempReg));

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_wss525_data(0x%08x)\n\r"), g_pSdoutConfig->SDO_ARMWSS525));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_wss625_data
(BOOL bSurrounfSound,
BOOL bCopyright,
BOOL bCopyProtection,
BOOL bTextSubtitles,
SDOUT_625_DATA_OPEN_SUBTITLES openSubtitles,
SDOUT_625_DATA_CAMERA_FILM_MODE cameraFilm,
SDOUT_625_DATA_COLOR_ENCODING colorEncoding,
BOOL bHelperSignal,
SDOUT_625_DATA_DISPLAY_ASPECT_RATIO displayRatio)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_wss625_data(%d,%d,%d,%d,%d,%d,%d,%d,%d)\n\r"),
	bSurrounfSound,bCopyright,bCopyProtection,bTextSubtitles,openSubtitles,cameraFilm,colorEncoding,bHelperSignal,displayRatio));

	if(bSurrounfSound)
	{
		uiTempReg = SDO_WSS625_SURROUND_SOUND_ENABLE;
	}
	else
	{
		uiTempReg = SDO_WSS625_SURROUND_SOUND_DISABLE;
	}

	if(bCopyright)
	{
		uiTempReg |= SDO_WSS625_COPYRIGHT;
	}
	else
	{
		uiTempReg |= SDO_WSS625_NO_COPYRIGHT;
	}

	if(bCopyProtection)
	{
		uiTempReg |= SDO_WSS625_COPY_RESTRICTED;
	}
	else
	{
		uiTempReg |= SDO_WSS625_COPY_NOT_RESTRICTED;
	}

	if(bTextSubtitles)
	{
		uiTempReg |= SDO_WSS625_TELETEXT_SUBTITLES;
	}
	else
	{
		uiTempReg |= SDO_WSS625_TELETEXT_NO_SUBTITLES;
	}

	switch(openSubtitles)
	{
		case SDO_625_NO_OPEN_SUBTITLES:
			uiTempReg |= SDO_WSS625_NO_OPEN_SUBTITLES;
			break;
		case SDO_625_INACT_OPEN_SUBTITLES:
			uiTempReg |= SDO_WSS625_INACT_OPEN_SUBTITLES;
			break;
		case SDO_625_OUTACT_OPEN_SUBTITLES:
			uiTempReg |= SDO_WSS625_OUTACT_OPEN_SUBTITLES;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss625_data() : invalid openSubtitles parameter(%d)\n\r"),openSubtitles));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(cameraFilm)
	{
		case SDO_625_CAMERA:
			uiTempReg |= SDO_WSS625_CAMERA;
			break;
		case SDO_625_FILM:
			uiTempReg |= SDO_WSS625_FILM;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss625_data() : invalid cameraFilm parameter(%d)\n\r"),cameraFilm));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(colorEncoding)
	{
		case SDO_625_NORMAL_PAL:
			uiTempReg |= SDO_WSS625_NORMAL_PAL;
			break;
		case SDO_625_MOTION_ADAPTIVE_COLORPLUS:
			uiTempReg |= SDO_WSS625_MOTION_ADAPTIVE_COLORPLUS;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss625_data() : invalid colorEncoding parameter(%d)\n\r"),colorEncoding));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(bHelperSignal)
	{
		uiTempReg |= SDO_WSS625_HELPER_SIG;
	}
	else
	{
		uiTempReg |= SDO_WSS625_HELPER_NO_SIG;
	}

	switch(displayRatio)
	{
		case SDO_625_4_3_FULL_576:
			uiTempReg |= SDO_WSS625_4_3_FULL_576;
			break;
		case SDO_625_14_9_LETTERBOX_CENTER_504:
			uiTempReg |= SDO_WSS625_14_9_LETTERBOX_CENTER_504;
			break;
		case SDO_625_14_9_LETTERBOX_TOP_504:
			uiTempReg |= SDO_WSS625_14_9_LETTERBOX_TOP_504;
			break;
		case SDO_625_16_9_LETTERBOX_CENTER_430:
			uiTempReg |= SDO_WSS625_16_9_LETTERBOX_CENTER_430;
			break;
		case SDO_625_16_9_LETTERBOX_TOP_430:
			uiTempReg |= SDO_WSS625_16_9_LETTERBOX_TOP_430;
			break;
		case SDO_625_16_9_LETTERBOX_CENTER:
			uiTempReg |= SDO_WSS625_16_9_LETTERBOX_CENTER;
			break;
		case SDO_625_14_9_FULL_CENTER_576:
			uiTempReg |= SDO_WSS625_14_9_FULL_CENTER_576;
			break;
		case SDO_625_16_9_ANAMORPIC_576:
			uiTempReg |= SDO_WSS625_16_9_ANAMORPIC_576;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_wss625_data() : invalid displayRatio parameter(%d)\n\r"),displayRatio));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_ARMWSS625 = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_wss625_data(0x%08x)\n\r"), g_pSdoutConfig->SDO_ARMWSS625));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_cgmsa525_data
(SDOUT_525_DATA_COPY_PERMIT_MODE copyPermit,
SDOUT_525_DATA_MV_PSP_MODE mvPsp,
SDOUT_525_DATA_COPY_INFO_MODE copyInfo,
BOOL bAnalogOn,
SDOUT_525_DATA_DISPLAY_ASPECT_RATIO displayRatio)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_cgmsa525_data(%d,%d,%d,%d)\n\r"),copyPermit,mvPsp,copyInfo,displayRatio));

	switch(copyPermit)
	{
		case SDO_525_COPY_PERMIT:
			uiTempReg = SDO_WORD2_CGMS525_COPY_PERMIT;
			break;
		case SDO_525_ONECOPY_PERMIT:
			uiTempReg = SDO_WORD2_CGMS525_ONECOPY_PERMIT;
			break;
		case SDO_525_NOCOPY_PERMIT:
			uiTempReg = SDO_WORD2_CGMS525_NOCOPY_PERMIT;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgmsa525_data() : invalid copyPermit parameter(%d)\n\r"),copyPermit));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(mvPsp)
	{
		case SDO_525_MV_PSP_OFF:
			uiTempReg |= SDO_WORD2_CGMS525_MV_PSP_OFF;
			break;
		case SDO_525_MV_PSP_ON_2LINE_BURST:
			uiTempReg |= SDO_WORD2_CGMS525_MV_PSP_ON_2LINE_BURST;
			break;
		case SDO_525_MV_PSP_ON_BURST_OFF:
			uiTempReg |= SDO_WORD2_CGMS525_MV_PSP_ON_BURST_OFF;
			break;
		case SDO_525_MV_PSP_ON_4LINE_BURST:
			uiTempReg |= SDO_WORD2_CGMS525_MV_PSP_ON_4LINE_BURST;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgmsa525_data() : invalid mvPsp parameter(%d)\n\r"),mvPsp));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(copyInfo)
	{
		case SDO_525_COPY_INFO:
			uiTempReg |= SDO_WORD1_CGMS525_COPY_INFO;
			break;
		case SDO_525_DEFAULT:
			uiTempReg |= SDO_WORD1_CGMS525_DEFAULT;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgmsa525_data() : invalid copyInfo parameter(%d)\n\r"),copyInfo));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(bAnalogOn)
	{
		uiTempReg |= SDO_WORD2_CGMS525_ANALOG_ON;
	}
	else
	{
		uiTempReg |= SDO_WORD2_CGMS525_ANALOG_OFF;
	}

	switch(displayRatio)
	{
		case SDO_525_COPY_PERMIT:
			uiTempReg |= SDO_WORD0_CGMS525_4_3_NORMAL;
			break;
		case SDO_525_ONECOPY_PERMIT:
			uiTempReg |= SDO_WORD0_CGMS525_16_9_ANAMORPIC;
			break;
		case SDO_525_NOCOPY_PERMIT:
			uiTempReg |= SDO_WORD0_CGMS525_4_3_LETTERBOX;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgmsa525_data() : invalid displayRatio parameter(%d)\n\r"),displayRatio));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_ARMCGMS525 = uiTempReg |
	SDO_CRC_CGMS525(Sdout_initialize_wss_cgms_crc(uiTempReg));

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_cgmsa525_data(0x%08x)\n\r"), g_pSdoutConfig->SDO_ARMCGMS525));

	return SDOUT_NO_ERROR;
}

SDOUT_ERROR
Sdout_initialize_cgmsa625_data
(BOOL bSurrounfSound,
BOOL bCopyright,
BOOL bCopyProtection,
BOOL bTextSubtitles,
SDOUT_625_DATA_OPEN_SUBTITLES openSubtitles,
SDOUT_625_DATA_CAMERA_FILM_MODE cameraFilm,
SDOUT_625_DATA_COLOR_ENCODING colorEncoding,
BOOL bHelperSignal,
SDOUT_625_DATA_DISPLAY_ASPECT_RATIO displayRatio)
{
	unsigned int uiTempReg = 0;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_cgms625_data(%d,%d,%d,%d,%d,%d,%d,%d,%d)\n\r"),
	bSurrounfSound,bCopyright,bCopyProtection,bTextSubtitles,openSubtitles,cameraFilm,colorEncoding,bHelperSignal,displayRatio));

	if(bSurrounfSound)
	{
		uiTempReg = SDO_CGMS625_SURROUND_SOUND_ENABLE;
	}
	else
	{
		uiTempReg = SDO_CGMS625_SURROUND_SOUND_DISABLE;
	}

	if(bCopyright)
	{
		uiTempReg |= SDO_CGMS625_COPYRIGHT;
	}
	else
	{
		uiTempReg |= SDO_CGMS625_NO_COPYRIGHT;
	}

	if(bCopyProtection)
	{
		uiTempReg |= SDO_CGMS625_COPY_RESTRICTED;
	}
	else
	{
		uiTempReg |= SDO_CGMS625_COPY_NOT_RESTRICTED;
	}

	if(bTextSubtitles)
	{
		uiTempReg |= SDO_CGMS625_TELETEXT_SUBTITLES;
	}
	else
	{
		uiTempReg |= SDO_CGMS625_TELETEXT_NO_SUBTITLES;
	}

	switch(openSubtitles)
	{
		case SDO_625_NO_OPEN_SUBTITLES:
			uiTempReg |= SDO_CGMS625_NO_OPEN_SUBTITLES;
			break;
		case SDO_625_INACT_OPEN_SUBTITLES:
			uiTempReg |= SDO_CGMS625_INACT_OPEN_SUBTITLES;
			break;
		case SDO_625_OUTACT_OPEN_SUBTITLES:
			uiTempReg |= SDO_CGMS625_OUTACT_OPEN_SUBTITLES;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgms625_data() : invalid openSubtitles parameter(%d)\n\r"),openSubtitles));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(cameraFilm)
	{
		case SDO_625_CAMERA:
			uiTempReg |= SDO_CGMS625_CAMERA;
			break;
		case SDO_625_FILM:
			uiTempReg |= SDO_CGMS625_FILM;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgms625_data() : invalid cameraFilm parameter(%d)\n\r"),cameraFilm));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	switch(colorEncoding)
	{
		case SDO_625_NORMAL_PAL:
			uiTempReg |= SDO_CGMS625_NORMAL_PAL;
			break;
		case SDO_625_MOTION_ADAPTIVE_COLORPLUS:
			uiTempReg |= SDO_CGMS625_MOTION_ADAPTIVE_COLORPLUS;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgms625_data() : invalid colorEncoding parameter(%d)\n\r"),colorEncoding));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	if(bHelperSignal)
	{
		uiTempReg |= SDO_CGMS625_HELPER_SIG;
	}
	else
	{
		uiTempReg |= SDO_CGMS625_HELPER_NO_SIG;
	}

	switch(displayRatio)
	{
		case SDO_625_4_3_FULL_576:
			uiTempReg |= SDO_CGMS625_4_3_FULL_576;
			break;
		case SDO_625_14_9_LETTERBOX_CENTER_504:
			uiTempReg |= SDO_CGMS625_14_9_LETTERBOX_CENTER_504;
			break;
		case SDO_625_14_9_LETTERBOX_TOP_504:
			uiTempReg |= SDO_CGMS625_14_9_LETTERBOX_TOP_504;
			break;
		case SDO_625_16_9_LETTERBOX_CENTER_430:
			uiTempReg |= SDO_CGMS625_16_9_LETTERBOX_CENTER_430;
			break;
		case SDO_625_16_9_LETTERBOX_TOP_430:
			uiTempReg |= SDO_CGMS625_16_9_LETTERBOX_TOP_430;
			break;
		case SDO_625_16_9_LETTERBOX_CENTER:
			uiTempReg |= SDO_CGMS625_16_9_LETTERBOX_CENTER;
			break;
		case SDO_625_14_9_FULL_CENTER_576:
			uiTempReg |= SDO_CGMS625_14_9_FULL_CENTER_576;
			break;
		case SDO_625_16_9_ANAMORPIC_576:
			uiTempReg |= SDO_CGMS625_16_9_ANAMORPIC_576;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_cgms625_data() : invalid displayRatio parameter(%d)\n\r"),displayRatio));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	g_pSdoutConfig->SDO_ARMCGMS625 = uiTempReg;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_cgms625_data(0x%08x)\n\r"), g_pSdoutConfig->SDO_ARMCGMS625));

	return SDOUT_NO_ERROR;
}


// start
//  - start functions are only called under stopping sdout
void
Sdout_start
(void)
{
//int i;
//unsigned int *puiTemp = (unsigned int*)g_pSdoutConfig;
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_start()\n\r")));

//g_pSdoutConfig->SDO_PAUSE = 1;
//g_pSdoutConfig->SDO_BB_CTRL = 0;

	g_pSdoutConfig->SDO_CLKCON |= SDO_TVOUT_CLOCK_ON;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_start(0x%x)\n\r"),g_pSdoutConfig->SDO_CLKCON));
//for(i=0;i<(0x3DC/4);i++)
//{
//    DBGMSG(TV_USR1,(_T("[SDOUT] 0x%x = 0x%x\n\r"),puiTemp+i,*(puiTemp+i)));
//}
}


// stop
//  - stop functions are only called under running sdout
void
Sdout_stop
(void)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_stop()\n\r")));

	g_pSdoutConfig->SDO_CLKCON &= ~(SDO_TVOUT_CLOCK_ON);

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_stop(0x%x)\n\r"),g_pSdoutConfig->SDO_CLKCON));
}


// reset
//  - reset function
void
Sdout_sw_reset
(BOOL bActive)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_sw_reset(%d)\n\r"),bActive));

	if(bActive)
		g_pSdoutConfig->SDO_CLKCON |= SDO_TVOUT_SW_RESET;
	else
		g_pSdoutConfig->SDO_CLKCON &= ~SDO_TVOUT_SW_RESET;

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_sw_reset(0x%x)\n\r"),g_pSdoutConfig->SDO_CLKCON));
}


// interrupt
//  -
void
Sdout_set_interrupt_enable
(BOOL bVsyncIntrEn)
{
	DBGMSG((TV_FUNC&&TV_UNDER_RUN),(_T("[SDOUT]++Sdout_set_interrupt_enable(%d)\n\r"),bVsyncIntrEn));

	if(bVsyncIntrEn)
		g_pSdoutConfig->SDO_IRQMASK &= ~SDO_VSYNC_IRQ_DISABLE;
	else
		g_pSdoutConfig->SDO_IRQMASK |= SDO_VSYNC_IRQ_DISABLE;

	DBGMSG((TV_FUNC&&TV_UNDER_RUN),(_T("[SDOUT]--Sdout_set_interrupt_enable(0x%x)\n\r"),g_pSdoutConfig->SDO_IRQMASK));
}

void
Sdout_clear_interrupt_pending
(void)//bVsyncIntr
{
	DBGMSG((TV_FUNC&&TV_UNDER_RUN),(_T("[SDOUT]++Sdout_clear_interrupt_pending(0x%x)\n\r"),g_pSdoutConfig->SDO_IRQ));

	g_pSdoutConfig->SDO_IRQ |= SDO_VSYNC_IRQ_PEND;

	DBGMSG((TV_FUNC&&TV_UNDER_RUN),(_T("[SDOUT]--Sdout_clear_interrupt_pending(0x%x)\n\r"),g_pSdoutConfig->SDO_IRQ));
}

BOOL
Sdout_get_interrupt_pending
(void)
{
	DBGMSG((TV_FUNC&&TV_UNDER_RUN),(_T("[SDOUT]++Sdout_get_interrupt_pending(0x%x)\n\r"),g_pSdoutConfig->SDO_IRQ));

	DBGMSG((TV_FUNC&&TV_UNDER_RUN),(_T("[SDOUT]--Sdout_get_interrupt_pending()\n\r")));

	return (g_pSdoutConfig->SDO_IRQ | SDO_VSYNC_IRQ_PEND)? TRUE:FALSE;
}

// static functions
static unsigned int
Sdout_initialize_wss_cgms_crc
(unsigned int uiValue)
{
	unsigned int i;
	unsigned char CGMS[14], CRC[6], OLD_CRC;
	unsigned int TEMP_IN;

	TEMP_IN = uiValue;
	for ( i=0; i<14; i++ )
		CGMS[i] = (unsigned char)(TEMP_IN>>i ) & 0x1 ;

	// initialize state
	for ( i=0; i<6; i++ )
		CRC[i] = 0x1;

	// round 20
	for ( i=0; i<14; i++ )
	{
		OLD_CRC = CRC[0];
		CRC[0] = CRC[1];
		CRC[1] = CRC[2];
		CRC[2] = CRC[3];
		CRC[3] = CRC[4];
		CRC[4] = OLD_CRC ^ CGMS[i] ^ CRC[5];
		CRC[5] = OLD_CRC ^ CGMS[i];
	}

	// recompose to return crc
	TEMP_IN &= 0x3fff;

	for( i=0; i<6; i++ )
	{
		TEMP_IN |= ( (unsigned int)(CRC[i]&0x1)<<i);
	}

	return TEMP_IN;
}

static SDOUT_ERROR
Sdout_initialize_antialias_filter_coeff_default
(SDOUT_LEVEL compositeLevel,
SDOUT_VIDEO_SYNC_RATIO compositeRatio,
TVOUT_OUTPUT_MODE outMode)
{
	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_antialias_filter_coeff_default(%d,%d,%d)\n\r"),compositeLevel,compositeRatio,outMode));

	switch(compositeLevel)
	{
		case SDOUT_LEVEL_0IRE:
			switch(compositeRatio)
			{
				case SDOUT_VTOS_RATIO_10_4:
					switch(outMode)
					{
						case TVOUT_OUTPUT_COMPOSITE:
							g_pSdoutConfig->SDO_Y0 = 0x00000000;
							g_pSdoutConfig->SDO_Y1 = 0x00000000;
							g_pSdoutConfig->SDO_Y2 = 0x00000000;
							g_pSdoutConfig->SDO_Y3 = 0x00000000;
							g_pSdoutConfig->SDO_Y4 = 0x00000000;
							g_pSdoutConfig->SDO_Y5 = 0x00000000;
							g_pSdoutConfig->SDO_Y6 = 0x00000000;
							g_pSdoutConfig->SDO_Y7 = 0x00000000;
							g_pSdoutConfig->SDO_Y8 = 0x00000000;
							g_pSdoutConfig->SDO_Y9 = 0x00000000;
							g_pSdoutConfig->SDO_Y10= 0x00000000;
							g_pSdoutConfig->SDO_Y11= 0x0000029a;
							g_pSdoutConfig->SDO_CB0= 0x00000000;
							g_pSdoutConfig->SDO_CB1= 0x00000000;
							g_pSdoutConfig->SDO_CB2= 0x00000000;
							g_pSdoutConfig->SDO_CB3= 0x00000000;
							g_pSdoutConfig->SDO_CB4= 0x00000000;
							g_pSdoutConfig->SDO_CB5= 0x00000001;
							g_pSdoutConfig->SDO_CB6= 0x00000007;
							g_pSdoutConfig->SDO_CB7= 0x00000015;
							g_pSdoutConfig->SDO_CB8= 0x0000002b;
							g_pSdoutConfig->SDO_CB9= 0x00000045;
							g_pSdoutConfig->SDO_CB10=0x00000059;
							g_pSdoutConfig->SDO_CB11=0x00000061;
							g_pSdoutConfig->SDO_CR1 =0x00000000;
							g_pSdoutConfig->SDO_CR2 =0x00000000;
							g_pSdoutConfig->SDO_CR3 =0x00000000;
							g_pSdoutConfig->SDO_CR4 =0x00000000;
							g_pSdoutConfig->SDO_CR5 =0x00000002;
							g_pSdoutConfig->SDO_CR6 =0x0000000a;
							g_pSdoutConfig->SDO_CR7 =0x0000001e;
							g_pSdoutConfig->SDO_CR8 =0x0000003d;
							g_pSdoutConfig->SDO_CR9 =0x00000061;
							g_pSdoutConfig->SDO_CR10=0x0000007a;
							g_pSdoutConfig->SDO_CR11=0x0000008f;
							break;
						default:
							DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_antialias_filter_coeff_default() : invalid outMode parameter(%d)\n\r"),outMode));
							return SDOUT_ERROR_INVALID_PARAM;
							break;
					}
					break;
					
				case SDOUT_VTOS_RATIO_7_3:
					g_pSdoutConfig->SDO_Y0  = 0x00000000;
					g_pSdoutConfig->SDO_Y1  = 0x00000000;
					g_pSdoutConfig->SDO_Y2  = 0x00000000;
					g_pSdoutConfig->SDO_Y3  = 0x00000000;
					g_pSdoutConfig->SDO_Y4  = 0x00000000;
					g_pSdoutConfig->SDO_Y5  = 0x00000000;
					g_pSdoutConfig->SDO_Y6  = 0x00000000;
					g_pSdoutConfig->SDO_Y7  = 0x00000000;
					g_pSdoutConfig->SDO_Y8  = 0x00000000;
					g_pSdoutConfig->SDO_Y9  = 0x00000000;
					g_pSdoutConfig->SDO_Y10 = 0x00000000;
					g_pSdoutConfig->SDO_Y11 = 0x00000281;
					g_pSdoutConfig->SDO_CB0 = 0x00000000;
					g_pSdoutConfig->SDO_CB1 = 0x00000000;
					g_pSdoutConfig->SDO_CB2 = 0x00000000;
					g_pSdoutConfig->SDO_CB3 = 0x00000000;
					g_pSdoutConfig->SDO_CB4 = 0x00000000;
					g_pSdoutConfig->SDO_CB5 = 0x00000001;
					g_pSdoutConfig->SDO_CB6 = 0x00000007;
					g_pSdoutConfig->SDO_CB7 = 0x00000015;
					g_pSdoutConfig->SDO_CB8 = 0x0000002a;
					g_pSdoutConfig->SDO_CB9 = 0x00000044;
					g_pSdoutConfig->SDO_CB10= 0x00000057;
					g_pSdoutConfig->SDO_CB11= 0x0000005f;
					g_pSdoutConfig->SDO_CR1 = 0x00000000;
					g_pSdoutConfig->SDO_CR2 = 0x00000000;
					g_pSdoutConfig->SDO_CR3 = 0x00000000;
					g_pSdoutConfig->SDO_CR4 = 0x00000000;
					g_pSdoutConfig->SDO_CR5 = 0x00000002;
					g_pSdoutConfig->SDO_CR6 = 0x0000000a;
					g_pSdoutConfig->SDO_CR7 = 0x0000001d;
					g_pSdoutConfig->SDO_CR8 = 0x0000003c;
					g_pSdoutConfig->SDO_CR9 = 0x0000005f;
					g_pSdoutConfig->SDO_CR10= 0x0000007b;
					g_pSdoutConfig->SDO_CR11= 0x00000086;
					break;
				default:
					DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_antialias_filter_coeff_default() : invalid compositeRatio parameter(%d)\n\r"),compositeRatio));
					return SDOUT_ERROR_INVALID_PARAM;
					break;
			}
			break;
		case SDOUT_LEVEL_75IRE:
			switch(compositeRatio)
			{
				case SDOUT_VTOS_RATIO_10_4:
					g_pSdoutConfig->SDO_Y0  =0x00000000;
					g_pSdoutConfig->SDO_Y1  =0x00000000;
					g_pSdoutConfig->SDO_Y2  =0x00000000;
					g_pSdoutConfig->SDO_Y3  =0x00000000;
					g_pSdoutConfig->SDO_Y4  =0x00000000;
					g_pSdoutConfig->SDO_Y5  =0x00000000;
					g_pSdoutConfig->SDO_Y6  =0x00000000;
					g_pSdoutConfig->SDO_Y7  =0x00000000;
					g_pSdoutConfig->SDO_Y8  =0x00000000;
					g_pSdoutConfig->SDO_Y9  =0x00000000;
					g_pSdoutConfig->SDO_Y10 =0x00000000;
					g_pSdoutConfig->SDO_Y11 =0x0000025d;
					g_pSdoutConfig->SDO_CB0 =0x00000000;
					g_pSdoutConfig->SDO_CB1 =0x00000000;
					g_pSdoutConfig->SDO_CB2 =0x00000000;
					g_pSdoutConfig->SDO_CB3 =0x00000000;
					g_pSdoutConfig->SDO_CB4 =0x00000000;
					g_pSdoutConfig->SDO_CB5 =0x00000001;
					g_pSdoutConfig->SDO_CB6 =0x00000007;
					g_pSdoutConfig->SDO_CB7 =0x00000014;
					g_pSdoutConfig->SDO_CB8 =0x00000028;
					g_pSdoutConfig->SDO_CB9 =0x0000003f;
					g_pSdoutConfig->SDO_CB10=0x00000052;
					g_pSdoutConfig->SDO_CB11=0x0000005a;
					g_pSdoutConfig->SDO_CR1 =0x00000000;
					g_pSdoutConfig->SDO_CR2 =0x00000000;
					g_pSdoutConfig->SDO_CR3 =0x00000000;
					g_pSdoutConfig->SDO_CR4 =0x00000000;
					g_pSdoutConfig->SDO_CR5 =0x00000001;
					g_pSdoutConfig->SDO_CR6 =0x00000009;
					g_pSdoutConfig->SDO_CR7 =0x0000001c;
					g_pSdoutConfig->SDO_CR8 =0x00000039;
					g_pSdoutConfig->SDO_CR9 =0x0000005a;
					g_pSdoutConfig->SDO_CR10=0x00000074;
					g_pSdoutConfig->SDO_CR11=0x0000007e;
					break;
				case SDOUT_VTOS_RATIO_7_3:
					g_pSdoutConfig->SDO_Y0  =0x00000000;
					g_pSdoutConfig->SDO_Y1  =0x00000000;
					g_pSdoutConfig->SDO_Y2  =0x00000000;
					g_pSdoutConfig->SDO_Y3  =0x00000000;
					g_pSdoutConfig->SDO_Y4  =0x00000000;
					g_pSdoutConfig->SDO_Y5  =0x00000000;
					g_pSdoutConfig->SDO_Y6  =0x00000000;
					g_pSdoutConfig->SDO_Y7  =0x00000000;
					g_pSdoutConfig->SDO_Y8  =0x00000000;
					g_pSdoutConfig->SDO_Y9  =0x00000000;
					g_pSdoutConfig->SDO_Y10 =0x00000000;
					g_pSdoutConfig->SDO_Y11 =0x00000251;
					g_pSdoutConfig->SDO_CB0 =0x00000000;
					g_pSdoutConfig->SDO_CB1 =0x00000000;
					g_pSdoutConfig->SDO_CB2 =0x00000000;
					g_pSdoutConfig->SDO_CB3 =0x00000000;
					g_pSdoutConfig->SDO_CB4 =0x00000000;
					g_pSdoutConfig->SDO_CB5 =0x00000001;
					g_pSdoutConfig->SDO_CB6 =0x00000006;
					g_pSdoutConfig->SDO_CB7 =0x00000013;
					g_pSdoutConfig->SDO_CB8 =0x00000028;
					g_pSdoutConfig->SDO_CB9 =0x0000003f;
					g_pSdoutConfig->SDO_CB10=0x00000051;
					g_pSdoutConfig->SDO_CB11=0x00000056;
					g_pSdoutConfig->SDO_CR1 =0x00000000;
					g_pSdoutConfig->SDO_CR2 =0x00000000;
					g_pSdoutConfig->SDO_CR3 =0x00000000;
					g_pSdoutConfig->SDO_CR4 =0x00000000;
					g_pSdoutConfig->SDO_CR5 =0x00000002;
					g_pSdoutConfig->SDO_CR6 =0x00000005;
					g_pSdoutConfig->SDO_CR7 =0x00000018;
					g_pSdoutConfig->SDO_CR8 =0x00000037;
					g_pSdoutConfig->SDO_CR9 =0x0000005A;
					g_pSdoutConfig->SDO_CR10=0x00000076;
					g_pSdoutConfig->SDO_CR11=0x0000007e;
					break;
				default:
					DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_antialias_filter_coeff_default() : invalid compositeRatio parameter(%d)\n\r"),compositeRatio));
					return SDOUT_ERROR_INVALID_PARAM;
					break;
			}
				break;
			break;
		default:
			DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_antialias_filter_coeff_default() : invalid compositeLevel parameter(%d)\n\r"),compositeLevel));
			return SDOUT_ERROR_INVALID_PARAM;
			break;
	}

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_antialias_filter_coeff_default()\n\r")));

	return SDOUT_NO_ERROR;

}

static SDOUT_ERROR
Sdout_initialize_oversampling_filter_coeff_default
(TVOUT_OUTPUT_MODE outMode)
{
	unsigned int* puiTempReg = 0;
	int i;

	DBGMSG(TV_FUNC,(_T("[SDOUT]++Sdout_initialize_oversampling_filter_coeff_default(%d)\n\r"),outMode));

	switch(outMode)
	{
	case TVOUT_OUTPUT_COMPOSITE:
		puiTempReg = (unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_0);
		for (i=0; i<3; i++) // Setting SFR data from SDOUT_OSFC00_0 to SDOUT_OSFC23_2
		{
			puiTempReg = (i==0)? (unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_0) :
			(i==1)? (unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_1) :
			(unsigned int*)&(g_pSdoutConfig->SDO_OSFC00_2);

			*(puiTempReg+0)  = ((-2&0xfff)<<0)|((-3&0xfff)<<0);
			*(puiTempReg+1)  = 0;
			*(puiTempReg+2)  = (4<<0)|(5<<16);
			*(puiTempReg+3)  = ((-1&0xfff)<<0)|(0<<16);
			*(puiTempReg+4)  = ((-6&0xfff)<<0)|((-9&0xfff)<<16);
			*(puiTempReg+5)  = (1<<0)|(0<<16);
			*(puiTempReg+6)  = (10<<0)|(14<<16);
			*(puiTempReg+7)  = ((-1&0xfff)<<0)|(0<<16);
			*(puiTempReg+8)  = ((-14&0xfff)<<0)|((-20&0xfff)<<16);
			*(puiTempReg+9)  = (1<<0)|(0<<16);
			*(puiTempReg+10) = (20<<0)|(29<<16);
			*(puiTempReg+11) = ((-2&0xfff)<<0)|(0<<16);
			*(puiTempReg+12) = ((-28&0xfff)<<0)|((-40&0xfff)<<16);
			*(puiTempReg+13) = (2<<0)|(0<<16);
			*(puiTempReg+14) = (40<<0)|(56<<16);
			*(puiTempReg+15) = ((-3&0xfff)<<0)|(0<<16);
			*(puiTempReg+16) = ((-57&0xfff)<<0)|((-80&0xfff)<<16);
			*(puiTempReg+17) = (5<<0)|(0<<16);
			*(puiTempReg+18) = (86<<0)|(121<<16);
			*(puiTempReg+19) = ((-10&0xfff)<<0)|(0<<16);
			*(puiTempReg+20) = ((-154&0xfff)<<0)|((-212&0xfff)<<16);
			*(puiTempReg+21) = (27<<0)|(0<<16);
			*(puiTempReg+22) = (613<<0)|(651<<16);
			*(puiTempReg+23) = ((-308&0xfff)<<0)|(1024<<16);
		}
		break;

	default:
		DBGMSG(TV_USR1,(_T("[SDOUT:ERR] Sdout_initialize_oversampling_filter_coeff_default() : invalid outMode parameter(%d)\n\r"),outMode));
		return SDOUT_ERROR_INVALID_PARAM;
		break;
	}

	DBGMSG(TV_FUNC,(_T("[SDOUT]--Sdout_initialize_oversampling_filter_coeff_default()\n\r")));

	return SDOUT_NO_ERROR;
}

