//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
// Copyright (c) Samsung Electronics. Co. LTD. All rights reserved.

/*++

THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:

power_control_lib.c   Power Controller Library

Abstract:

   Low Level HW Block Power control Library

Functions:

    

Notes:

--*/

#include "precomp.h"

static volatile CMU_CLK_REG *g_pCMUCLKRegs = NULL;
static volatile CMU_MISC_REG *g_pCMUMISCRegs = NULL;
static volatile PMU_PM_REG *g_pPMUPMRegs = NULL;
static volatile PMU_MISC_REG *g_pPMUMISCRegs = NULL;

/**
*    @fn    PwrCon_initialize_register_address(void *pCMUCLKRegs, void *pCMUMISCRegs, void *pPMUPMRegs, void *pPMUMISCRegs)
*    @param pCMUCLKRegs/pCMUMISCRegs/pPMUPMRegs/pPMUMISCRegs    System Controller Register(CMU/PMU) Block's Start Address(Virtual)
*    @note  This funciton just save System Controller Register Block's Start address into global variable
*/
BOOL PwrCon_initialize_register_address(void *pCMUCLKRegs, void *pCMUMISCRegs, void *pPMUPMRegs, void *pPMUMISCRegs)
{
    RETAILMSG(PWC_FUNC,(TEXT("[PWRCON]++%s(0x%08x)\n"), _T(__FUNCTION__), pCMUCLKRegs));

    if ((pCMUCLKRegs == NULL)||(pCMUMISCRegs == NULL)||(pPMUPMRegs == NULL)||(pPMUMISCRegs == NULL))
    {
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] PwrCon_initialize_register_address() : NULL pointer parameter\n")));
        return FALSE;
    }
    else
    {
        g_pCMUCLKRegs  = (CMU_CLK_REG *)pCMUCLKRegs;
        g_pCMUMISCRegs = (CMU_MISC_REG *)pCMUMISCRegs;
        g_pPMUPMRegs   = (PMU_PM_REG *)pPMUPMRegs;
        g_pPMUMISCRegs = (PMU_MISC_REG *)pPMUMISCRegs;
        
        DBGMSG(PWC_DBG,(TEXT("[PWRCON:INF] g_pCMUCLKRegs = 0x%08x, g_pCMUMISCRegs = 0x%08x\n"), g_pCMUCLKRegs, g_pCMUMISCRegs));
        DBGMSG(PWC_DBG,(TEXT("[PWRCON:INF] g_pPMUPMRegs = 0x%08x, g_pPMUMISCRegs = 0x%08x\n"), g_pPMUPMRegs, g_pPMUMISCRegs));
    }

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));
    return TRUE;
}

/**
*    @fn    PwrCon_set_block_power_on(BLKPWR_DOMAIN eDomain)
*    @param eDomain    The IP Block number in NORMAL_CFG Register to turn on
*    @note  This funciton turn on specific IP Block's power
*           when the system is in NORMAL Mode, not in other Power mode
*/
BOOL PwrCon_set_block_power_on(BLKPWR_DOMAIN eDomain)
{
    UINT32 TimeOut;

    assert(g_pPMUPMRegs);

    DBGMSG(PWC_FUNC, (TEXT("[PWRCON]++%s(%d)\n"), _T(__FUNCTION__), eDomain));
    
    TimeOut = 10000000L;   // This is big value.

    switch(eDomain)
    {
    case BLKPWR_DOMAIN_AUDIOSS:
        if(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_AUDIOSS_POWER_GATE<<BP_AUDIOSS_POWER_GATE))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_AUDIOSS_POWER_GATE<<BP_AUDIOSS_POWER_GATE))|
                                            (PWR_ON<<BP_AUDIOSS_POWER_GATE);
        while(!(g_pPMUPMRegs->STATUS_REG.BLK_PWR_STAT & BLK_PWR_ON_STAT<<BP_AUDIOSS_PWR_STAT)&& 
              !(TimeOut-- == 0));
        if(TimeOut == 0) goto Block_Pwr_ON_FAIL;        
        break;
    case BLKPWR_DOMAIN_CAM:
        if(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_CAM_POWER_GATE<<BP_CAM_POWER_GATE))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }        
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_CAM_POWER_GATE<<BP_CAM_POWER_GATE))|
                                            (PWR_ON<<BP_CAM_POWER_GATE);
        while(!(g_pPMUPMRegs->STATUS_REG.BLK_PWR_STAT & BLK_PWR_ON_STAT<<BP_CAM_PWR_STAT)&& 
              !(TimeOut-- == 0))
        if(TimeOut == 0) goto Block_Pwr_ON_FAIL;        
        break;
    case BLKPWR_DOMAIN_TV:
        if(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_TV_POWER_GATE<<BP_TV_POWER_GATE))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }  
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_TV_POWER_GATE<<BP_TV_POWER_GATE))|
                                            (PWR_ON<<BP_TV_POWER_GATE);
        while(!(g_pPMUPMRegs->STATUS_REG.BLK_PWR_STAT & BLK_PWR_ON_STAT<<BP_TV_PWR_STAT)&& 
              !(TimeOut-- == 0))
        if(TimeOut == 0) goto Block_Pwr_ON_FAIL;        
        break;                        
    case BLKPWR_DOMAIN_LCD:
        if(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_LCD_POWER_GATE<<BP_LCD_POWER_GATE))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }       
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_LCD_POWER_GATE<<BP_LCD_POWER_GATE))|
                                            (PWR_ON<<BP_LCD_POWER_GATE);
        while(!(g_pPMUPMRegs->STATUS_REG.BLK_PWR_STAT & BLK_PWR_ON_STAT<<BP_LCD_PWR_STAT)&& 
              !(TimeOut-- == 0))
        if(TimeOut == 0) goto Block_Pwr_ON_FAIL;        
        break;
    case BLKPWR_DOMAIN_G3D:
        if(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_G3D_POWER_GATE<<BP_G3D_POWER_GATE))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }          
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_G3D_POWER_GATE<<BP_G3D_POWER_GATE))|
                                            (PWR_ON<<BP_G3D_POWER_GATE);
        while(!(g_pPMUPMRegs->STATUS_REG.BLK_PWR_STAT & BLK_PWR_ON_STAT<<BP_G3D_PWR_STAT)&&
              !(TimeOut-- == 0))
        if(TimeOut == 0) goto Block_Pwr_ON_FAIL;        
        break;
    case BLKPWR_DOMAIN_MFC: 
        if(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_MFC_POWER_GATE<<BP_MFC_POWER_GATE))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }         
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_MFC_POWER_GATE<<BP_MFC_POWER_GATE))|
                                            (PWR_ON<<BP_MFC_POWER_GATE);
        while(!(g_pPMUPMRegs->STATUS_REG.BLK_PWR_STAT & BLK_PWR_ON_STAT<<BP_MFC_PWR_STAT)&& 
              !(TimeOut-- == 0))
        if(TimeOut == 0) goto Block_Pwr_ON_FAIL;        
        break;
    default:
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] %s() : Unknown Domain = %d\n"), _T(__FUNCTION__), eDomain));
        return FALSE;
    }

    DBGMSG(PWC_FUNC, (TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));

    return TRUE;

Block_Pwr_ON_FAIL:
    DBGMSG(TRUE, (_T("[PWRCON:ERR] %s() : Time OUT!! Domain = %d\n"), _T(__FUNCTION__), eDomain));
    
    return FALSE;
}

/**
*    @fn    PwrCon_set_block_power_off(BLKPWR_DOMAIN eDomain)
*    @param eDomain    The IP Block number in NORMAL_CFG Register to turn off
*    @note  This funciton turn off specific IP Block's power
*           when the system is in NORMAL Mode, not in other Power mode
*/
BOOL PwrCon_set_block_power_off(BLKPWR_DOMAIN eDomain)
{
    
    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]++%s(%d)\n"), _T(__FUNCTION__), eDomain));

    switch(eDomain)
    {
    case BLKPWR_DOMAIN_AUDIOSS:  
        if(!(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_AUDIOSS_POWER_GATE<<BP_AUDIOSS_POWER_GATE)))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }         
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_AUDIOSS_POWER_GATE<<BP_AUDIOSS_POWER_GATE))|
                                            (PWR_OFF<<BP_AUDIOSS_POWER_GATE);
        break;
    case BLKPWR_DOMAIN_CAM:         
        if(!(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_CAM_POWER_GATE<<BP_CAM_POWER_GATE)))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }   
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_CAM_POWER_GATE<<BP_CAM_POWER_GATE))|
                                            (PWR_OFF<<BP_CAM_POWER_GATE);
        break;                
    case BLKPWR_DOMAIN_TV:         
        if(!(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_TV_POWER_GATE<<BP_TV_POWER_GATE)))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eDomain)); 
            break;
        } 
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_TV_POWER_GATE<<BP_TV_POWER_GATE))|
                                            (PWR_OFF<<BP_TV_POWER_GATE);
        break;
    case BLKPWR_DOMAIN_LCD: 
		//break; //added by for test 2012.06.28
		
        if(!(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_LCD_POWER_GATE<<BP_LCD_POWER_GATE)))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eDomain)); 
            break;
        } 
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_LCD_POWER_GATE<<BP_LCD_POWER_GATE))|
                                            (PWR_OFF<<BP_LCD_POWER_GATE);
        break;
    case BLKPWR_DOMAIN_G3D:     
        if(!(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_G3D_POWER_GATE<<BP_G3D_POWER_GATE)))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }         
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_G3D_POWER_GATE<<BP_G3D_POWER_GATE))|
                                            (PWR_OFF<<BP_G3D_POWER_GATE);
        break;
    case BLKPWR_DOMAIN_MFC:     
        if(!(g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & (BW_MFC_POWER_GATE<<BP_MFC_POWER_GATE)))
        {
           DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eDomain)); 
            break;
        }         
        g_pPMUPMRegs->PWR_CONF.NORMAL_CFG = (g_pPMUPMRegs->PWR_CONF.NORMAL_CFG & ~(BW_MFC_POWER_GATE<<BP_MFC_POWER_GATE))|
                                            (PWR_OFF<<BP_MFC_POWER_GATE);
        break;
    default:
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] %s() : Unknown Domain = %d\n"), _T(__FUNCTION__), eDomain));
        return FALSE;
    }

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));

    return TRUE;
}

/**
*    @fn    PwrCon_set_analog_power_on(PWR_IP_LIST eIPPower)
*    @param eIPPower    The IP number in power control register in analog device
*    @note  This funciton turn on specific IP's power
*           when the system is in NORMAL Mode, not in other Power mode
*/
BOOL PwrCon_set_analog_power_on(PWR_IP_LIST eIPPower)
{

    assert(g_pPMUMISCRegs);

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]++%s(%d)\n"), _T(__FUNCTION__), eIPPower));
    
    switch(eIPPower)
    {
    case PWR_IP_MIE_SLEEP:
        if(g_pPMUMISCRegs->IP_CON_REG.MIE_CONTROL &(BW_MIE_I_DISPON<<BP_MIE_I_DISPON))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }
        g_pPMUMISCRegs->IP_CON_REG.MIE_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.MIE_CONTROL & ~(BW_MIE_I_DISPON<<BP_MIE_I_DISPON))|
                                                 (PWR_ON<<BP_MIE_I_DISPON);
        break;
    case PWR_IP_HDMI_PHY:
        if(g_pPMUMISCRegs->IP_CON_REG.HDMI_CONTROL &(BW_ENABLE_HDMIPHY<<BP_ENABLE_HDMIPHY))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }        
        g_pPMUMISCRegs->IP_CON_REG.HDMI_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.HDMI_CONTROL & ~(BW_ENABLE_HDMIPHY<<BP_ENABLE_HDMIPHY))|
                                                  (PHY_ENABLE<<BP_ENABLE_HDMIPHY);
        break;
    case PWR_IP_USB_PHY0:
        if(g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL &(BW_ENABLE_USBPHY0<<BP_ENABLE_USBPHY0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL & ~(BW_ENABLE_USBPHY0<<BP_ENABLE_USBPHY0))|
                                                     (PHY_ENABLE<<BP_ENABLE_USBPHY0);
        break;                 
    case PWR_IP_USB_PHY1:
        if(g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL &(BW_ENABLE_USBPHY1<<BP_ENABLE_USBPHY1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL & ~(BW_ENABLE_USBPHY1<<BP_ENABLE_USBPHY1))|
                                                     (PHY_ENABLE<<BP_ENABLE_USBPHY1);
        break;
    case PWR_IP_DAC:
        if(g_pPMUMISCRegs->IP_CON_REG.DAC_CONTROL &(BW_ENABLE_DAC<<BP_ENABLE_DAC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.DAC_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.DAC_CONTROL & ~(BW_ENABLE_DAC<<BP_ENABLE_DAC))|
                                                 (DAC_ENABLE<<BP_ENABLE_DAC);
        break;
    case PWR_IP_MIPI_DSI_DPHY:
        if((g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_DSI_DPHY<<BP_ENABLE_MIPI_DSI_DPHY))&&(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_WHOLE_DPHY<<BP_ENABLE_MIPI_WHOLE_DPHY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }

        g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= (g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL & ~(BW_ENABLE_MIPI_DSI_DPHY<<BP_ENABLE_MIPI_DSI_DPHY))|
                                                      (MIPI_DSI_DPHY_ENABLE<<BP_ENABLE_MIPI_DSI_DPHY);

        if(!(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_WHOLE_DPHY<<BP_ENABLE_MIPI_WHOLE_DPHY)))
        {
            g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL |
                                                          (MIPI_DPHY_ENABLE<<BP_ENABLE_MIPI_WHOLE_DPHY);            
        }
        break;
    case PWR_IP_MIPI_CSI_DPHY:
        if((g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_CSI_DPHY<<BP_ENABLE_MIPI_CSI_DPHY))&&(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_WHOLE_DPHY<<BP_ENABLE_MIPI_WHOLE_DPHY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }

        g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= (g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL & ~(BW_ENABLE_MIPI_CSI_DPHY<<BP_ENABLE_MIPI_CSI_DPHY))|
                                                      (MIPI_CSI_DPHY_ENABLE<<BP_ENABLE_MIPI_CSI_DPHY);

        if(!(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_WHOLE_DPHY<<BP_ENABLE_MIPI_WHOLE_DPHY)))
        {
            g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL |
                                                          (MIPI_DPHY_ENABLE<<BP_ENABLE_MIPI_WHOLE_DPHY);            
        }
        break;      
    case PWR_IP_TS_ADC:
        if(g_pPMUMISCRegs->IP_CON_REG.ADC_CONTROL &(BW_ENABLE_TSADC<<BP_ENABLE_TSADC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.ADC_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.ADC_CONTROL & ~(BW_ENABLE_TSADC<<BP_ENABLE_TSADC))|
                                                  (TSADC_ENABLE<<BP_ENABLE_TSADC);
        break;
    default:
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] %s() : Unknown Power IP = %d\n"), _T(__FUNCTION__), eIPPower));
        return FALSE;
    }

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));

    return TRUE;

}

/**
*    @fn    PwrCon_set_analog_power_off(PWR_IP_LIST eIPPower)
*    @param eIPPower    The IP number in power control register in analog device
*    @note  This funciton turn off specific IP's power
*           when the system is in NORMAL Mode, not in other Power mode
*/
BOOL PwrCon_set_analog_power_off(PWR_IP_LIST eIPPower)
{

    assert(g_pPMUMISCRegs);

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]++%s(%d)\n"), _T(__FUNCTION__), eIPPower));
    
    switch(eIPPower)
    {
    case PWR_IP_MIE_SLEEP:
        if(!(g_pPMUMISCRegs->IP_CON_REG.MIE_CONTROL &(BW_MIE_I_DISPON<<BP_MIE_I_DISPON)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }        
        g_pPMUMISCRegs->IP_CON_REG.MIE_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.MIE_CONTROL & ~(BW_MIE_I_DISPON<<BP_MIE_I_DISPON))|
                                                 (PWR_OFF<<BP_MIE_I_DISPON);
        break;
    case PWR_IP_HDMI_PHY:
        if(!(g_pPMUMISCRegs->IP_CON_REG.HDMI_CONTROL &(BW_ENABLE_HDMIPHY<<BP_ENABLE_HDMIPHY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }        
        g_pPMUMISCRegs->IP_CON_REG.HDMI_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.HDMI_CONTROL & ~(BW_ENABLE_HDMIPHY<<BP_ENABLE_HDMIPHY))|
                                                  (PHY_DISABLE<<BP_ENABLE_HDMIPHY);
        break;
    case PWR_IP_USB_PHY0:
        if(!(g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL &(BW_ENABLE_USBPHY0<<BP_ENABLE_USBPHY0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL & ~(BW_ENABLE_USBPHY0<<BP_ENABLE_USBPHY0))|
                                                     (PHY_DISABLE<<BP_ENABLE_USBPHY0);
        break;                 
    case PWR_IP_USB_PHY1:
        if(!(g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL &(BW_ENABLE_USBPHY1<<BP_ENABLE_USBPHY1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        } 
        g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.USB_PHY_CONTROL & ~(BW_ENABLE_USBPHY1<<BP_ENABLE_USBPHY1))|
                                                     (PHY_DISABLE<<BP_ENABLE_USBPHY1);
        break;
    case PWR_IP_DAC:
        if(!(g_pPMUMISCRegs->IP_CON_REG.DAC_CONTROL &(BW_ENABLE_DAC<<BP_ENABLE_DAC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.DAC_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.DAC_CONTROL & ~(BW_ENABLE_DAC<<BP_ENABLE_DAC))|
                                                 (DAC_DISABLE<<BP_ENABLE_DAC);
        break;
    case PWR_IP_MIPI_DSI_DPHY:
        if(!(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_DSI_DPHY<<BP_ENABLE_MIPI_DSI_DPHY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }
        
        g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= (g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL & ~(BW_ENABLE_MIPI_DSI_DPHY<<BP_ENABLE_MIPI_DSI_DPHY))|
                                                      (MIPI_DSI_DPHY_DISABLE<<BP_ENABLE_MIPI_DSI_DPHY);

        if(!(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_CSI_DPHY<<BP_ENABLE_MIPI_CSI_DPHY)))
        {
            g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= (g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL & ~(BW_ENABLE_MIPI_WHOLE_DPHY<<BP_ENABLE_MIPI_WHOLE_DPHY))|
                                                         (MIPI_DPHY_DISABLE<<BP_ENABLE_MIPI_WHOLE_DPHY);         
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s MIPI DPHY Whole Off\n"), _T(__FUNCTION__))); 
        }
        break;
    case PWR_IP_MIPI_CSI_DPHY:
        if(!(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_CSI_DPHY<<BP_ENABLE_MIPI_CSI_DPHY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= (g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL & ~(BW_ENABLE_MIPI_CSI_DPHY<<BP_ENABLE_MIPI_CSI_DPHY))|
                                                      (MIPI_CSI_DPHY_DISABLE<<BP_ENABLE_MIPI_CSI_DPHY);
        
        if(!(g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL &(BW_ENABLE_MIPI_DSI_DPHY<<BP_ENABLE_MIPI_DSI_DPHY)))
        {
            g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL= (g_pPMUMISCRegs->IP_CON_REG.MIPI_DPHY_CONTROL & ~(BW_ENABLE_MIPI_WHOLE_DPHY<<BP_ENABLE_MIPI_WHOLE_DPHY))|
                                                         (MIPI_DPHY_DISABLE<<BP_ENABLE_MIPI_WHOLE_DPHY);           
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s MIPI DPHY Whole Off\n"), _T(__FUNCTION__))); 
        }        
        break;        
    case PWR_IP_TS_ADC:
        if(!(g_pPMUMISCRegs->IP_CON_REG.ADC_CONTROL &(BW_ENABLE_TSADC<<BP_ENABLE_TSADC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), eIPPower)); 
            break;           
        }         
        g_pPMUMISCRegs->IP_CON_REG.ADC_CONTROL = (g_pPMUMISCRegs->IP_CON_REG.ADC_CONTROL & ~(BW_ENABLE_TSADC<<BP_ENABLE_TSADC))|
                                                  (TSADC_DISABLE<<BP_ENABLE_TSADC);
        break;
    default:
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] %s() : Unknown Power IP = %d\n"), _T(__FUNCTION__), eIPPower));
        return FALSE;
    }

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));

    return TRUE;

}

/**
*    @fn    PwrCon_set_block_clock_on(CLK_IP_LIST)
*    @param CLK_IP_LIST     
*    @note  This funciton turn on specific IP's Clock
*           when the system is in NORMAL Mode, not in other Power mode
*/
BOOL PwrCon_set_IP_clock_on(CLK_IP_LIST CLK_GATE_IP)
{
    assert(g_pCMUCLKRegs);

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]++%s(%d)\n"), _T(__FUNCTION__), CLK_GATE_IP));
    

    switch(CLK_GATE_IP)
    {
    case CLK_IP0_CSIS:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_CSIS))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_CSIS);
        break;
    case CLK_IP0_IPC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_IPC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }        
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_IPC);
        break;
    case CLK_IP0_ROTATOR:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_ROTATOR))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }          
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_ROTATOR); 
        break;
    case CLK_IP0_JPEG:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_JPEG))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_JPEG);
        break;
    case CLK_IP0_FIMC2:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_FIMC2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_FIMC2);
        break;
    case CLK_IP0_FIMC1:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_FIMC1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_FIMC1);
        break;
    case CLK_IP0_FIMC0:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_FIMC0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_FIMC0);
        break;
    case CLK_IP0_MFC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_MFC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_MFC);
        break;
#if (S5PV210_EVT>0)        
    case CLK_IP0_G2D:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_G2D))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_G2D);
        break;
#endif        
    case CLK_IP0_G3D:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_G3D))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_G3D);
        break;
    case CLK_IP0_IMEM:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_IMEM))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_IMEM);
        break;
    case CLK_IP0_PDMA1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_PDMA1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_PDMA1);
        break;
    case CLK_IP0_PDMA0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_PDMA0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_PDMA0);
        break;
    case CLK_IP0_MDMA:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_MDMA))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_MDMA);
        break;
    case CLK_IP0_DMC1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_DMC1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_DMC1);
        break;
    case CLK_IP0_DMC0:    
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_DMC0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 |= (CLK_ON << BP_CLK_IP_DMC0);
        break;
    case CLK_IP1_NFCON:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_NFCON))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_NFCON);
        break;
    case CLK_IP1_SROMC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_SROMC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_SROMC);
        break;
    case CLK_IP1_CFCON:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_CFCON))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_CFCON);
        break;
    case CLK_IP1_ONENAND:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_ONENANDXL))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_ONENANDXL);
        break;
    case CLK_IP1_USBHOST:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_USBHOST))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_USBHOST);
        break;
    case CLK_IP1_USBOTG:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_USBOTG))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_USBOTG);
        break;
    case CLK_IP1_HDMI:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_HDMI))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_HDMI);
        break;
    case CLK_IP1_TVENC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_TVENC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_TVENC);
        break;
    case CLK_IP1_MIXER:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_MIXER))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_MIXER);
        break;
    case CLK_IP1_VP:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_VP))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_VP);
        break;
    case CLK_IP1_DSIM:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_DSIM))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_DSIM);
        break;
    case CLK_IP1_MIE:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_MIE))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_MIE);
        break;
    case CLK_IP1_FIMD:    
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_FIMD))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 |= (CLK_ON << BP_CLK_IP_FIMD);
        break;
    case CLK_IP2_TZIC3:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC3))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_TZIC3);
        break;
    case CLK_IP2_TZIC2:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_TZIC2);
        break;
    case CLK_IP2_TZIC1:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_TZIC1);
        break;
    case CLK_IP2_TZIC0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_TZIC0);
        break;
    case CLK_IP2_VIC3:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC3))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_VIC3);
        break;
    case CLK_IP2_VIC2:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_VIC2);
        break;
    case CLK_IP2_VIC1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_VIC1);
        break;
    case CLK_IP2_VIC0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_VIC0);
        break;
    case CLK_IP2_TSI:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TSI))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_TSI);
        break;
    case CLK_IP2_HSMMC3:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC3))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_MMC3);
        break;
    case CLK_IP2_HSMMC2:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_MMC2);
        break;
    case CLK_IP2_HSMMC1:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_MMC1);
        break;
    case CLK_IP2_HSMMC0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_MMC0);
        break;
    case CLK_IP2_SECJTAG:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_SECJTAG))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_SECJTAG);
        break;
    case CLK_IP2_HOSTIF:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_HOSTIF))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_HOSTIF);
        break;
    case CLK_IP2_MODEM:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MODEM))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_MODEM);
        break;
    case CLK_IP2_CORESIGHT:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_CORESIGHT))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_CORESIGHT);
        break;
    case CLK_IP2_SDM:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_SDM))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_SDM);
        break;
    case CLK_IP2_SECSS:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_SECSS))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 |= (CLK_ON << BP_CLK_IP_SECSS);//4
        break;
    case CLK_IP3_PCM2:     //4
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PCM2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_PCM2);
        break;
    case CLK_IP3_PCM1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PCM1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }        
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_PCM1);
        break;
    case CLK_IP3_PCM0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PCM0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_PCM0);
        break;
    case CLK_IP3_SYSCON:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SYSCON))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_SYSCON);
        break;
    case CLK_IP3_GPIO:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_GPIO))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_GPIO);
        break;
    case CLK_IP3_TSADC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_TSADC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_TSADC);
        break;
    case CLK_IP3_PWM:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PWM))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_PWM);
        break;
    case CLK_IP3_WDT:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_WDT))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_WDT);
        break;
    case CLK_IP3_KEYIF:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_KEYIF))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_KEYIF);
        break;
    case CLK_IP3_UART3:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART3))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_UART3);
        break;
    case CLK_IP3_UART2:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_UART2);
        break;
    case CLK_IP3_UART1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_UART1);
        break;
    case CLK_IP3_UART0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_UART0);
        break;
    case CLK_IP3_SYSTIMER:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SYSTIMER))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_SYSTIMER);
        break;
    case CLK_IP3_RTC:     
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_RTC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_RTC);
        break;
    case CLK_IP3_SPI2:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPI2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_SPI2);
        break;
    case CLK_IP3_SPI1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPI1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_SPI1);
        break;
    case CLK_IP3_SPI0:   
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPI0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_SPI0);
        break;
    case CLK_IP3_I2C_HDMI_PHY:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C_HDMI_PHY))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2C_HDMI_PHY);
        break;
    case CLK_IP3_I2C_HDMI_DDC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C_HDMI_DDC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2C_HDMI_DDC);
        break;
    case CLK_IP3_I2C2:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2C2);
        break;
    case CLK_IP3_I2C1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2C1);
        break;
    case CLK_IP3_I2C0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2C0);
        break;
    case CLK_IP3_I2S2: 
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2S2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2S2);
        break;
    case CLK_IP3_I2S1:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2S1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2S1);
        break;
    case CLK_IP3_I2S0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2S0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_I2S0);
        break;
    case CLK_IP3_AC97:     
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_AC97))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_AC97);
        break;
    case CLK_IP3_SPDIF:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPDIF))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 |= (CLK_ON << BP_CLK_IP_SPDIF);
        break;
    case CLK_IP4_TZPC3:  
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC3))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_TZPC3);
        break;
    case CLK_IP4_TZPC2:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC2))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_TZPC2);
        break;
    case CLK_IP4_TZPC1:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC1))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_TZPC1);
        break;
    case CLK_IP4_TZPC0:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC0))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_TZPC0);
        break;
    case CLK_IP4_SECKEY:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_SECKEY))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_SECKEY);
        break;
    case CLK_IP4_IEM_APC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_IEM_APC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_IEM_APC);
        break;
    case CLK_IP4_IEM_IEC:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_IEM_IEC))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_IEM_IEC);
        break;
    case CLK_IP4_CHIP_ID:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_CHIP_ID))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already On\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 |= (CLK_ON << BP_CLK_IP_CHIP_ID);
        break;

    default:
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] %s() : Unknown Clcok IP = %d\n"), _T(__FUNCTION__), CLK_GATE_IP));
        return FALSE;
    }

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));

    return TRUE;
}

/**
*    @fn    PwrCon_set_block_clock_off(CLK_IP_LIST)
*    @param CLK_IP_LIST     
*    @note  This funciton turn off specific IP's Clock
*           when the system is in NORMAL Mode, not in other Power mode
*/
BOOL PwrCon_set_IP_clock_off(CLK_IP_LIST CLK_GATE_IP)
{
    assert(g_pCMUCLKRegs);

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]++%s(%d)\n"), _T(__FUNCTION__), CLK_GATE_IP));
    

    switch(CLK_GATE_IP)
    {
    case CLK_IP0_CSIS:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_CSIS)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_CSIS);
        break;
    case CLK_IP0_IPC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_IPC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }        
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_IPC);
        break;
    case CLK_IP0_ROTATOR:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_ROTATOR)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }          
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_ROTATOR); 
        break;
    case CLK_IP0_JPEG:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_JPEG)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_JPEG);
        break;
    case CLK_IP0_FIMC2:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_FIMC2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_FIMC2);
        break;
    case CLK_IP0_FIMC1:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_FIMC1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_FIMC1);
        break;
    case CLK_IP0_FIMC0:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_FIMC0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_FIMC0);
        break;
    case CLK_IP0_MFC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_MFC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_MFC);
        break;
#if (S5PV210_EVT>0)        
    case CLK_IP0_G2D:
        if(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_G2D))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_G2D);
        break;
#endif         
    case CLK_IP0_G3D:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_G3D)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_G3D);
        break;
    case CLK_IP0_IMEM:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_IMEM)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_IMEM);
        break;
    case CLK_IP0_PDMA1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_PDMA1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_PDMA1);
        break;
    case CLK_IP0_PDMA0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_PDMA0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_PDMA0);
        break;
    case CLK_IP0_MDMA:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_MDMA)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_MDMA);
        break;
    case CLK_IP0_DMC1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_DMC1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_DMC1);
        break;
    case CLK_IP0_DMC0:    
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &(1<<BP_CLK_IP_DMC0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }  
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP0 &= ~(CLK_ON << BP_CLK_IP_DMC0);
        break;
    case CLK_IP1_NFCON:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_NFCON)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_NFCON);
        break;
    case CLK_IP1_SROMC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_SROMC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_SROMC);
        break;
    case CLK_IP1_CFCON:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_CFCON)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_CFCON);
        break;
    case CLK_IP1_ONENAND:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_ONENANDXL)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_ONENANDXL);
        break;
    case CLK_IP1_USBHOST:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_USBHOST)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_USBHOST);
        break;
    case CLK_IP1_USBOTG:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_USBOTG)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_USBOTG);
        break;
    case CLK_IP1_HDMI:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_HDMI)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_HDMI);
        break;
    case CLK_IP1_TVENC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_TVENC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_TVENC);
        break;
    case CLK_IP1_MIXER:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_MIXER)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_MIXER);
        break;
    case CLK_IP1_VP:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_VP)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_VP);
        break;
    case CLK_IP1_DSIM:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_DSIM)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_DSIM);
        break;
    case CLK_IP1_MIE:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_MIE)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_MIE);
        break;
    case CLK_IP1_FIMD:    
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &(1<<BP_CLK_IP_FIMD)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP1 &= ~(CLK_ON << BP_CLK_IP_FIMD);
        break;
    case CLK_IP2_TZIC3:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC3)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_TZIC3);
        break;
    case CLK_IP2_TZIC2:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_TZIC2);
        break;
    case CLK_IP2_TZIC1:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_TZIC1);
        break;
    case CLK_IP2_TZIC0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TZIC0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_TZIC0);
        break;
    case CLK_IP2_VIC3:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC3)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_VIC3);
        break;
    case CLK_IP2_VIC2:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_VIC2);
        break;
    case CLK_IP2_VIC1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_VIC1);
        break;
    case CLK_IP2_VIC0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_VIC0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_VIC0);
        break;
    case CLK_IP2_TSI:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_TSI)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_TSI);
        break;
    case CLK_IP2_HSMMC3:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC3)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_MMC3);
        break;
    case CLK_IP2_HSMMC2:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_MMC2);
        break;
    case CLK_IP2_HSMMC1:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_MMC1);
        break;
    case CLK_IP2_HSMMC0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MMC0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_MMC0);
        break;
    case CLK_IP2_SECJTAG:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_SECJTAG)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_SECJTAG);
        break;
    case CLK_IP2_HOSTIF:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_HOSTIF)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_HOSTIF);
        break;
    case CLK_IP2_MODEM:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_MODEM)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_MODEM);
        break;
    case CLK_IP2_CORESIGHT:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_CORESIGHT)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_CORESIGHT);
        break;
    case CLK_IP2_SDM:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_SDM)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_SDM);
        break;
    case CLK_IP2_SECSS:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &(1<<BP_CLK_IP_SECSS)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP2 &= ~(CLK_ON << BP_CLK_IP_SECSS);//4
        break;
    case CLK_IP3_PCM2:     //4
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PCM2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_PCM2);
        break;
    case CLK_IP3_PCM1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PCM1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }        
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_PCM1);
        break;
    case CLK_IP3_PCM0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PCM0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_PCM0);
        break;
    case CLK_IP3_SYSCON:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SYSCON)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_SYSCON);
        break;
    case CLK_IP3_GPIO:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_GPIO)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_GPIO);
        break;
    case CLK_IP3_TSADC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_TSADC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_TSADC);
        break;
    case CLK_IP3_PWM:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_PWM)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_PWM);
        break;
    case CLK_IP3_WDT:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_WDT)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_WDT);
        break;
    case CLK_IP3_KEYIF:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_KEYIF)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_KEYIF);
        break;
    case CLK_IP3_UART3:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART3)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_UART3);
        break;
    case CLK_IP3_UART2:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_UART2);
        break;
    case CLK_IP3_UART1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_UART1);
        break;
    case CLK_IP3_UART0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_UART0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_UART0);
        break;
    case CLK_IP3_SYSTIMER:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SYSTIMER)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_SYSTIMER);
        break;
    case CLK_IP3_RTC:     
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_RTC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_RTC);
        break;
    case CLK_IP3_SPI2:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPI2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_SPI2);
        break;
    case CLK_IP3_SPI1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPI1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_SPI1);
        break;
    case CLK_IP3_SPI0:   
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPI0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_SPI0);
        break;
    case CLK_IP3_I2C_HDMI_PHY:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C_HDMI_PHY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2C_HDMI_PHY);
        break;
    case CLK_IP3_I2C_HDMI_DDC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C_HDMI_DDC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2C_HDMI_DDC);
        break;
    case CLK_IP3_I2C2:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2C2);
        break;
    case CLK_IP3_I2C1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2C1);
        break;
    case CLK_IP3_I2C0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2C0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2C0);
        break;
    case CLK_IP3_I2S2: 
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2S2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2S2);
        break;
    case CLK_IP3_I2S1:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2S1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2S1);
        break;
    case CLK_IP3_I2S0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_I2S0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_I2S0);
        break;
    case CLK_IP3_AC97:     
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_AC97)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_AC97);
        break;
    case CLK_IP3_SPDIF:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &(1<<BP_CLK_IP_SPDIF)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP3 &= ~(CLK_ON << BP_CLK_IP_SPDIF);
        break;
    case CLK_IP4_TZPC3:  
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC3)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_TZPC3);
        break;
    case CLK_IP4_TZPC2:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC2)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_TZPC2);
        break;
    case CLK_IP4_TZPC1:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC1)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_TZPC1);
        break;
    case CLK_IP4_TZPC0:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_TZPC0)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_TZPC0);
        break;
    case CLK_IP4_SECKEY:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_SECKEY)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_SECKEY);
        break;
    case CLK_IP4_IEM_APC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_IEM_APC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_IEM_APC);
        break;
    case CLK_IP4_IEM_IEC:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_IEM_IEC)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_IEM_IEC);
        break;
    case CLK_IP4_CHIP_ID:
        if(!(g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &(1<<BP_CLK_IP_CHIP_ID)))
        {
            DBGMSG(PWC_INFO,(TEXT("[PWRCON]++%s(%d) already Off\n"), _T(__FUNCTION__), CLK_GATE_IP));
            break;
        }
        g_pCMUCLKRegs->CLK_GATE.CLK_GATE_IP4 &= ~(CLK_ON << BP_CLK_IP_CHIP_ID);
        break;

    default:
        DBGMSG(PWC_INFO,(TEXT("[PWRCON:ERR] %s() : Unknown Clcok IP = %d\n"), _T(__FUNCTION__), CLK_GATE_IP));
        return FALSE;
    }

    DBGMSG(PWC_FUNC,(TEXT("[PWRCON]--%s()\n"), _T(__FUNCTION__)));

    return TRUE;
}
