//
// Copyright (c) Samsung Electronics CO., LTD.  All rights reserved.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    csi_s.cpp

Abstract:       implementation of register controller for MIPI CSI

Functions:

Notes:

--*/

#include <bsp.h>
#include <drvlib_mem.h>
#include <drvmsg.h>
#include <assert.h>
#include "csi_s.h"

#define MODULE_NAME _T("MIPICSI")

#define MIPI_MSG(x)    DBGMSG(CIF_FUNC && CIF_USR3 , x)
#define MIPI_INF(x)    DBGMSG(CIF_INFO && CIF_USR3 , x)
#define MIPI_ERR(x)    ERRMSG(x)

CSI_S::CSI_S(DWORD dwPhyBaseAddress)  : m_pMIPICSI_Reg(NULL)
{
    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));

    // Clock Control Virtual alloc
    m_pMIPICSI_Reg = (MIPICSI_REG *)DrvLib_MapIoSpace(dwPhyBaseAddress, sizeof(MIPICSI_REG), FALSE);
    if (m_pMIPICSI_Reg == NULL)
    {
        MIPI_ERR( (_T("[%s] %s - m_pMIPICSI_Reg Mapping Register Physical Base Address to Virtual Address Failed\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    }

    MIPI_MSG( (_T("[%s]-- %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));

}

CSI_S::~CSI_S()
{
    if (m_pMIPICSI_Reg != NULL)
    {
        DrvLib_UnmapIoSpace((PVOID)m_pMIPICSI_Reg);
        m_pMIPICSI_Reg = NULL;
    }
    if (m_pPKTDATA_Reg != NULL)
    {
        DrvLib_UnmapIoSpace((PVOID)m_pPKTDATA_Reg);
        m_pPKTDATA_Reg = NULL;
    }    
}

// [Reset IP]
void CSI_S::ResetIp(void)
{
    UINT32 uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uCtrlRegVal = m_pMIPICSI_Reg->CSIS_CONTROL;
    
    uCtrlRegVal = uCtrlRegVal | (BM_CSIS_SWRST);
    
    m_pMIPICSI_Reg->CSIS_CONTROL = uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONTROL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONTROL));
}

// [Set number of data lane]
void CSI_S::SetNumOfDataLane(const CSI_S_DATA_LANE eDataLane)
{
    UINT32 uConfigRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uConfigRegVal = m_pMIPICSI_Reg->CSIS_CONFIG;
    uConfigRegVal = uConfigRegVal & ~(BM_CSIS_NUMOFDATLANE);

    switch(eDataLane)
    {
        case DATA_LANE_1:
        case DATA_LANE_2:        
        case DATA_LANE_3:
        case DATA_LANE_4:        
            uConfigRegVal = uConfigRegVal |(eDataLane<<BP_CSIS_NUMOFDATLANE);
            break;
        default:
            assert(0);
            break;
    }
    m_pMIPICSI_Reg->CSIS_CONFIG = uConfigRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONFIG=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONFIG));
}



// [Set WrapperTiming]
void CSI_S::SetWrapperTiming(DWORD dwLintv, DWORD dwSintv, DWORD dwEintv)
{
    UINT32 uConfigRegVal;
    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    uConfigRegVal = m_pMIPICSI_Reg->CSIS_CONFIG;
    uConfigRegVal = uConfigRegVal & ~(BM_CSIS_HSYNC_LINTV);
    uConfigRegVal = uConfigRegVal & ~(BM_CSIS_VSYNC_SINTV);
    uConfigRegVal = uConfigRegVal & ~(BM_CSIS_VSYNC_EINTV);

    uConfigRegVal = uConfigRegVal | (dwLintv << BP_CSIS_HSYNC_LINTV) | (dwSintv << BP_CSIS_VSYNC_SINTV) | (dwEintv << BP_CSIS_VSYNC_EINTV);

    m_pMIPICSI_Reg->CSIS_CONFIG = uConfigRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONFIG=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONFIG));
}


// [Set data Format]
void CSI_S::SetImgFormat(CSI_S_IMG_FMT eFinImgType)
{
    UINT32 uConfigRegVal;
    UINT32 uImgType=0;
    MIPI_MSG( (_T("[%s] %s, eFinImgType= %d\r\n"), MODULE_NAME, _T(__FUNCTION__),eFinImgType));

    uConfigRegVal = m_pMIPICSI_Reg->CSIS_CONFIG;
    uConfigRegVal = uConfigRegVal & ~(BM_CSIS_DATA_FORMAT);

    switch(eFinImgType)
    {
        case FMT_YUV422:
            MIPI_INF((TEXT("FMT_YUV422\r\n")));
            uImgType = 0x1E;
            break;
        case FMT_BAYER_8:        
            MIPI_INF((TEXT("FMT_BAYER_8\r\n")));
            uImgType = 0x2A;
            break;
        case FMT_BAYER_10:
            MIPI_INF((TEXT("FMT_BAYER_10\r\n")));
            uImgType = 0x2B;
            break;
        case FMT_BAYER_12:        
            MIPI_INF((TEXT("FMT_BAYER_12\r\n")));
            uImgType = 0x2C;
            break;
    	case FMT_USER_DEFINED_TYPE1:
            MIPI_INF((TEXT("USER_DEFINED_TYPE1\r\n")));
    		uImgType = 0x30;
            break;

        case FMT_USER_DEFINED_TYPE2:
            MIPI_INF((TEXT("USER_DEFINED_TYPE2\r\n")));
    		uImgType = 0x31;
            break;

        case FMT_USER_DEFINED_TYPE3:
            MIPI_INF((TEXT("USER_DEFINED_TYPE3\r\n")));
    		uImgType = 0x32;
            break;
        case FMT_USER_DEFINED_TYPE4:
            MIPI_INF((TEXT("USER_DEFINED_TYPE4\r\n")));
    		uImgType = 0x33;
            break;

        default:
            assert(0);
            break;
    }
    uConfigRegVal = uConfigRegVal |(uImgType<<BP_CSIS_DATA_FORMAT);

    m_pMIPICSI_Reg->CSIS_CONFIG = uConfigRegVal;
    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONFIG=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONFIG));

}




// [Enable interrupt]
void CSI_S::EnableInterrupt(void)
{
    UINT32 uIntMaskRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uIntMaskRegVal = m_pMIPICSI_Reg->CSIS_INTMSK;
    
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_EVENBEFORE;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_EVENAFTER;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ODDBEFORE;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ODDAFTER;
    
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ERR_SOT_HS;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ERR_CTRL;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ERR_ECC;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ERR_CRC;
    uIntMaskRegVal = uIntMaskRegVal | BM_CSIS_MSK_ERR_ID;
    
    m_pMIPICSI_Reg->CSIS_INTMSK = uIntMaskRegVal;
}

// [Disable interrupt]
void CSI_S::DisableInterrupt(void)
{
    UINT32 uIntMaskRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uIntMaskRegVal = m_pMIPICSI_Reg->CSIS_INTMSK;
    
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_EVENBEFORE);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_EVENAFTER);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ODDBEFORE);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ODDAFTER);
    
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ERR_SOT_HS);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ERR_CTRL);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ERR_ECC);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ERR_CRC);
    uIntMaskRegVal = uIntMaskRegVal & ~(BM_CSIS_MSK_ERR_ID);
    
    m_pMIPICSI_Reg->CSIS_INTMSK = uIntMaskRegVal;
}

// [System on]
void CSI_S::SystemOn(void)
{
    UINT32 uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));

    uCtrlRegVal = m_pMIPICSI_Reg->CSIS_CONTROL;
    uCtrlRegVal = uCtrlRegVal | (BM_CSIS_ENABLE);

    m_pMIPICSI_Reg->CSIS_CONTROL = uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONTROL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONTROL));
}

// [System off]
void CSI_S::SystemOff(void)
{
    UINT32 uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uCtrlRegVal = m_pMIPICSI_Reg->CSIS_CONTROL;    
    uCtrlRegVal = uCtrlRegVal & ~(BM_CSIS_ENABLE);

    m_pMIPICSI_Reg->CSIS_CONTROL = uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONTROL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONTROL));
}

// [Select Wrapper Clock Source]
void CSI_S::SetWClkSrc(CSI_S_WCLK_SRC eClkSrc)
{
    UINT32 uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));

    uCtrlRegVal = m_pMIPICSI_Reg->CSIS_CONTROL;
    uCtrlRegVal = uCtrlRegVal & ~(BM_CSIS_WCLK_SRC);

    switch(eClkSrc)
    {
        case PCLK_SRC:
        case EXTCLK_SRC:        
            uCtrlRegVal = uCtrlRegVal |(eClkSrc<<BP_CSIS_WCLK_SRC);
            break;
        default:
            assert(0);
            break;
    }

    m_pMIPICSI_Reg->CSIS_CONTROL = uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONTROL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONTROL));
}

// [Select Data Align]
void CSI_S::SetDataAlign(CSI_S_DATA_ALIGN eAlign)
{
    UINT32 uCtrlRegVal;


    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));

    uCtrlRegVal = m_pMIPICSI_Reg->CSIS_CONTROL;
    uCtrlRegVal = uCtrlRegVal & ~(BM_CSIS_PARALLEL);

    switch(eAlign)
    {
        case BIT_ALIGN_24:
        case BIT_ALIGN_32:        
            uCtrlRegVal = uCtrlRegVal |(eAlign<<BP_CSIS_PARALLEL);
            break;
        default:
            assert(0);
            break;
    }

    m_pMIPICSI_Reg->CSIS_CONTROL = uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONTROL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONTROL));
}

#define DPHYON_DATA3        (1<<4)
#define DPHYON_DATA2        (1<<3)
#define DPHYON_DATA1        (1<<2)
#define DPHYON_DATA0        (1<<1)
#define DPHYON_CLK          (1<<0)

// [D phy on]
void CSI_S::DphyOn(CSI_S_DATA_LANE eDataLane)
{
    UINT32 uDphyCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uDphyCtrlRegVal = m_pMIPICSI_Reg->CSIS_DPHYCTRL;
  	uDphyCtrlRegVal &= ~(0xF);

    //uDphyCtrlRegVal = uDphyCtrlRegVal | (BM_CSIS_DPHYON);
	if(eDataLane == DATA_LANE_1) {
	    uDphyCtrlRegVal |= (DPHYON_CLK|DPHYON_DATA0);
		//phyCtrlRegVal |= 0x1;
	}
	else if(eDataLane == DATA_LANE_2) {
        uDphyCtrlRegVal |= (DPHYON_CLK|DPHYON_DATA0|DPHYON_DATA1);
		//phyCtrlRegVal |= 0x3;
	}
	else if(eDataLane == DATA_LANE_3) {
        uDphyCtrlRegVal |= (DPHYON_CLK|DPHYON_DATA0|DPHYON_DATA1|DPHYON_DATA2);
		//phyCtrlRegVal |= 0x7;
	}
	else if(eDataLane == DATA_LANE_4) {
        uDphyCtrlRegVal |= (DPHYON_CLK|DPHYON_DATA0|DPHYON_DATA1|DPHYON_DATA2|DPHYON_DATA3);
		//phyCtrlRegVal |= 0xF;
	}


    m_pMIPICSI_Reg->CSIS_DPHYCTRL = uDphyCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->DPHYCTRL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_DPHYCTRL));
}

// [D phy off]
void CSI_S::DphyOff(void)
{
    UINT32 uDphyCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uDphyCtrlRegVal = m_pMIPICSI_Reg->CSIS_DPHYCTRL;
    uDphyCtrlRegVal = uDphyCtrlRegVal & ~(BM_CSIS_DPHYON);
    
    m_pMIPICSI_Reg->CSIS_DPHYCTRL = uDphyCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->DPHYCTRL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_DPHYCTRL));
}

void CSI_S::SetHSsettle(unsigned char eHSsettle)
{
	UINT32 uDphyCtrlRegVal;
    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));

    uDphyCtrlRegVal = m_pMIPICSI_Reg->CSIS_DPHYCTRL;

	uDphyCtrlRegVal &= ~(0xF8000000);
	uDphyCtrlRegVal |= (eHSsettle<<27);

    m_pMIPICSI_Reg->CSIS_DPHYCTRL = uDphyCtrlRegVal;
    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->DPHYCTRL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_DPHYCTRL));
}

void CSI_S::SetImageResolution(DWORD Hor, DWORD Vert)
{
    MIPI_MSG( (_T("[%s] %s width = %d Height = %d\r\n"), MODULE_NAME, _T(__FUNCTION__), Hor, Vert));

    m_pMIPICSI_Reg->CSIS_RESOL = ((Hor << BP_CSIS_HRESOL) | (Vert << BP_CSIS_VRESOL));

    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->RESOL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_DPHYCTRL));
}


// [Get interrupt source]
void CSI_S::GetIntrSrc(CSI_S_INTR_SRC& eSrc)
{
    UINT32 uStatusRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uStatusRegVal = m_pMIPICSI_Reg->CSIS_INTSRC;

    if(IS_ON(uStatusRegVal, BM_CSIS_SRC_EVENBEFORE))
    {
        eSrc = EVEN_BEFORE;
    }
    else if(IS_ON(uStatusRegVal, BM_CSIS_SRC_EVENAFTER))
    {
        eSrc = EVEN_AFTER;
    }
    else if(IS_ON(uStatusRegVal, BM_CSIS_SRC_ODDBEFORE))
    {
        eSrc = ODD_BEFORE;
    }
    else if(IS_ON(uStatusRegVal, BM_CSIS_SRC_ODDAFTER))
    {
        eSrc = ODD_AFTER;
    }
    else if(IS_ON(uStatusRegVal, (0x1<<(BP_CSIS_SRC_ERR_SOT_HS+1))))
    {
        eSrc = ERR_SOT_HS_1;
    }
    else if(IS_ON(uStatusRegVal, (0x1<<(BP_CSIS_SRC_ERR_SOT_HS))))
    {
        eSrc = ERR_SOT_HS_0;
    }    
    else if(IS_ON(uStatusRegVal, (0x1<<(BP_CSIS_SRC_ERR_CTRL+1))))
    {
        eSrc = ERR_CTRL_1;
    }    
    else if(IS_ON(uStatusRegVal, (0x1<<(BP_CSIS_SRC_ERR_CTRL))))
    {
        eSrc = ERR_CTRL_0;
    }    
    else if(IS_ON(uStatusRegVal, BM_CSIS_SRC_ERR_ECC))
    {
        eSrc = ERR_ECC;
    }    
    else if(IS_ON(uStatusRegVal, BM_CSIS_SRC_ERR_CRC))
    {
        eSrc = ERR_CRC;
    }
    else if(IS_ON(uStatusRegVal, BM_CSIS_SRC_ERR_ID))
    {
        eSrc = ERR_ID;
    }    
    else
    {
        assert(0);
    }
}

// [Clear interrupt]
void CSI_S::ClearInterrupt(const CSI_S_INTR_SRC eSrc)
{
    UINT32 uIntSrcRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uIntSrcRegVal = m_pMIPICSI_Reg->CSIS_INTSRC;
    
    switch(eSrc)
    {
        case EVEN_BEFORE:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_EVENBEFORE;
            break;
        case EVEN_AFTER:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_EVENAFTER;
            break;
        case ODD_BEFORE:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_ODDBEFORE;
            break;
        case ODD_AFTER:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_ODDAFTER;
            break;
        case ERR_SOT_HS_1:
            uIntSrcRegVal = uIntSrcRegVal | (0x1<<(BP_CSIS_SRC_ERR_SOT_HS+1));
            break;
        case ERR_SOT_HS_0:
            uIntSrcRegVal = uIntSrcRegVal | (0x1<<(BP_CSIS_SRC_ERR_SOT_HS));
            break;            
            break;        
        case ERR_CTRL_1:
            uIntSrcRegVal = uIntSrcRegVal | (0x1<<(BP_CSIS_SRC_ERR_CTRL+1));
            break;
        case ERR_CTRL_0:
            uIntSrcRegVal = uIntSrcRegVal | (0x1<<(BP_CSIS_SRC_ERR_CTRL));
            break;        
        case ERR_ECC:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_ERR_ECC;
            break;
        case ERR_CRC:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_ERR_CRC;
            break;
        case ERR_ID:
            uIntSrcRegVal = uIntSrcRegVal | BM_CSIS_SRC_ERR_ID;
            break;
        default:
            assert(0);
            break;
    }

    m_pMIPICSI_Reg->CSIS_INTSRC = uIntSrcRegVal;
    
}

// [Stop IP]
void CSI_S::StopIp(void)
{
    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    DisableInterrupt();
    SystemOff();
    DphyOff();
    // After this, MIPI DPhy power block should be off.
}

void CSI_S::UpdateShadow(void)
{
	// User have to assert software reset when camera module is turned off.
    UINT32 uCtrlRegVal;

    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    
    uCtrlRegVal = m_pMIPICSI_Reg->CSIS_CONTROL;
	
	uCtrlRegVal |= (1 << 16);
    m_pMIPICSI_Reg->CSIS_CONTROL = uCtrlRegVal;
    MIPI_MSG( (_T("[%s] %s - m_pMIPICSI_Reg->CONTROL=0x%08x\r\n"), MODULE_NAME, _T(__FUNCTION__), m_pMIPICSI_Reg->CSIS_CONTROL));

}

// [Initialize IP]
void CSI_S::InitIp(CSI_S_DATA_LANE eDataLane,CSI_S_IMG_FMT eImgType, 
                CSI_S_DATA_ALIGN eDataAlign ,CSI_S_WCLK_SRC eWCLKSrc,
                unsigned char eHSsettle,  DWORD eHorizon, DWORD eVertical, 
                DWORD dwLintv, DWORD dwSintv, DWORD dwEintv)
{
    MIPI_MSG( (_T("[%s] %s\r\n"), MODULE_NAME, _T(__FUNCTION__)));
    MIPI_INF((TEXT("DataLane = %d, ImgType = %d, DataAlign = %d, WCLKSRC = %d, \nHSSettle = %d, Width = %d, Height = %d, Lintv = %d, Sintv = %d, Eintv = %d\r\n"),
                       eDataLane,eImgType ,eDataAlign ,eWCLKSrc ,eHSsettle ,eHorizon ,eVertical ,dwLintv ,dwSintv , dwEintv));
    // Before this, MIPI DPhy power block should be on.
    ResetIp();
    Sleep(10);
    SetHSsettle(eHSsettle);
    SetNumOfDataLane(eDataLane);
    SetDataAlign(eDataAlign);
    SetWClkSrc(eWCLKSrc);
    SetImgFormat(eImgType);
    SetImageResolution( eHorizon,  eVertical);
    SetWrapperTiming(dwLintv, dwSintv, dwEintv);

    UpdateShadow();
    
    EnableInterrupt();

    DphyOn(eDataLane);

    SystemOn();
    Sleep(10);
    
} 
void CSI_S::DUMPRegs(void)
{
    MIPI_MSG((TEXT("[%s] 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, 0x%x\r\n"), _T(__FUNCTION__),
        m_pMIPICSI_Reg->CSIS_CONTROL,
        m_pMIPICSI_Reg->CSIS_DPHYCTRL,
        m_pMIPICSI_Reg->CSIS_CONFIG,
        m_pMIPICSI_Reg->CSIS_DPHYSTS,
        m_pMIPICSI_Reg->CSIS_INTMSK,
        m_pMIPICSI_Reg->CSIS_INTSRC,
        m_pMIPICSI_Reg->CSIS_RESOL));

}

BOOL CSI_S::CheckPacketHeaderECC()
{
    return (m_pMIPICSI_Reg->CSIS_INTSRC & BM_CSIS_SRC_ERR_ECC)>>BP_CSIS_SRC_ERR_ECC;
}

BOOL CSI_S::CheckPacketDataCRC()
{
    return (m_pMIPICSI_Reg->CSIS_INTSRC & BM_CSIS_SRC_ERR_CRC)>>BM_CSIS_SRC_ERR_CRC;
}
