/**************************************************************************
**  Copyright 2009 Samsung Electronics Corporation. All Rights Reserved.
**
**
**  Module Name: Menu_PMIC.c
**
**  Module Auther : woochang810.lee@samsung.com
**
**  Module Purpose : Control and Test for PMIC(MAX8698)
**
**  Module History : 09.01.13 by wc Lee - First release
**			     
**
**************************************************************************/

//////////////////////////////////////////////////////////////////////////
// INCLUDES & DEFINES
/////////////////////////////////////////////////////////////////////////
#include <windows.h>
#include <bsp.h>
#include <menu.h>
#include <MAX8698.h>
#include <MAX17040.h>
#include <oal_i2cEmul.h>
#include <Delay.h>

#define REG_ADDR_START			8
#define REG_ADDR_MASK			0xFF00
#define REG_DATA_MASK			0x00FF
#define LOWBIT_DETECT			0x00F0
#define HIGHBIT_MASK				0xF0
#define LOWBIT_MASK				0x0F
#define HIGHBIT_START			4

BYTE g_SelectedDVSReg[3] = {0,0,0};

const char * MAX8698RegisterName[MAX_MAX8698_REG_NUM] =
{
	"ONOFF1     ", 	"ONOFF2     ",  "ADISCHG_EN1", "ADISCHG_EN2", 
	"DVSARM1_2  ", "DVSARM3_4  ", "DVSINT1_2  ", "BUCK3      ",
	"LDO2_3     ", "LDO4       ", "LDO5       ", "LDO6       ",
	"LDO7       ", "LDO8       ", "LDO9       ", "LBCNFG     "
};

const char * MAX8698_ONOFF1_ValueName[8] =
{
	 " ", "ELDO5", "ELDO4", "ELDO3", "ELDO2", "EN3", "EN2", "EN1"
};

const char * MAX8698_ONOFF2_ValueName[8] =
{
	 "ELBCNF1", " ", " ", " ", "ELDO9", "ELDO8", "ELDO7", "ELDO6"
};

const char * MAX8698_ADISCHG_EN1_ValueName[8] =
{
	"LDO6 ADEN", "LDO5 ADEN", "LDO4 ADEN", "LDO3 ADEN",	 "LDO2 ADEN",
	"BUCK3 ADEN", "BUCK2 ADEN", "BUCK1 ADEN"
};

const char * MAX8698_ADISCHG_EN2_ValueName[8] =
{
	 " ", " ", " ", " ", " ", "LDO9 ADEN", "LDO8 ADEN", "LDO7 ADEN"
/*
	"RAMP 1mV/us", "RAMP 2mV/us", "RAMP 3mV/us", "RAMP 4mV/us", "RAMP 5mV/us", "RAMP 6mV/us",
	"RAMP 7mV/us", "RAMP 8mV/us", "RAMP 9mV/us", "RAMP 10mV/us", "RAMP 11mV/us", "RAMP 12mV/us"
*/
};

const char * MAX8698_DVSReg_Name[6] =
{
	 "DVSARM1", "DVSARM2", "DVSARM3", "DVSARM4", "DVSINT1", "DVSINT2"
};

char * MAX8698_DVS_Voltage[16] =
{
	"0.75V", "0.80V", "0.85V","0.90V", "0.95V", "1.00V", "1.05V","1.10V",
	"1.15V", "1.20V", "1.25V","1.30V", "1.35V", "1.40V", "1.45V","1.50V"
};

const char * MAX8698_LBTH_Voltage[8] =
{
	"2.8V", "2.9V", "3.0V", "3.1V", "3.2V", "3.3V", "3.4V", "3.5V"
};

/*
char * MAX8698_DVSARM1234_ValueName[16] =
{
	"DVSARM 0.75V", "DVSARM 0.80V", "DVSARM 0.85V","DVSARM 0.90V",
	"DVSARM 0.95V", "DVSARM 1.00V", "DVSARM 1.05V","DVSARM 1.10V",
	"DVSARM 1.15V", "DVSARM 1.20V", "DVSARM 1.25V","DVSARM 1.30V",
	"DVSARM 1.35V", "DVSARM 1.40V", "DVSARM 1.45V","DVSARM 1.50V"
};

char * MAX8698_DVSINT12_ValueName[16] =
{
	"DVSINT 0.75V", "DVSINT 0.80V", "DVSINT 0.85V","DVSINT 0.90V",
	"DVSINT 0.95V", "DVSINT 1.00V", "DVSINT 1.05V","DVSINT 1.10V",
	"DVSINT 1.15V", "DVSINT 1.20V", "DVSINT 1.25V","DVSINT 1.30V",
	"DVSINT 1.35V", "DVSINT 1.40V", "DVSINT 1.45V","DVSINT 1.50V"
};

char * MAX8698_BUCK3_ValueName[21] =
{
	"BUCK3 1.6V", "BUCK3 1.7V", "BUCK3 1.8V", "BUCK3 1.9V", "BUCK3 2.0V", "BUCK3 2.1V", "BUCK3 2.2V",
	"BUCK3 2.3V", "BUCK3 2.4V", "BUCK3 2.5V", "BUCK3 2.6V", "BUCK3 2.7V", "BUCK3 2.8V", "BUCK3 2.9V",
	"BUCK3 3.0V", "BUCK3 3.1V", "BUCK3 3.2V", "BUCK3 3.3V", "BUCK3 3.4V", "BUCK3 3.5V", "BUCK3 3.6V"
};

char * MAX8698_LDO23_ValueName[16] =
{
	"LDO2_3 0.80V", "LDO2_3 0.85V", "LDO2_3 0.90V", "LDO2_3 0.95V", 
	"LDO2_3 1.00V", "LDO2_3 1.05V", "LDO2_3 1.10V", "LDO2_3 1.15V", 
	"LDO2_3 1.20V", "LDO2_3 1.25V", "LDO2_3 1.30V"
};

char * MAX8698_LDO4_ValueName[21] =
{
	"LDO4 1.6V", "LDO4 1.7V", "LDO4 1.8V", "LDO4 1.9V", "LDO4 2.0V", "LDO4 2.1V", "LDO4 2.2V",
	"LDO4 2.3V", "LDO4 2.4V", "LDO4 2.5V", "LDO4 2.6V", "LDO4 2.7V", "LDO4 2.8V", "LDO4 2.9V",
	"LDO4 3.0V", "LDO4 3.1V", "LDO4 3.2V", "LDO4 3.3V", "LDO4 3.4V", "LDO4 3.5V", "LDO4 3.6V"
};

char * MAX8698_LDO5_ValueName[21] =
{
	"LDO5 1.6V", "LDO5 1.7V", "LDO5 1.8V", "LDO5 1.9V", "LDO5 2.0V", "LDO5 2.1V", "LDO5 2.2V",
	"LDO5 2.3V", "LDO5 2.4V", "LDO5 2.5V", "LDO5 2.6V", "LDO5 2.7V", "LDO5 2.8V", "LDO5 2.9V",
	"LDO5 3.0V", "LDO5 3.1V", "LDO5 3.2V", "LDO5 3.3V", "LDO5 3.4V", "LDO5 3.5V", "LDO5 3.6V"
};

char * MAX8698_LDO6_ValueName[21] =
{
	"LDO6 1.6V", "LDO6 1.7V", "LDO6 1.8V", "LDO6 1.9V", "LDO6 2.0V", "LDO6 2.1V", "LDO6 2.2V",
	"LDO6 2.3V", "LDO6 2.4V", "LDO6 2.5V", "LDO6 2.6V", "LDO6 2.7V", "LDO6 2.8V", "LDO6 2.9V",
	"LDO6 3.0V", "LDO6 3.1V", "LDO6 3.2V", "LDO6 3.3V", "LDO6 3.4V", "LDO6 3.5V", "LDO6 3.6V"
};

char * MAX8698_LDO7_ValueName[21] =
{
	"LDO7 1.6V", "LDO7 1.7V", "LDO7 1.8V", "LDO7 1.9V", "LDO7 2.0V", "LDO7 2.1V", "LDO7 2.2V",
	"LDO7 2.3V", "LDO7 2.4V", "LDO7 2.5V", "LDO7 2.6V", "LDO7 2.7V", "LDO7 2.8V", "LDO7 2.9V",
	"LDO7 3.0V", "LDO7 3.1V", "LDO7 3.2V", "LDO7 3.3V", "LDO7 3.4V", "LDO7 3.5V", "LDO7 3.6V"
};

char * MAX8698_LDO8_ValueName[12] =
{
	"LDO8 3.0V", "LDO8 3.1V", "LDO8 3.2V", "LDO8 3.3V", "LDO8 3.4V", "LDO8 3.5V", "LDO8 3.6V",
	"BKCHR 2.9V", "BKCHR 3.0V", "BKCHR 3.1V", "BKCHR 3.2V", "BKCHR 3.3V"
};

char * MAX8698_LDO9_ValueName[21] =
{
	"LDO9 1.6V", "LDO9 1.7V", "LDO9 1.8V", "LDO9 1.9V", "LDO9 2.0V", "LDO9 2.1V", "LDO9 2.2V",
	"LDO9 2.3V", "LDO9 2.4V", "LDO9 2.5V", "LDO9 2.6V", "LDO9 2.7V", "LDO9 2.8V", "LDO9 2.9V",
	"LDO9 3.0V", "LDO9 3.1V", "LDO9 3.2V", "LDO9 3.3V", "LDO9 3.4V", "LDO9 3.5V", "LDO9 3.6V"
};

const char * MAX8698_LBCNFG_ValueName[] =
{
	"LBHYST 100mV", "LBHYST 200mV", "LBHYST 300mV", "LBHYST 400mV",
	"LBTH 2.8V", "LBTH 2.9V", "LBTH 3.0V", "LBTH 3.1V",
	"LBTH 3.2V", "LBTH 3.3V", "LBTH 3.4V", "LBTH 3.5V"	
};
*/

enum
{
	SAVE_ONOFF1_VALUE_INIT = 0,
	SAVE_ONOFF2_VALUE_INIT,
	SAVE_ONOFF1_VALUE_SLEEP,
	SAVE_ONOFF2_VALUE_SLEEP
};

//////////////////////////////////////////////////////////////////////////
// GLOBAL VARIABELS 
/////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// FUNCTIONS
/////////////////////////////////////////////////////////////////////////
int MenuDisplayMAX8698Reg(void *arg)
{
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	BYTE regValue;
	int i;
	unsigned char szValue[3]={0,};

        I2C_EMUL_Init(PMIC_GROUP, (PVOID)pGPIOReg, (PVOID)g_pBSPArgs, PMIC_SLAVE_ADDR, 200);
		
	EdbgOutputDebugString("===========================\r\n");
	EdbgOutputDebugString("=    NAME     [Addr]:Data =\r\n");
       	EdbgOutputDebugString("=-------------------------=\r\n");

	for(i=0;i<MAX_MAX8698_REG_NUM;i++)
	{
		I2C_EMUL_Read(PMIC_GROUP, i,&regValue);
		Hex2Str(regValue,szValue,2);
		EdbgOutputDebugString("= %s|[0x0%x]:0x%s = \r\n",MAX8698RegisterName[i],i,szValue);		
	}

	EdbgOutputDebugString("===========================\r\n");
	return 0;
}


int MenuVoidFunction(void *arg)
{
	return 0;
}


BYTE GetBitCnt(BYTE num)
{
	int i = 0;

	for(i=-1;i<=7;i++)
	{
		if(num == 0) 		return i;
		else				num = num>>1;
	}
	return i;
}


void MenuTitleMAX8698OnOffReg(void *arg)
{
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	BYTE regAddr, regData, regValue, bitCnt;
	
	unsigned short para = (unsigned short)arg;
	regAddr = (BYTE)((para&REG_ADDR_MASK)>>REG_ADDR_START);
	regData = (BYTE)(para&REG_DATA_MASK);

    I2C_EMUL_Init(PMIC_GROUP, (PVOID)pGPIOReg, (PVOID)g_pBSPArgs,  PMIC_SLAVE_ADDR, 200);		
	I2C_EMUL_Read(PMIC_GROUP, regAddr,&regValue);

	//get bit number
	bitCnt = GetBitCnt(regData);

	switch(regAddr)
	{		
		case ONOFF1:
			EdbgOutputDebugString("On/Off %s : (%s)", MAX8698_ONOFF1_ValueName[bitCnt], (regValue&regData?"On":"Off"));
			break;

		case ONOFF2:
			EdbgOutputDebugString("On/Off %s : (%s)", MAX8698_ONOFF2_ValueName[bitCnt], (regValue&regData?"On":"Off"));
			break;

		case ADISCHG_EN1:
			EdbgOutputDebugString("Switch %s : (%s)", MAX8698_ADISCHG_EN1_ValueName[bitCnt], (regValue&regData?"En":"Dis"));
			break;
			
		case ADISCHG_EN2:
			EdbgOutputDebugString("Switch %s : (%s)",MAX8698_ADISCHG_EN1_ValueName[bitCnt], (regValue&regData?"En":"Dis"));
			break;

		default: 
			EdbgOutputDebugString("Wrong value to display value!!\r\n");			
			break;			

	}	
}


void MenuTitleMAX8698EachReg(void *arg)
{
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	BYTE regValue;
	unsigned char szValue[3]={0,};

	BYTE i = (BYTE)arg;

    I2C_EMUL_Init(PMIC_GROUP, (PVOID)pGPIOReg, (PVOID)g_pBSPArgs,  PMIC_SLAVE_ADDR, 200);	
	I2C_EMUL_Read(PMIC_GROUP, i,&regValue);
	
	Hex2Str(regValue,szValue,2);
	EdbgOutputDebugString("%s Value is [0x%s]",MAX8698RegisterName[i],szValue);
}


void MenuTitleSwitchDVSReg(void *arg)
{
	BYTE i = (BYTE)arg;
	BYTE dvsReg = (i-DVSARM1_2) *2;
	BYTE selectedDVSReg = dvsReg + g_SelectedDVSReg[(i-DVSARM1_2)];

	EdbgOutputDebugString("Switch Register [%s]", MAX8698_DVSReg_Name[selectedDVSReg]);
}


int MenuSwitchDVSReg(void *arg)
{
	BYTE i = (BYTE)arg;	
	BYTE selectedDVSRegValue = g_SelectedDVSReg[(i-DVSARM1_2)];

	if(selectedDVSRegValue == 0)
		g_SelectedDVSReg[(i-DVSARM1_2)] = 1;
	else
		g_SelectedDVSReg[(i-DVSARM1_2)] = 0;

	return 0;
}


void MenuTitleContolDVSReg(void *arg)
{
	BYTE regAddr, regData, dvsReg, selectedReg;	
	unsigned short para = (unsigned short)arg;
	
	regAddr = (BYTE)((para&REG_ADDR_MASK)>>REG_ADDR_START);
	regData = (BYTE)(para&REG_DATA_MASK);
	
	dvsReg = (regAddr-DVSARM1_2) *2;
	selectedReg = dvsReg + g_SelectedDVSReg[(regAddr-DVSARM1_2)];

	if(g_SelectedDVSReg[(regAddr-DVSARM1_2)] == 0)
		EdbgOutputDebugString("Set %s %s [0x_%x]", MAX8698_DVSReg_Name[selectedReg], MAX8698_DVS_Voltage[regData], regData);
	else
		EdbgOutputDebugString("Set %s %s [0x%x_]", MAX8698_DVSReg_Name[selectedReg], MAX8698_DVS_Voltage[regData], regData);
}


int MenuControlMAX8698EachReg(void *arg)
{
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	BYTE regAddr, regData, setData, currentData;
	unsigned char szValue[3]={0,};

	unsigned short para = (unsigned short)arg;
	
	regAddr = (BYTE)((para&REG_ADDR_MASK)>>REG_ADDR_START);
	regData = (BYTE)(para&REG_DATA_MASK);

	//get current reg. value
    I2C_EMUL_Init(PMIC_GROUP, (PVOID)pGPIOReg, (PVOID)g_pBSPArgs,  PMIC_SLAVE_ADDR, 200);			
	I2C_EMUL_Read(PMIC_GROUP, regAddr,&currentData);	
	
	switch(regAddr)
	{		
		case ONOFF1:						
		case ONOFF2:
		case ADISCHG_EN1:
			if(currentData & regData)
				setData = currentData & ~regData;
			else					
				setData = currentData |regData;
			break;
						
		case ADISCHG_EN2:
			if(regData>=LOWBIT_DETECT)
				setData = (currentData& ~LOWBIT_MASK)|(regData&LOWBIT_MASK);
			else
			{	
				if(currentData & regData)
					setData = currentData & ~regData;
				else					
					setData = currentData |regData;
			}
			break;
			
		case DVSARM1_2:
		case DVSARM3_4:
		case DVSINT1_2:
			if(g_SelectedDVSReg[(regAddr-DVSARM1_2)] == 0)
				setData = (currentData& ~LOWBIT_MASK)|(regData&LOWBIT_MASK);
			else				
				setData = (currentData& ~HIGHBIT_MASK)|(regData<<HIGHBIT_START);
			break;
			
		case BUCK3:
		case LDO4:
		case LDO5:
		case LDO6:
		case LDO7:
		case LDO9:
			setData = regData;
			break;
			
		case LDO2_3:
		case LDO8:
			if(regData>=LOWBIT_DETECT)
				setData = (currentData& ~LOWBIT_MASK)|(regData&LOWBIT_MASK);
			else
				setData = (currentData& ~HIGHBIT_MASK)|(regData&HIGHBIT_MASK);
			break;			
		case LBCNFG:
			if(regData>=LOWBIT_DETECT)
				setData = (currentData& ~LBTH_MASK)|(regData&LBTH_MASK);
			else
				setData = (currentData& ~LBHYST_MASK)|(regData&LBHYST_MASK);
			break;
			
		default: 
			EdbgOutputDebugString("Wrong value to control PMIC!!\r\n");			
			break;			
	}
	//set PMIC reg value
    	I2C_EMUL_Write(PMIC_GROUP, regAddr, setData);
	
	return 0;
}


int MenuSaveMAX8698RegToArgs(void *arg)
{
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	BSP_ARGS *pBSPArgs = g_pBSPArgs/*(volatile BSP_ARGS*)OALPAtoVA(IMAGE_SHARE_ARGS_PA_START, FALSE)*/;

	unsigned short para = (unsigned short)arg;
	
	//get current reg. value
    I2C_EMUL_Init(PMIC_GROUP, (PVOID)pGPIOReg, (PVOID)g_pBSPArgs,  PMIC_SLAVE_ADDR, 200);		
	
	switch(para)
	{		
		case SAVE_ONOFF1_VALUE_INIT:
			I2C_EMUL_Read(PMIC_GROUP, ONOFF1,&(pBSPArgs->PMICSetting_Init[0]));	
			EdbgOutputDebugString("PMIC setting of Init (ONOFF1) : 0x%X[0x%X]\r\n", pBSPArgs->PMICSetting_Init[0], &(pBSPArgs->PMICSetting_Init[0]));
			break;
		case SAVE_ONOFF2_VALUE_INIT:
			I2C_EMUL_Read(PMIC_GROUP, ONOFF1,&(pBSPArgs->PMICSetting_Init[1]));	
			EdbgOutputDebugString("PMIC setting of Init (ONOFF2) : 0x%X[0x%X]\r\n", pBSPArgs->PMICSetting_Init[1], &(pBSPArgs->PMICSetting_Init[0]));			
			break;
		case SAVE_ONOFF1_VALUE_SLEEP:
			I2C_EMUL_Read(PMIC_GROUP, ONOFF1,&(pBSPArgs->PMICSetting_Sleep[0]));	
			EdbgOutputDebugString("PMIC setting of Sleep (ONOFF1) : 0x%X[0x%X]\r\n", pBSPArgs->PMICSetting_Sleep[0], &(pBSPArgs->PMICSetting_Sleep[0]));			
			break;
		case SAVE_ONOFF2_VALUE_SLEEP:
			I2C_EMUL_Read(PMIC_GROUP, ONOFF1,&(pBSPArgs->PMICSetting_Sleep[1]));	
			EdbgOutputDebugString("PMIC setting of Sleep (ONOFF2) : 0x%X[0x%X]\r\n", pBSPArgs->PMICSetting_Sleep[1], &(pBSPArgs->PMICSetting_Sleep[1]));
			break;
		default: 
			EdbgOutputDebugString("Wrong value to control PMIC!!\r\n");			
			break;			
	}
	return 0;
}


#if(BSP_HW_ID>=4)
void MenuTitleSelectedSetDVS(void *arg)
{
	BYTE selectedDVSARMValue, selectedDVSINTValue;
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);

	selectedDVSARMValue = (BYTE)(Get_PinData(pGPIOReg, SET_DVS1)|(Get_PinData(pGPIOReg, SET_DVS2)<<1));
	selectedDVSINTValue = (BYTE)Get_PinData(pGPIOReg, SET_DVS3);

	EdbgOutputDebugString("Selected DVS [0x%x|0x%x][%s|%s]", selectedDVSARMValue, selectedDVSINTValue
			, MAX8698_DVSReg_Name[selectedDVSARMValue], MAX8698_DVSReg_Name[selectedDVSINTValue+4]);
}


int MenuControlSetDVS(void *arg)
{
	volatile GPIO_REG *pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	BYTE i = (BYTE)arg;	
	
	switch(i)
	{
		case 0:
			Set_PinData(pGPIOReg, SET_DVS1, FALSE);
			Set_PinData(pGPIOReg, SET_DVS2, FALSE);
			break;
		case 1:
			Set_PinData(pGPIOReg, SET_DVS1, TRUE);
			Set_PinData(pGPIOReg, SET_DVS2, FALSE);
			break;
		case 2:
			Set_PinData(pGPIOReg, SET_DVS1, FALSE);
			Set_PinData(pGPIOReg, SET_DVS2, TRUE);
			break;
		case 3:
			Set_PinData(pGPIOReg, SET_DVS1, TRUE);
			Set_PinData(pGPIOReg, SET_DVS2, TRUE);
			break;
		case 4:
			Set_PinData(pGPIOReg, SET_DVS3, FALSE);
			break;
		case 5:
			Set_PinData(pGPIOReg, SET_DVS3, TRUE);
			break;
	}
	
	return 0;
}

int MenuTestLowBattMon(void *arg)
{
	int InChar, value;

	volatile SYSCON_PM_REG *pSysConPMReg = (volatile SYSCON_PM_REG *)OALPAtoVA(BASE_REG_PA_SYSCON_PM, FALSE);
	volatile GPIO_REG *pGPIOReg = (volatile GPIO_REG*)OALPAtoVA(BASE_REG_PA_IOPORT, FALSE);
	volatile VIC_REG *pINTRReg = (volatile VIC_REG *)OALPAtoVA(BASE_REG_PA_VIC0, FALSE);	
	BSP_ARGS *pBSPArgs = g_pBSPArgs;
	BYTE msb=0, lsb=0, battTH=0;	
	unsigned char szValue[6]={0,};

	//Enable for Interrupt of PowerButton
	pINTRReg->VICINTENABLE |= (1<<PHYIRQ_BATF);
	pSysConPMReg->PWR_CFG |= (1 << 3);

	//get current reg. value
 	I2C_EMUL_Init(PMIC_GROUP, (PVOID)pGPIOReg, (PVOID)g_pBSPArgs,  PMIC_SLAVE_ADDR, 200);
	I2C_EMUL_Read(PMIC_GROUP, LBCNFG, &battTH);
	battTH = (battTH&LBTH_MASK)>>LBTH_START;

	EdbgOutputDebugString("***** Test Low Battery Monitor ('q' = Quit) *****\r\n");
	while(1)
	{
		//Current battery value
		I2C_EMUL_Read(BATTERY_GROUP, VCELL_MSB, &msb);
	   	I2C_EMUL_Read(BATTERY_GROUP, VCELL_LSB, &lsb);
		value = ((msb<<8) | lsb)>>4;
		value = value*125;
		Dec2Str(value,szValue,6);
		EdbgOutputDebugString("Battery Voltage = %c.%c%c%c%c[V], Low Battery Threshold = %s\r\n"
			, szValue[0],szValue[1],szValue[2],szValue[3],szValue[4], MAX8698_LBTH_Voltage[battTH]);

		OALWaitMsec(1000);
		InChar=OEMReadDebugByte();
		if((InChar=='Q')||(InChar=='q'))
			break;
	}	

	//Disable for Interrupt of PowerButton
	pINTRReg->VICINTENCLEAR |= (1<<PHYIRQ_BATF);
	pSysConPMReg->PWR_CFG &= ~(1 << 3);
	return 0;
}
#endif


MENU_ITEM m_menuREG0_Control[] =
{
	{ ' ', "View ONOFF1 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)0},
	{ '0', "On/Off BUCK1", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(EN1|(ONOFF1<<REG_ADDR_START))},
	{ '1', "On/off BUCK2", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(EN2|(ONOFF1<<REG_ADDR_START))},
	{ '2', "On/off BUCK3", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(EN3|(ONOFF1<<REG_ADDR_START))},
	{ '3', "On/off LDO2", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO2|(ONOFF1<<REG_ADDR_START))},
	{ '4', "On/off LDO3", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO3|(ONOFF1<<REG_ADDR_START))},
	{ '5', "On/off LDO4", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO4|(ONOFF1<<REG_ADDR_START))},
	{ '6', "On/off LDO5", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO5|(ONOFF1<<REG_ADDR_START))},
	{ 'I', "Save to the Init Value", NULL, MenuSaveMAX8698RegToArgs, NULL, NULL, NULL, (void *)0},
	{ 'S', "Save to the Sleep Value", NULL, MenuSaveMAX8698RegToArgs, NULL, NULL, NULL, (void *)2},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG1_Control[] =
{
	{ ' ', "View ONOFF2 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)1},
	{ '0', "On/Off LDO6", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO6|(ONOFF2<<REG_ADDR_START))},
	{ '1', "On/off LDO7", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO7|(ONOFF2<<REG_ADDR_START))},
	{ '2', "On/off LDO8", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO8|(ONOFF2<<REG_ADDR_START))},
	{ '3', "On/off LDO9", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELDO9|(ONOFF2<<REG_ADDR_START))},
	{ '4', "On/off LBCNFG", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(ELBCNFG|(ONOFF2<<REG_ADDR_START))},
	{ 'I', "Save to the Init Value", NULL, MenuSaveMAX8698RegToArgs, NULL, NULL, NULL, (void *)1},
	{ 'S', "Save to the Sleep Value", NULL, MenuSaveMAX8698RegToArgs, NULL, NULL, NULL, (void *)3},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};

MENU_ITEM m_menuREG2_Control[] =
{
	{ ' ', "View ADISCHG_EN1 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)2},
	{ '0', "Switch BUCK1 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK1_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '1', "Switch BUCK2 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK2_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '2', "Switch BUCK3 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '3', "Switch LDO2 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO2_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '4', "Switch LDO3 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO3_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '5', "Switch LDO4 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '6', "Switch LDO5 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO5_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '7', "Switch LDO6 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO6_ADEN|(ADISCHG_EN1<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG3_Control[] =
{
	{ ' ', "View ADISCHG_EN2 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)3},
	{ '0', "Set RAMP 1mV/us [0x_0]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_1MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},
	{ '1', "Set RAMP 2mV/us [0x_1]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_2MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},
	{ '2', "Set RAMP 3mV/us [0x_2]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_3MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},
	{ '3', "Set RAMP 4mV/us [0x_3]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_4MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ '4', "Set RAMP 5mV/us [0x_4]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_5MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ '5', "Set RAMP 6mV/us [0x_5]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_6MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ '6', "Set RAMP 7mV/us [0x_6]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_7MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ '7', "Set RAMP 8mV/us [0x_7]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_8MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ '8', "Set RAMP 9mV/us [0x_8]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_9MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ '9', "Set RAMP 10mV/us [0x_9]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_10MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ 'A', "Set RAMP 11mV/us [0x_A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_11MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},	
	{ 'B', "Set RAMP 12mV/us [0x_B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((RAMP_12MV_US|LOWBIT_DETECT)|(ADISCHG_EN2<<REG_ADDR_START))},		
	{ 'C', "Switch LDO7 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO7_ADEN|(ADISCHG_EN2<<REG_ADDR_START))},
	{ 'D', "Switch LDO7 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO8_ADEN|(ADISCHG_EN2<<REG_ADDR_START))},
	{ 'E', "Switch LDO7 ADEN", MenuTitleMAX8698OnOffReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_ADEN|(ADISCHG_EN2<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG4_Control[] =
{
	{ ' ', "View DVSARM1_2 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)4},
	{ 'R', "Switch Register(DVSARM1/DVSARM2)", MenuTitleSwitchDVSReg, MenuSwitchDVSReg, NULL, NULL, NULL, (void *)DVSARM1_2},
	{ '0', "Set DVSARM1_2 0.75V [0x00]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_75|(DVSARM1_2<<REG_ADDR_START))},
	{ '1', "Set DVSARM1_2 0.80V [0x11]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_80|(DVSARM1_2<<REG_ADDR_START))},
	{ '2', "Set DVSARM1_2 0.85V [0x22]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_85|(DVSARM1_2<<REG_ADDR_START))},
	{ '3', "Set DVSARM1_2 0.90V [0x33]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_90|(DVSARM1_2<<REG_ADDR_START))},
	{ '4', "Set DVSARM1_2 0.95V [0x44]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_95|(DVSARM1_2<<REG_ADDR_START))},
	{ '5', "Set DVSARM1_2 1.00V [0x55]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_00|(DVSARM1_2<<REG_ADDR_START))},
	{ '6', "Set DVSARM1_2 1.05V [0x66]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_05|(DVSARM1_2<<REG_ADDR_START))},
	{ '7', "Set DVSARM1_2 1.10V [0x77]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_10|(DVSARM1_2<<REG_ADDR_START))},
	{ '8', "Set DVSARM1_2 1.15V [0x88]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_15|(DVSARM1_2<<REG_ADDR_START))},
	{ '9', "Set DVSARM1_2 1.20V [0x99]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_20|(DVSARM1_2<<REG_ADDR_START))},
	{ 'A', "Set DVSARM1_2 1.25V [0xAA]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_25|(DVSARM1_2<<REG_ADDR_START))},
	{ 'B', "Set DVSARM1_2 1.30V [0xBB]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_30|(DVSARM1_2<<REG_ADDR_START))},
	{ 'C', "Set DVSARM1_2 1.35V [0xCC]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_35|(DVSARM1_2<<REG_ADDR_START))},
	{ 'D', "Set DVSARM1_2 1.40V [0xDD]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_40|(DVSARM1_2<<REG_ADDR_START))},
	{ 'E', "Set DVSARM1_2 1.45V [0xEE]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_45|(DVSARM1_2<<REG_ADDR_START))},
	{ 'F', "Set DVSARM1_2 1.50V [0xFF]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_50|(DVSARM1_2<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG5_Control[] =
{
	{ ' ', "View DVSARM3_4 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)5},
	{ 'R', "Switch Register(DVSARM3/DVSARM4)", MenuTitleSwitchDVSReg, MenuSwitchDVSReg, NULL, NULL, NULL, (void *)DVSARM3_4},
	{ '0', "Set DVSARM3_4 0.75V [0x00]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_75|(DVSARM3_4<<REG_ADDR_START))},
	{ '1', "Set DVSARM3_4 0.80V [0x11]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_80|(DVSARM3_4<<REG_ADDR_START))},
	{ '2', "Set DVSARM3_4 0.85V [0x22]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_85|(DVSARM3_4<<REG_ADDR_START))},
	{ '3', "Set DVSARM3_4 0.90V [0x33]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_90|(DVSARM3_4<<REG_ADDR_START))},
	{ '4', "Set DVSARM3_4 0.95V [0x44]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_0_95|(DVSARM3_4<<REG_ADDR_START))},
	{ '5', "Set DVSARM3_4 1.00V [0x55]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_00|(DVSARM3_4<<REG_ADDR_START))},
	{ '6', "Set DVSARM3_4 1.05V [0x66]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_05|(DVSARM3_4<<REG_ADDR_START))},
	{ '7', "Set DVSARM3_4 1.10V [0x77]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_10|(DVSARM3_4<<REG_ADDR_START))},
	{ '8', "Set DVSARM3_4 1.15V [0x88]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_15|(DVSARM3_4<<REG_ADDR_START))},
	{ '9', "Set DVSARM3_4 1.20V [0x99]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_20|(DVSARM3_4<<REG_ADDR_START))},
	{ 'A', "Set DVSARM3_4 1.25V [0xAA]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_25|(DVSARM3_4<<REG_ADDR_START))},
	{ 'B', "Set DVSARM3_4 1.30V [0xBB]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_30|(DVSARM3_4<<REG_ADDR_START))},
	{ 'C', "Set DVSARM3_4 1.35V [0xCC]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_35|(DVSARM3_4<<REG_ADDR_START))},
	{ 'D', "Set DVSARM3_4 1.40V [0xDD]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_40|(DVSARM3_4<<REG_ADDR_START))},
	{ 'E', "Set DVSARM3_4 1.45V [0xEE]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_45|(DVSARM3_4<<REG_ADDR_START))},
	{ 'F', "Set DVSARM3_4 1.50V [0xFF]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSARM_1_50|(DVSARM3_4<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG6_Control[] =
{
	{ ' ', "View DVSINT1_2 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)6},
	{ 'R', "Switch Register(DVSINT1/DVSINT2)", MenuTitleSwitchDVSReg, MenuSwitchDVSReg, NULL, NULL, NULL, (void *)DVSINT1_2},
	{ '0', "Set DVSINT1_2 0.75V [0x00]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_0_75|(DVSINT1_2<<REG_ADDR_START))},
	{ '1', "Set DVSINT1_2 0.80V [0x11]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_0_80|(DVSINT1_2<<REG_ADDR_START))},
	{ '2', "Set DVSINT1_2 0.85V [0x22]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_0_85|(DVSINT1_2<<REG_ADDR_START))},
	{ '3', "Set DVSINT1_2 0.90V [0x33]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_0_90|(DVSINT1_2<<REG_ADDR_START))},
	{ '4', "Set DVSINT1_2 0.95V [0x44]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_0_95|(DVSINT1_2<<REG_ADDR_START))},
	{ '5', "Set DVSINT1_2 1.00V [0x55]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_00|(DVSINT1_2<<REG_ADDR_START))},
	{ '6', "Set DVSINT1_2 1.05V [0x66]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_05|(DVSINT1_2<<REG_ADDR_START))},
	{ '7', "Set DVSINT1_2 1.10V [0x77]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_10|(DVSINT1_2<<REG_ADDR_START))},
	{ '8', "Set DVSINT1_2 1.15V [0x88]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_15|(DVSINT1_2<<REG_ADDR_START))},
	{ '9', "Set DVSINT1_2 1.20V [0x99]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_20|(DVSINT1_2<<REG_ADDR_START))},
	{ 'A', "Set DVSINT1_2 1.25V [0xAA]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_25|(DVSINT1_2<<REG_ADDR_START))},
	{ 'B', "Set DVSINT1_2 1.30V [0xBB]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_30|(DVSINT1_2<<REG_ADDR_START))},
	{ 'C', "Set DVSINT1_2 1.35V [0xCC]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_35|(DVSINT1_2<<REG_ADDR_START))},
	{ 'D', "Set DVSINT1_2 1.40V [0xDD]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_40|(DVSINT1_2<<REG_ADDR_START))},
	{ 'E', "Set DVSINT1_2 1.45V [0xEE]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_45|(DVSINT1_2<<REG_ADDR_START))},
	{ 'F', "Set DVSINT1_2 1.50V [0xFF]", MenuTitleContolDVSReg, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(DVSINT_1_50|(DVSINT1_2<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG7_Control[] =
{
	{ ' ', "View BUCK3 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)7},
	{ '0', "Set BUCK3 1.6V [0x00]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_1_60|(BUCK3<<REG_ADDR_START))},
	{ '1', "Set BUCK3 1.7V [0x01]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_1_70|(BUCK3<<REG_ADDR_START))},
	{ '2', "Set BUCK3 1.8V [0x02]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_1_80|(BUCK3<<REG_ADDR_START))},
	{ '3', "Set BUCK3 1.9V [0x03]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_1_90|(BUCK3<<REG_ADDR_START))},
	{ '4', "Set BUCK3 2.0V [0x04]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_00|(BUCK3<<REG_ADDR_START))},
	{ '5', "Set BUCK3 2.1V [0x05]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_10|(BUCK3<<REG_ADDR_START))},
	{ '6', "Set BUCK3 2.2V [0x06]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_20|(BUCK3<<REG_ADDR_START))},
	{ '7', "Set BUCK3 2.3V [0x07]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_30|(BUCK3<<REG_ADDR_START))},
	{ '8', "Set BUCK3 2.4V [0x08]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_40|(BUCK3<<REG_ADDR_START))},
	{ '9', "Set BUCK3 2.5V [0x09]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_50|(BUCK3<<REG_ADDR_START))},
	{ 'A', "Set BUCK3 2.6V [0x0A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_60|(BUCK3<<REG_ADDR_START))},
	{ 'B', "Set BUCK3 2.7V [0x0B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_70|(BUCK3<<REG_ADDR_START))},
	{ 'C', "Set BUCK3 2.8V [0x0C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_80|(BUCK3<<REG_ADDR_START))},
	{ 'D', "Set BUCK3 2.9V [0x0D]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_2_90|(BUCK3<<REG_ADDR_START))},
	{ 'E', "Set BUCK3 3.0V [0x0E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_00|(BUCK3<<REG_ADDR_START))},
	{ 'F', "Set BUCK3 3.1V [0x0F]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_10|(BUCK3<<REG_ADDR_START))},
	{ 'G', "Set BUCK3 3.2V [0x10]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_20|(BUCK3<<REG_ADDR_START))},
	{ 'H', "Set BUCK3 3.3V [0x11]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_30|(BUCK3<<REG_ADDR_START))},
	{ 'I', "Set BUCK3 3.4V [0x12]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_40|(BUCK3<<REG_ADDR_START))},
	{ 'J', "Set BUCK3 3.5V [0x13]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_50|(BUCK3<<REG_ADDR_START))},
	{ 'K', "Set BUCK3 3.6V [0x14]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(BUCK3_3_60|(BUCK3<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG8_Control[] =
{
	{ ' ', "View LDO2 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)8},
	{ '0', "Set LDO2 0.80V [0x_0]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_80|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '1', "Set LDO2 0.85V [0x_1]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_85|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '2', "Set LDO2 0.90V [0x_2]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_90|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '3', "Set LDO2 0.95V [0x_3]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_95|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '4', "Set LDO2 1.00V [0x_4]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_00|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '5', "Set LDO2 1.05V [0x_5]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_05|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '6', "Set LDO2 1.10V [0x_6]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_10|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '7', "Set LDO2 1.15V [0x_7]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_15|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '8', "Set LDO2 1.20V [0x_8]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_20|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ '9', "Set LDO2 1.25V [0x_9]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_25|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ 'A', "Set LDO2 1.30V [0x_A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_30|LOWBIT_DETECT)|(LDO2_3<<REG_ADDR_START))},
	{ 'B', "Set LDO3 0.80V [0x0_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_80<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'C', "Set LDO3 0.85V [0x1_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_85<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'D', "Set LDO3 0.90V [0x2_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_90<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'E', "Set LDO3 0.95V [0x3_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_0_95<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'F', "Set LDO3 1.00V [0x4_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_00<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'G', "Set LDO3 1.05V [0x5_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_05<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'H', "Set LDO3 1.10V [0x6_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_10<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'I', "Set LDO3 1.15V [0x7_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_15<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'J', "Set LDO3 1.20V [0x8_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_20<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'K', "Set LDO3 1.25V [0x9_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_25<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ 'L', "Set LDO3 1.30V [0xA_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO2TO3_1_30<<HIGHBIT_START)|(LDO2_3<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
	
};


MENU_ITEM m_menuREG9_Control[] =
{
	{ ' ', "View LDO4 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)9},
	{ '0', "Set LDO4 1.6V [0x00]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_60|(LDO4<<REG_ADDR_START))},
	{ '1', "Set LDO4 1.7V [0x01]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_70|(LDO4<<REG_ADDR_START))},
	{ '2', "Set LDO4 1.8V [0x02]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_80|(LDO4<<REG_ADDR_START))},
	{ '3', "Set LDO4 1.9V [0x03]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_90|(LDO4<<REG_ADDR_START))},
	{ '4', "Set LDO4 2.0V [0x04]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_00|(LDO4<<REG_ADDR_START))},
	{ '5', "Set LDO4 2.1V [0x05]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_10|(LDO4<<REG_ADDR_START))},
	{ '6', "Set LDO4 2.2V [0x06]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_20|(LDO4<<REG_ADDR_START))},
	{ '7', "Set LDO4 2.3V [0x07]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_30|(LDO4<<REG_ADDR_START))},
	{ '8', "Set LDO4 2.4V [0x08]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_40|(LDO4<<REG_ADDR_START))},
	{ '9', "Set LDO4 2.5V [0x09]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_50|(LDO4<<REG_ADDR_START))},
	{ 'A', "Set LDO4 2.6V [0x0A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_60|(LDO4<<REG_ADDR_START))},
	{ 'B', "Set LDO4 2.7V [0x0B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_70|(LDO4<<REG_ADDR_START))},
	{ 'C', "Set LDO4 2.8V [0x0C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_80|(LDO4<<REG_ADDR_START))},
	{ 'D', "Set LDO4 2.9V [0x0D]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_90|(LDO4<<REG_ADDR_START))},
	{ 'E', "Set LDO4 3.0V [0x0E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_00|(LDO4<<REG_ADDR_START))},
	{ 'F', "Set LDO4 3.1V [0x0F]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_10|(LDO4<<REG_ADDR_START))},
	{ 'G', "Set LDO4 3.2V [0x10]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_20|(LDO4<<REG_ADDR_START))},
	{ 'H', "Set LDO4 3.3V [0x11]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_30|(LDO4<<REG_ADDR_START))},
	{ 'I', "Set LDO4 3.4V [0x12]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_40|(LDO4<<REG_ADDR_START))},
	{ 'J', "Set LDO4 3.5V [0x13]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_50|(LDO4<<REG_ADDR_START))},
	{ 'K', "Set LDO4 3.6V [0x14]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_60|(LDO4<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG10_Control[] =
{
	{ ' ', "View LDO5 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)10},
	{ '0', "Set LDO5 1.6V [0x00]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_60|(LDO5<<REG_ADDR_START))},
	{ '1', "Set LDO5 1.7V [0x01]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_70|(LDO5<<REG_ADDR_START))},
	{ '2', "Set LDO5 1.8V [0x02]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_80|(LDO5<<REG_ADDR_START))},
	{ '3', "Set LDO5 1.9V [0x03]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_90|(LDO5<<REG_ADDR_START))},
	{ '4', "Set LDO5 2.0V [0x04]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_00|(LDO5<<REG_ADDR_START))},
	{ '5', "Set LDO5 2.1V [0x05]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_10|(LDO5<<REG_ADDR_START))},
	{ '6', "Set LDO5 2.2V [0x06]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_20|(LDO5<<REG_ADDR_START))},
	{ '7', "Set LDO5 2.3V [0x07]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_30|(LDO5<<REG_ADDR_START))},
	{ '8', "Set LDO5 2.4V [0x08]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_40|(LDO5<<REG_ADDR_START))},
	{ '9', "Set LDO5 2.5V [0x09]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_50|(LDO5<<REG_ADDR_START))},
	{ 'A', "Set LDO5 2.6V [0x0A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_60|(LDO5<<REG_ADDR_START))},
	{ 'B', "Set LDO5 2.7V [0x0B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_70|(LDO5<<REG_ADDR_START))},
	{ 'C', "Set LDO5 2.8V [0x0C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_80|(LDO5<<REG_ADDR_START))},
	{ 'D', "Set LDO5 2.9V [0x0D]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_90|(LDO5<<REG_ADDR_START))},
	{ 'E', "Set LDO5 3.0V [0x0E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_00|(LDO5<<REG_ADDR_START))},
	{ 'F', "Set LDO5 3.1V [0x0F]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_10|(LDO5<<REG_ADDR_START))},
	{ 'G', "Set LDO5 3.2V [0x10]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_20|(LDO5<<REG_ADDR_START))},
	{ 'H', "Set LDO5 3.3V [0x11]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_30|(LDO5<<REG_ADDR_START))},
	{ 'I', "Set LDO5 3.4V [0x12]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_40|(LDO5<<REG_ADDR_START))},
	{ 'J', "Set LDO5 3.5V [0x13]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_50|(LDO5<<REG_ADDR_START))},
	{ 'K', "Set LDO5 3.6V [0x14]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_60|(LDO5<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG11_Control[] =
{
	{ ' ', "View LDO6 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)11},
	{ '0', "Set LDO6 1.6V [0x00]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_60|(LDO6<<REG_ADDR_START))},
	{ '1', "Set LDO6 1.7V [0x01]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_70|(LDO6<<REG_ADDR_START))},
	{ '2', "Set LDO6 1.8V [0x02]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_80|(LDO6<<REG_ADDR_START))},
	{ '3', "Set LDO6 1.9V [0x03]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_90|(LDO6<<REG_ADDR_START))},
	{ '4', "Set LDO6 2.0V [0x04]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_00|(LDO6<<REG_ADDR_START))},
	{ '5', "Set LDO6 2.1V [0x05]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_10|(LDO6<<REG_ADDR_START))},
	{ '6', "Set LDO6 2.2V [0x06]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_20|(LDO6<<REG_ADDR_START))},
	{ '7', "Set LDO6 2.3V [0x07]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_30|(LDO6<<REG_ADDR_START))},
	{ '8', "Set LDO6 2.4V [0x08]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_40|(LDO6<<REG_ADDR_START))},
	{ '9', "Set LDO6 2.5V [0x09]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_50|(LDO6<<REG_ADDR_START))},
	{ 'A', "Set LDO6 2.6V [0x0A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_60|(LDO6<<REG_ADDR_START))},
	{ 'B', "Set LDO6 2.7V [0x0B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_70|(LDO6<<REG_ADDR_START))},
	{ 'C', "Set LDO6 2.8V [0x0C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_80|(LDO6<<REG_ADDR_START))},
	{ 'D', "Set LDO6 2.9V [0x0D]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_90|(LDO6<<REG_ADDR_START))},
	{ 'E', "Set LDO6 3.0V [0x0E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_00|(LDO6<<REG_ADDR_START))},
	{ 'F', "Set LDO6 3.1V [0x0F]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_10|(LDO6<<REG_ADDR_START))},
	{ 'G', "Set LDO6 3.2V [0x10]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_20|(LDO6<<REG_ADDR_START))},
	{ 'H', "Set LDO6 3.3V [0x11]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_30|(LDO6<<REG_ADDR_START))},
	{ 'I', "Set LDO6 3.4V [0x12]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_40|(LDO6<<REG_ADDR_START))},
	{ 'J', "Set LDO6 3.5V [0x13]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_50|(LDO6<<REG_ADDR_START))},
	{ 'K', "Set LDO6 3.6V [0x14]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_60|(LDO6<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG12_Control[] =
{
	{ ' ', "View LDO7 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)12},
	{ '0', "Set LDO7 1.6V [0x00]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_60|(LDO7<<REG_ADDR_START))},
	{ '1', "Set LDO7 1.7V [0x01]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_70|(LDO7<<REG_ADDR_START))},
	{ '2', "Set LDO7 1.8V [0x02]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_80|(LDO7<<REG_ADDR_START))},
	{ '3', "Set LDO7 1.9V [0x03]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_1_90|(LDO7<<REG_ADDR_START))},
	{ '4', "Set LDO7 2.0V [0x04]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_00|(LDO7<<REG_ADDR_START))},
	{ '5', "Set LDO7 2.1V [0x05]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_10|(LDO7<<REG_ADDR_START))},
	{ '6', "Set LDO7 2.2V [0x06]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_20|(LDO7<<REG_ADDR_START))},
	{ '7', "Set LDO7 2.3V [0x07]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_30|(LDO7<<REG_ADDR_START))},
	{ '8', "Set LDO7 2.4V [0x08]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_40|(LDO7<<REG_ADDR_START))},
	{ '9', "Set LDO7 2.5V [0x09]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_50|(LDO7<<REG_ADDR_START))},
	{ 'A', "Set LDO7 2.6V [0x0A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_60|(LDO7<<REG_ADDR_START))},
	{ 'B', "Set LDO7 2.7V [0x0B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_70|(LDO7<<REG_ADDR_START))},
	{ 'C', "Set LDO7 2.8V [0x0C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_80|(LDO7<<REG_ADDR_START))},
	{ 'D', "Set LDO7 2.9V [0x0D]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_2_90|(LDO7<<REG_ADDR_START))},
	{ 'E', "Set LDO7 3.0V [0x0E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_00|(LDO7<<REG_ADDR_START))},
	{ 'F', "Set LDO7 3.1V [0x0F]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_10|(LDO7<<REG_ADDR_START))},
	{ 'G', "Set LDO7 3.2V [0x10]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_20|(LDO7<<REG_ADDR_START))},
	{ 'H', "Set LDO7 3.3V [0x11]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_30|(LDO7<<REG_ADDR_START))},
	{ 'I', "Set LDO7 3.4V [0x12]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_40|(LDO7<<REG_ADDR_START))},
	{ 'J', "Set LDO7 3.5V [0x13]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_50|(LDO7<<REG_ADDR_START))},
	{ 'K', "Set LDO7 3.6V [0x14]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO4TO7_3_60|(LDO7<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG13_Control[] =
{
	{ ' ', "View LDO8 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)13},
	{ '0', "Set BKCHR 2.9V [0x_0]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((BKCHR_2_90|LOWBIT_DETECT)|(LDO8<<REG_ADDR_START))},
	{ '1', "Set BKCHR 3.0V [0x_1]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((BKCHR_3_00|LOWBIT_DETECT)|(LDO8<<REG_ADDR_START))},
	{ '2', "Set BKCHR 3.1V [0x_2]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((BKCHR_3_10|LOWBIT_DETECT)|(LDO8<<REG_ADDR_START))},
	{ '3', "Set BKCHR 3.2V [0x_3]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((BKCHR_3_20|LOWBIT_DETECT)|(LDO8<<REG_ADDR_START))},
	{ '4', "Set BKCHR 3.3V [0x_4]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((BKCHR_3_30|LOWBIT_DETECT)|(LDO8<<REG_ADDR_START))},
	{ 'A', "Set LDO8 3.0V [0x0_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_00<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ 'B', "Set LDO8 3.1V [0x1_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_10<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ 'C', "Set LDO8 3.2V [0x2_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_20<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ 'D', "Set LDO8 3.3V [0x3_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_30<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ 'E', "Set LDO8 3.4V [0x4_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_40<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ 'F', "Set LDO8 3.5V [0x5_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_50<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ 'G', "Set LDO8 3.6V [0x6_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LDO8_3_60<<HIGHBIT_START)|(LDO8<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG14_Control[] =
{
	{ ' ', "View LDO9 Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)14},
	{ '0', "Set LDO9 1.6V [0x00]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_1_60|(LDO9<<REG_ADDR_START))},
	{ '1', "Set LDO9 1.7V [0x01]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_1_70|(LDO9<<REG_ADDR_START))},
	{ '2', "Set LDO9 1.8V [0x02]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_1_80|(LDO9<<REG_ADDR_START))},
	{ '3', "Set LDO9 1.9V [0x03]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_1_90|(LDO9<<REG_ADDR_START))},
	{ '4', "Set LDO9 2.0V [0x04]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_00|(LDO9<<REG_ADDR_START))},
	{ '5', "Set LDO9 2.1V [0x05]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_10|(LDO9<<REG_ADDR_START))},
	{ '6', "Set LDO9 2.2V [0x06]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_20|(LDO9<<REG_ADDR_START))},
	{ '7', "Set LDO9 2.3V [0x07]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_30|(LDO9<<REG_ADDR_START))},
	{ '8', "Set LDO9 2.4V [0x08]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_40|(LDO9<<REG_ADDR_START))},
	{ '9', "Set LDO9 2.5V [0x09]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_50|(LDO9<<REG_ADDR_START))},
	{ 'A', "Set LDO9 2.6V [0x0A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_60|(LDO9<<REG_ADDR_START))},
	{ 'B', "Set LDO9 2.7V [0x0B]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_70|(LDO9<<REG_ADDR_START))},
	{ 'C', "Set LDO9 2.8V [0x0C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_80|(LDO9<<REG_ADDR_START))},
	{ 'D', "Set LDO9 2.9V [0x0D]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_2_90|(LDO9<<REG_ADDR_START))},
	{ 'E', "Set LDO9 3.0V [0x0E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_00|(LDO9<<REG_ADDR_START))},
	{ 'F', "Set LDO9 3.1V [0x0F]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_10|(LDO9<<REG_ADDR_START))},
	{ 'G', "Set LDO9 3.2V [0x10]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_20|(LDO9<<REG_ADDR_START))},
	{ 'H', "Set LDO9 3.3V [0x11]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_30|(LDO9<<REG_ADDR_START))},
	{ 'I', "Set LDO9 3.4V [0x12]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_40|(LDO9<<REG_ADDR_START))},
	{ 'J', "Set LDO9 3.5V [0x13]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_50|(LDO9<<REG_ADDR_START))},
	{ 'K', "Set LDO9 3.6V [0x14]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(LDO9_3_60|(LDO9<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuREG15_Control[] =
{
	{ ' ', "View LBCNFG Value", MenuTitleMAX8698EachReg, MenuVoidFunction, NULL, NULL, NULL, (void *)15},
	{ '0', "Set LBTH 2.8V [0x_0]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_2_80<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '1', "Set LBTH 2.9V [0x_2]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_2_90<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '2', "Set LBTH 3.0V [0x_4]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_3_00<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '3', "Set LBTH 3.1V [0x_6]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_3_10<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '4', "Set LBTH 3.2V [0x_8]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_3_20<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '5', "Set LBTH 3.3V [0x_A]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_3_30<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '6', "Set LBTH 3.4V [0x_C]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_3_40<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ '7', "Set LBTH 3.5V [0x_E]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)(((LBTH_3_50<<LBTH_START)|LOWBIT_DETECT)|(LBCNFG<<REG_ADDR_START))},
	{ 'A', "Set LBHYST 100mV [0x0_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LBHYST_100MV<<LBHYST_START)|(LBCNFG<<REG_ADDR_START))},
	{ 'B', "Set LBHYST 200mV [0x1_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LBHYST_200MV<<LBHYST_START)|(LBCNFG<<REG_ADDR_START))},
	{ 'C', "Set LBHYST 300mV [0x2_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LBHYST_300MV<<LBHYST_START)|(LBCNFG<<REG_ADDR_START))},
	{ 'D', "Set LBHYST 400mV [0x3_]", NULL, MenuControlMAX8698EachReg, NULL, NULL, NULL, (void *)((LBHYST_400MV<<LBHYST_START)|(LBCNFG<<REG_ADDR_START))},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};


MENU_ITEM m_menuPMIC_Control[] =
{
	{ '0', "Control Reg[0x00] (ONOFF1)", NULL, NULL, m_menuREG0_Control, NULL, NULL,NULL},
	{ '1', "Control Reg[0x01] (ONOFF2)", NULL, NULL, m_menuREG1_Control, NULL, NULL,NULL},
	{ '2', "Control Reg[0x02] (ADISCHG_EN1)", NULL, NULL, m_menuREG2_Control, NULL, NULL, NULL},	
	{ '3', "Control Reg[0x03] (ADISCHG_EN2)", NULL, NULL, m_menuREG3_Control, NULL, NULL, NULL},	
	{ '4', "Control Reg[0x04] (DVSARM1_2)", NULL, NULL, m_menuREG4_Control, NULL, NULL, NULL},	
	{ '5', "Control Reg[0x05] (DVSARM3_4)", NULL, NULL, m_menuREG5_Control, NULL, NULL, NULL},	
	{ '6', "Control Reg[0x06] (DVSINT1_2)", NULL, NULL, m_menuREG6_Control, NULL, NULL, NULL},	
	{ '7', "Control Reg[0x07] (BUCK3)", NULL, NULL, m_menuREG7_Control, NULL, NULL, NULL},	
	{ '8', "Control Reg[0x08] (LDO2_3)", NULL, NULL, m_menuREG8_Control, NULL, NULL, NULL},	
	{ '9', "Control Reg[0x09] (LDO4)", NULL, NULL, m_menuREG9_Control, NULL, NULL, NULL},	
	{ 'A', "Control Reg[0x0A] (LDO5)", NULL, NULL, m_menuREG10_Control, NULL, NULL, NULL},	
	{ 'B', "Control Reg[0x0B] (LDO6)", NULL, NULL, m_menuREG11_Control, NULL, NULL, NULL},	
	{ 'C', "Control Reg[0x0C] (LDO7)", NULL, NULL, m_menuREG12_Control, NULL, NULL, NULL},	
	{ 'D', "Control Reg[0x0D] (LDO8)", NULL, NULL, m_menuREG13_Control, NULL, NULL, NULL},	
	{ 'E', "Control Reg[0x0E] (LDO9)", NULL, NULL, m_menuREG14_Control, NULL, NULL, NULL},	
	{ 'F', "Control Reg[0x0F] (LBCNFG)", NULL, NULL, m_menuREG15_Control, NULL, NULL, NULL},	
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};

#if(BSP_HW_ID>=4)
MENU_ITEM m_menuPMIC_SetDVS[] =
{
	{ ' ', "Selected DVS [DVSARM1|DVSINT1]", MenuTitleSelectedSetDVS, NULL, NULL, NULL, NULL,NULL},
	{ '0', "Set DVSARM to DVSARM1", NULL, MenuControlSetDVS, NULL, NULL, NULL,(void*)0},
	{ '1', "Set DVSARM to DVSARM2", NULL, MenuControlSetDVS, NULL, NULL, NULL,(void*)1},
	{ '2', "Set DVSARM to DVSARM3", NULL, MenuControlSetDVS, NULL, NULL, NULL,(void*)2},
	{ '3', "Set DVSARM to DVSARM4", NULL, MenuControlSetDVS, NULL, NULL, NULL,(void*)3},
	{ '4', "Set DVSINT to DVSINT1", NULL, MenuControlSetDVS, NULL, NULL, NULL,(void*)4},
	{ '5', "Set DVSINT to DVSINT2", NULL, MenuControlSetDVS, NULL, NULL, NULL,(void*)5},
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};
#endif

MENU_ITEM m_menuPMICTest[] =
{
	{ '0', "Display MAX8698 Reg. Value", NULL,MenuDisplayMAX8698Reg, NULL, NULL, NULL,NULL},
	{ '1', "Control MAX8698 Reg.", NULL, NULL, m_menuPMIC_Control, NULL, NULL, NULL},	
#if(BSP_HW_ID>=4)
	{ '2', "Change SetDVS", NULL, NULL, m_menuPMIC_SetDVS, NULL, NULL, NULL},	
	{ '3', "Test Low Battery Monitor", NULL, MenuTestLowBattMon, NULL, NULL, NULL, NULL},
#endif
	{ '\0', "", NULL, NULL, NULL, NULL, NULL}
};
