//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  File: interrupt.c
//
//  This file implement major part of interrupt module for S3C3210X SoC.
//
#include <windows.h>
#include <ceddk.h>
#include <Nkintr.h>

#include <oal.h>
#include "soc_cfg.h"
#include "register_map.h"
#include "base_regs.h"
#include "bsp_gpio.h"
#include "bsp_args.h"
#include "image_cfg.h"


#ifdef BSP_USEDVFS
#include <dvfs.h>
#endif
#define VIC_MASK
#define HEART_BEAT_DURATION     (1000)

static USB_HOST_INTR eUSB_HOST = USBHOST_EHCI;


static const UINT32 g_VirIrq2PhyIrq[IRQ_MAX_S5PV210] =
{
//                      Virtual Num  Physical Num
//.....................................................

     PHYIRQ_EINT0,          // 0          // 0
     PHYIRQ_EINT1,          // 1          // 1
     PHYIRQ_EINT2,          // 2          // 2
     PHYIRQ_EINT3,          // 3          // 3
     PHYIRQ_EINT4,          // 4          // 4
     PHYIRQ_EINT5,          // 5          // 5
     PHYIRQ_EINT6,          // 6          // 6
     PHYIRQ_EINT7,          // 7          // 7
     PHYIRQ_EINT8,          // 8          // 8
     PHYIRQ_EINT9,          // 9          // 9
     PHYIRQ_EINT10,         // 10         // 10
     PHYIRQ_EINT11,         // 11         // 11
     PHYIRQ_EINT12,         // 12         // 12
     PHYIRQ_EINT13,         // 13         // 13
     PHYIRQ_EINT14,         // 14         // 14
     PHYIRQ_EINT15,         // 15         // 15
     PHYIRQ_EINT16_31,      // 16         // 16
     PHYIRQ_EINT16_31,      // 17         // 16
     PHYIRQ_EINT16_31,      // 18         // 16
     PHYIRQ_EINT16_31,      // 19         // 16
     PHYIRQ_EINT16_31,      // 20         // 16
     PHYIRQ_EINT16_31,      // 21         // 16
     PHYIRQ_EINT16_31,      // 22         // 16
     PHYIRQ_EINT16_31,      // 23         // 16
     PHYIRQ_EINT16_31,      // 24         // 16
     PHYIRQ_EINT16_31,      // 25         // 16
     PHYIRQ_EINT16_31,      // 26         // 16
     PHYIRQ_EINT16_31,      // 27         // 16
     PHYIRQ_EINT16_31,      // 28         // 16
     PHYIRQ_EINT16_31,      // 29         // 16
     PHYIRQ_EINT16_31,      // 30         // 16
     PHYIRQ_EINT16_31,      // 31         // 16
     PHYIRQ_RESERVED0,      // 32         // 17
     PHYIRQ_MDMA,           // 33         // 18
     PHYIRQ_MDMA,           // 34         // 18
     PHYIRQ_MDMA,           // 35         // 18
     PHYIRQ_MDMA,           // 36         // 18
     PHYIRQ_MDMA,           // 37         // 18
     PHYIRQ_MDMA,           // 38         // 18
     PHYIRQ_MDMA,           // 39         // 18
     PHYIRQ_MDMA,           // 40         // 18
     PHYIRQ_PDMA0,          // 41         // 19
     PHYIRQ_PDMA0,          // 42         // 19
     PHYIRQ_PDMA0,          // 43         // 19
     PHYIRQ_PDMA0,          // 44         // 19
     PHYIRQ_PDMA0,          // 45         // 19
     PHYIRQ_PDMA0,          // 46         // 19
     PHYIRQ_PDMA0,          // 47         // 19
     PHYIRQ_PDMA0,          // 48         // 19
     PHYIRQ_PDMA1,          // 49         // 20
     PHYIRQ_PDMA1,          // 50         // 20
     PHYIRQ_PDMA1,          // 51         // 20
     PHYIRQ_PDMA1,          // 52         // 20
     PHYIRQ_PDMA1,          // 53         // 20
     PHYIRQ_PDMA1,          // 54         // 20
     PHYIRQ_PDMA1,          // 55         // 20
     PHYIRQ_PDMA1,          // 56         // 20
     PHYIRQ_TIMER0,         // 57         // 21
     PHYIRQ_TIMER1,         // 58         // 22
     PHYIRQ_TIMER2,         // 59         // 23
     PHYIRQ_TIMER3,         // 60         // 24
     PHYIRQ_TIMER4,         // 61         // 25
     PHYIRQ_SYSTIMER,       // 62         // 26
     PHYIRQ_WDT,            // 63         // 27
     PHYIRQ_RTC_ALARM,      // 64         // 28
     PHYIRQ_RTC_TIC,        // 65         // 29
     PHYIRQ_GPIO,           // 66         // 30
     PHYIRQ_FIMC3,          // 67         // 31
     PHYIRQ_CORTEX0,        // 68         // 32
     PHYIRQ_CORTEX1,        // 69         // 33
     PHYIRQ_CORTEX2,        // 70         // 34
     PHYIRQ_CORTEX3,        // 71         // 35
     PHYIRQ_CORTEX4,        // 72         // 36
     PHYIRQ_IEMAPC,         // 73         // 37
     PHYIRQ_IEMIEC,         // 74         // 38
     PHYIRQ_RESERVED1,      // 75         // 39 
     PHYIRQ_NFC,            // 76         // 40
     PHYIRQ_CFC,            // 77         // 41
     PHYIRQ_UART0,          // 78         // 42
     PHYIRQ_UART1,          // 79         // 43
     PHYIRQ_UART2,          // 80         // 44
     PHYIRQ_UART3,          // 81         // 45
     PHYIRQ_I2C,            // 82         // 46
     PHYIRQ_SPI0,           // 83         // 47
     PHYIRQ_SPI1,           // 84         // 48
     PHYIRQ_SPI2,           // 85         // 49
     PHYIRQ_AUDIO_SS,       // 86         // 50
     PHYIRQ_I2C_PMIC_LINK,  // 87         // 51
     PHYIRQ_I2C_HDMI_PHY,   // 88         // 52
     PHYIRQ_HSIRX,          // 89         // 53
     PHYIRQ_HSITX,          // 90         // 54
     PHYIRQ_UHOST,          // 91         // 55
     PHYIRQ_OTG,            // 92         // 56
     PHYIRQ_MSM,            // 93         // 57
     PHYIRQ_SDMMC0,         // 94         // 58
     PHYIRQ_SDMMC1,         // 95         // 59
     PHYIRQ_SDMMC2,         // 96         // 60
     PHYIRQ_MIPI_CSI,       // 97         // 61
     PHYIRQ_MIPI_DSI,       // 98         // 62
     PHYIRQ_ONENAND_AUDI,   // 99         // 63
     PHYIRQ_LCD0,           // 100        // 64
     PHYIRQ_LCD1,           // 101        // 65
     PHYIRQ_LCD2,           // 102        // 66
     PHYIRQ_LCD3,           // 103        // 67
     PHYIRQ_ROTATOR,        // 104        // 68
     PHYIRQ_FIMC0,          // 105        // 69
     PHYIRQ_FIMC1,          // 106        // 70
     PHYIRQ_FIMC2,          // 107        // 71
     PHYIRQ_JPEG,           // 108        // 72
     PHYIRQ_2D,             // 109        // 73
     PHYIRQ_3D,             // 110        // 74
     PHYIRQ_MIXER,          // 111        // 75
     PHYIRQ_HDMI,           // 112        // 76
     PHYIRQ_HDMI_I2C,       // 113        // 77
     PHYIRQ_MFC,            // 114        // 78
     PHYIRQ_TVENC,          // 115        // 79
     PHYIRQ_I2S0,           // 116        // 80
     PHYIRQ_I2S1,           // 117        // 81
     PHYIRQ_RESERVED2,      // 118        // 82  
     PHYIRQ_AC97,           // 119        // 83
     PHYIRQ_PCM0,           // 120        // 84
     PHYIRQ_PCM1,           // 121        // 85
     PHYIRQ_SPDIF,          // 122        // 86
     PHYIRQ_ADC,            // 123        // 87
     PHYIRQ_PENDN,          // 124        // 88
     PHYIRQ_KEYPAD,         // 125        // 89
     PHYIRQ_SPG,            // 126        // 90
     PHYIRQ_IntHash_SSS,    // 127        // 91
     PHYIRQ_IntFeedCtrl_SSS,// 128        // 92
     PHYIRQ_RESERVED3,      // 129        // 93
     PHYIRQ_SDM_IRQ,        // 130        // 94
     PHYIRQ_SDM_FIQ,        // 131        // 95
     PHYIRQ_IPC,            // 132        // 96
     PHYIRQ_HOSTIF,         // 133        // 97
     PHYIRQ_SDMMC3,         // 134        // 98
     PHYIRQ_CEC,            // 135        // 99
     PHYIRQ_TSI,            // 136        // 100
     PHYIRQ_MDNIE0,         // 137        // 101
     PHYIRQ_MDNIE1,         // 138        // 102
     PHYIRQ_MDNIE2,         // 139        // 103
     PHYIRQ_MDNIE3,         // 140        // 104
     PHYIRQ_ADC1,           // 141		  // 105
     PHYIRQ_PENDN1,         // 142        // 106
     PHYIRQ_SYSTIMER,       // 143        // 26  for Dummy for DIDLEWKUP     

#ifdef MULTI_USBHOST   
     PHYIRQ_UHOST,           // 144          // 55   for EHCI interrupt
     PHYIRQ_UHOST,           //145            // 55   for EHCI interrupt
#ifdef BSP_USEDVFS
     PHYIRQ_SYSTIMER,       // 146        // 26  for IRQ_DVFS
     PHYIRQ_GPIO            // 147        // 30   for GPJ0_INT5 Sample
/* Add the Physical IRQ ID if needed. */
#else
     PHYIRQ_GPIO             //146        // 30    for GPJ0_INT5 Sample
/* Add the Physical IRQ ID if needed. */     
#endif     

#else
     PHYIRQ_UHOST,           // 144          // 55   for EHCI interrupt
#ifdef BSP_USEDVFS
     PHYIRQ_SYSTIMER,       // 146        // 26  for IRQ_DVFS
     PHYIRQ_GPIO            // 147        // 30   for GPJ0_INT5 Sample
/* Add the Physical IRQ ID if needed. */
#else
     PHYIRQ_GPIO             //146        // 30    for GPJ0_INT5 Sample
/* Add the Physical IRQ ID if needed. */     
#endif     
#endif
};

static const UINT32 g_PhyIrq2VirIrq[PHYIRQ_MAX_S5PV210] =
{
//                      Virtual Num  Physical Num
//.....................................................

     IRQ_EINT0,          // 0          // 0
     IRQ_EINT1,          // 1          // 1
     IRQ_EINT2,          // 2          // 2
     IRQ_EINT3,          // 3          // 3
     IRQ_EINT4,          // 4          // 4
     IRQ_EINT5,          // 5          // 5
     IRQ_EINT6,          // 6          // 6
     IRQ_EINT7,          // 7          // 7
     IRQ_EINT8,          // 8          // 8
     IRQ_EINT9,          // 9          // 9
     IRQ_EINT10,         // 10         // 10
     IRQ_EINT11,         // 11         // 11
     IRQ_EINT12,         // 12         // 12
     IRQ_EINT13,         // 13         // 13
     IRQ_EINT14,         // 14         // 14
     IRQ_EINT15,         // 15         // 15
     IRQ_EINT16,         // 16         // 16
     IRQ_RESERVED0,      // 32         // 17
     IRQ_MDMA_CH0,       // 33         // 18
     IRQ_PDMA0_CH0,      // 41         // 19
     IRQ_PDMA1_CH0,      // 49         // 20
     IRQ_TIMER0,         // 57         // 21
     IRQ_TIMER1,         // 58         // 22
     IRQ_TIMER2,         // 59         // 23
     IRQ_TIMER3,         // 60         // 24
     IRQ_TIMER4,         // 61         // 25
     IRQ_SYSTIMER,       // 62         // 26
     IRQ_WDT,            // 63         // 27
     IRQ_RTC_ALARM,      // 64         // 28
     IRQ_RTC_TIC,        // 65         // 29
     IRQ_GPIO,           // 66         // 30
     IRQ_FIMC3,          // 67         // 31
     IRQ_CORTEX0,        // 68         // 32
     IRQ_CORTEX1,        // 69         // 33
     IRQ_CORTEX2,        // 70         // 34
     IRQ_CORTEX3,        // 71         // 35
     IRQ_CORTEX4,        // 72         // 36
     IRQ_IEMAPC,         // 73         // 37
     IRQ_IEMIEC,         // 74         // 38
     IRQ_RESERVED1,      // 75         // 39  
     IRQ_NFC,            // 76         // 40
     IRQ_CFC,            // 77         // 41
     IRQ_UART0,          // 78         // 42
     IRQ_UART1,          // 79         // 43
     IRQ_UART2,          // 80         // 44
     IRQ_UART3,          // 81         // 45
     IRQ_I2C,            // 82         // 46
     IRQ_SPI0,           // 83         // 47
     IRQ_SPI1,           // 84         // 48
     IRQ_SPI2,           // 85         // 49
     IRQ_AUDIO_SS,       // 86         // 50
     IRQ_I2C_PMIC_LINK,  // 87         // 51
     IRQ_I2C_HDMI_PHY,   // 88         // 52
     IRQ_HSIRX,          // 89         // 53
     IRQ_HSITX,          // 90         // 54
     IRQ_UHOST,          // 91         // 55
     IRQ_OTG,            // 92         // 56
     IRQ_MSM,            // 93         // 57
     IRQ_SDMMC0,         // 94         // 58
     IRQ_SDMMC1,         // 95         // 59
     IRQ_SDMMC2,         // 96         // 60
     IRQ_MIPI_CSI,       // 97         // 61
     IRQ_MIPI_DSI,       // 98         // 62
     IRQ_ONENAND_AUDI,   // 99         // 63
     IRQ_LCD0,           // 100        // 64
     IRQ_LCD1,           // 101        // 65
     IRQ_LCD2,           // 102        // 66
     IRQ_LCD3,           // 103        // 67
     IRQ_ROTATOR,        // 104        // 68
     IRQ_FIMC0,          // 105        // 69
     IRQ_FIMC1,          // 106        // 70
     IRQ_FIMC2,          // 107        // 71
     IRQ_JPEG,           // 108        // 72
     IRQ_2D,             // 109        // 73
     IRQ_3D,             // 110        // 74
     IRQ_MIXER,          // 111        // 75
     IRQ_HDMI,           // 112        // 76
     IRQ_HDMI_I2C,       // 113        // 77
     IRQ_MFC,            // 114        // 78
     IRQ_TVENC,          // 115        // 79
     IRQ_I2S0,           // 116        // 80
     IRQ_I2S1,           // 117        // 81
     IRQ_RESERVED2,      // 118        // 82  
     IRQ_AC97,           // 119        // 83
     IRQ_PCM0,           // 120        // 84
     IRQ_PCM1,           // 121        // 85
     IRQ_SPDIF,          // 122        // 86
     IRQ_ADC,            // 123        // 87
     IRQ_PENDN,          // 124        // 88
     IRQ_KEYPAD,         // 125        // 89
     IRQ_SPG,            // 126        // 90
     IRQ_IntHash_SSS,    // 127        // 91
     IRQ_IntFeedCtrl_SSS,// 128        // 92
     IRQ_RESERVED3,      // 129        // 93
     IRQ_SDM_IRQ,        // 130        // 94
     IRQ_SDM_FIQ,        // 131        // 95
     IRQ_IPC,            // 132        // 96
     IRQ_HOSTIF,         // 133        // 97
     IRQ_SDMMC3,         // 134        // 98
     IRQ_CEC,            // 135        // 99
     IRQ_TSI,            // 136        // 100
     IRQ_MDNIE0,         // 137        // 101
     IRQ_MDNIE1,         // 138        // 102
     IRQ_MDNIE2,         // 139        // 103
     IRQ_MDNIE3,         // 140        // 104
     IRQ_ADC1,           // 141        // 105
     IRQ_PENDN1          // 142        // 106
};

//------------------------------------------------------------------------------
//
//  Globals:  g_pIntrRegs/g_pPortRegs
//
//  The global variables are storing virual address for interrupt and port
//  registers for use in interrupt handling to avoid possible time consumig
//  call to OALPAtoVA function.
//
static volatile VIC_REG *g_pVIC0Reg;
static volatile VIC_REG *g_pVIC1Reg;
static volatile VIC_REG *g_pVIC2Reg;
static volatile VIC_REG *g_pVIC3Reg;
static volatile GPIO_REG *g_pGPIOReg;
static volatile DMAC_REG *g_pMDMACReg;
static volatile DMAC_REG *g_pPDMAC0Reg;
static volatile DMAC_REG *g_pPDMAC1Reg;
static volatile PWM_REG *pPWMRegs;
#ifdef RTC_TICK_ERROR_COMPENSATION
static volatile RTC_REG *g_pRTCReg;
static volatile BSP_ARGS *g_pBSPArg;
static UINT32 RTCCount10S = 0;
#endif
static volatile BSP_ARGS *pBSPArgs;
//  Function pointer to profiling timer ISR routine.
//
PFN_PROFILER_ISR g_pProfilerISR = NULL;


#ifdef MULTI_USBHOST
static volatile USBOHCIHOST_REG *g_pUSBOHCIHOSTReg;
static volatile USBEHCIHOST_REG *g_pUSBEHCIHOSTReg;
#define DOWNSTREAMPORT_CONNECTED    0x1
#define FRAMENUMBEROVERFLOW         0x20
#endif

static void PrepareEINTIntr(void);
static void PrepareDMACIntr(void);
static void InitializeVIC(void);
void VICTableInit(void);                    // Reference by OEMPowerOff() in "Off.c"

#ifndef SYSTIMER_FIXED  // This intercept is used only varibale idle system timer

BOOL DeepIdle_WakeupSrc(void * SharedArgs, DWORD PHYIRQ_ID);
#endif

//------------------------------------------------------------------------------
//
//  Function:  OALIntrInit
//
//  This function initialize interrupt mapping, hardware and call platform
//  specific initialization.
//
BOOL OALIntrInit()
{
    BOOL rc = FALSE;

    OALMSG( 1, (L"+OALInterruptInit\r\n") );

    // First get uncached virtual addresses
    g_pVIC0Reg = (VIC_REG *)OALPAtoVA(BASE_REG_PA_VIC0, FALSE);
    g_pVIC1Reg = (VIC_REG *)OALPAtoVA(BASE_REG_PA_VIC1, FALSE);
    g_pVIC2Reg = (VIC_REG *)OALPAtoVA(BASE_REG_PA_VIC2, FALSE);
    g_pVIC3Reg = (VIC_REG *)OALPAtoVA(BASE_REG_PA_VIC3, FALSE);
    g_pGPIOReg = (GPIO_REG*)OALPAtoVA(BASE_REG_PA_GPIO, FALSE);
    g_pMDMACReg = (DMAC_REG*)OALPAtoVA(BASE_REG_PA_MDMA, FALSE);
    g_pPDMAC0Reg = (DMAC_REG*)OALPAtoVA(BASE_REG_PA_PDMA0, FALSE);
    g_pPDMAC1Reg = (DMAC_REG*)OALPAtoVA(BASE_REG_PA_PDMA1, FALSE);
#ifdef RTC_TICK_ERROR_COMPENSATION
    g_pRTCReg = (RTC_REG*)OALPAtoVA(BASE_REG_PA_RTC, FALSE);
    g_pBSPArg = (BSP_ARGS*)IMAGE_SHARE_ARGS_UA_START; 
#endif
    pPWMRegs = (PWM_REG *)OALPAtoVA(BASE_REG_PA_PWMTIMER,FALSE);

    pBSPArgs =  (BSP_ARGS*)IMAGE_SHARE_ARGS_UA_START;
#ifdef MULTI_USBHOST
    g_pUSBOHCIHOSTReg = (USBOHCIHOST_REG*)OALPAtoVA(BASE_REG_PA_USBHOST_OHCI, FALSE);
    g_pUSBEHCIHOSTReg = (USBEHCIHOST_REG*)OALPAtoVA(BASE_REG_PA_USBHOST_EHCI, FALSE);
#endif
    // Initialize interrupt mapping
    OALIntrMapInit();

    PrepareEINTIntr();

    PrepareDMACIntr();

    // Initialize VIC
    InitializeVIC();

#ifdef OAL_BSP_CALLBACKS
    // Give BSP change to initialize subordinate controller
    rc = BSPIntrInit();
#else
    rc = TRUE;
#endif

    OALMSG(1, (L"-OALInterruptInit(rc = %d)\r\n", rc));
    return rc;
}


//------------------------------------------------------------------------------
//
//  Function:  OALIntrRequestIrqs
//
//  This function returns IRQs for CPU/SoC devices based on their
//  physical address.
//
BOOL OALIntrRequestIrqs(DEVICE_LOCATION *pDevLoc, UINT32 *pCount, UINT32 *pIrqs)
{
    BOOL rc = FALSE;

    OALMSG(OAL_INTR&&OAL_FUNC, (L"+OALIntrRequestIrqs(0x%08x->%d/%d/0x%08x/%d, 0x%08x, 0x%08x)\r\n",
                pDevLoc, pDevLoc->IfcType, pDevLoc->BusNumber, pDevLoc->LogicalLoc,
                pDevLoc->Pin, pCount, pIrqs));

    // This shouldn't happen
    if (*pCount < 1) goto cleanUp;

#ifdef OAL_BSP_CALLBACKS
    rc = BSPIntrRequestIrqs(pDevLoc, pCount, pIrqs);
#endif

cleanUp:

    OALMSG(OAL_INTR&&OAL_FUNC, (L"-OALIntrRequestIrqs(rc = %d)\r\n", rc));
    return rc;
}


//------------------------------------------------------------------------------
//
//  Function:  OALIntrEnableIrqs
//
BOOL OALIntrEnableIrqs(UINT32 count, const UINT32 *pIrqs)
{
    BOOL bRet = TRUE;
    UINT32 VirtualIRQ;
    UINT32 PhysicalIRQ;
    UINT32 i, mask;

    OALMSG(OAL_INTR&&OAL_FUNC, (L"+OALIntrEnableIrqs(%d, 0x%08x)\r\n", count, pIrqs));

    for (i = 0; i < count; i++)
    {
#ifndef OAL_BSP_CALLBACKS
        VirtualIRQ = pIrqs[i];
#else
        // Give BSP chance to enable irq on subordinate interrupt controller
        VirtualIRQ = BSPIntrEnableIrq(pIrqs[i]);
#endif

        if (VirtualIRQ == OAL_INTR_IRQ_UNDEFINED) continue;

        if(VirtualIRQ == IRQ_DIDLEWU) continue;

        // Translate to Physical IRQ
        PhysicalIRQ = g_VirIrq2PhyIrq[VirtualIRQ];

#ifndef ETH_EINT9
        //denis_wei add 2010-08-13 
        if (PhysicalIRQ == PHYIRQ_EINT10)
        {
            g_pGPIOReg->EINTMSK.EXT_INT1_MASK &=~(0x1<<2); //enable eint10
            OALMSG(1, (L"+OALIntrEnableIrqs(%d,)\r\n", PhysicalIRQ));
        }
#else
        //added by terry 20120320
        if (PhysicalIRQ == PHYIRQ_EINT9)
        {
            g_pGPIOReg->EINTMSK.EXT_INT1_MASK &=~(0x1<<1); //enable eint9
            OALMSG(1, (L"+OALIntrEnableIrqs(%d,)\r\n", PhysicalIRQ));
        }		
#endif
        if (PhysicalIRQ < VIC1_BIT_OFFSET)
        {
            // External Interrupt Handling
            mask = (0x1<<(VirtualIRQ - IRQ_EINT0));

            if(VirtualIRQ >= IRQ_EINT0 && VirtualIRQ < IRQ_EINT8)
            {
                mask = mask & 0xff;
                OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT0_PEND,	mask);
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT0_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT8 && VirtualIRQ < IRQ_EINT16)
            {
                mask = (mask>>8) & 0xff;
                OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT1_PEND,	mask);
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT1_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT16 && VirtualIRQ < IRQ_EINT24)
            {
                mask = (mask>>16) & 0xff;
                OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT2_PEND,	mask);
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT2_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT24 && VirtualIRQ <= IRQ_EINT31)
            {
                mask = (mask>>24) & 0xff;
                OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT3_PEND,	mask);
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT3_MASK,	mask);
            }
            else if(PhysicalIRQ == PHYIRQ_GPIO)
            {
                switch(VirtualIRQ)
                {    
                // TODO : Implement the GPIO_INT you need
                // This is Sample Code in case of using GPJ0_5 interrupt
                #if 0
                case IRQ_GPJ0_INT5:    
                    /* To use the several GPIO INT, you should implement code like this. */
				    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPJ0_INT_PEND, (1 << 5));
				    CLRREG32(&g_pGPIOReg->GPIOINTMSK.GPJ0_INT_MASK, (1 << 5));
                    break;
                default:
                    OALMSG(1, (L"Invalid GPIO_INT request(%d)\r\n", VirtualIRQ));
                    break;   
                #endif
                }
            }    
            OUTREG32(&g_pVIC0Reg->VICINTENABLE,  (0x1<<PhysicalIRQ));
        }
        else if (PhysicalIRQ < VIC2_BIT_OFFSET)
        {                     
            OUTREG32(&g_pVIC1Reg->VICINTENABLE, 0x1<<(PhysicalIRQ-VIC1_BIT_OFFSET));
        }
        else if (PhysicalIRQ < VIC3_BIT_OFFSET)
        {
            OUTREG32(&g_pVIC2Reg->VICINTENABLE, 0x1<<(PhysicalIRQ-VIC2_BIT_OFFSET));
        }
        else if (PhysicalIRQ < PHYIRQ_MAX_S5PV210)
        {
            OUTREG32(&g_pVIC3Reg->VICINTENABLE, 0x1<<(PhysicalIRQ-VIC3_BIT_OFFSET));
        }
        else
        {
            bRet = FALSE;
        }
    }

    OALMSG(OAL_INTR&&OAL_FUNC, (L"-OALIntrEnableIrqs(rc = %d)\r\n", bRet));

    return bRet;
}

//------------------------------------------------------------------------------
//
//  Function:  OALIntrDisableIrqs
//
VOID OALIntrDisableIrqs(UINT32 count, const UINT32 *pIrqs)
{
    UINT32 VirtualIRQ;
    UINT32 PhysicalIRQ;
    UINT32 i, mask;

    OALMSG(OAL_INTR&&OAL_FUNC, (L"+OALIntrDisableIrqs(%d, 0x%08x)\r\n", count, pIrqs));

    for (i = 0; i < count; i++)
    {
#ifndef OAL_BSP_CALLBACKS
        VirtualIRQ = pIrqs[i];
#else
        // Give BSP chance to disable irq on subordinate interrupt controller
        VirtualIRQ = BSPIntrDisableIrq(pIrqs[i]);
#endif
        if (VirtualIRQ == OAL_INTR_IRQ_UNDEFINED) continue;
        if(VirtualIRQ == IRQ_DIDLEWU) continue;

        // Translate to Physical IRQ
        PhysicalIRQ = g_VirIrq2PhyIrq[VirtualIRQ];


#ifndef ETH_EINT9 
	 if (PhysicalIRQ == PHYIRQ_EINT10)
        {
		//g_pGPIOReg->WEINT1_MASK |= (0x1<<2);           // Mask EINT10
		//g_pGPIOReg->WEINT1_PEND |= (0x1<<2);           // Clear pending EINT10
		g_pGPIOReg->EINTMSK.EXT_INT1_MASK |=(0x1<<2); //enable eint10
		g_pGPIOReg->EINTPND.EXT_INT1_PEND |=(0x1<<2);  // Clear pending EINT10
		  OALMSG(1, (L"+OALIntrDisableIrqs(%d,)\r\n", PhysicalIRQ));
		
        }
#else //eint9 // added by terry 2012.03.20

	 // ƿͻ(ETH EINT9)
        if (PhysicalIRQ == PHYIRQ_EINT9)
        {
		g_pGPIOReg->EINTMSK.EXT_INT1_MASK |=(0x1<<1); //enable eint9
		g_pGPIOReg->EINTPND.EXT_INT1_PEND |=(0x1<<1);  // Clear pending EINT9
		  OALMSG(1, (L"+OALIntrDisableIrqs(%d,)\r\n", PhysicalIRQ));		
        }
#endif		

        if (PhysicalIRQ < VIC1_BIT_OFFSET)
        {
            // External Interrupt Handling
            mask = (0x1<<(VirtualIRQ - IRQ_EINT0));

            if(VirtualIRQ >= IRQ_EINT0 && VirtualIRQ < IRQ_EINT8)
            {
                mask = mask & 0xff;
                SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT0_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT8 && VirtualIRQ < IRQ_EINT16)
            {
                mask = (mask>>8) & 0xff;
                SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT1_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT16 && VirtualIRQ < IRQ_EINT24)
            {
                mask = (mask>>16) & 0xff;
                SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT2_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT24 && VirtualIRQ <= IRQ_EINT31)
            {
                mask = (mask>>24) & 0xff;
                SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT3_MASK,	mask);
            }
            else if(PhysicalIRQ == PHYIRQ_GPIO)
            {
                switch(VirtualIRQ)
                {    
                // TODO : Implement the GPIO_INT you need
                // This is Sample Code in case of using GPJ0_5 interrupt
                #if 0
                case IRQ_GPJ0_INT5:    // QT602240 Touch Interrupt
                    /* To use the several GPIO INT, you should implement code like this. */
				    SETREG32(&g_pGPIOReg->GPIOINTMSK.GPJ0_INT_MASK, (1 << 5));
                    break;
                default:
                    OALMSG(OAL_INTR&&OAL_ERROR, (L"Invalid GPIO_INT done(%d)\r\n", VirtualIRQ));
                    break;
                #endif    
                }
            }
            else
            {
                OUTREG32(&g_pVIC0Reg->VICINTENCLEAR, (0x1<<PhysicalIRQ));
            }
        }
        else if (PhysicalIRQ < VIC2_BIT_OFFSET)
        {
            OUTREG32(&g_pVIC1Reg->VICINTENCLEAR, 0x1<<(PhysicalIRQ-VIC1_BIT_OFFSET));
        }
        else if (PhysicalIRQ < VIC3_BIT_OFFSET)
        {
            OUTREG32(&g_pVIC2Reg->VICINTENCLEAR, 0x1<<(PhysicalIRQ-VIC2_BIT_OFFSET));
        }        
        else if (PhysicalIRQ < PHYIRQ_MAX_S5PV210)
        {
            OUTREG32(&g_pVIC3Reg->VICINTENCLEAR, 0x1<<(PhysicalIRQ-VIC3_BIT_OFFSET));
        }
        else
        {
            OALMSG(TRUE, (L"[OAL:ERR] OALIntrDisable : Invalid Interrupt ID !!!!\r\n"));        
        }
    }

    OALMSG(OAL_INTR&&OAL_FUNC, (L"-OALIntrDisableIrqs\r\n"));
}

//------------------------------------------------------------------------------
//
//  Function:  OALIntrDoneIrqs
//
VOID OALIntrDoneIrqs(UINT32 count, const UINT32 *pIrqs)
{
    UINT32 VirtualIRQ;
    UINT32 PhysicalIRQ;
    UINT32 i, mask;

    OALMSG(OAL_INTR&&OAL_VERBOSE, (L"+OALIntrDoneIrqs(%d, 0x%08x)\r\n", count, pIrqs));

    for (i = 0; i < count; i++)
    {
#ifndef OAL_BSP_CALLBACKS
        VirtualIRQ = pIrqs[i];
#else
        // Give BSP chance to finish irq on subordinate interrupt controller
        VirtualIRQ = BSPIntrDoneIrq(pIrqs[i]);
#endif

        if(VirtualIRQ == IRQ_DIDLEWU) continue;

        // Translate to Physical IRQ
        PhysicalIRQ = g_VirIrq2PhyIrq[VirtualIRQ];

        		
        // denis_wei add 2010-08-13
        if (PhysicalIRQ == PHYIRQ_EINT10)
        {
		// g_pGPIOReg->WEINT1_MASK &= ~(0x1<<2);       // Mask EINT10 
		g_pGPIOReg->EINTMSK.EXT_INT1_MASK &= ~(0x1<<2);       // UNMask EINT10 
		//OALMSG(1, (L"+OALIntrDoneIrqs(%d,)\r\n", PhysicalIRQ));
		
        }
		
        if (PhysicalIRQ < VIC1_BIT_OFFSET)
        {
           // External Interrupt Handling
            mask = (0x1<<(VirtualIRQ - IRQ_EINT0));

            if(VirtualIRQ >= IRQ_EINT0 && VirtualIRQ < IRQ_EINT8)
            {
                mask = mask & 0xff;
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT0_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT8 && VirtualIRQ < IRQ_EINT16)
            {
                mask = (mask>>8) & 0xff;
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT1_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT16 && VirtualIRQ < IRQ_EINT24)
            {
                mask = (mask>>16) & 0xff;
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT2_MASK,	mask);
            }
            else if(VirtualIRQ >= IRQ_EINT24 && VirtualIRQ <= IRQ_EINT31)
            {
                mask = (mask>>24) & 0xff;
                CLRREG32(&g_pGPIOReg->EINTMSK.EXT_INT3_MASK,	mask);
            }
            else if(PhysicalIRQ == PHYIRQ_GPIO)
            {
                switch(VirtualIRQ)
                {    
                // TODO : Implement the GPIO_INT you need
                // This is Sample Code in case of using GPJ0_5 interrupt
                #if 0
                case IRQ_GPJ0_INT5:    // QT602240 Touch Interrupt
                    /* To use the several GPIO INT, you should implement code like this. */
                    CLRREG32(&g_pGPIOReg->GPIOINTMSK.GPJ0_INT_MASK, (1 << 5));
                    break;
                default:
                    OALMSG(OAL_INTR&&OAL_ERROR, (L"Invalid GPIO_INT done(%d)\r\n", VirtualIRQ));
                    break;
                #endif    
                }
            }
			else if(PhysicalIRQ == PHYIRQ_MDMA)	// MDMA
			{
				g_pMDMACReg->IntEnable |= (1<<(VirtualIRQ - IRQ_MDMA_CH0));
			}
			else if(PhysicalIRQ == PHYIRQ_PDMA0)	// PDMA 0
			{
				g_pPDMAC0Reg->IntEnable |= (1<<(VirtualIRQ - IRQ_PDMA0_CH0));
			}
			else if(PhysicalIRQ == PHYIRQ_PDMA1)	// PDMA 1
			{
				g_pPDMAC1Reg->IntEnable |= (1<<(VirtualIRQ - IRQ_PDMA1_CH0));
			}
            else
            {
                OUTREG32(&g_pVIC0Reg->VICINTENABLE, (0x1<<PhysicalIRQ));
            }
        }
        else if (PhysicalIRQ < VIC2_BIT_OFFSET)
        {
            OUTREG32(&g_pVIC1Reg->VICINTENABLE, 0x1<<(PhysicalIRQ-VIC1_BIT_OFFSET));
        }
        else if (PhysicalIRQ < VIC3_BIT_OFFSET)
        {
            OUTREG32(&g_pVIC2Reg->VICINTENABLE, 0x1<<(PhysicalIRQ-VIC2_BIT_OFFSET));
        }        
        else if (PhysicalIRQ < PHYIRQ_MAX_S5PV210)
        {
            OUTREG32(&g_pVIC3Reg->VICINTENABLE, 0x1<<(PhysicalIRQ-VIC3_BIT_OFFSET));
        }

    }

    OALMSG(OAL_INTR&&OAL_VERBOSE, (L"-OALIntrDoneIrqs\r\n"));
}

//------------------------------------------------------------------------------
//
//  Function:  OEMInterruptHandler
//

BOOL bLPIntercepted = FALSE;

ULONG OEMInterruptHandler(ULONG ra)
{
    UINT32 SysIntr = SYSINTR_NOP;
    UINT32 PhysicalIRQ;
    UINT32 VirtualIRQ;
    UINT32 IntEnVIC0, IntEnVIC1, IntEnVIC2, IntEnVIC3;
#ifdef MULTI_USBHOST
        UINT32 OHCIIntEnable, EHCIIntEnable;
#endif

	


    #if TIMER_LED_DISPLAY
    static DWORD HeartBeatCnt, HeartBeatStat;  //LED4 is used for heart beat
    #endif  // TIMER_LED_DISPLAY

    // Get Pending Interrupt Number
    PhysicalIRQ = g_pVIC0Reg->VICADDRESS;
//	RETAILMSG(1, (TEXT("*PhysicalIRQ = 0x%x\r\n"),PhysicalIRQ));

    // To Avoid low proirity interrupt lost
    IntEnVIC0 = g_pVIC0Reg->VICINTENABLE;
    IntEnVIC1 = g_pVIC1Reg->VICINTENABLE;
    IntEnVIC2 = g_pVIC2Reg->VICINTENABLE;
    IntEnVIC3 = g_pVIC3Reg->VICINTENABLE;
    g_pVIC0Reg->VICINTENCLEAR = 0xffffffff;
    g_pVIC1Reg->VICINTENCLEAR = 0xffffffff;
    g_pVIC2Reg->VICINTENCLEAR = 0xffffffff;
    g_pVIC3Reg->VICINTENCLEAR = 0xffffffff;

    // Translate Physical IRQ to Virtual IRQ (Except Flatten IRQ)
    VirtualIRQ = g_PhyIrq2VirIrq[PhysicalIRQ];

#ifdef BSP_USEDVFS   // For emergency level up
    if(EMGRCHECK_BEFOREHAND(pBSPArgs) && CheckEmergencyIntr(PhysicalIRQ))
    {
    			DVFS_TRANSIT_SET(pBSPArgs,FALSE);
                SysIntr = OALIntrTranslateIrq(IRQ_DVFS);
                goto CleanUp_ISR;
            }
#endif  // BSP_USEDVFS
#ifndef SYSTIMER_FIXED  // This intercept is used only varibale idle system timer
// Intercept Interrupt
    if(pBSPArgs->bAugStateLPMode && bLPIntercepted == FALSE && DeepIdle_WakeupSrc((BSP_ARGS *)pBSPArgs,PhysicalIRQ))
    {
        SysIntr = OALIntrTranslateIrq(IRQ_DIDLEWU);
        RETAILMSG(1,(_T("0x%x %d\r\n"), SysIntr, PhysicalIRQ));
        bLPIntercepted = TRUE;
        goto CleanUp_ISR;
    }

    if(bLPIntercepted && DeepIdle_WakeupSrc((BSP_ARGS *)pBSPArgs,PhysicalIRQ))
    {
        RETAILMSG(1,(_T("Intr Intercepted(%d %d)\r\n"),VirtualIRQ, PhysicalIRQ));
        bLPIntercepted = FALSE;
    }
#endif

    if(PhysicalIRQ == PHYIRQ_SYSTIMER)
    {
        #if TIMER_LED_DISPLAY
        // Heart Beat LED
        if (++HeartBeatCnt >= HEART_BEAT_DURATION)
        {
            HeartBeatCnt = 0;
            HeartBeatStat ^= 1;

            if (HeartBeatStat)
            {
                Set_PinData(g_pGPIOReg, LED9, 1);
            }
            else
            {
                Set_PinData(g_pGPIOReg, LED9, 0);
            }
        }
        #endif

        // Handling System Timer Interrupt
        SysIntr = OALTimerIntrHandler();
        
#ifdef BSP_USEDVFS
        if(SysIntr == SYSINTR_DVFS)
        {
            SysIntr = OALIntrTranslateIrq(IRQ_DVFS);
        }
#endif        
    }

    else if (PhysicalIRQ == PHYIRQ_TIMER2)
    {
        //-------------
        // Profiling Timer
        //-------------


        IntEnVIC0 &= ~(1<<PHYIRQ_TIMER2);

        // TODO: Clear Source Pending
        
        CLR_PWMINT(pPWMRegs->TINT_CSTAT, 2);

        // Handling Profiling Timer Interrupt
        if (g_pProfilerISR)
        {
            SysIntr = g_pProfilerISR(ra);
        }
    }
// System timer can have some accuracy error while system tick is passed in long time period
// Added for compensation error by using RTC tick 
// In this case, RTC tick has to be accurated    
#ifdef RTC_TICK_ERROR_COMPENSATION
    else if (PhysicalIRQ == PHYIRQ_RTC_TIC)
    {
        g_pRTCReg->INTP = (g_pRTCReg->INTP & ~(0x3)) |(1<<0); 
        if(g_pBSPArg->bTickCountInit)
        {
            g_pBSPArg->bTickCountInit = FALSE;
            RTCCount10S = CurMSec;
        }
        else
        {
            RTCCount10S += 10000;
        }

        if(RTCCount10S != CurMSec)
          CurMSec = RTCCount10S;
        //OALMSG(1,(TEXT("RCompare(%d %d)\r\n"),CurMSec, RTCCount10S));
    }    
#endif
    else
    {
#ifdef OAL_ILTIMING
        if (g_oalILT.active)
        {
            g_oalILT.isrTime1 = OALTimerCountsSinceSysTick();
            g_oalILT.savedPC = 0;
            g_oalILT.interrupts++;
        }
#endif
        if(VirtualIRQ >= IRQ_EINT0 && VirtualIRQ <= IRQ_EINT31)
        {
            if(VirtualIRQ >= IRQ_EINT0 && VirtualIRQ < IRQ_EINT8)
            {
                OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT0_PEND, (0x1 << (VirtualIRQ - IRQ_EINT0)));
                SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT0_MASK, (0x1 << (VirtualIRQ - IRQ_EINT0)));
            }
            else if(VirtualIRQ >= IRQ_EINT8 && VirtualIRQ < IRQ_EINT16)
            {
                if( IRQ_EINT10== VirtualIRQ) 
            	{
			//	RETAILMSG(1,(_T("[intr.c] [DM9000] Intr Intercepted(%d %d)\r\n"),VirtualIRQ, PhysicalIRQ));
            	}
                OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT1_PEND, (0x1 << (VirtualIRQ - IRQ_EINT8)));
                SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT1_MASK, (0x1 << (VirtualIRQ - IRQ_EINT8)));
            }
            else if(VirtualIRQ == IRQ_EINT16)
            {
                UINT32 Idx, EintPnd2, EintPnd3;

                EintPnd2 = 0xff & (g_pGPIOReg->EINTPND.EXT_INT2_PEND & ~(g_pGPIOReg->EINTMSK.EXT_INT2_MASK));
                EintPnd3 = 0xff & (g_pGPIOReg->EINTPND.EXT_INT3_PEND & ~(g_pGPIOReg->EINTMSK.EXT_INT3_MASK));

                EintPnd2 = EintPnd2 | (EintPnd3 << 8);

                for(Idx = 0 ; Idx<32 ; Idx++)
                {
                    if(EintPnd2 & (0x1 << Idx)) break;
                }

                VirtualIRQ = Idx + IRQ_EINT16;

                if(VirtualIRQ >= IRQ_EINT16 && VirtualIRQ < IRQ_EINT24)
                {
                    OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT2_PEND, (0x1 << (VirtualIRQ - IRQ_EINT16)));
                    SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT2_MASK, (0x1 << (VirtualIRQ - IRQ_EINT16)));
                }
                else if(VirtualIRQ >= IRQ_EINT24 && VirtualIRQ <= IRQ_EINT31)
                {
                    OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT3_PEND, (0x1 << (VirtualIRQ - IRQ_EINT24)));
                    SETREG32(&g_pGPIOReg->EINTMSK.EXT_INT3_MASK, (0x1 << (VirtualIRQ - IRQ_EINT24)));
                }
            }
            else
            {
                OALMSG(TRUE, (L"[OAL:ERR] OEMInterrupthanlder : Invalid EINT ID !!!!\r\n"));
            }
        }
        else if(PhysicalIRQ == PHYIRQ_GPIO)
        {
            // TODO : should implement GPIO_INT , if you needed.
            // This is Sample Code in case of using GPJ0_5 interrupt
            DWORD CurrentServiceGPIOINT;
            DWORD SVC_Group_Num, SVC_Num, SVC_Pend;
            CurrentServiceGPIOINT = INREG32(&g_pGPIOReg->GPIOINT_SERVICE);
            SVC_Group_Num = (CurrentServiceGPIOINT & (0x1f<<3)) >> 3;
            SVC_Num = (CurrentServiceGPIOINT & 0x7);
            SVC_Pend = INREG32(&g_pGPIOReg->GPIOINT_SERVICE_PEND);  

            switch(SVC_Group_Num)
            {
            #if 0
            case SVC_GPJ0_INT:
                /* To use the several GPIO INT, you should implement code like this.*/

                OUTREG32(&g_pGPIOReg->GPIOINTPND.GPJ0_INT_PEND, (0x1<<SVC_Num));
                SETREG32(&g_pGPIOReg->GPIOINTMSK.GPJ0_INT_MASK,	(0x1<<SVC_Num));

                if(SVC_Num == 5)
                {    
				    VirtualIRQ = IRQ_GPJ0_INT5;
                }// else if -> Add Another GPJ0 Interrupt if there is another GPJ0 interrupt
                else  
                {
                    OALMSG(OAL_INTR&&OAL_ERROR,(L"Not Registered GPJ Interrupt(%d)\r\n",SVC_Num));
                }
                
                    
                break;

            // TODO : should implement the another GPIO_INT group, if you needed.

            default:
                OALMSG(OAL_INTR&&OAL_ERROR, (L"Invalid GPIO_INT Handler(%d)\r\n", SVC_Group_Num));
                break;
            #endif    
            }
			
        }        
        else if (PhysicalIRQ == PHYIRQ_MDMA)
        {
            DWORD dwIntStatus;

            // By JOHNLAY. 2010.01.21 IntEnVIC0 &= ~(1<<PHYIRQ_MDMA);

            dwIntStatus = g_pMDMACReg->IntStatus;

			// JOHNLAY. 2010.01.21 
			// DMAC Interrupt Handler is changed. 
			// Previously, all the DMA CH Intr is blocked by VIC until InterruptDone is called in IST. 
			// This is not appropriate architecture, because the time from ISR to IST is too long, 
			// thus some other DMA CH Intr may occur within this long time. 

			// So the architecture of ISR is changed with similar to EINT. 

            if (dwIntStatus & 0x01) VirtualIRQ = IRQ_MDMA_CH0;        // channel 0
            else if (dwIntStatus & 0x02) VirtualIRQ = IRQ_MDMA_CH1;    // channel 1
            else if (dwIntStatus & 0x04) VirtualIRQ = IRQ_MDMA_CH2;    // channel 2
            else if (dwIntStatus & 0x08) VirtualIRQ = IRQ_MDMA_CH3;    // channel 3
            else if (dwIntStatus & 0x10) VirtualIRQ = IRQ_MDMA_CH4;    // channel 4
            else if (dwIntStatus & 0x20) VirtualIRQ = IRQ_MDMA_CH5;    // channel 5
            else if (dwIntStatus & 0x40) VirtualIRQ = IRQ_MDMA_CH6;    // channel 6
            else if (dwIntStatus & 0x80) VirtualIRQ = IRQ_MDMA_CH7;    // channel 7
            else 
            {
				OALMSG(OAL_INTR&&OAL_ERROR, (L"Invalid MDMA Interrupt\r\n"));
				goto CleanUp_ISR;
            }

			// Clear DMAC Interrupt
			g_pMDMACReg->IntClear = 1<<(VirtualIRQ - IRQ_MDMA_CH0);
			// Disable DMAC Interrupt
			g_pMDMACReg->IntEnable &= ~(1<<(VirtualIRQ - IRQ_MDMA_CH0));
        }
        else if (PhysicalIRQ == PHYIRQ_PDMA0)
        {
            DWORD dwIntStatus;

            // By JOHNLAY. 2010.01.21 IntEnVIC0 &= ~(1<<PHYIRQ_PDMA0);

            dwIntStatus = (g_pPDMAC0Reg->IntStatus) ;

            if (dwIntStatus & 0x01) VirtualIRQ = IRQ_PDMA0_CH0;        // channel 0
            else if (dwIntStatus & 0x02) VirtualIRQ = IRQ_PDMA0_CH1;    // channel 1
            else if (dwIntStatus & 0x04) VirtualIRQ = IRQ_PDMA0_CH2;    // channel 2
            else if (dwIntStatus & 0x08) VirtualIRQ = IRQ_PDMA0_CH3;    // channel 3
            else if (dwIntStatus & 0x10) VirtualIRQ = IRQ_PDMA0_CH4;    // channel 4
            else if (dwIntStatus & 0x20) VirtualIRQ = IRQ_PDMA0_CH5;    // channel 5
            else if (dwIntStatus & 0x40) VirtualIRQ = IRQ_PDMA0_CH6;    // channel 6
            else if (dwIntStatus & 0x80) VirtualIRQ = IRQ_PDMA0_CH7;    // channel 7
            else 
            {
				OALMSG(OAL_INTR&&OAL_ERROR, (L"Invalid PDMA0 Interrupt\r\n"));
				goto CleanUp_ISR;
            }

			// Clear DMAC Interrupt
			g_pPDMAC0Reg->IntClear = 1<<(VirtualIRQ - IRQ_PDMA0_CH0);
			// Disable DMAC Interrupt
			g_pPDMAC0Reg->IntEnable &= ~(1<<(VirtualIRQ - IRQ_PDMA0_CH0));
        }
        else if (PhysicalIRQ == PHYIRQ_PDMA1)
        {
            DWORD dwIntStatus;


            // By JOHNLAY. 2010.01.21 IntEnVIC0 &= ~(1<<PHYIRQ_PDMA1);

            dwIntStatus = g_pPDMAC1Reg->IntStatus;

            if (dwIntStatus & 0x01) VirtualIRQ = IRQ_PDMA1_CH0;        // channel 0
            else if (dwIntStatus & 0x02) VirtualIRQ = IRQ_PDMA1_CH1;    // channel 1
            else if (dwIntStatus & 0x04) VirtualIRQ = IRQ_PDMA1_CH2;    // channel 2
            else if (dwIntStatus & 0x08) VirtualIRQ = IRQ_PDMA1_CH3;    // channel 3
            else if (dwIntStatus & 0x10) VirtualIRQ = IRQ_PDMA1_CH4;    // channel 4
            else if (dwIntStatus & 0x20) VirtualIRQ = IRQ_PDMA1_CH5;    // channel 5
            else if (dwIntStatus & 0x40) VirtualIRQ = IRQ_PDMA1_CH6;    // channel 6
            else if (dwIntStatus & 0x80) VirtualIRQ = IRQ_PDMA1_CH7;    // channel 7
            else 
            {
				OALMSG(OAL_INTR&&OAL_ERROR, (L"Invalid PDMA1 Interrupt\r\n"));
				goto CleanUp_ISR;
            }

			// Clear DMAC Interrupt
			g_pPDMAC1Reg->IntClear = 1<<(VirtualIRQ - IRQ_PDMA1_CH0);
			// Disable DMAC Interrupt
			g_pPDMAC1Reg->IntEnable &= ~(1<<(VirtualIRQ - IRQ_PDMA1_CH0));
        }

        else if (PhysicalIRQ < VIC1_BIT_OFFSET)  
        {
            IntEnVIC0 &= ~(1<<PhysicalIRQ);
        }
        else if (PhysicalIRQ < VIC2_BIT_OFFSET)
        {
#ifdef MULTI_USBHOST
            if (PhysicalIRQ == PHYIRQ_UHOST)
            {
                OHCIIntEnable = g_pUSBOHCIHOSTReg->HcInterruptEnable;
                EHCIIntEnable = g_pUSBEHCIHOSTReg->USBINTR;

                  if(g_pUSBEHCIHOSTReg->USBSTS & EHCIIntEnable)
                    {
                          VirtualIRQ = IRQ_EHCI ;
                         if(g_pUSBEHCIHOSTReg->PORTSTASCTRL & DOWNSTREAMPORT_CONNECTED)
                             RETAILMSG(0,(_T("IRQ_EHCI : Conneted Device\r\n")));
                         else
                             RETAILMSG(0,(_T("IRQ_EHCI : Not Conneted Device\r\n")));

                      }
                  else if(g_pUSBOHCIHOSTReg->HcInterruptStatus & OHCIIntEnable)
                    {
                         VirtualIRQ = IRQ_OHCI ;
                         if(g_pUSBOHCIHOSTReg->HcRhPortStatus & DOWNSTREAMPORT_CONNECTED)
                             RETAILMSG(0,(_T("IRQ_OHCI : Conneted Device\r\n")));
                         else
                             RETAILMSG(0,(_T("IRQ_OHCI : Not Conneted Device\r\n")));
                      }
                  else{
                    
                    RETAILMSG(1,(_T("USB HOST Unknown Interrupt!!\r\n")));
                     RETAILMSG(1,(_T("OHCI INT ENABLE : %x\r\n"),OHCIIntEnable));
                     RETAILMSG(1,(_T("OHCI INT STATUS : %x\r\n\n"),g_pUSBOHCIHOSTReg->HcInterruptStatus));

                     RETAILMSG(1,(_T("EHCI INT ENABLE : %x\r\n"),EHCIIntEnable));                     
                     RETAILMSG(1,(_T("EHCI INT STATUS : %x\r\n\n"),g_pUSBEHCIHOSTReg->USBSTS));
                    
                    }

                  if((g_pUSBOHCIHOSTReg->HcInterruptStatus & OHCIIntEnable) &&
                    (g_pUSBEHCIHOSTReg->USBSTS & EHCIIntEnable)){
                    RETAILMSG(1,(_T("USB HOST MultiHost Interrupt!!\r\n")));
                     RETAILMSG(1,(_T("OHCI INT ENABLE : %x\r\n"),OHCIIntEnable));
                     RETAILMSG(1,(_T("OHCI INT STATUS : %x\r\n\n"),g_pUSBOHCIHOSTReg->HcInterruptStatus));

                     RETAILMSG(1,(_T("EHCI INT ENABLE : %x\r\n"),EHCIIntEnable));                     
                     RETAILMSG(1,(_T("EHCI INT STATUS : %x\r\n\n"),g_pUSBEHCIHOSTReg->USBSTS));
                     if(VirtualIRQ == IRQ_OHCI)
                           RETAILMSG(1,(_T("RETURN USB HOST INT : IRQ_OHCI!!\r\n")));                        
                     else
                           RETAILMSG(1,(_T("RETURN USB HOST INT : IRQ_EHCI!!\r\n")));                        
                    }
                }               

              IntEnVIC1 &= ~(1<<(PhysicalIRQ-VIC1_BIT_OFFSET));
#else
        IntEnVIC1 &= ~(1<<(PhysicalIRQ-VIC1_BIT_OFFSET));
#endif
        }
        else if (PhysicalIRQ < VIC3_BIT_OFFSET)
        {

           // RETAILMSG(1,(_T("PhysicalIRQ=%d  -----denis_wei add for test 2010-07-28\r\n"),PhysicalIRQ));
            IntEnVIC2 &= ~(1<<(PhysicalIRQ-VIC2_BIT_OFFSET));
        }        
        else if (PhysicalIRQ < PHYIRQ_MAX_S5PV210)
        {

            IntEnVIC3 &= ~(1<<(PhysicalIRQ-VIC3_BIT_OFFSET));

        }

            // We don't assume IRQ sharing, use static mapping
        SysIntr = OALIntrTranslateIrq(VirtualIRQ);
    }

CleanUp_ISR:
    // Clear Vector Address Register
    g_pVIC0Reg->VICADDRESS = 0x0;
    g_pVIC1Reg->VICADDRESS = 0x0;
    g_pVIC2Reg->VICADDRESS = 0x0;
    g_pVIC3Reg->VICADDRESS = 0x0;    

    // To Avoid low proirity interrupt lost
    g_pVIC0Reg->VICINTENABLE = IntEnVIC0;
    g_pVIC1Reg->VICINTENABLE = IntEnVIC1;
    g_pVIC2Reg->VICINTENABLE = IntEnVIC2;
    g_pVIC3Reg->VICINTENABLE = IntEnVIC3;


    return SysIntr;
}


static void PrepareEINTIntr(void)
{
    // Wakeup source EINT(GPH) Initialization
    OUTREG32(&g_pGPIOReg->EINTMSK.EXT_INT0_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTMSK.EXT_INT1_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTMSK.EXT_INT2_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTMSK.EXT_INT3_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT0_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT1_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT2_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->EINTPND.EXT_INT3_PEND, 0xFF);

    // General EINT Initialization
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPA0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPA1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPB_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPC0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPC1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPD0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPD1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPE0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPE1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPF0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPF1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPF2_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPF3_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPG0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPG1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPG2_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPG3_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPJ0_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPJ1_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPJ2_INT_MASK, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTMSK.GPJ3_INT_MASK, 0xFF);

    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPA0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPA1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPB_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPC0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPC1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPD0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPD1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPE0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPE1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPF0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPF1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPF2_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPF3_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPG0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPG1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPG2_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPG3_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPJ0_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPJ1_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPJ2_INT_PEND, 0xFF);
    OUTREG32(&g_pGPIOReg->GPIOINTPND.GPJ3_INT_PEND, 0xFF);
}

static void PrepareDMACIntr(void)
{
    // Disable Interrupt of All Channel
    // Mask TC and Error Interrupt
    // Clear Interrupt Pending

#if 0
    g_pDMAC0Reg->DMACC0Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC0Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC1Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC1Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC2Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC2Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC3Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC3Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC4Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC4Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC5Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC5Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC6Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC6Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC0Reg->DMACC7Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC0Reg->DMACC7Configuration &= (3<<14);    // TCINT & ErrINT Mask

    g_pDMAC0Reg->DMACIntTCClear = 0xff;            // TC Int Pending Clear
    g_pDMAC0Reg->DMACIntErrClear = 0xff;            // Err Int Pending Clear

    g_pDMAC1Reg->DMACC0Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC0Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC1Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC1Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC2Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC2Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC3Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC3Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC4Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC4Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC5Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC5Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC6Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC6Configuration &= (3<<14);    // TCINT & ErrINT Mask
    g_pDMAC1Reg->DMACC7Control0 &= ~(1<<31);        // TCINT Disable
    g_pDMAC1Reg->DMACC7Configuration &= (3<<14);    // TCINT & ErrINT Mask

    g_pDMAC1Reg->DMACIntTCClear = 0xff;            // TC Int Pending Clear
    g_pDMAC1Reg->DMACIntErrClear = 0xff;            // Err Int Pending Clear
#endif

}

static void InitializeVIC(void)
{
    // Disable All Interrupts
    g_pVIC0Reg->VICINTENCLEAR = 0xFFFFFFFF;
    g_pVIC1Reg->VICINTENCLEAR = 0xFFFFFFFF;
    g_pVIC2Reg->VICINTENCLEAR = 0xFFFFFFFF;
    g_pVIC3Reg->VICINTENCLEAR = 0xFFFFFFFF;
    g_pVIC0Reg->VICSOFTINTCLEAR = 0xFFFFFFFF;
    g_pVIC1Reg->VICSOFTINTCLEAR = 0xFFFFFFFF;
    g_pVIC2Reg->VICSOFTINTCLEAR = 0xFFFFFFFF;
    g_pVIC3Reg->VICSOFTINTCLEAR = 0xFFFFFFFF;

    // Clear Current Active Vector Address
    g_pVIC0Reg->VICADDRESS = 0x0;
    g_pVIC1Reg->VICADDRESS = 0x0;
    g_pVIC2Reg->VICADDRESS = 0x0;
    g_pVIC3Reg->VICADDRESS = 0x0;

    // System Timer Interrupt is highest priority
    g_pVIC0Reg->VICVECTPRIORITY26 = (g_pVIC0Reg->VICVECTPRIORITY26 & ~(0xf))|0x0;
    
    // Initialize Vector Address Table
    VICTableInit();

      //denis_wei add to enable keybd interrupt 2010-07-28
      g_pVIC2Reg->VICRAWINTR |= (0X1<<25);
	  g_pVIC2Reg->VICINTENABLE |= (0X1<<25);
	  
	

    // Enable IRQ Interrupt on CP15
    System_EnableIRQ();

    // Enable FIQ Interrupt on CP15
    System_EnableFIQ();
}

void VICTableInit(void)
{
    // This Function is reference by OEMPowerOff() in "Off.c"
    // Make Sure that Caller and This function is in Same Address Space

    // Fill Vector Address of VIC0
    // Actually, Filled with Physical IRQ Numbers.
    // Because We do not use vectored interrupt feature


    // VIC0
    g_pVIC0Reg->VICVECTADDR0 = PHYIRQ_EINT0;        // 0          // 0
    g_pVIC0Reg->VICVECTADDR1 = PHYIRQ_EINT1;        // 1          // 1
    g_pVIC0Reg->VICVECTADDR2 = PHYIRQ_EINT2;        // 2          // 2
    g_pVIC0Reg->VICVECTADDR3 = PHYIRQ_EINT3;        // 3          // 3
    g_pVIC0Reg->VICVECTADDR4 = PHYIRQ_EINT4;        // 4          // 4
    g_pVIC0Reg->VICVECTADDR5 = PHYIRQ_EINT5;        // 5          // 5
    g_pVIC0Reg->VICVECTADDR6 = PHYIRQ_EINT6;        // 6          // 6
    g_pVIC0Reg->VICVECTADDR7 = PHYIRQ_EINT7;        // 7          // 7
    g_pVIC0Reg->VICVECTADDR8 = PHYIRQ_EINT8;        // 8          // 8
    g_pVIC0Reg->VICVECTADDR9 = PHYIRQ_EINT9;        // 9          // 9
    g_pVIC0Reg->VICVECTADDR10 = PHYIRQ_EINT10;       // 10         // 10
    g_pVIC0Reg->VICVECTADDR11 = PHYIRQ_EINT11;       // 11         // 11
    g_pVIC0Reg->VICVECTADDR12 = PHYIRQ_EINT12;       // 12         // 12
    g_pVIC0Reg->VICVECTADDR13 = PHYIRQ_EINT13;       // 13         // 13
    g_pVIC0Reg->VICVECTADDR14 = PHYIRQ_EINT14;       // 14         // 14
    g_pVIC0Reg->VICVECTADDR15 = PHYIRQ_EINT15;       // 15         // 15
    g_pVIC0Reg->VICVECTADDR16 = PHYIRQ_EINT16_31;    // 16         // 16
    g_pVIC0Reg->VICVECTADDR17 = PHYIRQ_RESERVED0;    // 32         // 17  
    g_pVIC0Reg->VICVECTADDR18 = PHYIRQ_MDMA;         // 33         // 18
    g_pVIC0Reg->VICVECTADDR19 = PHYIRQ_PDMA0;        // 41         // 19
    g_pVIC0Reg->VICVECTADDR20 = PHYIRQ_PDMA1;        // 49         // 20
    g_pVIC0Reg->VICVECTADDR21 = PHYIRQ_TIMER0;       // 57         // 21
    g_pVIC0Reg->VICVECTADDR22 = PHYIRQ_TIMER1;       // 58         // 22
    g_pVIC0Reg->VICVECTADDR23 = PHYIRQ_TIMER2;       // 59         // 23
    g_pVIC0Reg->VICVECTADDR24 = PHYIRQ_TIMER3;       // 60         // 24
    g_pVIC0Reg->VICVECTADDR25 = PHYIRQ_TIMER4;       // 61         // 25
    g_pVIC0Reg->VICVECTADDR26 = PHYIRQ_SYSTIMER;     // 62         // 26
    g_pVIC0Reg->VICVECTADDR27 = PHYIRQ_WDT;          // 63         // 27
    g_pVIC0Reg->VICVECTADDR28 = PHYIRQ_RTC_ALARM;    // 64         // 28
    g_pVIC0Reg->VICVECTADDR29 = PHYIRQ_RTC_TIC;      // 65         // 29
    g_pVIC0Reg->VICVECTADDR30 = PHYIRQ_GPIO;         // 66         // 30
    g_pVIC0Reg->VICVECTADDR31 = PHYIRQ_FIMC3;        // 67         // 31

    // VIC1
    g_pVIC1Reg->VICVECTADDR0 = PHYIRQ_CORTEX0;      // 68         // 32
    g_pVIC1Reg->VICVECTADDR1 = PHYIRQ_CORTEX1;      // 69         // 33
    g_pVIC1Reg->VICVECTADDR2 = PHYIRQ_CORTEX2;      // 70         // 34
    g_pVIC1Reg->VICVECTADDR3 = PHYIRQ_CORTEX3;      // 71         // 35
    g_pVIC1Reg->VICVECTADDR4 = PHYIRQ_CORTEX4;      // 72         // 36
    g_pVIC1Reg->VICVECTADDR5 = PHYIRQ_IEMAPC;       // 73         // 37
    g_pVIC1Reg->VICVECTADDR6 = PHYIRQ_IEMIEC;       // 74         // 38
    g_pVIC1Reg->VICVECTADDR7 = PHYIRQ_RESERVED1;    // 75         // 39
    g_pVIC1Reg->VICVECTADDR8 = PHYIRQ_NFC;          // 76         // 40
    g_pVIC1Reg->VICVECTADDR9 = PHYIRQ_CFC;          // 77         // 41
    g_pVIC1Reg->VICVECTADDR10 = PHYIRQ_UART0;        // 78         // 42
    g_pVIC1Reg->VICVECTADDR11 = PHYIRQ_UART1;        // 79         // 43
    g_pVIC1Reg->VICVECTADDR12 = PHYIRQ_UART2;        // 80         // 44
    g_pVIC1Reg->VICVECTADDR13 = PHYIRQ_UART3;        // 81         // 45
    g_pVIC1Reg->VICVECTADDR14 = PHYIRQ_I2C;          // 82         // 46
    g_pVIC1Reg->VICVECTADDR15 = PHYIRQ_SPI0;         // 83         // 47
    g_pVIC1Reg->VICVECTADDR16 = PHYIRQ_SPI1;         // 84         // 48
    g_pVIC1Reg->VICVECTADDR17 = PHYIRQ_SPI2;         // 85         // 49
    g_pVIC1Reg->VICVECTADDR18 = PHYIRQ_AUDIO_SS;     // 86         // 50
    g_pVIC1Reg->VICVECTADDR19 = PHYIRQ_I2C_PMIC_LINK;// 87         // 51
    g_pVIC1Reg->VICVECTADDR20 = PHYIRQ_I2C_HDMI_PHY; // 88         // 52
    g_pVIC1Reg->VICVECTADDR21 = PHYIRQ_HSIRX;        // 89         // 53
    g_pVIC1Reg->VICVECTADDR22 = PHYIRQ_HSITX;        // 90         // 54
    g_pVIC1Reg->VICVECTADDR23 = PHYIRQ_UHOST;        // 91         // 55
    g_pVIC1Reg->VICVECTADDR24 = PHYIRQ_OTG;          // 92         // 56
    g_pVIC1Reg->VICVECTADDR25 = PHYIRQ_MSM;          // 93         // 57
    g_pVIC1Reg->VICVECTADDR26 = PHYIRQ_SDMMC0;       // 94         // 58
    g_pVIC1Reg->VICVECTADDR27 = PHYIRQ_SDMMC1;       // 95         // 59
    g_pVIC1Reg->VICVECTADDR28 = PHYIRQ_SDMMC2;       // 96         // 60
    g_pVIC1Reg->VICVECTADDR29 = PHYIRQ_MIPI_CSI;     // 97         // 61
    g_pVIC1Reg->VICVECTADDR30 = PHYIRQ_MIPI_DSI;     // 98         // 62
    g_pVIC1Reg->VICVECTADDR31 = PHYIRQ_ONENAND_AUDI; // 99         // 63

    // VIC2
    g_pVIC2Reg->VICVECTADDR0 = PHYIRQ_LCD0;         // 100        // 64
    g_pVIC2Reg->VICVECTADDR1 = PHYIRQ_LCD1;         // 101        // 65
    g_pVIC2Reg->VICVECTADDR2 = PHYIRQ_LCD2;         // 102        // 66
    g_pVIC2Reg->VICVECTADDR3 = PHYIRQ_LCD3;         // 103        // 67
    g_pVIC2Reg->VICVECTADDR4 = PHYIRQ_ROTATOR;      // 104        // 68
    g_pVIC2Reg->VICVECTADDR5 = PHYIRQ_FIMC0;        // 105        // 69
    g_pVIC2Reg->VICVECTADDR6 = PHYIRQ_FIMC1;        // 106        // 70
    g_pVIC2Reg->VICVECTADDR7 = PHYIRQ_FIMC2;        // 107        // 71
    g_pVIC2Reg->VICVECTADDR8 = PHYIRQ_JPEG;         // 108        // 72
    g_pVIC2Reg->VICVECTADDR9 = PHYIRQ_2D;           // 109        // 73
    g_pVIC2Reg->VICVECTADDR10 = PHYIRQ_3D;           // 110        // 74
    g_pVIC2Reg->VICVECTADDR11 = PHYIRQ_MIXER;        // 111        // 75
    g_pVIC2Reg->VICVECTADDR12 = PHYIRQ_HDMI;         // 112        // 76
    g_pVIC2Reg->VICVECTADDR13 = PHYIRQ_HDMI_I2C;     // 113        // 77
    g_pVIC2Reg->VICVECTADDR14 = PHYIRQ_MFC;          // 114        // 78
    g_pVIC2Reg->VICVECTADDR15 = PHYIRQ_TVENC;        // 115        // 79
    g_pVIC2Reg->VICVECTADDR16 = PHYIRQ_I2S0;         // 116        // 80
    g_pVIC2Reg->VICVECTADDR17 = PHYIRQ_I2S1;         // 117        // 81
    g_pVIC2Reg->VICVECTADDR18 = PHYIRQ_RESERVED2;    // 118        // 82 
    g_pVIC2Reg->VICVECTADDR19 = PHYIRQ_AC97;         // 119        // 83
    g_pVIC2Reg->VICVECTADDR20 = PHYIRQ_PCM0;         // 120        // 84
    g_pVIC2Reg->VICVECTADDR21 = PHYIRQ_PCM1;         // 121        // 85
    g_pVIC2Reg->VICVECTADDR22 = PHYIRQ_SPDIF;        // 122        // 86
    g_pVIC2Reg->VICVECTADDR23 = PHYIRQ_ADC;          // 123        // 87
    g_pVIC2Reg->VICVECTADDR24 = PHYIRQ_PENDN;        // 124        // 88
    g_pVIC2Reg->VICVECTADDR25 = PHYIRQ_KEYPAD;       // 125        // 89
    g_pVIC2Reg->VICVECTADDR26 = PHYIRQ_SPG;          // 126        // 90
    g_pVIC2Reg->VICVECTADDR27 = PHYIRQ_IntHash_SSS;  // 127        // 91
    g_pVIC2Reg->VICVECTADDR28 = PHYIRQ_IntFeedCtrl_SSS;       // 128        // 92
    g_pVIC2Reg->VICVECTADDR29 = PHYIRQ_RESERVED3;    // 129        // 93
    g_pVIC2Reg->VICVECTADDR30 = PHYIRQ_SDM_IRQ;      // 130        // 94
    g_pVIC2Reg->VICVECTADDR31 = PHYIRQ_SDM_FIQ;      // 131        // 95
    
    // VIC3
    g_pVIC3Reg->VICVECTADDR0 = PHYIRQ_IPC;           // 132        // 96
    g_pVIC3Reg->VICVECTADDR1 = PHYIRQ_HOSTIF;        // 133        // 97
    g_pVIC3Reg->VICVECTADDR2 = PHYIRQ_SDMMC3;        // 134        // 98
    g_pVIC3Reg->VICVECTADDR3 = PHYIRQ_CEC;           // 135        // 99
    g_pVIC3Reg->VICVECTADDR4 = PHYIRQ_TSI;           // 136        // 100
    g_pVIC3Reg->VICVECTADDR5 = PHYIRQ_MDNIE0;        // 137        // 101
    g_pVIC3Reg->VICVECTADDR6 = PHYIRQ_MDNIE1;        // 138        // 102
    g_pVIC3Reg->VICVECTADDR7 = PHYIRQ_MDNIE2;        // 139        // 103
    g_pVIC3Reg->VICVECTADDR8 = PHYIRQ_MDNIE3;        // 140        // 104    
    g_pVIC3Reg->VICVECTADDR9 = PHYIRQ_ADC1;          // 141        // 105    
    g_pVIC3Reg->VICVECTADDR10 = PHYIRQ_PENDN1;       // 142        // 106    
}

void SetUSBHostIntrFlag(USB_HOST_INTR eIntr)
{
    eUSB_HOST = eIntr;
}
USB_HOST_INTR GetUSBHostIntrFlag()
{
    return eUSB_HOST;
}


//------------------------------------------------------------------------------
