//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  Header: systimer_reg.h
//
//  Defines the PWM Timer register layout and associated types and constants.
//
#ifndef __S5PV210_SYSTIMER_H
#define __S5PV210_SYSTIMER_H

#if __cplusplus
extern "C" {
#endif

//------------------------------------------------------------------------------
//
//  Type:  SYSTIMER_REG
//
//  Defines the PWM Timer control register layout. This register bank is
//  located by the constant BASE_REG_XX_PWM in the configuration file
//  base_reg_cfg.h.
//


typedef struct
{
	UINT32 TCFG;    // E260_0000
	UINT32 TCON;    // E260_0004
	UINT32 TICNTB;  // E260_0008
	UINT32 TICNTO;  // E260_000C
	UINT32 TFCNTB;  // E260_0010
	UINT32 PAD;     // E260_0014
	UINT32 ICNTB;   // E260_0018
	UINT32 ICNTO;   // E260_001C
	UINT32 INT_CSTAT; //E260_0020
} SYSTIMER_REG, *PSYSTIMER_REG;

#define ST_TCON_WRITE_INT_ENABLE   (0x1<<10)
#define ST_ICNTB_WRITE_INT_ENABLE  (0x1<<9)
#define ST_TFCNTB_WRITE_INT_ENABLE (0x1<<8)
#define ST_TICNTB_WRITE_INT_ENABLE (0x1<<7)
#define ST_ICNT_EXPIRE_INT_ENABLE  (0x1<<6)
#define ST_TCON_WRITE_STS         (0x1<<5)
#define ST_ICNTB_WRITE_STS        (0x1<<4)
#define ST_TFCNTB_WRITE_STS       (0x1<<3)
#define ST_TICNTB_WRITE_STS       (0x1<<2)
#define ST_INTR_STS               (0x1<<1)
#define ST_INTR_ENABLE            (0x1<<0)

//------------------------------------------------------------------------------
#define ENABLE_TCON_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) | ST_TCON_WRITE_INT_ENABLE;           \
                                        } while(0)
                                        
#define ENABLE_ICNTB_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) | ST_ICNTB_WRITE_INT_ENABLE;           \
                                        } while(0)
                                        
#define ENABLE_TFCNTB_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) | ST_TFCNTB_WRITE_INT_ENABLE;           \
                                        } while(0)

#define ENABLE_TICNTB_EXPIRE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) | ST_TICNTB_WRITE_INT_ENABLE;           \
                                        } while(0)
                                        
#define ENABLE_ICNT_EXPIRE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) | ST_ICNT_EXPIRE_INT_ENABLE;           \
                                        } while(0)


#define DISABLE_ALL_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = Addr_INT_CSTAT & ~(ST_TCON_WRITE_INT_ENABLE | ST_ICNTB_WRITE_INT_ENABLE | ST_TFCNTB_WRITE_INT_ENABLE | ST_TICNTB_WRITE_INT_ENABLE | ST_ICNT_EXPIRE_INT_ENABLE);           \
                                        } while(0) 

#define DISABLE_TCON_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) & ~(ST_TCON_WRITE_INT_ENABLE);           \
                                        } while(0)
                                        
#define DISABLE_ICNTB_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) & ~(ST_ICNTB_WRITE_INT_ENABLE);           \
                                        } while(0)
                                        
#define DISABLE_TFCNTB_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) & ~(ST_TFCNTB_WRITE_INT_ENABLE);           \
                                        } while(0)

#define DISABLE_TICNTB_WRITE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) & ~(ST_TICNTB_WRITE_INT_ENABLE);           \
                                        } while(0)

#define DISABLE_ICNT_EXPIRE_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F<<1)) & ~(ST_ICNT_EXPIRE_INT_ENABLE);           \
                                        } while(0)

#define ENABLE_SYSTEM_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F)) | ST_INTR_ENABLE;           \
                                        } while(0)

#define DISABLE_SYSTEM_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F)) & ~(ST_INTR_ENABLE);           \
                                        } while(0)

#define CLEAR_ALL_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F)) | (ST_INTR_STS | ST_TCON_WRITE_STS | ST_ICNTB_WRITE_STS | ST_TFCNTB_WRITE_STS | ST_TICNTB_WRITE_STS); \
                                        } while(0)

#define CLEAR_SYSTEM_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F << 1)) | ST_INTR_STS;  \
                                        } while(0)

#define CLEAR_TCON_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F << 1)) | ST_TCON_WRITE_STS;  \
                                        } while(0)

#define CLEAR_ICNTB_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F << 1)) | ST_ICNTB_WRITE_STS;  \
                                        } while(0)
                                        
#define CLEAR_TFCNTB_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F << 1)) | ST_TFCNTB_WRITE_STS;  \
                                        } while(0)

#define CLEAR_TICNTB_INTR(Addr_INT_CSTAT)    do {                                                    \
                                                Addr_INT_CSTAT = (Addr_INT_CSTAT & ~(0x1F << 1)) | ST_TICNTB_WRITE_STS;  \
                                        } while(0)

#define WAIT_FOR_TCON_UPDATE(Addr_INT_CSTAT)    do {                                                    \
                                                while(!((Addr_INT_CSTAT) & ST_TCON_WRITE_STS));                                    \
                                                Addr_INT_CSTAT = Addr_INT_CSTAT & ~(0x1F << 1) | ST_TCON_WRITE_STS;  \
                                       } while(0)

#define WAIT_FOR_ICNTB_UPDATE(Addr_INT_CSTAT)    do {                                                  \
                                                while(!((Addr_INT_CSTAT) & ST_ICNTB_WRITE_STS));                                    \
                                                Addr_INT_CSTAT = Addr_INT_CSTAT & ~(0x1F << 1) | ST_ICNTB_WRITE_STS;  \
                                       } while(0)

#define WAIT_FOR_TFCNTB_UPDATE(Addr_INT_CSTAT)    do {                                                  \
                                                while(!((Addr_INT_CSTAT) & ST_TFCNTB_WRITE_STS));                                    \
                                                Addr_INT_CSTAT = Addr_INT_CSTAT & ~(0x1F << 1) | ST_TFCNTB_WRITE_STS;  \
                                       } while(0)

#define WAIT_FOR_TICNTB_UPDATE(Addr_INT_CSTAT)    do {                                                 \
                                                while(!((Addr_INT_CSTAT) & ST_TICNTB_WRITE_STS));                                    \
                                                Addr_INT_CSTAT = Addr_INT_CSTAT & ~(0x1F << 1) | ST_TICNTB_WRITE_STS;  \
                                       } while(0)

#define MultiplyHDRD(val)     ((val<<6)+(val<<5)+(val<<2))
//------------------------------------------------------------------------------

#if __cplusplus
}
#endif

#endif // __S5PV210_SYSTIMER_H

