//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this source code is subject to the terms of the Microsoft end-user
// license agreement (EULA) under which you licensed this SOFTWARE PRODUCT.
// If you did not accept the terms of the EULA, you are not authorized to use
// this source code. For a copy of the EULA, please see the LICENSE.RTF on your
// install media.
//
//------------------------------------------------------------------------------
//
//  Header:  syscon.h
//
//  Defines the System Controller (CMU, PMU) register layout and associated types
//  and constants.
//
//  CMU : CMU_XXX_REG (CLK, SRST, IEM, SYS, MISC)
//  PMU : PMU_XXX_REG  (OSC, RSTS, PM, MISC)
//
#ifndef __S5PV210_SYSCON_H
#define __S5PV210_SYSCON_H

#if __cplusplus
extern "C" {
#endif

#include "CMUEntry_reg.h"
#include "PMUEntry_reg.h"
//------------------------------------------------------------------------------
//
//  Type:  CMU_XXX_REG
//
//  System Clock Controller registers. This register bank is located by the
//  constant BASE_REG_CMU_XX in the configuration file
//  reg_base_cfg.h.
//

// S5PV210 Syscon Registers need to be separated to following structures..
//   - CMU_CLK      : Clock Controller
//   - CMU_SRST    : SW RESET
//   - CMU_IEM       : IEM
//   - CMU_SYS      : System Controller
//   - CMU_MISC    : Miscellaneous

typedef struct  // Base Addr : 0xE010_0000, BASE_REG_PA_CMU_CLK
{
    _PLL_LCK_REG	        PLL_LOCK;
    _PLL_CON_REG	        PLL_CON;
    _CLK_SRC_REG	        CLK_SRC;
    _CLK_DIV_REG	        CLK_DIV;
    _CLK_GAT_REG	        CLK_GATE;
    _CLK_OUT_REG	        CLK_OUTPUT;
    _CLK_GTS_REG	        CLK_DIV_STAT;
    _CLK_MTS_REG	        CLK_MUX_STAT;
}   CMU_CLK_REG, *PCMU_CLK_REG;

typedef struct  // Base Addr : 0xE010_2000, BASE_REG_PA_CMU_SRST
{
    _SWRESET_REG        SWRESET;
}   CMU_SRST_REG, *PCMU_SRST_REG;

typedef struct  // Base Addr : 0xE010_3000, BASE_REG_PA_CMU_IEM
{
    _IEM_CFG_REG	        CONFIG;
    _IEM_PST_REG	        PLL_SET;
    _IEM_DST_REG	        DIV_SET;
#if (S5PV210_EVT>0)    
    _IEM_PST2_REG	        PLL1_SET;
#endif
}   CMU_IEM_REG, *PCMU_IEM_REG;

typedef struct  // Base Addr : 0xE010_6100, BASE_REG_PA_CMU_GCTRL
{
    _GENERAL_CTRL        ARM_VOLT_CTRL;
}   CMU_GCTRL_REG, *PCMU_GCTRL_REG;

#if (S5PV210_EVT==0)
typedef struct  // Base Addr : 0xE010_6200, BASE_REG_PA_CMU_SYS
{
    _SYS_MEM_REG	        MEM_REG;
}   CMU_SYS_REG, *PCMU_SYS_REG;
#endif

typedef struct  // Base Addr : 0xE010_7000, BASE_REG_PA_CMU_MISC
{
    _MISC_MTM_REG	MULTIMEDIA;
    _MISC_TST_REG	TEST_REG;
}   CMU_MISC_REG, *PCMU_MISC_REG;

//------------------------------------------------------------------------------
//
//  Type:  PMU_XXX_REG
//
//  System PM Controller registers. This register bank is located by the
//  constant BASE_REG_PMU_XX in the configuration file
//  reg_base_cfg.h.
//

// S5PV210 Syscon Registers need to be separated to following structures..
//   - PMU_OSC      : OSC Clock control
//   - PMU_RSTS    : Reset Control
//   - PMU_PM        : Power Management
//   - PMU_MISC    : Miscellaneous

typedef struct  // Base Addr : 0xE010_8000, BASE_REG_PA_PMU_OSC
{
    _CLK_CON_REG        OSC_CON;
}   PMU_OSC_REG, *PPMU_OSC_REG;

typedef struct  // Base Addr : 0xE010_A000, BASE_REG_PA_PMU_RSTS
{
    _RST_STS_REG        RST_STAT;
}   PMU_RSTS_REG, *PPMU_RSTS_REG;

typedef struct  // Base Addr : 0xE010_C000, BASE_REG_PA_PMU_PM
{
    _PWR_CFG_REG       PWR_CONF;
    _PWR_STA_REG       STABLE_REG;
    _PWR_STS_REG       STATUS_REG;
    _PWR_ABB_REG       ABB_REG;
}   PMU_PM_REG, *PPMU_PM_REG;

typedef struct  // Base Addr : 0xE010_E000, BASE_REG_PA_PMU_MISC
{
    _SYS_CON_REG        SYS_CON;
    _SYS_STS_REG        STAT_REG;
    _SYS_IPC_REG        IP_CON_REG;
    _SYS_MIS_REG        MISC_REG;
}   PMU_MISC_REG, *PPMU_MISC_REG;


//// Bit Position(CMU)
// PLL CON
#define BP_PLL_ENABLE                   (31)
#define BP_PLL_LOCKED                   (29)
#define BP_PLL_ENABLE_LOCKED_DET        (28)
#define BP_PLL_MDIV                     (16)
#define BP_PLL_PDIV                     (8)
#define BP_PLL_SDIV                     (0)
#define BP_PLL_VCO_FREQ_SEL             (27)
#if (S5PV210_EVT>0)
#define BP_APLL_AFC_ENB                 (31)
#define BP_APLL_AFC                     (0)
#define BP_EPLL_DSM_K                   (0)
#endif

// CLK_SRC
#define BP_MUX_FLASH_SEL                (28)
#define BP_MUX_PSYS_SEL                 (24)
#define BP_MUX_DSYS_SEL                 (20)
#define BP_MUX_MSYS_SEL                 (16)
#define BP_MUX_VPLL_SEL                 (12)
#define BP_MUX_EPLL_SEL                 (8)
#define BP_MUX_MPLL_SEL                 (4)
#define BP_MUX_APLL_SEL                 (0)

#define BP_MUX_VPLLSRC_SEL              (28)
#define BP_MUX_CSIS_SEL                 (24)
#define BP_MUX_FIMD_SEL                 (20)
#define BP_MUX_CAM1_SEL                 (16)
#define BP_MUX_CAM0_SEL                 (12)
#define BP_MUX_DAC_SEL                  (8)
#define BP_MUX_MIXER_SEL                (4)
#define BP_MUX_HDMI_SEL                 (0)

#if (S5PV210_EVT>0)
#define BP_MUX_G2D_SEL                  (8)
#endif
#define BP_MUX_MFC_SEL                  (4)
#define BP_MUX_G3D_SEL                  (0)

#define BP_MUX_FIMC2_LCLK_SEL           (20)
#define BP_MUX_FIMC1_LCLK_SEL           (16)
#define BP_MUX_FIMC0_LCLK_SEL           (12)
#define BP_MUX_MDNIE_PWMCLK_SEL         (4)
#define BP_MUX_MDNIE_SEL                (0)

#define BP_MUX_UART3_SEL                (28)
#define BP_MUX_UART2_SEL                (24)
#define BP_MUX_UART1_SEL                (20)
#define BP_MUX_UART0_SEL                (16)
#define BP_MUX_MMC3_SEL                 (12)
#define BP_MUX_MMC2_SEL                 (8)
#define BP_MUX_MMC1_SEL                 (4)
#define BP_MUX_MMC0_SEL                 (0)

#define BP_MUX_PWM_SEL                  (12)
#define BP_MUX_SPI2_SEL                 (8)
#define BP_MUX_SPI1_SEL                 (4)
#define BP_MUX_SPI0_SEL                 (0)

#define BP_MUX_DMC0_SEL                 (24)
#define BP_MUX_PWI_SEL                  (20)
#define BP_MUX_HPM_SEL                  (16)
#define BP_MUX_SPDIF_SEL                (12)
#define BP_MUX_AUDIO2_SEL               (8)
#define BP_MUX_AUDIO1_SEL               (4)
#define BP_MUX_AUDIO0_SEL               (0)

//CLK_SRC_MASK
#define BP_MUX_PWI_MASK                 (29)
#define BP_MUX_SPDIF_MASK               (27)
#define BP_MUX_AUDIO2_MASK              (26)
#define BP_MUX_AUDIO1_MASK              (25)
#define BP_MUX_AUDIO0_MASK              (24)
#define BP_MUX_PWM_MASK                 (19)
#define BP_MUX_SPI2_MASK                (18)
#define BP_MUX_SPI1_MASK                (17)
#define BP_MUX_SPI0_MASK                (16)
#define BP_MUX_UART3_MASK               (15)
#define BP_MUX_UART2_MASK               (14)
#define BP_MUX_UART1_MASK               (13)
#define BP_MUX_UART0_MASK               (12)
#define BP_MUX_MMC3_MASK                (11)
#define BP_MUX_MMC2_MASK                (10)
#define BP_MUX_MMC1_MASK                (9)
#define BP_MUX_MMC0_MASK                (8)
#define BP_MUX_VPLLSRC_MASK             (7)
#define BP_MUX_CSIS_MASK                (6)
#define BP_MUX_FIMD_MASK                (5)
#define BP_MUX_CAM1_MASK                (4)
#define BP_MUX_CAM0_MASK                (3)
#define BP_MUX_DAC_MASK                 (2)
#define BP_MUX_MIXER_MASK               (1)
#define BP_MUX_HDMI_MASK                (0)

#define BP_MUX_FIMC2_LCLK_MASK          (4)
#define BP_MUX_FIMC1_LCLK_MASK          (3)
#define BP_MUX_FIMC0_LCLK_MASK          (2)
#define BP_MUX_MDNIE_PWM_MASK           (1)
#define BP_MUX_MDNIE_MASK               (0)

// CLK_DIV
#define BP_DIV_PCLK_PSYS_RATIO          (28)
#define BP_DIV_HCLK_PSYS_RATIO          (24)
#define BP_DIV_PCLK_DSYS_RATIO          (20)
#define BP_DIV_HCLK_DSYS_RATIO          (16)
#define BP_DIV_PCLK_MSYS_RATIO          (12)
#define BP_DIV_HCLK_MSYS_RATIO          (8)
#define BP_DIV_A2M_RATIO                (4)
#define BP_DIV_APLL_RATIO               (0)

#define BP_DIV_CSIS_RATIO               (28)
#define BP_DIV_FIMD_RATIO               (20)
#define BP_DIV_CAM1_RATIO               (16)
#define BP_DIV_CAM0_RATIO               (12)
#define BP_DIV_FIMC_RATIO               (8)
#define BP_DIV_TBLK_RATIO               (0)

#if (S5PV210_EVT>0)
#define BP_DIV_G2D_RATIO                (8)
#endif
#define BP_DIV_MFC_RATIO                (4)
#define BP_DIV_G3D_RATIO                (0)

#define BP_DIV_FIMC2_LCLK_RATIO         (20)
#define BP_DIV_FIMC1_LCLK_RATIO         (16)
#define BP_DIV_FIMC0_LCLK_RATIO         (12)
#define BP_DIV_MDNIE_PWM_RATIO          (4)
#define BP_DIV_MDNIE_RATIO              (0)

#define BP_DIV_UART3_RATIO              (28)
#define BP_DIV_UART2_RATIO              (24)
#define BP_DIV_UART1_RATIO              (20)
#define BP_DIV_UART0_RATIO              (16)
#define BP_DIV_MMC3_RATIO               (12)
#define BP_DIV_MMC2_RATIO               (8)
#define BP_DIV_MMC1_RATIO               (4)
#define BP_DIV_MMC0_RATIO               (0)

#define BP_DIV_PWM_RATIO                (12)
#define BP_DIV_SPI2_RATIO               (8)
#define BP_DIV_SPI1_RATIO               (4)
#define BP_DIV_SPI0_RATIO               (0)

#define BP_DIV_DMC0_RATIO            (28)
#define BP_DIV_PWI_RATIO                (24)
#define BP_DIV_HPM_RATIO                (20)
#define BP_DIV_COPY_RATIO               (16)
#define BP_DIV_ONENAND_RATIO            (12)
#define BP_DIV_AUDIO2_RATIO             (8)
#define BP_DIV_AUDIO1_RATIO             (4)
#define BP_DIV_AUDIO0_RATIO             (0)

#define BP_DIV_DPM_RATIO                (8)
#define BP_DIV_DVSEM                    (0)

// CLK_GATE
#define BP_CLK_IP_CSIS                  (31)
#define BP_CLK_IP_IPC                   (30)
#define BP_CLK_IP_ROTATOR               (29)
#define BP_CLK_IP_JPEG                  (28)
#define BP_CLK_IP_FIMC2                 (26)
#define BP_CLK_IP_FIMC1                 (25)
#define BP_CLK_IP_FIMC0                 (24)
#define BP_CLK_IP_MFC                   (16)
#if (S5PV210_EVT>0)
#define BP_CLK_IP_G2D                   (12)
#endif
#define BP_CLK_IP_G3D                   (8)
#define BP_CLK_IP_IMEM                  (5)
#define BP_CLK_IP_PDMA1                 (4)
#define BP_CLK_IP_PDMA0                 (3)
#define BP_CLK_IP_MDMA                  (2)
#define BP_CLK_IP_DMC1                  (1)
#define BP_CLK_IP_DMC0                  (0)

#define BP_CLK_IP_NFCON                 (28)
#define BP_CLK_IP_SROMC                 (26)
#define BP_CLK_IP_CFCON                 (25)
#define BP_CLK_IP_ONENANDXL             (24)
#define BP_CLK_IP_USBHOST               (17)
#define BP_CLK_IP_USBOTG                (16)
#define BP_CLK_IP_HDMI                  (11)
#define BP_CLK_IP_TVENC                 (10)
#define BP_CLK_IP_MIXER                 (9)
#define BP_CLK_IP_VP                    (8)
#define BP_CLK_IP_DSIM                  (2)
#define BP_CLK_IP_MIE                   (1)
#define BP_CLK_IP_FIMD                  (0)

#define BP_CLK_IP_TZIC3                 (31)
#define BP_CLK_IP_TZIC2                 (30)
#define BP_CLK_IP_TZIC1                 (29)
#define BP_CLK_IP_TZIC0                 (28)
#define BP_CLK_IP_VIC3                  (27)
#define BP_CLK_IP_VIC2                  (26)
#define BP_CLK_IP_VIC1                  (25)
#define BP_CLK_IP_VIC0                  (24)
#define BP_CLK_IP_TSI                   (20)
#define BP_CLK_IP_MMC3                  (19)
#define BP_CLK_IP_MMC2                  (18)
#define BP_CLK_IP_MMC1                  (17)
#define BP_CLK_IP_MMC0                  (16)
#define BP_CLK_IP_SECJTAG               (11)
#define BP_CLK_IP_HOSTIF                (10)
#define BP_CLK_IP_MODEM                 (9)
#define BP_CLK_IP_CORESIGHT             (8)
#define BP_CLK_IP_SDM                   (1)
#define BP_CLK_IP_SECSS                 (0)

#define BP_CLK_IP_PCM2                  (30)
#define BP_CLK_IP_PCM1                  (29)
#define BP_CLK_IP_PCM0                  (28)
#define BP_CLK_IP_SYSCON                (27)
#define BP_CLK_IP_GPIO                  (26)
#define BP_CLK_IP_TSADC                 (24)
#define BP_CLK_IP_PWM                   (23)
#define BP_CLK_IP_WDT                   (22)
#define BP_CLK_IP_KEYIF                 (21)
#define BP_CLK_IP_UART3                 (20)
#define BP_CLK_IP_UART2                 (19)
#define BP_CLK_IP_UART1                 (18)
#define BP_CLK_IP_UART0                 (17)
#define BP_CLK_IP_SYSTIMER              (16)
#define BP_CLK_IP_RTC                   (15)
#define BP_CLK_IP_SPI2                  (14)
#define BP_CLK_IP_SPI1                  (13)
#define BP_CLK_IP_SPI0                  (12)
#define BP_CLK_IP_I2C_HDMI_PHY          (11)
#define BP_CLK_IP_I2C_HDMI_DDC          (10)
#define BP_CLK_IP_I2C2                  (9)
#define BP_CLK_IP_I2C1                  (8)
#define BP_CLK_IP_I2C0                  (7)
#define BP_CLK_IP_I2S2                  (6)
#define BP_CLK_IP_I2S1                  (5)
#define BP_CLK_IP_I2S0                  (4)
#define BP_CLK_IP_AC97                  (1)
#define BP_CLK_IP_SPDIF                 (0)

#define BP_CLK_IP_TZPC3                 (8)
#define BP_CLK_IP_TZPC2                 (7)
#define BP_CLK_IP_TZPC1                 (6)
#define BP_CLK_IP_TZPC0                 (5)
#define BP_CLK_IP_SECKEY                (3)
#define BP_CLK_IP_IEM_APC               (2)
#define BP_CLK_IP_IEM_IEC               (1)
#define BP_CLK_IP_CHIP_ID               (0)

#define BP_CLK_BLK_INTC                 (10)
#define BP_CLK_BLK_MMC                  (9)
#define BP_CLK_BLK_DEBUG                (8)
#define BP_CLK_BLK_SECURITY             (7)
#define BP_CLK_BLK_MEMORY               (6)
#define BP_CLK_BLK_USB                  (5)
#define BP_CLK_BLK_TV                   (4)
#define BP_CLK_BLK_LCD                  (3)
#define BP_CLK_BLK_IMG                  (2)
#define BP_CLK_BLK_MFC                  (1)
#define BP_CLK_BLK_G3D                  (0)

// CLK_OUT
#define BP_CLKOUT_DIVVAL                (20)
#define BP_CLKOUT_CLKSEL                (12)
#define BP_CLKOUT_DCLKCMP               (8)
#define BP_CLKOUT_DCLKDIV               (4)
#define BP_CLKOUT_DCLKSEL               (1)
#define BP_CLKOUT_DCLKEN                (0)

// CLK_DIV_STAT
#define BP_DIV_UART3_STAT               (31)
#define BP_DIV_UART2_STAT               (30)
#define BP_DIV_UART1_STAT               (29)
#define BP_DIV_UART0_STAT               (28)
#define BP_DIV_MMC3_STAT                (27)               
#define BP_DIV_MMC2_STAT                (26)
#define BP_DIV_MMC1_STAT                (25)
#define BP_DIV_MMC0_STAT                (24)
#define BP_DIV_FIMC2_LCLK_STAT          (22)
#define BP_DIV_FIMC1_LCLK_STAT          (21)
#define BP_DIV_FIMC0_LCLK_STAT          (20)
#define BP_DIV_MDNIE_PWM_STAT           (19)
#define BP_DIV_MDNIE_STAT               (18)
#define BP_DIV_MFC_STAT                 (17)
#define BP_DIV_G3D_STAT                 (16)
#define BP_DIV_CSIS_STAT                (15)
#define BP_DIV_FIMD_STAT                (13)
#define BP_DIV_CAM1_STAT                (12)
#define BP_DIV_CAM0_STAT                (11)
#define BP_DIV_FIMC_STAT                (10)
#define BP_DIV_TBLK_STAT                (8)
#define BP_DIV_PCLK_PSYS_STAT           (7)
#define BP_DIV_HCLK_PSYS_STAT           (6)
#define BP_DIV_PCLK_DSYS_STAT           (5)
#define BP_DIV_HCLK_DSYS_STAT           (4)
#define BP_DIV_PCLK_MSYS_STAT           (3)
#define BP_DIV_HCLK_MSYS_STAT           (2)
#define BP_DIV_A2M_STAT                 (1)
#define BP_DIV_APLL_STAT                (0)

#if (S5PV210_EVT>0)
#define BP_DIV_G2D_STAT                 (20)
#endif
#define BP_DIV_DPM_STAT                 (17)
#define BP_DIV_DVSEM_STAT               (16)
#define BP_DIV_DMC0_STAT                (15)
#define BP_DIV_PWI_STAT                 (14)
#define BP_DIV_HPM_STAT                 (13)               
#define BP_DIV_COPY_STAT                (12)
#define BP_DIV_FLASH_STAT               (11)
#define BP_DIV_AUDIO2_STAT              (10)
#define BP_DIV_AUDIO1_STAT              (9)
#define BP_DIV_AUDIO0_STAT              (8)
#define BP_DIV_PWM_STAT                 (3)
#define BP_DIV_SPI2_PWM_STAT            (2)
#define BP_DIV_SPI1_STAT                (1)
#define BP_DIV_SPI0_STAT                (0)


// CLK_SRC_STAT
#define BP_MUX_FLASH_STAT                (28)
#define BP_MUX_PSYS_STAT                 (24)
#define BP_MUX_DSYS_STAT                 (20)
#define BP_MUX_MSYS_STAT                 (16)
#define BP_MUX_VPLL_STAT                 (12)
#define BP_MUX_EPLL_STAT                 (8)
#define BP_MUX_MPLL_STAT                 (4)
#define BP_MUX_APLL_STAT                 (0)

#if (S5PV210_EVT>0)
#define BP_MUX_DMC0_STAT                 (28)
#define BP_MUX_G2D_STAT                  (24)
#define BP_MUX_HPM_STAT                  (16)
#else
#define BP_MUX_HPM_STAT                  (16)
#define BP_MUX_DMC0_STAT                 (8)
#endif
#define BP_MUX_MFC_STAT                  (4)
#define BP_MUX_G3D_STAT                  (0)

// SWRESET
#define BP_SWRESET                       (0)

#define BP_MCS                           (0)

// System Control 
#if (S5PV210_EVT==0)
#define BP_DMC0_SEL_CLK                  (0)
#endif

//MISCELLANEOUS
#define BP_CAM_MUX_SEL                   (0)

#define BP_MIXER_OUT_SEL                 (0)

#define BP_DISPLAY_PATH_SEL              (0)

#define BP_RP_R_ENDIAN                   (3)
#define BP_RP_W_ENDIAN                   (2)
#define BP_ARM_R_ENDIAN                  (1)
#define BP_ARM_W_ENDIAN                  (0)

#define BP_MIPI_PLL_BYPASS_SEL           (31)
#define BP_MIPI_FORCE_SWAP_EN            (27)
#define BP_M_DPDN_SWAP                   (25)
#define BP_S_DPDN_SWAP                   (24)


////Bit Width(CMU)
// PLL CON
#define BW_PLL_ENABLE                   (0x1)
#define BW_PLL_LOCKED                   (0x1)
#define BW_PLL_ENABLE_LOCKED_DET        (0x1)
#if (S5PV210_EVT>0)
#define BW_APLL_AFC_ENB                 (0x1)
#define BW_APLL_AFC                     (0x1F)
#define BW_EPLL_DSM_K                   (0xFFFF)
#endif
// CLK_SRC
#define BW_MUX_FLASH_SEL                (0x1)
#define BW_MUX_PSYS_SEL                 (0x1)
#define BW_MUX_DSYS_SEL                 (0x1)
#define BW_MUX_MSYS_SEL                 (0x1)
#define BW_MUX_VPLL_SEL                 (0x1)
#define BW_MUX_EPLL_SEL                 (0x1)
#define BW_MUX_MPLL_SEL                 (0x1)
#define BW_MUX_APLL_SEL                 (0x1)

#define BW_MUX_VPLLSRC_SEL              (0x1)
#define BW_MUX_CSIS_SEL                 (0xF)
#define BW_MUX_FIMD_SEL                 (0xF)
#define BW_MUX_CAM1_SEL                 (0xF)
#define BW_MUX_CAM0_SEL                 (0xF)
#define BW_MUX_DAC_SEL                  (0x1)
#define BW_MUX_MIXER_SEL                (0x1)
#define BW_MUX_HDMI_SEL                 (0x1)

#if (S5PV210_EVT>0)
#define BW_MUX_G2D_SEL                  (0x3)
#endif
#define BW_MUX_MFC_SEL                  (0x3)
#define BW_MUX_G3D_SEL                  (0x3)

#define BW_MUX_FIMC2_LCLK_SEL           (0xF)
#define BW_MUX_FIMC1_LCLK_SEL           (0xF)
#define BW_MUX_FIMC0_LCLK_SEL           (0xF)
#define BW_MUX_MDNIE_PWMCLK_SEL         (0xF)
#define BW_MUX_MDNIE_SEL                (0xF)

#define BW_MUX_UART3_SEL                (0xF)
#define BW_MUX_UART2_SEL                (0xF)
#define BW_MUX_UART1_SEL                (0xF)
#define BW_MUX_UART0_SEL                (0xF)
#define BW_MUX_MMC3_SEL                 (0xF)
#define BW_MUX_MMC2_SEL                 (0xF)
#define BW_MUX_MMC1_SEL                 (0xF)
#define BW_MUX_MMC0_SEL                 (0xF)

#define BW_MUX_PWM_SEL                  (0xF)
#define BW_MUX_SPI2_SEL                 (0xF)
#define BW_MUX_SPI1_SEL                 (0xF)
#define BW_MUX_SPI0_SEL                 (0xF)

#define BW_MUX_DMC0_SEL                 (0x3)
#define BW_MUX_PWI_SEL                  (0xF)
#define BW_MUX_HPM_SEL                  (0x1)
#define BW_MUX_SPDIF_SEL                (0x3)
#define BW_MUX_AUDIO2_SEL               (0xF)
#define BW_MUX_AUDIO1_SEL               (0xF)
#define BW_MUX_AUDIO0_SEL               (0xF)

//CLK_SRC_MASK
#define BW_MUX_PWI_MASK                 (0x1)
#define BW_MUX_SPDIF_MASK               (0x1)
#define BW_MUX_AUDIO2_MASK              (0x1)
#define BW_MUX_AUDIO1_MASK              (0x1)
#define BW_MUX_AUDIO0_MASK              (0x1)
#define BW_MUX_PWM_MASK                 (0x1)
#define BW_MUX_SPI2_MASK                (0x1)
#define BW_MUX_SPI1_MASK                (0x1)
#define BW_MUX_SPI0_MASK                (0x1)
#define BW_MUX_UART3_MASK               (0x1)
#define BW_MUX_UART2_MASK               (0x1)
#define BW_MUX_UART1_MASK               (0x1)
#define BW_MUX_UART0_MASK               (0x1)
#define BW_MUX_MMC3_MASK                (0x1)
#define BW_MUX_MMC2_MASK                (0x1)
#define BW_MUX_MMC1_MASK                (0x1)
#define BW_MUX_MMC0_MASK                (0x1)
#define BW_MUX_VPLLSRC_MASK             (0x1)
#define BW_MUX_CSIS_MASK                (0x1)
#define BW_MUX_FIMD_MASK                (0x1)
#define BW_MUX_CAM1_MASK                (0x1)
#define BW_MUX_CAM0_MASK                (0x1)
#define BW_MUX_DAC_MASK                 (0x1)
#define BW_MUX_MIXER_MASK               (0x1)
#define BW_MUX_HDMI_MASK                (0x1)

#define BW_MUX_FIMC2_LCLK_MASK          (0x1)
#define BW_MUX_FIMC1_LCLK_MASK          (0x1)
#define BW_MUX_FIMC0_LCLK_MASK          (0x1)
#define BW_MUX_MDNIE_PWM_MASK           (0x1)
#define BW_MUX_MDNIE_MASK               (0x1)

// CLK_DIV
#define BW_DIV_PCLK_PSYS_RATIO          (0x7)
#define BW_DIV_HCLK_PSYS_RATIO          (0xF)
#define BW_DIV_PCLK_DSYS_RATIO          (0x7)
#define BW_DIV_HCLK_DSYS_RATIO          (0xF)
#define BW_DIV_PCLK_MSYS_RATIO          (0x7)
#define BW_DIV_HCLK_MSYS_RATIO          (0x7)
#define BW_DIV_A2M_RATIO                (0x7)
#define BW_DIV_APLL_RATIO               (0x7)

#define BW_DIV_CSIS_RATIO               (0xF)
#define BW_DIV_FIMD_RATIO               (0xF)
#define BW_DIV_CAM1_RATIO               (0xF)
#define BW_DIV_CAM0_RATIO               (0xF)
#define BW_DIV_FIMC_RATIO               (0xF)
#define BW_DIV_TBLK_RATIO               (0xF)

#if (S5PV210_EVT>0)
#define BW_DIV_G2D_RATIO                (0xF)
#endif
#define BW_DIV_MFC_RATIO                (0xF)
#define BW_DIV_G3D_RATIO                (0xF)

#define BW_DIV_FIMC2_LCLK_RATIO         (0xF)
#define BW_DIV_FIMC1_LCLK_RATIO         (0xF)
#define BW_DIV_FIMC0_LCLK_RATIO         (0xF)
#define BW_DIV_MDNIE_PWM_RATIO          (0x7F)
#define BW_DIV_MDNIE_RATIO              (0xF)

#define BW_DIV_UART3_RATIO              (0xF)
#define BW_DIV_UART2_RATIO              (0xF)
#define BW_DIV_UART1_RATIO              (0xF)
#define BW_DIV_UART0_RATIO              (0xF)
#define BW_DIV_MMC3_RATIO               (0xF)
#define BW_DIV_MMC2_RATIO               (0xF)
#define BW_DIV_MMC1_RATIO               (0xF)
#define BW_DIV_MMC0_RATIO               (0xF)

#define BW_DIV_PWM_RATIO                (0xF)
#define BW_DIV_SPI2_RATIO               (0xF)
#define BW_DIV_SPI1_RATIO               (0xF)
#define BW_DIV_SPI0_RATIO               (0xF)

#define BW_DIV_DMC0_RATIO               (0xF)
#define BW_DIV_PWI_RATIO                (0xF)
#define BW_DIV_HPM_RATIO                (0x7)
#define BW_DIV_COPY_RATIO               (0x7)
#define BW_DIV_ONENAND_RATIO            (0x7)
#define BW_DIV_AUDIO2_RATIO             (0xF)
#define BW_DIV_AUDIO1_RATIO             (0xF)
#define BW_DIV_AUDIO0_RATIO             (0xF)

#define BW_DIV_DPM_RATIO                (0x7F)
#define BW_DIV_DVSEM                    (0x7F)


// CLK_OUT
#define BW_CLKOUT_CLKSEL                (0x1F)
#define BW_CLKOUT_DCLKCMP               (0xF)
#define BW_CLKOUT_DCLKDIV               (0xF)
#define BW_CLKOUT_DCLKSEL               (0x7)
#define BW_CLKOUT_DCLKEN                (0x1)

#define BW_MCS                           (0x3)

// System Control 
#define BW_DMC0_SEL_CLK                  (0x1)

//MISCELLANEOUS
#define BW_CAM_MUX_SEL                   (0x3)

#define BW_MIXER_OUT_SEL                 (0x1)

#define BW_DISPLAY_PATH_SEL              (0x3)

#define BW_RP_R_ENDIAN                   (0x1)
#define BW_RP_W_ENDIAN                   (0x1)
#define BW_ARM_R_ENDIAN                  (0x1)
#define BW_ARM_W_ENDIAN                  (0x1)

#define BW_MIPI_PLL_BYPASS_SEL           (0x1)
#define BW_MIPI_FORCE_SWAP_EN            (0x1)
#define BW_M_DPDN_SWAP                   (0x1)
#define BW_S_DPDN_SWAP                   (0x1)  

//// CLK_DEF
#define PLL_ON                           (0x1)
#define PLL_OFF                          (0x0)
#define PLL_LOCKED_HWDETECT_ON           (0x1)
#define PLL_LOCKED_HWDETECT_OFF          (0x0)          
#define MUXONENAND_HCLK_PSYS             (0x0)
#define MUXONENAND_HCLK_DSYS             (0x1)
#define MUXPSYS_SCLKMPLL                 (0x0)
#define MUXPSYS_SCLKA2M                  (0x1)
#define MUXDSYS_SCLKMPLL                 (0x0)
#define MUXDSYS_SCLKA2M                  (0x1)
#define MUXMSYS_SCLKAPLL                 (0x0)
#define MUXMSYS_SCLKMPLL                 (0x1)
#define MUXVPLL_FINVPLL                  (0x0)
#define MUXVPLL_FOUTVPLL                 (0x1)
#define MUXEPLL_FINPLL                   (0x0)
#define MUXEPLL_FOUTEPLL                 (0x1)
#define MUXMPLL_FINPLL                   (0x0)
#define MUXMPLL_FOUTMPLL                 (0x1)
#define MUXAPLL_FINPLL                   (0x0)
#define MUXAPLL_FOUTAPLL                 (0x1)

#define MUXVPLLSRC_SCLK_HDMI27M          (0x1)
#define CLK_XXTI                         (0x0)
#define CLK_XUSBXTI                      (0x1)
#define CLK_SCLK_HDMI27M                 (0x2)
#define CLK_SCLK_USBPHY0                 (0x3) 
#define CLK_SCLK_USBPHY1                 (0x4)
#define CLK_SCLK_HDMIPHY                 (0x5)
#define CLK_SCLKMPLL                     (0x6)  
#define CLK_SCLKEPLL                     (0x7)
#define CLK_SCLKVPLL                     (0x8)
#define MUXDAC_SCLK_DAC                  (0x0)
#define MUXDAC_SCLK_HDMIPHY              (0x1)
#define MUXMIXER_SCLK_DAC                (0x0)
#define MUXMIXER_SCLK_HDMIPHY            (0x1)
#define MUXHDMI_SCLK_PIXEL               (0x0)
#define MUXHDMI_SCLK_HDMI_PHY            (0x1)

#define MUXDMC0_SCLKA2M                  (0x0)
#define MUXDMC0_SCLKMPLL                 (0x1)
#define MUXDMC0_SCLKEPLL                 (0x2)
#define MUXDMC0_SCLKVPLL                 (0x3)
#if (S5PV210_EVT>0)
#define MUXG2D_SCLKA2M                  (0x0)
#define MUXG2D_SCLKMPLL                 (0x1)
#define MUXG2D_SCLKEPLL                 (0x2)
#define MUXG2D_SCLKVPLL                 (0x3)
#endif
#define MUXMFC_SCLKA2M                   (0x0)
#define MUXMFC_SCLKMPLL                  (0x1)
#define MUXMFC_SCLKEPLL                  (0x2)
#define MUXMFC_SCLKVPLL                  (0x3)
#define MUXG3D_SCLKA2M                   (0x0)
#define MUXG3D_SCLKMPLL                  (0x1)
#define MUXG3D_SCLKEPLL                  (0x2)
#define MUXG3D_SCLKVPLL                  (0x3)

#define MUXHPM_SCLKAPLL                  (0x0)
#define MUXHPM_SCLKMPLL                  (0x1)
#define MUXSPDIF_SCLKAUDIO0              (0x0)
#define MUXSPDIF_SCLKAUDIO1              (0x1)
#define MUXSPDIF_SCLKAUDIO2              (0x2)

#define MUX_OUT_OFF                      (0x0)
#define MUX_OUT_ON                       (0x1)

#define CLK_OFF                          (0x0)
#define CLK_ON                           (0x1)   

#define CLKOUT_FOUTAPLLD4                (0x0)
#define CLKOUT_FOUTMPLLD2                (0x1)
#define CLKOUT_FOUTEPLL                  (0x2)
#define CLKOUT_FOUTVPLL                  (0x3)
#define CLKOUT_SCLK_USBPHY0              (0x4)
#define CLKOUT_SCLK_USBPHY1              (0x5)
#define CLKOUT_SCLK_HDMIPHY              (0x6)
#define CLKOUT_RTC                       (0x7)
#define CLKOUT_RTC_TICK_SRC              (0x8)
#define CLKOUT_HCLK_MSYS                 (0x9)
#define CLKOUT_PCLK_MSYS                 (0xA)
#define CLKOUT_HCLK_DSYS                 (0xB)
#define CLKOUT_PCLK_DSYS                 (0xC)
#define CLKOUT_HCLK_PSYS                 (0xD)
#define CLKOUT_PCLK_PSYS                 (0xE)
#define CLKOUT_ARMCLKD4                  (0xF)
#define CLKOUT_SCLK_HPM                  (0x10)
#define CLKOUT_XXTI                      (0x11)

#define DIV_STAT_STABLE                  (0x0)
#define DIV_STAT_ON_CHANGING             (0x1)

#define MUX_STAT_ON_CHANGING             (0x4)

#define SWRESET_VALUE                    (0x1)

#define DMC0_CFG_SCLK_DMC0               (0x0)
#define DMC0_CFG_HCLK_MSYS               (0x1) 

#define MIXER_OUT_SDOUT                  (0x0)
#define MIXER_OUT_HDMI                   (0x1)

#define DISPLAY_PATH_MIE                 (0x0)
#define DISPLAY_PATH_MDNIE               (0x1)
#define DISPLAY_PATH_FIMD                (0x2)

#define ARM_HIGHV                        (0x1)
#define ARM_LOWV                         (0x3)

//// Bit Positon(PMU)
#define BP_OSCUSB_EN                     (1)
#define BP_OSC_EN                        (0)

#define BP_BY_DIDLE_WAKEUP                  (19)
#define BP_BY_DSTOP_WAKEUP                  (18)
#define BP_BY_SLEEP_WAKEUP                  (16)
#define BP_BY_SWRESETESET                   (3)
#define BP_BY_nWDTRESET                     (2)
#define BP_BY_nWRESET                       (1)
#define BP_BY_nRESET                        (0)  

#define BP_CFG_STANDBYWFI                (8)

#define BP_EINT31_WAKEUP_MASK            (31)
#define BP_EINT30_WAKEUP_MASK            (30)
#define BP_EINT29_WAKEUP_MASK            (29)
#define BP_EINT28_WAKEUP_MASK            (28)
#define BP_EINT27_WAKEUP_MASK            (27)
#define BP_EINT26_WAKEUP_MASK            (26)
#define BP_EINT25_WAKEUP_MASK            (25)
#define BP_EINT24_WAKEUP_MASK            (24)
#define BP_EINT23_WAKEUP_MASK            (23)
#define BP_EINT22_WAKEUP_MASK            (22)
#define BP_EINT21_WAKEUP_MASK            (21)
#define BP_EINT20_WAKEUP_MASK            (20)
#define BP_EINT19_WAKEUP_MASK            (19)
#define BP_EINT18_WAKEUP_MASK            (18)
#define BP_EINT17_WAKEUP_MASK            (17)
#define BP_EINT16_WAKEUP_MASK            (16)
#define BP_EINT15_WAKEUP_MASK            (15)
#define BP_EINT14_WAKEUP_MASK            (14)
#define BP_EINT13_WAKEUP_MASK            (13)
#define BP_EINT12_WAKEUP_MASK            (12)
#define BP_EINT11_WAKEUP_MASK            (11)
#define BP_EINT10_WAKEUP_MASK            (10)
#define BP_EINT9_WAKEUP_MASK             (9)
#define BP_EINT8_WAKEUP_MASK             (8)
#define BP_EINT7_WAKEUP_MASK             (7)
#define BP_EINT6_WAKEUP_MASK             (6)
#define BP_EINT5_WAKEUP_MASK             (5)
#define BP_EINT4_WAKEUP_MASK             (4)
#define BP_EINT3_WAKEUP_MASK             (3)
#define BP_EINT2_WAKEUP_MASK             (2)
#define BP_EINT1_WAKEUP_MASK             (1)
#define BP_EINT0_WAKEUP_MASK             (0)
#define BP_EINT_WAKEUP_ALL_MASK         (0xFFFFFFFF)

#define BP_HDMICEC_WAKEUP_MASK           (15)
#define BP_ST_WAKEUP_MASK                (14)
#define BP_I2S_WAKEUP_MASK               (13)
#define BP_MMC3_WAKEUP_MASK              (12)
#define BP_MMC2_WAKEUP_MASK              (11)
#define BP_MMC1_WAKEUP_MASK              (10)
#define BP_MMC0_WAKEUP_MASK              (9)
#define BP_KEYIF_WAKEUP_MASK             (5)
#define BP_TS1_WAKEUP_MASK               (4)
#define BP_TS0_WAKEUP_MASK               (3)
#define BP_RTC_TICK_WAKEUP_MASK          (2)
#define BP_RTC_ALARM_WAKEUP_MASK         (1)
#define BP_WAKEUP_ALL_MASK               (0xFE3E)  

#define BP_IROM_POWER_GATE               (20)
#define BP_AUDIOSS_POWER_GATE            (7)
#define BP_CAM_POWER_GATE                (5)
#define BP_TV_POWER_GATE                 (4)
#define BP_LCD_POWER_GATE                (3)
#define BP_G3D_POWER_GATE                (2)
#define BP_MFC_POWER_GATE                (1)

#define BP_TOP_LOGIC                     (30)
#define BP_TOP_MEMORY                    (28)
#define BP_ARM_L2CACHE                   (26)
#define BP_CFG_DIDLE                     (0)

#define BP_TOP_ARM_LOGIC                 (24)
#define BP_STOP_OSCUSB_EN                (1)
#define BP_STOP_OSC_EN                   (0)

#define BP_STOP_ONENAND                  (8)
#define BP_STOP_MODEMIF                  (7)
#define BP_STOP_HOSTIF                   (6)
#define BP_STOP_USBOTG                   (5)
#define BP_STOP_HSMMC                    (4)
#define BP_STOP_CSSYS                    (3)
#define BP_STOP_SECSS                    (2)
#define BP_STOP_IRAM                     (1)

#define BP_SLEEP_OSCUSB_EN               (1)
#define BP_SLEEP_OSC_EN                  (0)

#define BP_OSC_FREQ_VALUE                (0)
#define BP_OSC_CNT_VALUE                 (0)
#define BP_PWR_CNT_VALUE                 (0)

#define BP_AUDIO_MTC_STABLE              (28)
#define BP_CAM_MTC_STABLE                (20)
#define BP_TV_MTC_STABLE                 (16)
#define BP_LCD_MTC_STABLE                (12)
#define BP_G3D_MTC_STABLE                (8)
#define BP_MFC_MTC_STABLE                (4)
#define BP_TOP_MTC_STABLE                (0)

#define BP_CLAMP_OFF_VALUE               (16)
#define BP_CLAMP_ON_VALUE                (0)

#define BP_HDMICEC_WAKEUP_STAT           (15)
#define BP_ST_WAKEUP_STAT                (14)
#define BP_I2S_WAKEUP_STAT               (13)
#define BP_MMC3_WAKEUP_STAT              (12)
#define BP_MMC2_WAKEUP_STAT              (11)
#define BP_MMC1_WAKEUP_STAT              (10)
#define BP_MMC0_WAKEUP_STAT              (9)
#define BP_MODEM_WAKEUP_STAT             (5)
#define BP_KEY_WAKEUP_STAT               (4)
#define BP_TS_WAKEUP_STAT                (3)
#define BP_RTC_TICK_WAKEUP_STAT          (2)
#define BP_RTC_ALARM_WAKEUP_STAT         (1)
#define BP_EINT_WAKEUP_STAT              (0)
#define BP_WAKEUP_STAT_ALL_MASK         (0xFE3F)

#define BP_AUDIOSS_PWR_STAT              (7)
#define BP_CAM_PWR_STAT                  (5)
#define BP_TV_PWR_STAT                   (4)
#define BP_LCD_PWR_STAT                  (3)
#define BP_G3D_PWR_STAT                  (2)
#define BP_MFC_PWR_STAT                  (1)
#define BP_TOP_PWR_STAT                  (0)

#define BP_RET_RELEAE_GPIO               (31)
#define BP_RET_RELEAE_MMC_IO             (29)
#define BP_RET_RELEAE_UART_IO            (28)
#define BP_ARM_PRESETn_TYPE              (17)
#define BP_CLKOUT_OTHER                  (8)
#define BP_CLEAR_DBGACK                  (1)
#define BP_SYSCON_INT_DISABLE            (0)

#define BP_MIE_I_SLPIN                   (1)
#define BP_MIE_I_DISPON                  (0)

#define BP_DIV_RATIO_HDMI                (16)
#define BP_ENABLE_HDMIPHY                (0)

#define BP_ENABLE_USBPHY1                (1)
#define BP_ENABLE_USBPHY0                (0)

#define BP_ENABLE_DAC                    (0)

#define BP_ENABLE_MIPI_WHOLE_DPHY        (0)
#define BP_ENABLE_MIPI_CSI_DPHY          (1)
#define BP_ENABLE_MIPI_DSI_DPHY          (2)

#define BP_ENABLE_TSADC                  (1)

#define BP_SR_PS_HOLD                    (14)
#define BP_DRV_PS_HOLD                   (12)
#define BP_PS_PS_HOLD                    (11)
#define BP_PE_PS_HOLD                    (10)
#define BP_DIR_PS_HOLD                   (9)
#define BP_DATA_PS_HOLD                  (8)
#define BP_PS_HOLD_OUT_EN                (0)                             

//// Bit Width(PMU)
#define BW_OSCUSB_EN                     (0x1)
#define BW_OSC_EN                        (0x1)

#define BW_DIDLE_WAKEUP                  (0x1)
#define BW_DSTOP_WAKEUP                  (0x1)
#define BW_SLEEP_WAKEUP                  (0x1)
#define BW_SWRESET                       (0x1)
#define BW_nWDTRESET                     (0x1)
#define BW_nWRESET                       (0x1)
#define BW_nRESET                        (0x1)  

#define BW_CFG_STANDBYWFI                (0x3)

#define BW_EINT31_WAKEUP_MASK            (0x1)
#define BW_EINT30_WAKEUP_MASK            (0x1)
#define BW_EINT29_WAKEUP_MASK            (0x1)
#define BW_EINT28_WAKEUP_MASK            (0x1)
#define BW_EINT27_WAKEUP_MASK            (0x1)
#define BW_EINT26_WAKEUP_MASK            (0x1)
#define BW_EINT25_WAKEUP_MASK            (0x1)
#define BW_EINT24_WAKEUP_MASK            (0x1)
#define BW_EINT23_WAKEUP_MASK            (0x1)
#define BW_EINT22_WAKEUP_MASK            (0x1)
#define BW_EINT21_WAKEUP_MASK            (0x1)
#define BW_EINT20_WAKEUP_MASK            (0x1)
#define BW_EINT19_WAKEUP_MASK            (0x1)
#define BW_EINT18_WAKEUP_MASK            (0x1)
#define BW_EINT17_WAKEUP_MASK            (0x1)
#define BW_EINT16_WAKEUP_MASK            (0x1)
#define BW_EINT15_WAKEUP_MASK            (0x1)
#define BW_EINT14_WAKEUP_MASK            (0x1)
#define BW_EINT13_WAKEUP_MASK            (0x1)
#define BW_EINT12_WAKEUP_MASK            (0x1)
#define BW_EINT11_WAKEUP_MASK            (0x1)
#define BW_EINT10_WAKEUP_MASK            (0x1)
#define BW_EINT9_WAKEUP_MASK             (0x1)
#define BW_EINT8_WAKEUP_MASK             (0x1)
#define BW_EINT7_WAKEUP_MASK             (0x1)
#define BW_EINT6_WAKEUP_MASK             (0x1)
#define BW_EINT5_WAKEUP_MASK             (0x1)
#define BW_EINT4_WAKEUP_MASK             (0x1)
#define BW_EINT3_WAKEUP_MASK             (0x1)
#define BW_EINT2_WAKEUP_MASK             (0x1)
#define BW_EINT1_WAKEUP_MASK             (0x1)
#define BW_EINT0_WAKEUP_MASK             (0x1)

#define BW_HDMICEC_WAKEUP_MASK           (0x1)
#define BW_ST_WAKEUP_MASK                (0x1)
#define BW_I2S_WAKEUP_MASK               (0x1)
#define BW_MMC3_WAKEUP_MASK              (0x1)
#define BW_MMC2_WAKEUP_MASK              (0x1)
#define BW_MMC1_WAKEUP_MASK              (0x1)
#define BW_MMC0_WAKEUP_MASK              (0x1)
#define BW_MODEM_WAKEUP_MASK             (0x1)
#define BW_KEYIF_WAKEUP_MASK             (0x1)
#define BW_TS_WAKEUP_MASK                (0x1)
#define BW_RTC_TICK_WAKEUP_MASK          (0x1)
#define BW_RTC_ALARM_WAKEUP_MASK         (0x1)

#define BW_IROM_POWER_GATE               (0x1)
#define BW_AUDIOSS_POWER_GATE            (0x1)
#define BW_CAM_POWER_GATE                (0x1)
#define BW_TV_POWER_GATE                 (0x1)
#define BW_LCD_POWER_GATE                (0x1)
#define BW_G3D_POWER_GATE                (0x1)
#define BW_MFC_POWER_GATE                (0x1)

#define BW_TOP_LOGIC                     (0x3)
#define BW_TOP_MEMORY                    (0x3)
#define BW_ARM_L2CACHE                   (0x3)
#define BW_CFG_DIDLE                     (0x1)

#define BW_TOP_ARM_LOGIC                 (0x3)
#define BW_STOP_OSCUSB_EN                (0x1)
#define BW_STOP_OSC_EN                   (0x1)

#define BW_STOP_ONENAND                  (0x1)
#define BW_STOP_MODEMIF                  (0x1)
#define BW_STOP_HOSTIF                   (0x1)
#define BW_STOP_USBOTG                   (0x1)
#define BW_STOP_HSMMC                    (0x1)
#define BW_STOP_CSSYS                    (0x1)
#define BW_STOP_SECSS                    (0x1)
#define BW_STOP_IRAM                     (0x1)

#define BW_SLEEP_OSCUSB_EN               (0x1)
#define BW_SLEEP_OSC_EN                  (0x1)

#define BW_OSC_FREQ_VALUE                (0xF)
#define BW_OSC_CNT_VALUE                 (0xFFFFF)
#define BW_PWR_CNT_VALUE                 (0xFFFFF)

#define BW_AUDIO_MTC_STABLE              (0xF)
#define BW_CAM_MTC_STABLE                (0xF)
#define BW_TV_MTC_STABLE                 (0xF)
#define BW_LCD_MTC_STABLE                (0xF)
#define BW_G3D_MTC_STABLE                (0xF)
#define BW_MFC_MTC_STABLE                (0xF)
#define BW_TOP_MTC_STABLE                (0xF)

#define BW_CLAMP_OFF_VALUE               (0x3FF)
#define BW_CLAMP_ON_VALUE                (0x3FF)

#define BW_HDMICEC_WAKEUP_STAT           (0x1)
#define BW_ST_WAKEUP_STAT                (0x1)
#define BW_I2S_WAKEUP_STAT               (0x1)
#define BW_MMC3_WAKEUP_STAT              (0x1)
#define BW_MMC2_WAKEUP_STAT              (0x1)
#define BW_MMC1_WAKEUP_STAT              (0x1)
#define BW_MMC0_WAKEUP_STAT              (0x1)
#define BW_MODEM_WAKEUP_STAT             (0x1)
#define BW_KEY_WAKEUP_STAT               (0x1)
#define BW_TS_WAKEUP_STAT                (0x1)
#define BW_RTC_TICK_WAKEUP_STAT          (0x1)
#define BW_RTC_ALARM_WAKEUP_STAT         (0x1)
#define BW_EINT_WAKEUP_STAT              (0x1)

#define BW_AUDIOSS_PWR_STAT              (0x1)
#define BW_CAM_PWR_STAT                  (0x1)
#define BW_TV_PWR_STAT                   (0x1)
#define BW_LCD_PWR_STAT                  (0x1)
#define BW_G3D_PWR_STAT                  (0x1)
#define BW_MFC_PWR_STAT                  (0x1)
#define BW_TOP_PWR_STAT                  (0x1)

#define BW_RET_RELEAE_GPIO               (0x1)
#define BW_RET_RELEAE_MMC_IO             (0x1)
#define BW_RET_RELEAE_UART_IO            (0x1)
#define BW_ARM_PRESETn_TYPE              (0x1)
#define BW_CLKOUT_OTHER                  (0x3)
#define BW_CLEAR_DBGACK                  (0x1)
#define BW_SYSCON_INT_DISABLE            (0x1)

#define BW_MIE_I_SLPIN                   (0x1)
#define BW_MIE_I_DISPON                  (0x1)

#define BW_DIV_RATIO_HDMI                (0x3FF)
#define BW_ENABLE_HDMIPHY                (0x1)

#define BW_ENABLE_USBPHY1                (0x1)
#define BW_ENABLE_USBPHY0                (0x1)

#define BW_ENABLE_DAC                    (0x1)

#define BW_ENABLE_MIPI_WHOLE_DPHY        (0x1)
#define BW_ENABLE_MIPI_DSI_DPHY          (0x1)
#define BW_ENABLE_MIPI_CSI_DPHY          (0x1)
#define BW_ENABLE_TSADC                  (0x1)

#define BW_SR_PS_HOLD                    (0x1)
#define BW_DRV_PS_HOLD                   (0x3)
#define BW_PS_PS_HOLD                    (0x1)
#define BW_PE_PS_HOLD                    (0x1)
#define BW_DIR_PS_HOLD                   (0x1)
#define BW_DATA_PS_HOLD                  (0x1)
#define BW_PS_HOLD_OUT_EN                (0x1)          


// PMU Define
#define OSC_ENABLE                       (1)
#define OSC_DISABLE                      (0)

#define CFG_IGNORE                       (0x0)
#define CFG_ENTER_IDLE                   (0x1)
#define CFG_ENTER_STOP                   (0x2)
#define CFG_ENTER_SLEEP                  (0x3)

#define ENABLE_WAKEUP_SRC                (0x0)
#define DISABLE_WAKEUP_SRC               (0x1)

#define PWR_ON                           (0x1)
#define PWR_OFF                          (0x0)

#define TOP_LOGIC_RET                    (0x1)
#define TOP_LOGIC_ON                     (0x2)
#define TOP_MEMORY_RET                   (0x1)
#define TOP_MEMORY_ON                    (0x2)
#define ARM_L2CACHE_RET                 (0x1)
#define ARM_L2CACHE_OFF                  (0x0)
#define CFG_DEEP_IDLE                    (0x1)
#define CFG_NODEEP_IDLE                  (0x0)

#define ARM_LOGIC_ON                     (0x2)
#define ARM_LOGIC_OFF                    (0x0)

#define STOP_MEM_RET                     (0x1)
#define STOP_MEM_OFF                     (0x0)

#define BLK_PWR_ON_STAT                  (0x1)
#define BLK_PWR_OFF_STAT                 (0x0)

#define RET_RELEASE_ON                   (0x1)   

#define PHY_ENABLE                       (0x1)
#define PHY_DISABLE                      (0x0)

#define MIPI_DSI_DPHY_ENABLE             (0x1)
#define MIPI_CSI_DPHY_ENABLE             (0x1)
#define MIPI_DPHY_ENABLE                 (0x1)

#define MIPI_DSI_DPHY_DISABLE            (0x0)
#define MIPI_CSI_DPHY_DISABLE            (0x0)
#define MIPI_DPHY_DISABLE                (0x0)

#define DAC_ENABLE                       (0x1)
#define DAC_DISABLE                      (0x0)

#define TSADC_ENABLE                     (0x1)
#define TSADC_DISABLE                    (0x0)

#define HDMICEC_WAKEUP_STAT           (1<<BP_HDMICEC_WAKEUP_STAT)
#define ST_WAKEUP_STAT                 (1<<BP_ST_WAKEUP_STAT)
#define I2S_WAKEUP_STAT                (1<<BP_I2S_WAKEUP_STAT)
#define MMC3_WAKEUP_STAT               (1<<BP_MMC3_WAKEUP_STAT)
#define MMC2_WAKEUP_STAT               (1<<BP_MMC2_WAKEUP_STAT)
#define MMC1_WAKEUP_STAT               (1<<BP_MMC1_WAKEUP_STAT)
#define MMC0_WAKEUP_STAT               (1<<BP_MMC0_WAKEUP_STAT)
#define MODEM_WAKEUP_STAT              (1<<BP_MODEM_WAKEUP_STAT)
#define KEY_WAKEUP_STAT                (1<<BP_KEY_WAKEUP_STAT)
#define TS_WAKEUP_STAT                 (1<<BP_TS_WAKEUP_STAT)
#define RTC_TICK_WAKEUP_STAT          (1<<BP_RTC_TICK_WAKEUP_STAT)
#define RTC_ALARM_WAKEUP_STAT         (1<<BP_RTC_ALARM_WAKEUP_STAT)
#define EINT_WAKEUP_STAT               (1<<BP_EINT_WAKEUP_STAT)
                       
#define BP_OTHERS_ENABLE_GPIO           (31)
#define BP_OTHERS_ENABLE_CF_IO          (30)
#define BP_OTHERS_ENABLE_MMC_IO         (29)
#define BP_OTHERS_ENABLE_UART_IO        (28)
#define BP_OTHERS_SYSCON_INT_DISABLE    (0)

#define ENABLE_GPIO                     (1<<BP_OTHERS_ENABLE_GPIO)
#define ENABLE_CF_IO                    (1<<BP_OTHERS_ENABLE_CF_IO)
#define ENABLE_MMC_IO                   (1<<BP_OTHERS_ENABLE_MMC_IO)
#define ENABLE_UART_IO                  (1<<BP_OTHERS_ENABLE_UART_IO)
                
#if __cplusplus
    }
#endif

#endif    // __S5PV210_SYSCON_H

