//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
//
// Use of this sample source code is subject to the terms of the Microsoft
// license agreement under which you licensed this sample source code. If
// you did not accept the terms of the license agreement, you are not
// authorized to use this sample source code. For the terms of the license,
// please see the license agreement between you and Microsoft or, if applicable,
// see the LICENSE.RTF on your install media or the root of your tools installation.
// THE SAMPLE SOURCE CODE IS PROVIDED "AS IS", WITH NO WARRANTIES OR INDEMNITIES.
//
//
// Copyright (c) Samsung Electronics. Co. LTD.  All rights reserved.
//
/*++
THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Module Name:    dvfs.h

Abstract:       Dynamic Voltage and Frequency Scaling Interface & defines

Functions:


Notes:


--*/

#ifndef _S5PV210_DVFS_H_
#define _S5PV210_DVFS_H_

#include "pmplatform.h"



/****************************************************
    SYMBOL DEFINE : Emergency Level up target power level for the INTERRUPT MONITOR

        UPPERMOST_LIMIT     : Target power level when the emergency level up.
                                          When supporting the 833MHZ DVFS, IEM need to be operated, and
                                          it takes several time. It would be better to promote up the L1 rather
                                          than the L0 for avoiding the additional SW load of IEM.

        MIN_SUSTAIN_TIME   : Initial Endurance time after emergency level up

        SNAPSHOT_PERIOD   : Time to monitor snapshot


****************************************************/
#define UPPERMOST_LIMIT     SYS_L1

#define SNAPSHOT_PERIOD     (OEM_COUNT_1MS * 16)
#define MIN_SUSTAIN_TIME        SNAPSHOT_PERIOD * 16

#define MAX_CPU_RATIO       (0x1 << 7)
#define RATIO_PERCENTAGE(v) (v * MAX_CPU_RATIO / 100)


/****************************************************
    FUNCTION MACRO : Function to check the simple condition. This kind of macro prevent the load of function call.

        EMGRCHECK_BEFOREHAND() : Pre-check the Emergency levelup check in INTERRUPT MONITOR

        DVFSCHECK_BEFOREHAND() : Pre-check the DVFS execution

        DVFS_TRANSIT_SET()     : DVFS_TransitionDone setting

****************************************************/
#define EMGRCHECK_BEFOREHAND(val)         ((val->CurrentPowerLevel > UPPERMOST_LIMIT) && (val->DVFSTransitionDone == TRUE))
#define DVFSCHECK_BEFOREHAND(bspargs)   (bspargs->DVFSTransitionDone == TRUE)
#define DVFS_TRANSIT_SET(bspargs,val)	{			\
					bspargs->DVFSTransitionDone = val;	\
			} while(0)
/****************************************************
    FUNCTION MACRO : Function to intercept the timer interrupt. This kind of macro prevent the load of function call.

        SetIntercepted()           : Set the intercept flag.

****************************************************/
#define SetIntercepted(InterceptFlag, Flag)             {   \
                    InterceptFlag = Flag;                                 \
            } while(0)

/****************************************************
    FUNCTION MACRO : Function to update Snapshot. This kind of macro prevent the load of function call.

        UpdateTotalSnapshot(),  UpdateIdleSnapshot(), ResetSnapshot() : Update Total/Idle snapshot

        ExpireSnapshot() : expire the snapshot, and reset snapshot.

****************************************************/
#define UpdateTotalSnapshot(BSPARG, Total, cnt)    {      \
                    Total += cnt;                                                                               \
                    if(BSPARG->DVFSTransitionDone == TRUE)  BSPARG->CPU_RunTime += cnt;     \
            } while(0)

#define UpdateIdleSnapshot(BSPARG, Idle, cnt)     {                       \
                    Idle += cnt;                                                                                 \
                    if(BSPARG->DVFSTransitionDone == TRUE) BSPARG->CPU_IdleTime += cnt;         \
            } while(0)

#define ResetSnapshot(Total, Idle)                                         {                      \
                    Total = 0;                                                                                     \
                    Idle  = 0;                                                                                   \
           } while(0)

#define ExpireSnapshot(Total, Idle)                                      {              \
                    if(Total > SNAPSHOT_PERIOD) ResetSnapshot(Total, Idle);           \
            } while(0)

/****************************************************
    FUNCTION  : Function to control Dynamic  Voltage and Frequency Scaling.

        InitializeDVFS()            : Initialize DVFS.

        CheckEmergencyIntr() : Interrupt Monitor - Check the pre-defined interrupt of which
                                                                        the power level is to be promoted up immediately.

        DecideDVFS()              : Adjust Endurance time, Determines the DVFS level.

****************************************************/
BOOL    InitializeDVFS(void);
BOOL    CheckEmergencyIntr(DWORD PhyIRQ_ID);
BOOL    DecideDVFS(DWORD TotalSnapshot, DWORD IdleSnapshot);

#endif //_S5PV210_DVFS_H_

