//
// Copyright (c) Samsung Electronics.  All rights reserved.
//
//
//------------------------------------------------------------------------------
//
//  Header:  drvmsg.h
//
//  This header file defines OAL Driver Message interface. The module implements
//  functions and varibles related to runtime debug message handling in drivers
//
//  Export for kernel/public interface:
//      * OEMCacheRangeFlush
//
//  Export for other OAL modules/protected interface:
//      * OALCacheGlobalsInit
//
//  Intenal variables/functions
//      * g_oalCacheInfo
//

#ifndef __OAL_DRVMSG_H
#define __OAL_DRVMSG_H

#include "oal.h"

#if __cplusplus
extern "C" {
#endif

//------------------------------------------------------------------------------
//
//  Define: total number of DBGMSG Variables
//
#define DEF_DRVLOG_MASK			0
#define DEF_DEVELOPER_MASK		(1<<MSG_INFO)
#define INITIAL_LOGMASK_VAL 	(DEF_DRVLOG_MASK|(DEF_DRVLOG_MASK<<8)|\
								(DEF_DRVLOG_MASK<<16)|(DEF_DRVLOG_MASK<<24))
#define DRVLOG_PART_NUM			15


//------------------------------------------------------------------------------
//
//  Extern:  g_DrvMsgMask[N]
//
//  Exports the global trace mask used by the DBGMSG macro.
//

#if 1
#ifdef DRVMSG_STATIC_LIBRARY
	#ifdef DMSGEXTERN
		#undef DMSGEXTERN
	#endif
	#define DMSGEXTERN	extern
#else
	#ifdef DRVMSG_LIBRARY
		//For used in drvmsg.c
		#ifdef DMSGEXTERN
			#undef DMSGEXTERN
		#endif
		#define DMSGEXTERN	__declspec(dllexport)
		__declspec(dllexport) void Print_DrvMsgMask(void); //for debugging - JMKO
	#else
		//For use with driver
		#ifdef DMSGEXTERN
			#undef DMSGEXTERN
		#endif
		#define DMSGEXTERN	extern __declspec(dllimport)
		__declspec(dllimport) void Print_DrvMsgMask(void); //for debugging - JMKO
	#endif
#endif
#endif //#if1

#if 0
#ifdef DRVMSG_LIBRARY  
//For used in drvmsg.c
#define DMSGEXTERN	extern __declspec(dllexport)
#elif defined(DRVMSG_STATIC_LIBRARY)
#define DMSGEXTERN	extern 
#else
//For use with driver
#define DMSGEXTERN	extern __declspec(dllimport) 
#endif
#endif //#if0

DMSGEXTERN unsigned int g_DrvMsgMask[DRVLOG_PART_NUM];

DMSGEXTERN void DrvMSGInitialize(void);
DMSGEXTERN void DrvMSGDeinitialize(void);

//------------------------------------------------------------------------------
// Miscellaneout definitions
//
#define SN_DRVMSG_ROOT			HKEY_LOCAL_MACHINE
#define SN_DRVMSG_PATH			TEXT("System\\State\\drvmsg")
#define SN_DRVMSG_VALUE 		TEXT("drvmask")
#define SN_DRVMSG_BITMASK		0xFFF

// For colors in ANSI terminal
#define ANSI_CLEAR				L"\x1b[0m"
#define ANSI_RED				L"\x1b[31m"
#define ANSI_L_RED 				L"\x1b[47;31m"
#define ANSI_GREEN				L"\x1b[32m"
#define ANSI_L_GREEN			L"\x1b[47;32m"
#define ANSI_YELLOW 			L"\x1b[33m"
#define ANSI_L_YELLOW			L"\x1b[47;33m"
#define ANSI_BLUE				L"\x1b[34m"
#define ANSI_L_BLUE				L"\x1b[47;34m"
#define ANSI_MAGENTA			L"\x1b[35m"
#define ANSI_L_MAGENTA			L"\x1b[47;35m"
#define ANSI_CYAN				L"\x1b[36m"
#define ANSI_L_CYAN				L"\x1b[47;36m"
#define ANSI_L_WHITE			L"\x1b[47;37m"


//------------------------------------------------------------------------------
//
//  Macro:  DBGMSG
//
//  This macro should be used for logging in Drivers. It supports log zones. All
//  tracing messages based on this macro is removed when SHIP_BUILD is
//  defined.
//

#ifdef ERRMSG //"public\wpc\cesysgen\oak\inc\wavedbg.h" Խ redefined warining  - wavemain.h
#undef ERRMSG
#endif

#ifdef WRNMSG
#undef WRNMSG
#endif

#ifdef SHIP_BUILD
#define DBGMSG(cond, exp)  		((void)FALSE)
#define ERRMSG(exp)				(void)(NKDbgPrintfW exp)
#define WRNMSG(exp)				(void)(NKDbgPrintfW exp)
#define FONMSG(exp)				(void)(NKDbgPrintfW exp)
#else
#define DBGMSG(cond, exp)   	((void)((cond)?(NKDbgPrintfW exp), TRUE : FALSE))
#define ERRMSG(exp)				{NKDbgPrintfW(ANSI_RED); (void)(NKDbgPrintfW exp);NKDbgPrintfW(ANSI_CLEAR);}
#define WRNMSG(exp)				{NKDbgPrintfW(ANSI_BLUE);(void)(NKDbgPrintfW exp);NKDbgPrintfW(ANSI_CLEAR);}
#define FONMSG(exp)				((void)FALSE)
#endif


//------------------------------------------------------------------------------
//
//  Define:  DRVZONE
//
//  Tests if a zone is being traced. Used in conditional in OALMSG macro.
//
#define DRVZONE(n, m)		(g_DrvMsgMask[n/4]&(1<<((n%4)*8+m)))


//------------------------------------------------------------------------------
//
//  Define:  MSG Flags Bit : Uses only 8 bit for 8 differnt messages per driver.
//
#define MSG_INFO	0
#define MSG_FUNC	1
#define MSG_DEBUG	2
#define MSG_PM		3
#define MSG_USER1	4
#define MSG_USER2	5
#define MSG_USER3	6
#define MSG_USER4	7


//------------------------------------------------------------------------------
//
//  Define:  Driver List
//
typedef enum
{
	DRIVER_DISPLAY = 0,
	DRIVER_ROTATOR,
	DRIVER_POST,
	DRIVER_TVOUT,

	DRIVER_ISM,
	DRIVER_CAMERAFILTER,
	DRIVER_CAMERACIF,
	DRIVER_CAMIF,

	DRIVER_AUDIO,
	DRIVER_AC97,
	DRIVER_SPI,
	DRIVER_ECHO,

	DRIVER_PM,
	DRIVER_POWER,
	DRIVER_PWC,
	DRIVER_BATTERY,
	
	DRIVER_LED,
	DRIVER_KEYPAD,
	DRIVER_TOUCH,
	DRIVER_BACKLIGHT,
	
	DRIVER_ONENAND,
	DRIVER_FLASH,
	DRIVER_SDHC,
	DRIVER_PCMCIA,	

	DRIVER_DMA,
	DRIVER_MFD,
	DRIVER_MMI,
	DRIVER_IIC,

	DRIVER_SERIAL,
	DRIVER_BTSERIAL,
	DRIVER_BLUETOOTH,
	DRIVER_BTPROFILE,

	DRIVER_RTC,
	DRIVER_RIL,
	DRIVER_RCD,
	DRIVER_DPRAM,

	DRIVER_SIM,
	DRIVER_WIBRO,
	DRIVER_WLAN,
	DRIVER_TDMB,
	
	DRIVER_ACC,
	DRIVER_GPS,	
	DRIVER_MFC,
	DRIVER_JPG,

	DRIVER_MITSMAN,
	DRIVER_USBFN,
	DRIVER_USBH,
	DRIVER_USBMODEM,
	
	DRIVER_ProximitySensors,
	DRVLIB_SPEAKERPHONE,
	DRVLIB_HEADSET,
	DRVLIB_FMRADIO,

	DRIVER_PWRBTN,
	DRIVER_DRVLIB,
	DRIVER_INTERNAL,
	DRIVER_RESERVED55,
	
	DRIVER_RESERVED56,
	DRIVER_RESERVED57,
	DRVLIB_USE,
	INTERNAL_USE
} DrvList;


//------------------------------------------------------------------------------
//
//  Define:  DRIVER_XXX  (Use 3 character for the driver acronym. 4 characters max!)
//
//  Driver trace zones for DBGMSG function. The following trace zones may
//  be combined, via boolean operators, to support Driver debug tracing.
//

/* Display Driver */
#define DISP_INFO		DRVZONE(DRIVER_DISPLAY, MSG_INFO)
#define DISP_FUNC		DRVZONE(DRIVER_DISPLAY, MSG_FUNC)
#define DISP_DBG		DRVZONE(DRIVER_DISPLAY, MSG_DEBUG)
#define DISP_PM			DRVZONE(DRIVER_DISPLAY, MSG_PM)
#define DISP_USR1		DRVZONE(DRIVER_DISPLAY, MSG_USER1)
#define DISP_USR2		DRVZONE(DRIVER_DISPLAY, MSG_USER2)
#define DISP_USR3		DRVZONE(DRIVER_DISPLAY, MSG_USER3)
#define DISP_USR4		DRVZONE(DRIVER_DISPLAY, MSG_USER4)

/* Rotator */
#define ROT_INFO		DRVZONE(DRIVER_ROTATOR, MSG_INFO)
#define ROT_FUNC		DRVZONE(DRIVER_ROTATOR, MSG_FUNC)
#define ROT_DBG			DRVZONE(DRIVER_ROTATOR, MSG_DEBUG)
#define ROT_PM			DRVZONE(DRIVER_ROTATOR, MSG_PM)
#define ROT_USR1		DRVZONE(DRIVER_ROTATOR, MSG_USER1)
#define ROT_USR2		DRVZONE(DRIVER_ROTATOR, MSG_USER2)
#define ROT_USR3		DRVZONE(DRIVER_ROTATOR, MSG_USER3)
#define ROT_USR4		DRVZONE(DRIVER_ROTATOR, MSG_USER4)

/* Post */
#define POST_INFO		DRVZONE(DRIVER_POST, MSG_INFO)
#define POST_FUNC		DRVZONE(DRIVER_POST, MSG_FUNC)
#define POST_DBG		DRVZONE(DRIVER_POST, MSG_DEBUG)
#define POST_PM			DRVZONE(DRIVER_POST, MSG_PM)
#define POST_USR1		DRVZONE(DRIVER_POST, MSG_USER1)
#define POST_USR2		DRVZONE(DRIVER_POST, MSG_USER2)
#define POST_USR3		DRVZONE(DRIVER_POST, MSG_USER3)
#define POST_USR4		DRVZONE(DRIVER_POST, MSG_USER4)



#if 0
/* TVOut */
#define TV_INFO			DRVZONE(DRIVER_TVOUT, MSG_INFO)
#define TV_FUNC			DRVZONE(DRIVER_TVOUT, MSG_FUNC)
#define TV_DBG			DRVZONE(DRIVER_TVOUT, MSG_DEBUG)
#define TV_PM			DRVZONE(DRIVER_TVOUT, MSG_PM)
#define TV_USR1			DRVZONE(DRIVER_TVOUT, MSG_USER1)
#define TV_USR2			DRVZONE(DRIVER_TVOUT, MSG_USER2)
#define TV_USR3			DRVZONE(DRIVER_TVOUT, MSG_USER3)
#define TV_USR4			DRVZONE(DRIVER_TVOUT, MSG_USER4)

#else       //modified by terry for test
#define TV_INFO			0
#define TV_FUNC			0
#define TV_DBG			0
#define TV_PM			0
#define TV_USR1			0
#define TV_USR2			0
#define TV_USR3			0
#define TV_USR4			0 
#endif


/* ISM2300 Driver */
#define ISM_INFO		DRVZONE(DRIVER_ISM, MSG_INFO)
#define ISM_FUNC		DRVZONE(DRIVER_ISM, MSG_FUNC)
#define ISM_DBG			DRVZONE(DRIVER_ISM, MSG_DEBUG)
#define ISM_PM			DRVZONE(DRIVER_ISM, MSG_PM)
#define ISM_USR1		DRVZONE(DRIVER_ISM, MSG_USER1)
#define ISM_USR2		DRVZONE(DRIVER_ISM, MSG_USER2)
#define ISM_USR3		DRVZONE(DRIVER_ISM, MSG_USER3)
#define ISM_USR4		DRVZONE(DRIVER_ISM, MSG_USER4)

/* Camera Filter*/
#define CAMF_INFO		DRVZONE(DRIVER_CAMERAFILTER, MSG_INFO)
#define CAMF_FUNC		DRVZONE(DRIVER_CAMERAFILTER, MSG_FUNC)
#define CAMF_DBG		DRVZONE(DRIVER_CAMERAFILTER, MSG_DEBUG)
#define CAMF_PM			DRVZONE(DRIVER_CAMERAFILTER, MSG_PM)
#define CAMF_USR1		DRVZONE(DRIVER_CAMERAFILTER, MSG_USER1)
#define CAMF_USR2		DRVZONE(DRIVER_CAMERAFILTER, MSG_USER2)
#define CAMF_USR3		DRVZONE(DRIVER_CAMERAFILTER, MSG_USER3)
#define CAMF_USR4		DRVZONE(DRIVER_CAMERAFILTER, MSG_USER4)

/* CIF Camera */
#define CAMC_INFO		DRVZONE(DRIVER_CAMERACIF, MSG_INFO)
#define CAMC_FUNC		DRVZONE(DRIVER_CAMERACIF, MSG_FUNC)
#define CAMC_DBG		DRVZONE(DRIVER_CAMERACIF, MSG_DEBUG)
#define CAMC_PM			DRVZONE(DRIVER_CAMERACIF, MSG_PM)
#define CAMC_USR1		DRVZONE(DRIVER_CAMERACIF, MSG_USER1)
#define CAMC_USR2		DRVZONE(DRIVER_CAMERACIF, MSG_USER2)
#define CAMC_USR3		DRVZONE(DRIVER_CAMERACIF, MSG_USER3)
#define CAMC_USR4		DRVZONE(DRIVER_CAMERACIF, MSG_USER4)

/* Cam IF */
#define CIF_INFO		DRVZONE(DRIVER_CAMIF, MSG_INFO)
#define CIF_FUNC		DRVZONE(DRIVER_CAMIF, MSG_FUNC)
#define CIF_DBG			DRVZONE(DRIVER_CAMIF, MSG_DEBUG)
#define CIF_PM			DRVZONE(DRIVER_CAMIF, MSG_PM)
#define CIF_USR1		DRVZONE(DRIVER_CAMIF, MSG_USER1)
#define CIF_USR2		DRVZONE(DRIVER_CAMIF, MSG_USER2)
#define CIF_USR3		DRVZONE(DRIVER_CAMIF, MSG_USER3)
#define CIF_USR4		DRVZONE(DRIVER_CAMIF, MSG_USER4)

/* Wavedev Audio */
#define WAVE_INFO		DRVZONE(DRIVER_AUDIO, MSG_INFO)
#define WAVE_FUNC		DRVZONE(DRIVER_AUDIO, MSG_FUNC)
#define WAVE_DBG		DRVZONE(DRIVER_AUDIO, MSG_DEBUG)
#define WAVE_PM			DRVZONE(DRIVER_AUDIO, MSG_PM)
#define WAVE_USR1		DRVZONE(DRIVER_AUDIO, MSG_USER1)
#define WAVE_USR2		DRVZONE(DRIVER_AUDIO, MSG_USER2)
#define WAVE_USR3		DRVZONE(DRIVER_AUDIO, MSG_USER3)
#define WAVE_USR4		DRVZONE(DRIVER_AUDIO, MSG_USER4)

/* AC97 Link */
#define AC97_INFO		DRVZONE(DRIVER_AC97, MSG_INFO)
#define AC97_FUNC		DRVZONE(DRIVER_AC97, MSG_FUNC)
#define AC97_DBG		DRVZONE(DRIVER_AC97, MSG_DEBUG)
#define AC97_PM			DRVZONE(DRIVER_AC97, MSG_PM)
#define AC97_USR1		DRVZONE(DRIVER_AC97, MSG_USER1)
#define AC97_USR2		DRVZONE(DRIVER_AC97, MSG_USER2)
#define AC97_USR3		DRVZONE(DRIVER_AC97, MSG_USER3)
#define AC97_USR4		DRVZONE(DRIVER_AC97, MSG_USER4)


/* SPI */
#define SPI_INFO		DRVZONE(DRIVER_SPI, MSG_INFO)
#define SPI_FUNC		DRVZONE(DRIVER_SPI, MSG_FUNC)
#define SPI_DBG		        DRVZONE(DRIVER_SPI, MSG_DEBUG)
#define SPI_PM		        DRVZONE(DRIVER_SPI, MSG_PM)
#define SPI_USR1		DRVZONE(DRIVER_SPI, MSG_USER1)
#define SPI_USR2		DRVZONE(DRIVER_SPI, MSG_USER2)
#define SPI_USR3		DRVZONE(DRIVER_SPI, MSG_USER3)
#define SPI_USR4		DRVZONE(DRIVER_SPI, MSG_USER4)

/* Echo Cancellor */
#define ECHO_INFO		DRVZONE(DRIVER_ECHO, MSG_INFO)
#define ECHO_FUNC		DRVZONE(DRIVER_ECHO, MSG_FUNC)
#define ECHO_DBG		DRVZONE(DRIVER_ECHO, MSG_DEBUG)
#define ECHO_PM			DRVZONE(DRIVER_ECHO, MSG_PM)
#define ECHO_USR1		DRVZONE(DRIVER_ECHO, MSG_USER1)
#define ECHO_USR2		DRVZONE(DRIVER_ECHO, MSG_USER2)
#define ECHO_USR3		DRVZONE(DRIVER_ECHO, MSG_USER3)
#define ECHO_USR4		DRVZONE(DRIVER_ECHO, MSG_USER4)

/* PM */
#define PM_INFO			DRVZONE(DRIVER_PM, MSG_INFO)
#define PM_FUNC			DRVZONE(DRIVER_PM, MSG_FUNC)
#define PM_DBG			DRVZONE(DRIVER_PM, MSG_DEBUG)
#define PM_PM			DRVZONE(DRIVER_PM, MSG_PM)
#define PM_USR1			DRVZONE(DRIVER_PM, MSG_USER1)
#define PM_USR2			DRVZONE(DRIVER_PM, MSG_USER2)
#define PM_USR3			DRVZONE(DRIVER_PM, MSG_USER3)
#define PM_USR4			DRVZONE(DRIVER_PM, MSG_USER4)

/* Power */
#define PWR_INFO		DRVZONE(DRIVER_POWER, MSG_INFO)
#define PWR_FUNC		DRVZONE(DRIVER_POWER, MSG_FUNC)
#define PWR_DBG			DRVZONE(DRIVER_POWER, MSG_DEBUG)
#define PWR_PM			DRVZONE(DRIVER_POWER, MSG_PM)
#define PWR_USR1		DRVZONE(DRIVER_POWER, MSG_USER1)
#define PWR_USR2		DRVZONE(DRIVER_POWER, MSG_USER2)
#define PWR_USR3		DRVZONE(DRIVER_POWER, MSG_USER3)
#define PWR_USR4		DRVZONE(DRIVER_POWER, MSG_USER4)


/* PWC */
#define PWC_INFO		DRVZONE(DRIVER_PWC, MSG_INFO)
#define PWC_FUNC		DRVZONE(DRIVER_PWC, MSG_FUNC)
#define PWC_DBG			DRVZONE(DRIVER_PWC, MSG_DEBUG)
#define PWC_PM			DRVZONE(DRIVER_PWC, MSG_PM)
#define PWC_USR1		DRVZONE(DRIVER_PWC, MSG_USER1)
#define PWC_USR2		DRVZONE(DRIVER_PWC, MSG_USER2)
#define PWC_USR3		DRVZONE(DRIVER_PWC, MSG_USER3)
#define PWC_USR4		DRVZONE(DRIVER_PWC, MSG_USER4)


/* Battery */
#define BATT_INFO		DRVZONE(DRIVER_BATTERY, MSG_INFO)
#define BATT_FUNC		DRVZONE(DRIVER_BATTERY, MSG_FUNC)
#define BATT_DBG		DRVZONE(DRIVER_BATTERY, MSG_DEBUG)
#define BATT_PM			DRVZONE(DRIVER_BATTERY, MSG_PM)
#define BATT_USR1		DRVZONE(DRIVER_BATTERY, MSG_USER1)
#define BATT_USR2		DRVZONE(DRIVER_BATTERY, MSG_USER2)
#define BATT_USR3		DRVZONE(DRIVER_BATTERY, MSG_USER3)
#define BATT_USR4		DRVZONE(DRIVER_BATTERY, MSG_USER4)

/* LED */
#define LED_INFO		DRVZONE(DRIVER_LED, MSG_INFO)
#define LED_FUNC		DRVZONE(DRIVER_LED, MSG_FUNC)
#define LED_DBG			DRVZONE(DRIVER_LED, MSG_DEBUG)
#define LED_PM			DRVZONE(DRIVER_LED, MSG_PM)
#define LED_USR1		DRVZONE(DRIVER_LED, MSG_USER1)
#define LED_USR2		DRVZONE(DRIVER_LED, MSG_USER2)
#define LED_USR3		DRVZONE(DRIVER_LED, MSG_USER3)
#define LED_USR4		DRVZONE(DRIVER_LED, MSG_USER4)

/* Keypad Driver */
#define KEY_INFO		DRVZONE(DRIVER_KEYPAD, MSG_INFO)
#define KEY_FUNC		DRVZONE(DRIVER_KEYPAD, MSG_FUNC)
#define KEY_DBG			DRVZONE(DRIVER_KEYPAD, MSG_DEBUG)
#define KEY_PM			DRVZONE(DRIVER_KEYPAD, MSG_PM)
#define KEY_USR1		DRVZONE(DRIVER_KEYPAD, MSG_USER1)
#define KEY_USR2		DRVZONE(DRIVER_KEYPAD, MSG_USER2)
#define KEY_USR3		DRVZONE(DRIVER_KEYPAD, MSG_USER3)
#define KEY_USR4		DRVZONE(DRIVER_KEYPAD, MSG_USER4)

/* Touch Driver */
#define TCH_INFO		DRVZONE(DRIVER_TOUCH, MSG_INFO)
#define TCH_FUNC		DRVZONE(DRIVER_TOUCH, MSG_FUNC)
#define TCH_DBG			DRVZONE(DRIVER_TOUCH, MSG_DEBUG)
#define TCH_PM			DRVZONE(DRIVER_TOUCH, MSG_PM)
#define TCH_USR1		DRVZONE(DRIVER_TOUCH, MSG_USER1)
#define TCH_USR2		DRVZONE(DRIVER_TOUCH, MSG_USER2)
#define TCH_USR3		DRVZONE(DRIVER_TOUCH, MSG_USER3)
#define TCH_USR4		DRVZONE(DRIVER_TOUCH, MSG_USER4)				// X/Y position information
#define TCH_XY          TCH_USR4




/* Backlight Driver */
#define BKL_INFO		DRVZONE(DRIVER_BACKLIGHT, MSG_INFO)
#define BKL_FUNC		DRVZONE(DRIVER_BACKLIGHT, MSG_FUNC)
#define BKL_DBG			DRVZONE(DRIVER_BACKLIGHT, MSG_DEBUG)
#define BKL_PM			DRVZONE(DRIVER_BACKLIGHT, MSG_PM)
#define BKL_USR1		DRVZONE(DRIVER_BACKLIGHT, MSG_USER1)
#define BKL_USR2		DRVZONE(DRIVER_BACKLIGHT, MSG_USER2)
#define BKL_USR3		DRVZONE(DRIVER_BACKLIGHT, MSG_USER3)
#define BKL_USR4		DRVZONE(DRIVER_BACKLIGHT, MSG_USER4)


/* ProximitySensors Driver */
#define PROX_INFO		DRVZONE(DRIVER_ProximitySensors, MSG_INFO)
#define PROX_FUNC		DRVZONE(DRIVER_ProximitySensors, MSG_FUNC)
#define PROX_DBG		DRVZONE(DRIVER_ProximitySensors, MSG_DEBUG)
#define PROX_PM			DRVZONE(DRIVER_ProximitySensors, MSG_PM)
#define PROX_USR1		DRVZONE(DRIVER_ProximitySensors, MSG_USER1)
#define PROX_USR2		DRVZONE(DRIVER_ProximitySensors, MSG_USER2)
#define PROX_USR3		DRVZONE(DRIVER_ProximitySensors, MSG_USER3)
#define PROX_USR4		DRVZONE(DRIVER_ProximitySensors, MSG_USER4)


/* OneNAND */
#define OND_INFO		DRVZONE(DRIVER_ONENAND, MSG_INFO)
#define OND_FUNC		DRVZONE(DRIVER_ONENAND, MSG_FUNC)
#define OND_DBG			DRVZONE(DRIVER_ONENAND, MSG_DEBUG)
#define OND_PM			DRVZONE(DRIVER_ONENAND, MSG_PM)
#define OND_USR1		DRVZONE(DRIVER_ONENAND, MSG_USER1)
#define OND_USR2		DRVZONE(DRIVER_ONENAND, MSG_USER2)
#define OND_USR3		DRVZONE(DRIVER_ONENAND, MSG_USER3)
#define OND_USR4		DRVZONE(DRIVER_ONENAND, MSG_USER4)

/* Flash */
#define FMD_INFO		DRVZONE(DRIVER_FLASH, MSG_INFO)
#define FMD_FUNC		DRVZONE(DRIVER_FLASH, MSG_FUNC)
#define FMD_DBG			DRVZONE(DRIVER_FLASH, MSG_DEBUG)
#define FMD_PM			DRVZONE(DRIVER_FLASH, MSG_PM)
#define FMD_ECC			DRVZONE(DRIVER_FLASH, MSG_USER1)
#define FMD_USR2		DRVZONE(DRIVER_FLASH, MSG_USER2)
#define FMD_USR3		DRVZONE(DRIVER_FLASH, MSG_USER3)
#define FMD_USR4		DRVZONE(DRIVER_FLASH, MSG_USER4)

/* SD Host Controller */
#define SDHC_INFO		DRVZONE(DRIVER_SDHC, MSG_INFO)
#define SDHC_FUNC		DRVZONE(DRIVER_SDHC, MSG_FUNC)
#define SDHC_DBG		DRVZONE(DRIVER_SDHC, MSG_DEBUG)
#define SDHC_PM			DRVZONE(DRIVER_SDHC, MSG_PM)
#define SDHC_USR1		DRVZONE(DRIVER_SDHC, MSG_USER1)
#define SDHC_USR2		DRVZONE(DRIVER_SDHC, MSG_USER2)
#define SDHC_USR3		DRVZONE(DRIVER_SDHC, MSG_USER3)
#define SDHC_USR4		DRVZONE(DRIVER_SDHC, MSG_USER4)

/* PCMCIA */
#define PCMCIA_INFO		DRVZONE(DRIVER_PCMCIA, MSG_INFO)
#define PCMCIA_FUNC		DRVZONE(DRIVER_PCMCIA, MSG_FUNC)
#define PCMCIA_DBG		DRVZONE(DRIVER_PCMCIA, MSG_DEBUG)
#define PCMCIA_PM		DRVZONE(DRIVER_PCMCIA, MSG_PM)
#define PCMCIA_USR1		DRVZONE(DRIVER_PCMCIA, MSG_USER1)
#define PCMCIA_USR2		DRVZONE(DRIVER_PCMCIA, MSG_USER2)
#define PCMCIA_USR3		DRVZONE(DRIVER_PCMCIA, MSG_USER3)
#define PCMCIA_USR4		DRVZONE(DRIVER_PCMCIA, MSG_USER4)

/* DMA */
#define DMA_INFO		DRVZONE(DRIVER_DMA, MSG_INFO)
#define DMA_FUNC		DRVZONE(DRIVER_DMA, MSG_FUNC)
#define DMA_DBG			DRVZONE(DRIVER_DMA, MSG_DEBUG)
#define DMA_PM			DRVZONE(DRIVER_DMA, MSG_PM)
#define DMA_USR1		DRVZONE(DRIVER_DMA, MSG_USER1)
#define DMA_USR2		DRVZONE(DRIVER_DMA, MSG_USER2)
#define DMA_USR3		DRVZONE(DRIVER_DMA, MSG_USER3)
#define DMA_USR4		DRVZONE(DRIVER_DMA, MSG_USER4)

/* MFD */
#define MFD_INFO		DRVZONE(DRIVER_MFD, MSG_INFO)
#define MFD_FUNC		DRVZONE(DRIVER_MFD, MSG_FUNC)
#define MFD_DBG			DRVZONE(DRIVER_MFD, MSG_DEBUG)
#define MFD_PM			DRVZONE(DRIVER_MFD, MSG_PM)
#define MFD_USR1		DRVZONE(DRIVER_MFD, MSG_USER1)
#define MFD_USR2		DRVZONE(DRIVER_MFD, MSG_USER2)
#define MFD_USR3		DRVZONE(DRIVER_MFD, MSG_USER3)
#define MFD_USR4		DRVZONE(DRIVER_MFD, MSG_USER4)

/* MMI */
#define MMI_INFO		DRVZONE(DRIVER_MMI, MSG_INFO)
#define MMI_FUNC		DRVZONE(DRIVER_MMI, MSG_FUNC)
#define MMI_DBG			DRVZONE(DRIVER_MMI, MSG_DEBUG)
#define MMI_PM			DRVZONE(DRIVER_MMI, MSG_PM)
#define MMI_USR1		DRVZONE(DRIVER_MMI, MSG_USER1)
#define MMI_USR2		DRVZONE(DRIVER_MMI, MSG_USER2)
#define MMI_USR3		DRVZONE(DRIVER_MMI, MSG_USER3)
#define MMI_USR4		DRVZONE(DRIVER_MMI, MSG_USER4)

/* IIC */
#define IIC_INFO		DRVZONE(DRIVER_IIC, MSG_INFO)
#define IIC_FUNC		DRVZONE(DRIVER_IIC, MSG_FUNC)
#define IIC_DBG			DRVZONE(DRIVER_IIC, MSG_DEBUG)
#define IIC_PM			DRVZONE(DRIVER_IIC, MSG_PM)
#define IIC_USR1		DRVZONE(DRIVER_IIC, MSG_USER1)
#define IIC_USR2		DRVZONE(DRIVER_IIC, MSG_USER2)
#define IIC_USR3		DRVZONE(DRIVER_IIC, MSG_USER3)
#define IIC_USR4		DRVZONE(DRIVER_IIC, MSG_USER4)

/* UART */
#define UART_INFO		DRVZONE(DRIVER_SERIAL, MSG_INFO)
#define UART_FUNC		DRVZONE(DRIVER_SERIAL, MSG_FUNC)
#define UART_DBG		DRVZONE(DRIVER_SERIAL, MSG_DEBUG)
#define UART_PM			DRVZONE(DRIVER_SERIAL, MSG_PM)
#define UART_USR1		DRVZONE(DRIVER_SERIAL, MSG_USER1)
#define UART_USR2		DRVZONE(DRIVER_SERIAL, MSG_USER2)
#define UART_USR3		DRVZONE(DRIVER_SERIAL, MSG_USER3)
#define UART_USR4		DRVZONE(DRIVER_SERIAL, MSG_USER4)

/* BT Serial */
#define BTS_INFO		DRVZONE(DRIVER_BTSERIAL, MSG_INFO)
#define BTS_FUNC		DRVZONE(DRIVER_BTSERIAL, MSG_FUNC)
#define BTS_DBG			DRVZONE(DRIVER_BTSERIAL, MSG_DEBUG)
#define BTS_PM			DRVZONE(DRIVER_BTSERIAL, MSG_PM)
#define BTS_USR1		DRVZONE(DRIVER_BTSERIAL, MSG_USER1)
#define BTS_USR2		DRVZONE(DRIVER_BTSERIAL, MSG_USER2)
#define BTS_USR3		DRVZONE(DRIVER_BTSERIAL, MSG_USER3)
#define BTS_USR4		DRVZONE(DRIVER_BTSERIAL, MSG_USER4)

/* Bluetooth */
#define BT_INFO			DRVZONE(DRIVER_BLUETOOTH, MSG_INFO)
#define BT_FUNC			DRVZONE(DRIVER_BLUETOOTH, MSG_FUNC)
#define BT_DBG			DRVZONE(DRIVER_BLUETOOTH, MSG_DEBUG)
#define BT_PM			DRVZONE(DRIVER_BLUETOOTH, MSG_PM)
#define BT_USR1			DRVZONE(DRIVER_BLUETOOTH, MSG_USER1)
#define BT_USR2			DRVZONE(DRIVER_BLUETOOTH, MSG_USER2)
#define BT_USR3			DRVZONE(DRIVER_BLUETOOTH, MSG_USER3)
#define BT_USR4			DRVZONE(DRIVER_BLUETOOTH, MSG_USER4)

/* Bluetooth Profile */
#define BTP_INFO		DRVZONE(DRIVER_BTPROFILE, MSG_INFO)
#define BTP_FUNC		DRVZONE(DRIVER_BTPROFILE, MSG_FUNC)
#define BTP_DBG			DRVZONE(DRIVER_BTPROFILE, MSG_DEBUG)
#define BTP_PM			DRVZONE(DRIVER_BTPROFILE, MSG_PM)
#define BTP_USR1		DRVZONE(DRIVER_BTPROFILE, MSG_USER1)
#define BTP_USR2		DRVZONE(DRIVER_BTPROFILE, MSG_USER2)
#define BTP_USR3		DRVZONE(DRIVER_BTPROFILE, MSG_USER3)
#define BTP_USR4		DRVZONE(DRIVER_BTPROFILE, MSG_USER4)

/* RTC */
#define RTC_INFO		DRVZONE(DRIVER_RTC, MSG_INFO)
#define RTC_FUNC		DRVZONE(DRIVER_RTC, MSG_FUNC)
#define RTC_DBG			DRVZONE(DRIVER_RTC, MSG_DEBUG)
#define RTC_PM			DRVZONE(DRIVER_RTC, MSG_PM)
#define RTC_USR1		DRVZONE(DRIVER_RTC, MSG_USER1)
#define RTC_USR2		DRVZONE(DRIVER_RTC, MSG_USER2)
#define RTC_USR3		DRVZONE(DRIVER_RTC, MSG_USER3)
#define RTC_USR4		DRVZONE(DRIVER_RTC, MSG_USER4)

/* RIL */
#define RIL_INFO		DRVZONE(DRIVER_RIL, MSG_INFO)
#define RIL_FUNC		DRVZONE(DRIVER_RIL, MSG_FUNC)
#define RIL_DBG			DRVZONE(DRIVER_RIL, MSG_DEBUG)
#define RIL_PM			DRVZONE(DRIVER_RIL, MSG_PM)
#define RIL_USR1		DRVZONE(DRIVER_RIL, MSG_USER1)
#define RIL_USR2		DRVZONE(DRIVER_RIL, MSG_USER2)
#define RIL_USR3		DRVZONE(DRIVER_RIL, MSG_USER3)
#define RIL_USR4		DRVZONE(DRIVER_RIL, MSG_USER4)

/* RCD */
#define RCD_INFO		DRVZONE(DRIVER_RCD, MSG_INFO)
#define RCD_FUNC		DRVZONE(DRIVER_RCD, MSG_FUNC)
#define RCD_DBG			DRVZONE(DRIVER_RCD, MSG_DEBUG)
#define RCD_PM			DRVZONE(DRIVER_RCD, MSG_PM)
#define RCD_USR1		DRVZONE(DRIVER_RCD, MSG_USER1)
#define RCD_USR2		DRVZONE(DRIVER_RCD, MSG_USER2)
#define RCD_USR3		DRVZONE(DRIVER_RCD, MSG_USER3)
#define RCD_USR4		DRVZONE(DRIVER_RCD, MSG_USER4)

/* DPRAM */
#define DPR_INFO		DRVZONE(DRIVER_DPRAM, MSG_INFO)
#define DPR_FUNC		DRVZONE(DRIVER_DPRAM, MSG_FUNC)
#define DPR_DBG			DRVZONE(DRIVER_DPRAM, MSG_DEBUG)
#define DPR_PM			DRVZONE(DRIVER_DPRAM, MSG_PM)
#define DPR_USR1		DRVZONE(DRIVER_DPRAM, MSG_USER1)
#define DPR_USR2		DRVZONE(DRIVER_DPRAM, MSG_USER2)
#define DPR_USR3		DRVZONE(DRIVER_DPRAM, MSG_USER3)
#define DPR_USR4		DRVZONE(DRIVER_DPRAM, MSG_USER4)

/* SIM */
#define SIM_INFO		DRVZONE(DRIVER_SIM, MSG_INFO)
#define SIM_FUNC		DRVZONE(DRIVER_SIM, MSG_FUNC)
#define SIM_DBG			DRVZONE(DRIVER_SIM, MSG_DEBUG)
#define SIM_PM			DRVZONE(DRIVER_SIM, MSG_PM)
#define SIM_USR1		DRVZONE(DRIVER_SIM, MSG_USER1)
#define SIM_USR2		DRVZONE(DRIVER_SIM, MSG_USER2)
#define SIM_USR3		DRVZONE(DRIVER_SIM, MSG_USER3)
#define SIM_USR4		DRVZONE(DRIVER_SIM, MSG_USER4)

/* Wibro */
#define WIBRO_INFO		DRVZONE(DRIVER_WIBRO, MSG_INFO)
#define WIBRO_FUNC		DRVZONE(DRIVER_WIBRO, MSG_FUNC)
#define WIBRO_DBG		DRVZONE(DRIVER_WIBRO, MSG_DEBUG)
#define WIBRO_PM		DRVZONE(DRIVER_WIBRO, MSG_PM)
#define WIBRO_USR1		DRVZONE(DRIVER_WIBRO, MSG_USER1)
#define WIBRO_USR2		DRVZONE(DRIVER_WIBRO, MSG_USER2)
#define WIBRO_USR3		DRVZONE(DRIVER_WIBRO, MSG_USER3)
#define WIBRO_USR4		DRVZONE(DRIVER_WIBRO, MSG_USER4)

/* WLAN */
#define WLAN_INFO		DRVZONE(DRIVER_WLAN, MSG_INFO)
#define WLAN_FUNC		DRVZONE(DRIVER_WLAN, MSG_FUNC)
#define WLAN_DBG		DRVZONE(DRIVER_WLAN, MSG_DEBUG)
#define WLAN_PM			DRVZONE(DRIVER_WLAN, MSG_PM)
#define WLAN_USR1		DRVZONE(DRIVER_WLAN, MSG_USER1)
#define WLAN_USR2		DRVZONE(DRIVER_WLAN, MSG_USER2)
#define WLAN_USR3		DRVZONE(DRIVER_WLAN, MSG_USER3)
#define WLAN_USR4		DRVZONE(DRIVER_WLAN, MSG_USER4)

/* TDMB */
#define TDMB_INFO		DRVZONE(DRIVER_TDMB, MSG_INFO)
#define TDMB_FUNC		DRVZONE(DRIVER_TDMB, MSG_FUNC)
#define TDMB_DBG		DRVZONE(DRIVER_TDMB, MSG_DEBUG)
#define TDMB_PM			DRVZONE(DRIVER_TDMB, MSG_PM)
#define TDMB_USR1		DRVZONE(DRIVER_TDMB, MSG_USER1)
#define TDMB_USR2		DRVZONE(DRIVER_TDMB, MSG_USER2)
#define TDMB_USR3		DRVZONE(DRIVER_TDMB, MSG_USER3)
#define TDMB_USR4		DRVZONE(DRIVER_TDMB, MSG_USER4)

/* ACC */
#define ACC_INFO		DRVZONE(DRIVER_ACC, MSG_INFO)
#define ACC_FUNC		DRVZONE(DRIVER_ACC, MSG_FUNC)
#define ACC_DBG			DRVZONE(DRIVER_ACC, MSG_DEBUG)
#define ACC_PM			DRVZONE(DRIVER_ACC, MSG_PM)
#define ACC_USR1		DRVZONE(DRIVER_ACC, MSG_USER1)
#define ACC_USR2		DRVZONE(DRIVER_ACC, MSG_USER2)
#define ACC_USR3		DRVZONE(DRIVER_ACC, MSG_USER3)
#define ACC_USR4		DRVZONE(DRIVER_ACC, MSG_USER4)

/* GPS */
#define GPS_INFO		DRVZONE(DRIVER_GPS, MSG_INFO)
#define GPS_FUNC		DRVZONE(DRIVER_GPS, MSG_FUNC)
#define GPS_DBG			DRVZONE(DRIVER_GPS, MSG_DEBUG)
#define GPS_PM			DRVZONE(DRIVER_GPS, MSG_PM)
#define GPS_USR1		DRVZONE(DRIVER_GPS, MSG_USER1)
#define GPS_USR2		DRVZONE(DRIVER_GPS, MSG_USER2)
#define GPS_USR3		DRVZONE(DRIVER_GPS, MSG_USER3)
#define GPS_USR4		DRVZONE(DRIVER_GPS, MSG_USER4)

/* MFC */
#define MFC_INFO		DRVZONE(DRIVER_MFC, MSG_INFO)
#define MFC_FUNC		DRVZONE(DRIVER_MFC, MSG_FUNC)
#define MFC_DBG			DRVZONE(DRIVER_MFC, MSG_DEBUG)
#define MFC_PM			DRVZONE(DRIVER_ACC, MSG_PM)
#define MFC_USR1		DRVZONE(DRIVER_MFC, MSG_USER1)
#define MFC_USR2		DRVZONE(DRIVER_MFC, MSG_USER2)
#define MFC_USR3		DRVZONE(DRIVER_MFC, MSG_USER3)
#define MFC_USR4		DRVZONE(DRIVER_MFC, MSG_USER4)

/* JPG */
#define JPG_INFO		DRVZONE(DRIVER_JPG, MSG_INFO)
#define JPG_FUNC		DRVZONE(DRIVER_JPG, MSG_FUNC)
#define JPG_DBG			DRVZONE(DRIVER_JPG, MSG_DEBUG)
#define JPG_PM			DRVZONE(DRIVER_JPG, MSG_PM)
#define JPG_USR1		DRVZONE(DRIVER_JPG, MSG_USER1)
#define JPG_USR2		DRVZONE(DRIVER_JPG, MSG_USER2)
#define JPG_USR3		DRVZONE(DRIVER_JPG, MSG_USER3)
#define JPG_USR4		DRVZONE(DRIVER_JPG, MSG_USER4)

/* Mitsman */
#define MMAN_INFO		DRVZONE(DRIVER_MITSMAN, MSG_INFO)
#define MMAN_FUNC		DRVZONE(DRIVER_MITSMAN, MSG_FUNC)
#define MMAN_DBG		DRVZONE(DRIVER_MITSMAN, MSG_DEBUG)
#define MMAN_PM			DRVZONE(DRIVER_MITSMAN, MSG_PM)
#define MMAN_USR1		DRVZONE(DRIVER_MITSMAN, MSG_USER1)
#define MMAN_USR2		DRVZONE(DRIVER_MITSMAN, MSG_USER2)
#define MMAN_USR3		DRVZONE(DRIVER_MITSMAN, MSG_USER3)
#define MMAN_USR4		DRVZONE(DRIVER_MITSMAN, MSG_USER4)

/* USB Client */
#define USB_INFO		DRVZONE(DRIVER_USBFN, MSG_INFO)
#define USB_FUNC		DRVZONE(DRIVER_USBFN, MSG_FUNC)
#define USB_DBG			DRVZONE(DRIVER_USBFN, MSG_DEBUG)
#define USB_PM			DRVZONE(DRIVER_USBFN, MSG_PM)
#define USB_USR1		DRVZONE(DRIVER_USBFN, MSG_USER1)
#define USB_USR2		DRVZONE(DRIVER_USBFN, MSG_USER2)
#define USB_USR3		DRVZONE(DRIVER_USBFN, MSG_USER3)
#define USB_USR4		DRVZONE(DRIVER_USBFN, MSG_USER4)

/* USB HCD */
#define USBH_INFO		DRVZONE(DRIVER_USBH, MSG_INFO)
#define USBH_FUNC		DRVZONE(DRIVER_USBH, MSG_FUNC)
#define USBH_DBG		DRVZONE(DRIVER_USBH, MSG_DEBUG)
#define USBH_PM			DRVZONE(DRIVER_USBH, MSG_PM)
#define USBH_USR1		DRVZONE(DRIVER_USBH, MSG_USER1)
#define USBH_USR2		DRVZONE(DRIVER_USBH, MSG_USER2)
#define USBH_USR3		DRVZONE(DRIVER_USBH, MSG_USER3)
#define USBH_USR4		DRVZONE(DRIVER_USBH, MSG_USER4)

/* USB Modem */
#define USBMODEM_INFO	DRVZONE(DRIVER_USBMODEM, MSG_INFO)
#define USBMODEM_FUNC	DRVZONE(DRIVER_USBMODEM, MSG_FUNC)
#define USBMODEM_DBG	DRVZONE(DRIVER_USBMODEM, MSG_DEBUG)
#define USBMODEM_PM		DRVZONE(DRIVER_USBMODEM, MSG_PM)
#define USBMODEM_USR1	DRVZONE(DRIVER_USBMODEM, MSG_USER1)
#define USBMODEM_USR2	DRVZONE(DRIVER_USBMODEM, MSG_USER2)
#define USBMODEM_USR3	DRVZONE(DRIVER_USBMODEM, MSG_USER3)
#define USBMODEM_USR4	DRVZONE(DRIVER_USBMODEM, MSG_USER4)

/* Reserved for other drivers */
#define RSVD48_INFO		DRVZONE(DRIVER_RESERVED48, MSG_INFO)
#define RSVD48_FUNC		DRVZONE(DRIVER_RESERVED48, MSG_FUNC)
#define RSVD48_DBG		DRVZONE(DRIVER_RESERVED48, MSG_DEBUG)
#define RSVD48_PM		DRVZONE(DRIVER_RESERVED48, MSG_PM)
#define RSVD48_USR1		DRVZONE(DRIVER_RESERVED48, MSG_USER1)
#define RSVD48_USR2		DRVZONE(DRIVER_RESERVED48, MSG_USER2)
#define RSVD48_USR3		DRVZONE(DRIVER_RESERVED48, MSG_USER3)
#define RSVD48_USR4		DRVZONE(DRIVER_RESERVED48, MSG_USER4)

/* Speakerphone driver */
#define SPKPHONE_INFO	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_INFO)
#define SPKPHONE_FUNC	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_FUNC)
#define SPKPHONE_DBG	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_DEBUG)
#define SPKPHONE_PM	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_PM)
#define SPKPHONE_USR1	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_USER1)
#define SPKPHONE_USR2	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_USER2)
#define SPKPHONE_USR3	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_USER3)
#define SPKPHONE_USR4	DRVZONE(DRVLIB_SPEAKERPHONE, MSG_USER4)

/* Headset driver */
#define HEADSET_INFO	DRVZONE(DRVLIB_HEADSET, MSG_INFO)
#define HEADSET_FUNC	DRVZONE(DRVLIB_HEADSET, MSG_FUNC)
#define HEADSET_DBG	DRVZONE(DRVLIB_HEADSET, MSG_DEBUG)
#define HEADSET_PM		DRVZONE(DRVLIB_HEADSET, MSG_PM)
#define HEADSET_USR1	DRVZONE(DRVLIB_HEADSET, MSG_USER1)
#define HEADSET_USR2	DRVZONE(DRVLIB_HEADSET, MSG_USER2)
#define HEADSET_USR3	DRVZONE(DRVLIB_HEADSET, MSG_USER3)
#define HEADSET_USR4	DRVZONE(DRVLIB_HEADSET, MSG_USER4)

/* Reserved for other drivers */
#define FMRADIO_INFO		DRVZONE(DRVLIB_FMRADIO, MSG_INFO)
#define FMRADIO_FUNC		DRVZONE(DRVLIB_FMRADIO, MSG_FUNC)
#define FMRADIO_DBG		DRVZONE(DRVLIB_FMRADIO, MSG_DEBUG)
#define FMRADIO_PM			DRVZONE(DRVLIB_FMRADIO, MSG_PM)
#define FMRADIO_USR1		DRVZONE(DRVLIB_FMRADIO, MSG_USER1)
#define FMRADIO_USR2		DRVZONE(DRVLIB_FMRADIO, MSG_USER2)
#define FMRADIO_USR3		DRVZONE(DRVLIB_FMRADIO, MSG_USER3)
#define FMRADIO_USR4		DRVZONE(DRVLIB_FMRADIO, MSG_USER4)

/* Reserved for other drivers */
#define PWRBTN_INFO		DRVZONE(DRIVER_PWRBTN, MSG_INFO)
#define PWRBTN_FUNC		DRVZONE(DRIVER_PWRBTN, MSG_FUNC)
#define PWRBTN_DBG		DRVZONE(DRIVER_PWRBTN, MSG_DEBUG)
#define PWRBTN_PM		DRVZONE(DRIVER_PWRBTN, MSG_PM)
#define PWRBTN_USR1		DRVZONE(DRIVER_PWRBTN, MSG_USER1)
#define PWRBTN_USR2		DRVZONE(DRIVER_PWRBTN, MSG_USER2)
#define PWRBTN_USR3		DRVZONE(DRIVER_PWRBTN, MSG_USER3)
#define PWRBTN_USR4		DRVZONE(DRIVER_PWRBTN, MSG_USER4)

/* for DRVLIB */
#define DRVLIB_INFO		DRVZONE(DRVLIB_USE, MSG_INFO)
#define DRVLIB_FUNC		DRVZONE(DRVLIB_USE, MSG_FUNC)
#define DRVLIB_DBG		DRVZONE(DRVLIB_USE, MSG_DEBUG)
#define DRVLIB_PM		DRVZONE(DRVLIB_USE, MSG_PM)
#define DRVLIB_USR1		DRVZONE(DRVLIB_USE, MSG_USER1)
#define DRVLIB_USR2		DRVZONE(DRVLIB_USE, MSG_USER2)
#define DRVLIB_USR3		DRVZONE(DRVLIB_USE, MSG_USER3)
#define DRVLIB_USR4		DRVZONE(DRVLIB_USE, MSG_USER4)

/* Reserved for global configuration */
#define INTERNAL_INFO	DRVZONE(INTERNAL_USE, MSG_INFO)
#define INTERNAL_FUNC	DRVZONE(INTERNAL_USE, MSG_FUNC)
#define INTERNAL_DBG	DRVZONE(INTERNAL_USE, MSG_DEBUG)
#define INTERNAL_PM		DRVZONE(INTERNAL_USE, MSG_PM)
#define INTERNAL_USR1	DRVZONE(INTERNAL_USE, MSG_USER1)
#define INTERNAL_USR2	DRVZONE(INTERNAL_USE, MSG_USER2)
#define INTERNAL_USR3	DRVZONE(INTERNAL_USE, MSG_USER3)
#define INTERNAL_USR4	DRVZONE(INTERNAL_USE, MSG_USER4)

/* Reserved for other drivers */
#define RSVD55_INFO		DRVZONE(DRIVER_RESERVED55, MSG_INFO)
#define RSVD55_FUNC		DRVZONE(DRIVER_RESERVED55, MSG_FUNC)
#define RSVD55_DBG		DRVZONE(DRIVER_RESERVED55, MSG_DEBUG)
#define RSVD55_PM		DRVZONE(DRIVER_RESERVED55, MSG_PM)
#define RSVD55_USR1		DRVZONE(DRIVER_RESERVED55, MSG_USER1)
#define RSVD55_USR2		DRVZONE(DRIVER_RESERVED55, MSG_USER2)
#define RSVD55_USR3		DRVZONE(DRIVER_RESERVED55, MSG_USER3)
#define RSVD55_USR4		DRVZONE(DRIVER_RESERVED55, MSG_USER4)

/* Reserved for other drivers */
#define RSVD56_INFO		DRVZONE(DRIVER_RESERVED56, MSG_INFO)
#define RSVD56_FUNC		DRVZONE(DRIVER_RESERVED56, MSG_FUNC)
#define RSVD56_DBG		DRVZONE(DRIVER_RESERVED56, MSG_DEBUG)
#define RSVD56_PM		DRVZONE(DRIVER_RESERVED56, MSG_PM)
#define RSVD56_USR1		DRVZONE(DRIVER_RESERVED56, MSG_USER1)
#define RSVD56_USR2		DRVZONE(DRIVER_RESERVED56, MSG_USER2)
#define RSVD56_USR3		DRVZONE(DRIVER_RESERVED56, MSG_USER3)
#define RSVD56_USR4		DRVZONE(DRIVER_RESERVED56, MSG_USER4)

/* Reserved for other drivers */
#define RSVD57_INFO		DRVZONE(DRIVER_RESERVED57, MSG_INFO)
#define RSVD57_FUNC		DRVZONE(DRIVER_RESERVED57, MSG_FUNC)
#define RSVD57_DBG		DRVZONE(DRIVER_RESERVED57, MSG_DEBUG)
#define RSVD57_PM		DRVZONE(DRIVER_RESERVED57, MSG_PM)
#define RSVD57_USR1		DRVZONE(DRIVER_RESERVED57, MSG_USER1)
#define RSVD57_USR2		DRVZONE(DRIVER_RESERVED57, MSG_USER2)
#define RSVD57_USR3		DRVZONE(DRIVER_RESERVED57, MSG_USER3)
#define RSVD57_USR4		DRVZONE(DRIVER_RESERVED57, MSG_USER4)




#if __cplusplus
}
#endif
#endif //__OAL_DRVMSG_H
