01 /**********************************************************************
02 * 文件名称： can_controller.c
03 * 功能描述： CAN抽象结构体框架的具体实现函数
04 *            主要实现调用硬件驱动的接口封装
05 * 文件目的： 掌握CAN底层驱动的抽象
06 * 修改日期             版本号        修改人           修改内容
07 * -----------------------------------------------
08 * 2020/05/13         V1.0             bert            创建
09 ***********************************************************************/
10 
11 /**************头文件**************************************************/
12 #include <stdio.h>
13 #include <stdlib.h>
14 #include <string.h>
15 #include <unistd.h>
16 
17 #include <net/if.h>
18 #include <sys/ioctl.h>
19 #include <sys/socket.h>
20 
21 #include <linux/can.h>
22 #include <linux/can/raw.h>
23 
24 #include <fcntl.h>
25 #include <pthread.h>
26 
27 #include "can_controller.h"
28 
29 /**************宏定义**************************************************/
30 
31 /* 将CAN0波特率设置为500000 bps */
32 #define ip_cmd_set_can_params  "ip link set can0 type can bitrate 500000 triple-sampling on"
33 
34 /* 打开CAN0 */
35 #define ip_cmd_open            "ifconfig can0 up"     
36 
37 /* 关闭CAN0 */    
38 #define ip_cmd_close           "ifconfig can0 down"       
39 
40 //#define CAN_SFF_MASK 0x000007ffU
41 
42 /**************全局变量定义*********************************************/
43 /*接收线程中应用层处理的指针函数*/
44 pCanInterrupt    g_pCanInterrupt = NULL;
45 /*接收线程线程ID*/
46 pthread_t ntid;
47 
48 /**************函数声明*************************************************/
49 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller);
50 void *CAN1_RX0_IRQHandler(void *arg);
51 
52 /***********************************************************************
53 ****************CAN控制器抽象接口实现代码*******************************
54 ***********************************************************************/
55 
56 /**********************************************************************
57 * 函数名称： int CAN_Set_Controller( void )
58 * 功能描述： CAN控制器初始化，包括GPIO(TX,RX），CAN外设时钟，波特率，过滤器等
59 * 输入参数： 无
60 * 输出参数： 无
61 * 返 回 值： int can_port:返回对应CAN控制器的通道号，类比socketcan中的套接口
62 * 修改日期             版本号        修改人           修改内容
63 * -----------------------------------------------
64 * 2020/05/13         V1.0             bert            创建
65 ***********************************************************************/
66 int CAN_Set_Controller( void )
67 {    
68     /*************************************************************/
69     /*定义套接口变量： sock_fd*/
70     int sock_fd;
71     /**/
72     struct sockaddr_can addr;
73     /**/
74     struct ifreq ifr;
75 
76     /*************************************************************/
77     /* 通过system调用ip命令设置CAN波特率 */
78     system(ip_cmd_close);               
79     system(ip_cmd_set_can_params);
80     system(ip_cmd_open);
81     
82     /*************************************************************/
83     /* 创建套接口 sock_fd */
84     sock_fd = socket(AF_CAN, SOCK_RAW, CAN_RAW);
85 	if(sock_fd < 0)
86 	{
87 		perror("socket create error!\n");
88 		return -1;
89 	}
90     
91     /*************************************************************/
92     //将套接字与 can0 绑定
93     strcpy(ifr.ifr_name, "can0");
94 	ioctl(sock_fd, SIOCGIFINDEX,&ifr); // can0 device
95 
96 	ifr.ifr_ifindex = if_nametoindex(ifr.ifr_name);
97 	printf("ifr_name:%s \n",ifr.ifr_name);
98 	printf("can_ifindex:%d \n",ifr.ifr_ifindex);
99 
100 	addr.can_family = AF_CAN;
101 	addr.can_ifindex = ifr.ifr_ifindex;
102 		
103 	if( bind(sock_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0 )
104 	{
105 		perror("bind error!\n");
106 		return -1;
107 	}
108 	
109 	/*************************************************************/
110 	//定义接收规则，只接收表示符等于 0x11 的报文
111 	struct can_filter rfilter[1];
112 	rfilter[0].can_id = 0x201;
113 	rfilter[0].can_mask = CAN_SFF_MASK;
114 	//设置过滤规则
115 	setsockopt(sock_fd, SOL_CAN_RAW, CAN_RAW_FILTER, &rfilter, sizeof(rfilter));
116 
117     /*************************************************************/
118     //设置read()和write()函数设置为非堵塞方式
119     int flags;
120     flags = fcntl(sock_fd, F_GETFL);
121     flags |= O_NONBLOCK;
122     fcntl(sock_fd, F_SETFL, flags);  
123 
124     /*************************************************************/
125     /*返回套接口*/
126     return sock_fd;  
127     /*************************************************************/
128 }
129 
130 
131 /**********************************************************************
132 * 函数名称： void CAN_Set_Interrupt(int can_port,  pCanInterrupt callback)
133 * 功能描述： 创建CAN接收线程，并传入应用的的回调函数，回调函数主要处理应用层的功能
134 * 输入参数： can_port,端口号
135 *            callback： 中断具体处理应用功能的回调函数
136 * 输出参数： 无
137 * 返 回 值： 无
138 * 修改日期             版本号        修改人           修改内容
139 * -----------------------------------------------
140 * 2020/05/13         V1.0             bert            创建
141 ***********************************************************************/
142 void CAN_Set_Interrupt(int can_port,  pCanInterrupt callback)
143 {
144     int err;
145     
146     if ( NULL != callback ) 
147     {
148         g_pCanInterrupt = callback;
149     }
150     
151     err = pthread_create(&ntid, NULL,CAN1_RX0_IRQHandler, NULL );
152     if( err !=0 )
153     {
154         printf("create thread fail! \n");
155         return ;
156     }
157     printf("create thread success!\n");
158     
159 
160     return ;
161 }
162 
163 
164 
165 /**********************************************************************
166 * 函数名称： void CAN_Read(int can_port, CanRxMsg* recv_msg)
167 * 功能描述： CAN读取接收寄存器，取出接收到的报文
168 * 输入参数： can_port,端口号     
169 * 输出参数： recv_msg：接收报文
170 * 返 回 值： 无
171 * 修改日期             版本号        修改人           修改内容
172 * -----------------------------------------------
173 * 2020/05/13         V1.0             bert            创建
174 ***********************************************************************/
175 void CAN_Read(int can_port, CanRxMsg* recv_msg)
176 { 
177     unsigned char i;
178     static unsigned int rxcounter =0;
179     
180     int nbytes;
181     struct can_frame rxframe;
182     
183     
184     nbytes = read(can_port, &rxframe, sizeof(struct can_frame));
185 	if(nbytes>0)
186 	{
187 	    printf("nbytes = %d \n",nbytes );
188 	    
189 	    recv_msg->StdId = rxframe.can_id;
190 	    recv_msg->DLC = rxframe.can_dlc;
191 	    memcpy( recv_msg->Data, &rxframe.data[0], rxframe.can_dlc);
192 	    
193 		rxcounter++;
194 		printf("rxcounter=%d, ID=%03X, DLC=%d, data=%02X %02X %02X %02X %02X %02X %02X %02X \n",  \
195 			rxcounter,
196 			rxframe.can_id, rxframe.can_dlc,  \
197 			rxframe.data[0],\
198 			rxframe.data[1],\
199 			rxframe.data[2],\
200 			rxframe.data[3],\
201 			rxframe.data[4],\
202 			rxframe.data[5],\
203 			rxframe.data[6],\
204 			rxframe.data[7] );
205 	}
206 
207     return ;
208 }
209  
210 /**********************************************************************
211 * 函数名称： void CAN_Write(int can_port, CanTxMsg send_msg)
212 * 功能描述： CAN报文发送接口，调用发送寄存器发送报文
213 * 输入参数： can_port,端口号     
214 * 输出参数： send_msg：发送报文
215 * 返 回 值： 无
216 * 修改日期             版本号        修改人           修改内容
217 * -----------------------------------------------
218 * 2020/05/13         V1.0             bert            创建
219 ***********************************************************************/
220 void CAN_Write(int can_port, CanTxMsg send_msg)
221 {
222     unsigned char i;
223     static unsigned int txcounter=0;
224     int nbytes;
225     
226     struct can_frame txframe;
227     
228     txframe.can_id = send_msg.StdId;
229     txframe.can_dlc = send_msg.DLC;
230     memcpy(&txframe.data[0], &send_msg.Data[0], txframe.can_dlc);
231 
232     nbytes = write(can_port, &txframe, sizeof(struct can_frame)); //发送 frame[0]
233 	
234 	if(nbytes == sizeof(txframe))
235 	{
236 	    txcounter++;
237 	    printf("txcounter=%d, ID=%03X, DLC=%d, data=%02X %02X %02X %02X %02X %02X %02X %02X \n",  \
238 			txcounter,
239 			txframe.can_id, txframe.can_dlc,  \
240 			txframe.data[0],\
241 			txframe.data[1],\
242 			txframe.data[2],\
243 			txframe.data[3],\
244 			txframe.data[4],\
245 			txframe.data[5],\
246 			txframe.data[6],\
247 			txframe.data[7] );
248     }
249     else
250 	{
251 		//printf("Send Error frame[0], nbytes=%d\n!",nbytes);
252 	}
253 
254     return ;
255 }
256 
257 /**********************************************************************
258 * 函数名称： void CAN1_RX0_IRQHandler(void)
259 * 功能描述： CAN接收中断函数
260 * 输入参数： 无  
261 * 输出参数： 无
262 * 返 回 值： 无
263 * 修改日期             版本号        修改人           修改内容
264 * -----------------------------------------------
265 * 2020/05/13         V1.0             bert            创建
266 ***********************************************************************/
267 void *CAN1_RX0_IRQHandler(void *arg)
268 {
269     /* 接收报文定义 */
270     while( 1 )
271     {
272     /* 如果回调函数存在，则执行回调函数 */
273         if( g_pCanInterrupt != NULL)
274         {
275             g_pCanInterrupt();
276         }
277         usleep(10000);
278     }
279 }
280 
281 
282 
283 /**********************************************************************
284 * 名称：     can1_controller
285 * 功能描述： CAN1结构体初始化
286 * 修改日期             版本号        修改人           修改内容
287 * -----------------------------------------------
288 * 2020/05/13         V1.0             bert            创建
289 ***********************************************************************/
290  CAN_COMM_STRUCT can1_controller = {
291     .name                   = "can0",
292 	.can_port               = -1,//默认，应用层传入此参数
293 	.can_set_controller     = CAN_Set_Controller,
294     .can_set_interrput      = CAN_Set_Interrupt,
295     .can_read               = CAN_Read,
296     .can_write              = CAN_Write, 
297 };
298 
299 
300 /**********************************************************************
301 * 函数名称： void CAN1_contoller_add(void)
302 * 功能描述： CAN结构体注册接口，应用层在使用can1_controller前调用
303 * 输入参数： 无  
304 * 输出参数： 无
305 * 返 回 值： 无
306 * 修改日期             版本号        修改人           修改内容
307 * -----------------------------------------------
308 * 2020/05/13         V1.0             bert            创建
309 ***********************************************************************/
310 void CAN1_contoller_add(void)
311 {
312     /*将can1_controller传递给应用层*/
313     register_can_controller( &can1_controller );
314 }
315 
316 
317 /***********************************************************************
318 ****************End Of File*********************************************
319 ***********************************************************************/
320 
