01 /**********************************************************************
02 * 文件名称： app_can.c
03 * 功能描述： CAN应用层代码，实现功能如下：
04 *            功能1：周期发送CAN报文；
05 *            功能2：接收报文，并转发报文
06 * 文件目的： 掌握CAN报文的基本发送和接收处理
07 * 修改日期             版本号        修改人           修改内容
08 * -----------------------------------------------
09 * 2020/05/13         V1.0             bert            创建
10 ***********************************************************************/
11 
12 /**************头文件**************************************************/
13 
14 #include <stdio.h>
15 #include <stdlib.h>
16 #include <string.h>
17 #include <unistd.h>
18 
19 #include <net/if.h>
20 #include <sys/ioctl.h>
21 #include <sys/socket.h>
22 
23 #include <linux/can.h>
24 #include <linux/can/raw.h>
25 
26 #include "app_can.h"
27 #include "can_controller.h"
28 
29 
30 /**************宏定义**************************************************/
31 /* 本例程中测试周期发送的CAN报文ID */
32 #define TX_CAN_CYCLIC_ID    0X123
33 #define TX_CAN_EVENT_ID     0X124
34 #define TX_CAN_CE_ID        0X125
35 
36 /* 本例程中测试接收的CAN报文ID */
37 #define RX_CAN_ID           0x201   
38 
39 
40 /**************全局变量定义*********************************************/
41 /** 
42 * CAN应用层调用结构体指针变量
43 */
44 static CAN_COMM_STRUCT gCAN_COMM_STRUCT;
45 
46 /** 
47 *CAN应用层接收CAN报文后存储的CAN消息
48 */
49 static CanRxMsg  g_CAN1_Rx_Message;
50 
51 /** 
52 *CAN应用层接收报文(ID:0X201) Byte0的bit0位1标志
53 */
54 static unsigned char g_CAN1_Rx_Event_Flag =0; 
55 /** 
56 *CAN应用层接收报文(ID:0X201) Byte0的bit1位1标志
57 */
58 static unsigned char g_CAN1_Rx_CE_Flag =0; 
59 
60 /**************函数声明*************************************************/
61 /** 
62 *CAN中断中回调函数，在MCU中中断处理函数有特定的函数定义，
63 *此处将中断函数放在can_controller.c中，将中断中的具体处理内容放到app_can.c应用中，
64 *此处作为回调是为了与linux socket编程中类比，在linux socket中使用回调传入的是接收线程函数
65 */
66 void CAN_RX_IRQHandler_Callback(void);
67 
68 
69 
70 
71 /***********************************************************************
72 ****************应用层代码**********************************************
73 ***********************************************************************/
74 
75 /**********************************************************************
76 * 函数名称： int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
77 * 功能描述： 应用层进行CAN1结构体注册
78 * 输入参数： p_can_controller，CAN控制器抽象结构体
79 * 输出参数： 无
80 * 返 回 值： 无
81 * 修改日期             版本号        修改人           修改内容
82 * -----------------------------------------------
83 * 2020/05/13         V1.0             bert            创建
84 ***********************************************************************/
85 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
86 {
87     /* 判断传入的p_can_controller为非空，目的是确认这个结构体是实体*/
88     if( p_can_controller != NULL )
89     {
90         /* 将传入的参数p_can_controller赋值给应用层结构体gCAN_COMM_STRUCT */
91         
92         /*端口号，类比socketcan套接口*/
93         gCAN_COMM_STRUCT.can_port              = p_can_controller->can_port; 
94         /*CAN控制器配置函数*/
95         gCAN_COMM_STRUCT.can_set_controller    = p_can_controller->can_set_controller; 
96         /*CAN中断配置*/
97         gCAN_COMM_STRUCT.can_set_interrput     = p_can_controller->can_set_interrput;
98         /*CAN报文读函数*/
99         gCAN_COMM_STRUCT.can_read              = p_can_controller->can_read;
100         /*CAN报文发送函数*/
101         gCAN_COMM_STRUCT.can_write             = p_can_controller->can_write;
102         return 1;
103     }
104 	return 0;
105 }
106 
107 /**********************************************************************
108 * 函数名称： void app_can_init(void)
109 * 功能描述： CAN应用层初始化
110 * 输入参数： 无
111 * 输出参数： 无
112 * 返 回 值： 无
113 * 修改日期             版本号        修改人           修改内容
114 * -----------------------------------------------
115 * 2020/05/13         V1.0             bert            创建
116 ***********************************************************************/
117 void app_can_init(void)
118 {
119     /** 
120     * 应用层进行CAN1结构体注册
121     */
122     CAN1_contoller_add();
123     
124     /*
125     *调用can_set_controller进行CAN控制器配置，
126     *返回can_port，类比linux socketcan中的套接口，单片机例程中作为自定义CAN通道 
127     */
128     gCAN_COMM_STRUCT.can_port = gCAN_COMM_STRUCT.can_set_controller();
129     /** 
130     * 调用can_set_interrput配置CAN接收中断，类比socketcan中的接收线程
131     */
132     gCAN_COMM_STRUCT.can_set_interrput( gCAN_COMM_STRUCT.can_port, CAN_RX_IRQHandler_Callback );
133 }
134 
135 
136 /**********************************************************************
137 * 函数名称： void app_can_cyclicmsg_test(void)
138 * 功能描述： CAN应用层测试发送周期型报文(ID:0X123)
139 * 输入参数： 无
140 * 输出参数： 无
141 * 返 回 值： 无
142 * 修改日期             版本号        修改人           修改内容
143 * -----------------------------------------------
144 * 2020/05/13         V1.0             bert            创建
145 ***********************************************************************/
146 void app_can_cyclicmsg_test(void)
147 {
148     // 以10ms为基准，运行CAN测试程序
149     
150     unsigned char i=0;
151     
152     /* 发送报文定义 */
153     CanTxMsg TxMessage;
154     
155     /* 发送报文中用一个字节来作为计数器 */
156     static unsigned char tx_counter = 0;
157     
158     /* 以10ms为基准，通过timer计数器设置该处理函数后面运行代码的周期为1秒钟*/  
159     static unsigned int timer =0;
160     if(timer++>100)
161     {
162         timer = 0;
163     }
164     else
165     {
166         return ;
167     }
168     
169     /* 发送报文报文数据填充，此报文周期是1秒 */
170     TxMessage.StdId = TX_CAN_CYCLIC_ID;	  /* 标准标识符为0x000~0x7FF */
171     TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
172     TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
173     TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
174     TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
175     
176     /* 填充数据，此处可以根据实际应用填充 */
177     TxMessage.Data[0] = tx_counter++;       /* 用来识别报文发送计数器 */
178     for(i=1; i<TxMessage.DLC; i++)
179     {
180         TxMessage.Data[i] = i;            
181     }
182     
183     /*  调用can_write发送CAN报文 */
184     gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
185     
186 }
187 
188 /**********************************************************************
189 * 函数名称： void app_can_eventmsg_test(void)
190 * 功能描述： CAN应用层测试发送事件型报文(ID:0X124)
191 * 输入参数： 无
192 * 输出参数： 无
193 * 返 回 值： 无
194 * 修改日期             版本号        修改人           修改内容
195 * -----------------------------------------------
196 * 2020/05/13         V1.0             bert            创建
197 ***********************************************************************/
198 void app_can_eventmsg_test(void)
199 {
200     unsigned char i=0;
201 
202     /* 发送报文中用一个字节来作为事件触发计数器 */
203     static unsigned char tx_counter = 0;
204 
205     /* 发送报文定义 */
206     CanTxMsg TxMessage;
207 
208     if( g_CAN1_Rx_Event_Flag == 1 )
209     {
210 	g_CAN1_Rx_Event_Flag = 0;
211 	printf("Message:0x124 is Triggered!\n");
212 
213         /* 发送报文报文数据填充，此报文周期是1秒 */
214         TxMessage.StdId = TX_CAN_EVENT_ID;	  /* 标准标识符为0x000~0x7FF */
215         TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
216         TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
217         TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
218         TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
219         
220         /* 填充数据，此处可以根据实际应用填充 */
221         for(i=0; i<TxMessage.DLC; i++)
222         {
223             TxMessage.Data[i] = 0x00;            
224         }
225         /* 填充数据，此处可以根据实际应用填充 */
226 	tx_counter = 0;
227 	
228 	/*更新第1帧数据*/
229 	TxMessage.Data[1] = 0x02;
230 	TxMessage.Data[7] = (++tx_counter);
231         /*  调用can_write发送CAN报文，第1帧 */
232         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
233 	/*延时50ms,作为事件报文间隔*/
234 	usleep(50000);
235 
236 	/*更新第2帧数据*/
237 	TxMessage.Data[1] = 0x02;
238 	TxMessage.Data[7] = (++tx_counter);
239 	/*  调用can_write发送CAN报文，第2帧 */
240         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
241 	/*延时50ms,作为事件报文间隔*/
242 	usleep(50000);
243 
244 	/*更新第3帧数据*/
245 	TxMessage.Data[1] = 0x02;
246 	TxMessage.Data[7] = (++tx_counter);
247 	/*  调用can_write发送CAN报文，第3帧 */
248         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
249 	/*延时50ms,作为事件报文间隔*/
250 	usleep(50000);
251     }
252 }
253 
254 
255 /**********************************************************************
256 * 函数名称： void app_can_cycliceventmsg_test(void)
257 * 功能描述： CAN应用层测试发送周期事件混合报文(ID:0X125)
258 * 输入参数： 无
259 * 输出参数： 无
260 * 返 回 值： 无
261 * 修改日期             版本号        修改人           修改内容
262 * -----------------------------------------------
263 * 2020/05/13         V1.0             bert            创建
264 ***********************************************************************/
265 void app_can_cycliceventmsg_test(void)
266 {
267     unsigned char i=0;
268     
269     /* 发送报文定义 */
270     CanTxMsg TxMessage;
271     
272     /* 发送报文中用一个字节来作为事件触发计数器 */
273     static unsigned char tx_counter = 0;
274 
275     /* 以10ms为基准，通过timer计数器设置该处理函数后面运行代码的周期为1秒钟*/  
276     static unsigned int timer =0;
277 
278     if( g_CAN1_Rx_CE_Flag == 1)
279     {
280 	g_CAN1_Rx_CE_Flag = 0;
281 	printf("Message:0x125 is Triggered!\n");
282 
283 	/* 发送报文报文数据填充，此报文周期是1秒 */
284         TxMessage.StdId = TX_CAN_CE_ID;	     /* 标准标识符为0x000~0x7FF */
285         TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
286         TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
287         TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
288         TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
289         
290         /* 清零数据区 */
291         for(i=0; i<TxMessage.DLC; i++)
292         {
293             TxMessage.Data[i] = 0x00;            
294         }
295 	/* 填充数据，此处可以根据实际应用填充 */
296 	tx_counter = 0;
297 
298         /*更新第1帧数据*/
299 	TxMessage.Data[1] = 0x01;
300 	TxMessage.Data[7] = (++tx_counter);
301         /*  调用can_write发送CAN报文，第1帧 */
302         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
303 	/*延时50ms,作为事件报文间隔*/
304 	usleep(50000);
305 
306 	/*更新第2帧数据*/
307 	TxMessage.Data[1] = 0x01;
308 	TxMessage.Data[7] = (++tx_counter);
309 	/*  调用can_write发送CAN报文，第2帧 */
310         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
311 	/*延时50ms,作为事件报文间隔*/
312 	usleep(50000);
313 
314 	/*更新第3帧数据*/
315 	TxMessage.Data[1] = 0x01;
316 	TxMessage.Data[7] = (++tx_counter);
317 	/*  调用can_write发送CAN报文，第3帧 */
318         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
319 	/*延时50ms,作为事件报文间隔*/
320 	usleep(50000);
321     }
322 
323     /* 以10ms为基准，通过timer计数器设置该处理函数后面运行代码的周期为1秒钟*/  
324     if(timer++>100)
325     {
326         timer = 0;
327     }
328     else
329     {
330         return ;
331     }
332 
333     /* 发送报文报文数据填充，此报文周期是1秒 */
334     TxMessage.StdId = TX_CAN_CE_ID;	  /* 标准标识符为0x000~0x7FF */
335     TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
336     TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
337     TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
338     TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
339         
340     /* 填充数据，此处可以根据实际应用填充 */
341     for(i=1; i<TxMessage.DLC; i++)
342     {
343 	TxMessage.Data[i] = 0x00;            
344     }
345      
346     /*  调用can_write发送CAN报文 */
347     gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
348 }
349 
350 
351 /**********************************************************************
352 * 函数名称： void CAN_RX_IRQHandler_Callback(void)
353 * 功能描述： CAN1接收中断函数；在linux中可以类比用线程，或定时器去读CAN数据
354 * 输入参数： 无
355 * 输出参数： 无
356 * 返 回 值： 无
357 * 修改日期             版本号        修改人           修改内容
358 * -----------------------------------------------
359 * 2020/05/13         V1.0             bert            创建
360 ***********************************************************************/
361 void CAN_RX_IRQHandler_Callback(void)
362 {
363     /* 接收报文定义 */
364     CanRxMsg RxMessage; 
365     
366     /* 接收报文清零 */
367     memset( &RxMessage, 0, sizeof(CanRxMsg) );
368    
369     /* 通过can_read接口读取寄存器已经接收到的报文 */
370     gCAN_COMM_STRUCT.can_read(gCAN_COMM_STRUCT.can_port, &RxMessage);
371 
372     /* 将读取到的CAN报文存拷贝到全局报文结构体g_CAN1_Rx_Message */
373     memcpy(&g_CAN1_Rx_Message, &RxMessage, sizeof( CanRxMsg ) );
374     
375     /* 设置当前接收完成标志，判断当前接收报文ID为RX_CAN_ID，则设置g_CAN1_Rx_Flag=1*/
376     if( g_CAN1_Rx_Message.StdId == RX_CAN_ID )
377     {
378 	/* 事件型报文触发条件判断 */
379 	if( g_CAN1_Rx_Message.Data[0] & 0x01 )
380 	{
381 		g_CAN1_Rx_Event_Flag = 1;
382 	}
383 	/* 周期事件型报文触发条件判断 */
384 	if( g_CAN1_Rx_Message.Data[0] & 0x02 )
385 	{
386 		g_CAN1_Rx_CE_Flag = 1;
387 	} 
388     }
389 }
390 
391 /**********************************************************************
392 * 函数名称： int main(int argc, char **argv)
393 * 功能描述： 主函数
394 * 输入参数： 无
395 * 输出参数： 无
396 * 返 回 值： 无
397 * 修改日期             版本号        修改人           修改内容
398 * -----------------------------------------------
399 * 2020/05/13         V1.0             bert            创建
400 ***********************************************************************/
401 int main(int argc, char **argv)
402 {
403     /* CAN应用层初始化 */
404     app_can_init();
405     
406     while(1)
407     {
408         /* CAN应用层发送周期报文（ID:0x123） */
409         app_can_cyclicmsg_test();
410         
411         /* CAN应用层发送事件型报文（ID:0x124） */
412         app_can_eventmsg_test();
413         
414  	/* CAN应用层发送事件型报文（ID:0x125） */
415 	app_can_cycliceventmsg_test();
416 
417         /* 利用linux的延时函数设计10ms的运行基准 */
418         usleep(10000);
419     }
420 }
421 
422 
423 
424 
425 /***********************************************************************
426 ****************End Of File*********************************************
427 ***********************************************************************/
428 
