01 /**********************************************************************
02 * 文件名称： app_can.c
03 * 功能描述： CAN应用层代码，实现功能如下：
04 *            功能1：周期发送CAN报文；
05 *            功能2：接收报文，并转发报文
06 * 文件目的： 掌握CAN报文的基本发送和接收处理
07 * 修改日期             版本号        修改人           修改内容
08 * -----------------------------------------------
09 * 2020/05/13         V1.0             bert            创建
10 ***********************************************************************/
11 
12 /**************头文件**************************************************/
13 
14 #include <stdio.h>
15 #include <stdlib.h>
16 #include <string.h>
17 #include <unistd.h>
18 
19 #include <net/if.h>
20 #include <sys/ioctl.h>
21 #include <sys/socket.h>
22 
23 #include <linux/can.h>
24 #include <linux/can/raw.h>
25 
26 #include "app_can.h"
27 #include "can_controller.h"
28 
29 
30 /**************宏定义**************************************************/
31 /* 本例程中测试周期发送的CAN报文ID */
32 #define TX_CAN_ID           0X101   
33 
34 /* 本例程中测试接收的CAN报文ID */
35 #define RX_CAN_ID           0x201   
36 
37 /* 本例程中接收到测试报文为RX_CAN_ID的报文，将ID改为RX_TO_TX_CAN_ID为ID的报文转发出去 */
38 #define RX_TO_TX_CAN_ID     0X301   
39 
40 
41 /**************全局变量定义*********************************************/
42 /** 
43 * CAN应用层调用结构体指针变量
44 */
45 static CAN_COMM_STRUCT gCAN_COMM_STRUCT;
46 
47 /** 
48 *CAN应用层接收CAN报文后存储的CAN消息
49 */
50 static CanRxMsg  g_CAN1_Rx_Message;
51 
52 /** 
53 *CAN应用层接收CAN报文标志，处理完后清零
54 */
55 static unsigned char g_CAN1_Rx_Flag =0; 
56 
57 
58 /**************函数声明*************************************************/
59 /** 
60 *CAN中断中回调函数，在MCU中中断处理函数有特定的函数定义，
61 *此处将中断函数放在can_controller.c中，将中断中的具体处理内容放到app_can.c应用中，
62 *此处作为回调是为了与linux socket编程中类比，在linux socket中使用回调传入的是接收线程函数
63 */
64 void CAN_RX_IRQHandler_Callback(void);
65 
66 
67 
68 
69 /***********************************************************************
70 ****************应用层代码**********************************************
71 ***********************************************************************/
72 
73 /**********************************************************************
74 * 函数名称： int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
75 * 功能描述： 应用层进行CAN1结构体注册
76 * 输入参数： p_can_controller，CAN控制器抽象结构体
77 * 输出参数： 无
78 * 返 回 值： 无
79 * 修改日期             版本号        修改人           修改内容
80 * -----------------------------------------------
81 * 2020/05/13         V1.0             bert            创建
82 ***********************************************************************/
83 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
84 {
85     /* 判断传入的p_can_controller为非空，目的是确认这个结构体是实体*/
86     if( p_can_controller != NULL )
87     {
88         /* 将传入的参数p_can_controller赋值给应用层结构体gCAN_COMM_STRUCT */
89         
90         /*端口号，类比socketcan套接口*/
91         gCAN_COMM_STRUCT.can_port              = p_can_controller->can_port; 
92         /*CAN控制器配置函数*/
93         gCAN_COMM_STRUCT.can_set_controller    = p_can_controller->can_set_controller; 
94         /*CAN中断配置*/
95         gCAN_COMM_STRUCT.can_set_interrput     = p_can_controller->can_set_interrput;
96         /*CAN报文读函数*/
97         gCAN_COMM_STRUCT.can_read              = p_can_controller->can_read;
98         /*CAN报文发送函数*/
99         gCAN_COMM_STRUCT.can_write             = p_can_controller->can_write;
100         return 1;
101     }
102 	return 0;
103 }
104 
105 /**********************************************************************
106 * 函数名称： void app_can_init(void)
107 * 功能描述： CAN应用层初始化
108 * 输入参数： 无
109 * 输出参数： 无
110 * 返 回 值： 无
111 * 修改日期             版本号        修改人           修改内容
112 * -----------------------------------------------
113 * 2020/05/13         V1.0             bert            创建
114 ***********************************************************************/
115 void app_can_init(void)
116 {
117     /** 
118     * 应用层进行CAN1结构体注册
119     */
120     CAN1_contoller_add();
121     
122     /*
123     *调用can_set_controller进行CAN控制器配置，
124     *返回can_port，类比linux socketcan中的套接口，单片机例程中作为自定义CAN通道 
125     */
126     gCAN_COMM_STRUCT.can_port = gCAN_COMM_STRUCT.can_set_controller();
127     /** 
128     * 调用can_set_interrput配置CAN接收中断，类比socketcan中的接收线程
129     */
130     gCAN_COMM_STRUCT.can_set_interrput( gCAN_COMM_STRUCT.can_port, CAN_RX_IRQHandler_Callback );
131 }
132 
133 /**********************************************************************
134 * 函数名称： void CAN_RX_IRQHandler_Callback(void)
135 * 功能描述： CAN1接收中断函数；在linux中可以类比用线程，或定时器去读CAN数据
136 * 输入参数： 无
137 * 输出参数： 无
138 * 返 回 值： 无
139 * 修改日期             版本号        修改人           修改内容
140 * -----------------------------------------------
141 * 2020/05/13         V1.0             bert            创建
142 ***********************************************************************/
143 void CAN_RX_IRQHandler_Callback(void)
144 {
145     /* 接收报文定义 */
146     CanRxMsg RxMessage; 
147     
148     /* 接收报文清零 */
149     memset( &RxMessage, 0, sizeof(CanRxMsg) );
150    
151     /* 通过can_read接口读取寄存器已经接收到的报文 */
152     gCAN_COMM_STRUCT.can_read(gCAN_COMM_STRUCT.can_port, &RxMessage);
153 
154     /* 将读取到的CAN报文存拷贝到全局报文结构体g_CAN1_Rx_Message */
155     memcpy(&g_CAN1_Rx_Message, &RxMessage, sizeof( CanRxMsg ) );
156     
157 }
158 
159 /**********************************************************************
160 * 函数名称： int main(int argc, char **argv)
161 * 功能描述： 主函数
162 * 输入参数： 无
163 * 输出参数： 无
164 * 返 回 值： 无
165 * 修改日期             版本号        修改人           修改内容
166 * -----------------------------------------------
167 * 2020/05/13         V1.0             bert            创建
168 ***********************************************************************/
169 int main(int argc, char **argv)
170 {
171     /* CAN应用层初始化 */
172     app_can_init();
173     
174     while(1)
175     {        
176         /* 利用linux的延时函数设计10ms的运行基准 */
177         usleep(10000);
178     }
179 }
180 
181 
182 
183 
184 /***********************************************************************
185 ****************End Of File*********************************************
186 ***********************************************************************/
187 
