01 /**********************************************************************
02 * 文件名称： can_controller.c
03 * 功能描述： CAN抽象结构体框架的具体实现函数
04 *            主要实现调用硬件驱动的接口封装
05 * 文件目的： 掌握CAN底层驱动的抽象
06 * 修改日期             版本号        修改人           修改内容
07 * -----------------------------------------------
08 * 2020/05/13         V1.0             bert            创建
09 ***********************************************************************/
10 
11 /**************头文件**************************************************/
12 #include <stdio.h>
13 #include <stdlib.h>
14 #include <string.h>
15 #include <unistd.h>
16 
17 #include <net/if.h>
18 #include <sys/ioctl.h>
19 #include <sys/socket.h>
20 
21 #include <linux/can.h>
22 #include <linux/can/raw.h>
23 
24 #include <fcntl.h>
25 #include <pthread.h>
26 
27 #include "can_controller.h"
28 
29 /**************宏定义**************************************************/
30 
31 /* 将CAN0波特率设置为500000 bps */
32 #define ip_cmd_set_can_params  "ip link set can0 type can bitrate 500000 triple-sampling on"
33 
34 /* 打开CAN0 */
35 #define ip_cmd_open            "ifconfig can0 up"     
36 
37 /* 关闭CAN0 */    
38 #define ip_cmd_close           "ifconfig can0 down"       
39 
40 //#define CAN_SFF_MASK 0x000007ffU
41 
42 /**************全局变量定义*********************************************/
43 /*接收线程中应用层处理的指针函数*/
44 pCanInterrupt    g_pCanInterrupt = NULL;
45 /*接收线程线程ID*/
46 pthread_t ntid;
47 
48 /**************函数声明*************************************************/
49 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller);
50 void *CAN1_RX0_IRQHandler(void *arg);
51 
52 /***********************************************************************
53 ****************CAN控制器抽象接口实现代码*******************************
54 ***********************************************************************/
55 
56 /**********************************************************************
57 * 函数名称： int CAN_Set_Controller( void )
58 * 功能描述： CAN控制器初始化，包括GPIO(TX,RX），CAN外设时钟，波特率，过滤器等
59 * 输入参数： 无
60 * 输出参数： 无
61 * 返 回 值： int can_port:返回对应CAN控制器的通道号，类比socketcan中的套接口
62 * 修改日期             版本号        修改人           修改内容
63 * -----------------------------------------------
64 * 2020/05/13         V1.0             bert            创建
65 ***********************************************************************/
66 int CAN_Set_Controller( void )
67 {    
68     /*************************************************************/
69     /*定义套接口变量： sock_fd*/
70     int sock_fd;
71     /**/
72     struct sockaddr_can addr;
73     /**/
74     struct ifreq ifr;
75 
76     /*************************************************************/
77     /* 通过system调用ip命令设置CAN波特率 */
78     system(ip_cmd_close);               
79     system(ip_cmd_set_can_params);
80     system(ip_cmd_open);
81     
82     /*************************************************************/
83     /* 创建套接口 sock_fd */
84     sock_fd = socket(AF_CAN, SOCK_RAW, CAN_RAW);
85 	if(sock_fd < 0)
86 	{
87 		perror("socket create error!\n");
88 		return -1;
89 	}
90     
91     /*************************************************************/
92     //将套接字与 can0 绑定
93     strcpy(ifr.ifr_name, "can0");
94 	ioctl(sock_fd, SIOCGIFINDEX,&ifr); // can0 device
95 
96 	ifr.ifr_ifindex = if_nametoindex(ifr.ifr_name);
97 	printf("ifr_name:%s \n",ifr.ifr_name);
98 	printf("can_ifindex:%d \n",ifr.ifr_ifindex);
99 
100 	addr.can_family = AF_CAN;
101 	addr.can_ifindex = ifr.ifr_ifindex;
102 		
103 	if( bind(sock_fd, (struct sockaddr *)&addr, sizeof(addr)) < 0 )
104 	{
105 		perror("bind error!\n");
106 		return -1;
107 	}
108 	
109 	/*************************************************************/
110 	//禁用过滤规则，本进程不接收报文，只负责发送
111     setsockopt(sock_fd, SOL_CAN_RAW, CAN_RAW_FILTER, NULL, 0);          
112 
113     /*************************************************************/
114     //设置read()和write()函数设置为非堵塞方式
115     int flags;
116     flags = fcntl(sock_fd, F_GETFL);
117     flags |= O_NONBLOCK;
118     fcntl(sock_fd, F_SETFL, flags);  
119 
120     /*************************************************************/
121     /*返回套接口*/
122     return sock_fd;  
123     /*************************************************************/
124 }
125 
126 
127 /**********************************************************************
128 * 函数名称： void CAN_Set_Interrupt(int can_port,  pCanInterrupt callback)
129 * 功能描述： 创建CAN接收线程，并传入应用的的回调函数，回调函数主要处理应用层的功能
130 * 输入参数： can_port,端口号
131 *            callback： 中断具体处理应用功能的回调函数
132 * 输出参数： 无
133 * 返 回 值： 无
134 * 修改日期             版本号        修改人           修改内容
135 * -----------------------------------------------
136 * 2020/05/13         V1.0             bert            创建
137 ***********************************************************************/
138 void CAN_Set_Interrupt(int can_port,  pCanInterrupt callback)
139 {
140     int err;
141     
142     if ( NULL != callback ) 
143     {
144         g_pCanInterrupt = callback;
145     }
146     
147     err = pthread_create(&ntid, NULL,CAN1_RX0_IRQHandler, NULL );
148     if( err !=0 )
149     {
150         printf("create thread fail! \n");
151         return ;
152     }
153     printf("create thread success!\n");
154     
155 
156     return ;
157 }
158 
159 
160 
161 /**********************************************************************
162 * 函数名称： void CAN_Read(int can_port, CanRxMsg* recv_msg)
163 * 功能描述： CAN读取接收寄存器，取出接收到的报文
164 * 输入参数： can_port,端口号     
165 * 输出参数： recv_msg：接收报文
166 * 返 回 值： 无
167 * 修改日期             版本号        修改人           修改内容
168 * -----------------------------------------------
169 * 2020/05/13         V1.0             bert            创建
170 ***********************************************************************/
171 void CAN_Read(int can_port, CanRxMsg* recv_msg)
172 { 
173     unsigned char i;
174     static unsigned int rxcounter =0;
175     
176     int nbytes;
177     struct can_frame rxframe;
178     
179     
180     nbytes = read(can_port, &rxframe, sizeof(struct can_frame));
181 	if(nbytes>0)
182 	{
183 	    printf("nbytes = %d \n",nbytes );
184 	    
185 	    recv_msg->StdId = rxframe.can_id;
186 	    recv_msg->DLC = rxframe.can_dlc;
187 	    memcpy( recv_msg->Data, &rxframe.data[0], rxframe.can_dlc);
188 	    
189 		rxcounter++;
190 		printf("rxcounter=%d, ID=%03X, DLC=%d, data=%02X %02X %02X %02X %02X %02X %02X %02X \n",  \
191 			rxcounter,
192 			rxframe.can_id, rxframe.can_dlc,  \
193 			rxframe.data[0],\
194 			rxframe.data[1],\
195 			rxframe.data[2],\
196 			rxframe.data[3],\
197 			rxframe.data[4],\
198 			rxframe.data[5],\
199 			rxframe.data[6],\
200 			rxframe.data[7] );
201 	}
202 
203     return ;
204 }
205  
206 /**********************************************************************
207 * 函数名称： void CAN_Write(int can_port, CanTxMsg send_msg)
208 * 功能描述： CAN报文发送接口，调用发送寄存器发送报文
209 * 输入参数： can_port,端口号     
210 * 输出参数： send_msg：发送报文
211 * 返 回 值： 无
212 * 修改日期             版本号        修改人           修改内容
213 * -----------------------------------------------
214 * 2020/05/13         V1.0             bert            创建
215 ***********************************************************************/
216 void CAN_Write(int can_port, CanTxMsg send_msg)
217 {
218     unsigned char i;
219     static unsigned int txcounter=0;
220     int nbytes;
221     
222     struct can_frame txframe;
223     
224     txframe.can_id = send_msg.StdId;
225     txframe.can_dlc = send_msg.DLC;
226     memcpy(&txframe.data[0], &send_msg.Data[0], txframe.can_dlc);
227 
228     nbytes = write(can_port, &txframe, sizeof(struct can_frame)); //发送 frame[0]
229 	
230 	if(nbytes == sizeof(txframe))
231 	{
232 	    txcounter++;
233 	    printf("txcounter=%d, ID=%03X, DLC=%d, data=%02X %02X %02X %02X %02X %02X %02X %02X \n",  \
234 			txcounter,
235 			txframe.can_id, txframe.can_dlc,  \
236 			txframe.data[0],\
237 			txframe.data[1],\
238 			txframe.data[2],\
239 			txframe.data[3],\
240 			txframe.data[4],\
241 			txframe.data[5],\
242 			txframe.data[6],\
243 			txframe.data[7] );
244     }
245     else
246 	{
247 		//printf("Send Error frame[0], nbytes=%d\n!",nbytes);
248 	}
249 
250     return ;
251 }
252 
253 /**********************************************************************
254 * 函数名称： void CAN1_RX0_IRQHandler(void)
255 * 功能描述： CAN接收线程函数
256 * 输入参数： 无  
257 * 输出参数： 无
258 * 返 回 值： 无
259 * 修改日期             版本号        修改人           修改内容
260 * -----------------------------------------------
261 * 2020/05/13         V1.0             bert            创建
262 ***********************************************************************/
263 void *CAN1_RX0_IRQHandler(void *arg)
264 {
265     /* 接收报文定义 */
266     while( 1 )
267     {
268     /* 如果回调函数存在，则执行回调函数 */
269         if( g_pCanInterrupt != NULL)
270         {
271             g_pCanInterrupt();
272         }
273         usleep(10000);
274     }
275 }
276 
277 
278 
279 /**********************************************************************
280 * 名称：     can1_controller
281 * 功能描述： CAN1结构体初始化
282 * 修改日期             版本号        修改人           修改内容
283 * -----------------------------------------------
284 * 2020/05/13         V1.0             bert            创建
285 ***********************************************************************/
286  CAN_COMM_STRUCT can1_controller = {
287     .name                   = "can0",
288 	.can_port               = -1,//默认，应用层传入此参数
289 	.can_set_controller     = CAN_Set_Controller,
290     .can_set_interrput      = CAN_Set_Interrupt,
291     .can_read               = CAN_Read,
292     .can_write              = CAN_Write, 
293 };
294 
295 
296 /**********************************************************************
297 * 函数名称： void CAN1_contoller_add(void)
298 * 功能描述： CAN结构体注册接口，应用层在使用can1_controller前调用
299 * 输入参数： 无  
300 * 输出参数： 无
301 * 返 回 值： 无
302 * 修改日期             版本号        修改人           修改内容
303 * -----------------------------------------------
304 * 2020/05/13         V1.0             bert            创建
305 ***********************************************************************/
306 void CAN1_contoller_add(void)
307 {
308     /*将can1_controller传递给应用层*/
309     register_can_controller( &can1_controller );
310 }
311 
312 
313 /***********************************************************************
314 ****************End Of File*********************************************
315 ***********************************************************************/
316 
