01 /**********************************************************************
02 * 文件名称： app_can.c
03 * 功能描述： CAN应用层代码，实现功能如下：
04 *            功能1：周期发送CAN报文；
05 *            功能2：接收报文，并转发报文
06 * 文件目的： 掌握CAN报文的基本发送和接收处理
07 * 修改日期             版本号        修改人           修改内容
08 * -----------------------------------------------
09 * 2020/05/13         V1.0             bert            创建
10 ***********************************************************************/
11 
12 /**************头文件**************************************************/
13 
14 #include <stdio.h>
15 #include <stdlib.h>
16 #include <string.h>
17 #include <unistd.h>
18 
19 #include <net/if.h>
20 #include <sys/ioctl.h>
21 #include <sys/socket.h>
22 
23 #include <linux/can.h>
24 #include <linux/can/raw.h>
25 
26 #include "app_can.h"
27 #include "can_controller.h"
28 
29 
30 /**************宏定义**************************************************/
31 /* 本例程中测试周期发送的CAN报文ID */
32 #define TX_CAN_ID           0X101   
33 
34 /* 本例程中测试接收的CAN报文ID */
35 #define RX_CAN_ID           0x201   
36 
37 /* 本例程中接收到测试报文为RX_CAN_ID的报文，将ID改为RX_TO_TX_CAN_ID为ID的报文转发出去 */
38 #define RX_TO_TX_CAN_ID     0X301   
39 
40 
41 /**************全局变量定义*********************************************/
42 /** 
43 * CAN应用层调用结构体指针变量
44 */
45 static CAN_COMM_STRUCT gCAN_COMM_STRUCT;
46 
47 /** 
48 *CAN应用层接收CAN报文后存储的CAN消息
49 */
50 static CanRxMsg  g_CAN1_Rx_Message;
51 
52 /** 
53 *CAN应用层接收CAN报文标志，处理完后清零
54 */
55 static unsigned char g_CAN1_Rx_Flag =0; 
56 
57 
58 /**************函数声明*************************************************/
59 /** 
60 *CAN中断中回调函数，在MCU中中断处理函数有特定的函数定义，
61 *此处将中断函数放在can_controller.c中，将中断中的具体处理内容放到app_can.c应用中，
62 *此处作为回调是为了与linux socket编程中类比，在linux socket中使用回调传入的是接收线程函数
63 */
64 void CAN_RX_IRQHandler_Callback(void);
65 
66 
67 
68 
69 /***********************************************************************
70 ****************应用层代码**********************************************
71 ***********************************************************************/
72 
73 /**********************************************************************
74 * 函数名称： int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
75 * 功能描述： 应用层进行CAN1结构体注册
76 * 输入参数： p_can_controller，CAN控制器抽象结构体
77 * 输出参数： 无
78 * 返 回 值： 无
79 * 修改日期             版本号        修改人           修改内容
80 * -----------------------------------------------
81 * 2020/05/13         V1.0             bert            创建
82 ***********************************************************************/
83 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
84 {
85     /* 判断传入的p_can_controller为非空，目的是确认这个结构体是实体*/
86     if( p_can_controller != NULL )
87     {
88         /* 将传入的参数p_can_controller赋值给应用层结构体gCAN_COMM_STRUCT */
89         
90         /*端口号，类比socketcan套接口*/
91         gCAN_COMM_STRUCT.can_port              = p_can_controller->can_port; 
92         /*CAN控制器配置函数*/
93         gCAN_COMM_STRUCT.can_set_controller    = p_can_controller->can_set_controller; 
94         /*CAN中断配置*/
95         gCAN_COMM_STRUCT.can_set_interrput     = p_can_controller->can_set_interrput;
96         /*CAN报文读函数*/
97         gCAN_COMM_STRUCT.can_read              = p_can_controller->can_read;
98         /*CAN报文发送函数*/
99         gCAN_COMM_STRUCT.can_write             = p_can_controller->can_write;
100         return 1;
101     }
102 	return 0;
103 }
104 
105 /**********************************************************************
106 * 函数名称： void app_can_init(void)
107 * 功能描述： CAN应用层初始化
108 * 输入参数： 无
109 * 输出参数： 无
110 * 返 回 值： 无
111 * 修改日期             版本号        修改人           修改内容
112 * -----------------------------------------------
113 * 2020/05/13         V1.0             bert            创建
114 ***********************************************************************/
115 void app_can_init(void)
116 {
117     /** 
118     * 应用层进行CAN1结构体注册
119     */
120     CAN1_contoller_add();
121     
122     /*
123     *调用can_set_controller进行CAN控制器配置，
124     *返回can_port，类比linux socketcan中的套接口，单片机例程中作为自定义CAN通道 
125     */
126     gCAN_COMM_STRUCT.can_port = gCAN_COMM_STRUCT.can_set_controller();
127     /** 
128     * 调用can_set_interrput配置CAN接收中断，类比socketcan中的接收线程，本例不用接收，因此回调函数传入NULL
129     */
130     gCAN_COMM_STRUCT.can_set_interrput( gCAN_COMM_STRUCT.can_port, NULL );
131 }
132 
133 
134 /**********************************************************************
135 * 函数名称： void app_can_tx_test(void)
136 * 功能描述： CAN应用层报文发送函数，用于测试周期发送报文
137 * 输入参数： 无
138 * 输出参数： 无
139 * 返 回 值： 无
140 * 修改日期             版本号        修改人           修改内容
141 * -----------------------------------------------
142 * 2020/05/13         V1.0             bert            创建
143 ***********************************************************************/
144 void app_can_tx_test(void)
145 {
146     // 以10ms为基准，运行CAN测试程序
147     
148     unsigned char i=0;
149     
150     /* 发送报文定义 */
151     CanTxMsg TxMessage;
152     
153     /* 发送报文中用一个字节来作为计数器 */
154     static unsigned char tx_counter = 0;
155     
156     /* 以10ms为基准，通过timer计数器设置该处理函数后面运行代码的周期为1秒钟*/  
157     static unsigned int timer =0;
158     if(timer++>100)
159     {
160         timer = 0;
161     }
162     else
163     {
164         return ;
165     }
166     
167     /* 发送报文报文数据填充，此报文周期是1秒 */
168     TxMessage.StdId = TX_CAN_ID;	      /* 标准标识符为0x000~0x7FF */
169     TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
170     TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
171     TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
172     TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
173     
174     /* 填充数据，此处可以根据实际应用填充 */
175     TxMessage.Data[0] = tx_counter++;       /* 用来识别报文发送计数器 */
176     for(i=1; i<TxMessage.DLC; i++)
177     {
178         TxMessage.Data[i] = i;            
179     }
180     
181     /*  调用can_write发送CAN报文 */
182     gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
183     
184 }
185 
186 
187 
188 /**********************************************************************
189 * 函数名称： int main(int argc, char **argv)
190 * 功能描述： 主函数
191 * 输入参数： 无
192 * 输出参数： 无
193 * 返 回 值： 无
194 * 修改日期             版本号        修改人           修改内容
195 * -----------------------------------------------
196 * 2020/05/13         V1.0             bert            创建
197 ***********************************************************************/
198 int main(int argc, char **argv)
199 {
200     /* CAN应用层初始化 */
201     app_can_init();
202     
203     while(1)
204     {
205         /* CAN应用层周期发送报文 */
206         app_can_tx_test();
207         
208         /* 利用linux的延时函数设计10ms的运行基准 */
209         usleep(10000);
210     }
211 }
212 
213 
214 
215 
216 /***********************************************************************
217 ****************End Of File*********************************************
218 ***********************************************************************/
