 /**********************************************************************
02 * 文件名称： can_controller.c
03 * 功能描述： CAN抽象结构体框架的具体实现函数
04 *            主要实现调用硬件驱动的接口封装
05 * 文件目的： 掌握CAN底层驱动的抽象
06 * 修改日期             版本号        修改人           修改内容
07 * -----------------------------------------------
08 * 2020/05/13         V1.0             bert            创建
09 ***********************************************************************/
10
11 /**************头文件**************************************************/
12 #include "can_controller.h"
13 #include "string.h"
14
15 /**************宏定义**************************************************/
16 #define CAN_BAUD_NUM    3            //可用配置波特率个数
17
18 /**************全局变量定义*********************************************/
19 const unsigned int CAN1_baud_table[CAN_BAUD_NUM][5] =
20 {
21     //波特率， CAN_SJW，     CAN_BS1，        CAN_BS2，    CAN_Prescaler
22      {250,   CAN_SJW_1tq,    CAN_BS1_15tq,   CAN_BS2_5tq,    8},                 //250k  ok
23      {500,   CAN_SJW_1tq,    CAN_BS1_15tq,   CAN_BS2_5tq,    4},                     //500K  ok
24      {1000,  CAN_SJW_1tq,    CAN_BS1_15tq,   CAN_BS2_5tq,    2},                     //1000K ok
25 };
26
27 pCanInterrupt    g_pCanInterrupt = NULL;
28
29
30 /**************函数声明*************************************************/
31 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller);
32
33
34
35 /***********************************************************************
36 ****************CAN控制器抽象接口实现代码*******************************
37 ***********************************************************************/
38
39 /**********************************************************************
40 * 函数名称： void CAN_Baud_Process(unsigned int Baud,CAN_InitTypeDef *CAN_InitStructure)
41 * 功能描述： 计算波特率
42 * 输入参数： 无
43 * 输出参数： 无
44 * 返 回 值： 无
45 * 注    意：CAN时钟为42M
46 * CAN_SJW : CAN_SJW_1tq - CAN_SJW_4tq          不能比任何一相位缓冲段长
47 * CAN_BS1 : CAN_BS1_1tq - CAN_BS1_16tq
48 * CAN_BS2 : CAN_BS2_1tq - CAN_BS2_8tq
49 * CAN_Prescaler : 1 - 1024
50 *    配置说明：
51 * CAN_SJW + CAN_BS1 / (CAN_SJW + CAN_BS1 + CAN_BS2)
52 *    0.75     baud > 800k
53 *    0.80     baud > 500k
54 *    0.875    baud <= 500k
55 *    baud = 42M / (CAN_SJW + CAN_BS1 + CAN_BS2) / CAN_Prescaler
56 * 修改日期             版本号        修改人           修改内容
57 * -----------------------------------------------
58 * 2020/05/13         V1.0             bert            创建
59 ***********************************************************************/
60 void CAN_Baud_Process(unsigned int Baud,CAN_InitTypeDef *CAN_InitStructure)
61 {
62      unsigned int i = 0;
63      for(i = 0;i < CAN_BAUD_NUM;i ++)
64      {
65              if(Baud == CAN1_baud_table[i][0])
66              {
67                      CAN_InitStructure->CAN_SJW = CAN1_baud_table[i][1];
68                      CAN_InitStructure->CAN_BS1 = CAN1_baud_table[i][2];
69                      CAN_InitStructure->CAN_BS2 = CAN1_baud_table[i][3];
70                      CAN_InitStructure->CAN_Prescaler = CAN1_baud_table[i][4];
71                      return;
72              }
73      }
74 }
75
76
77 /**********************************************************************
78 * 函数名称： int CAN_Set_Controller( void )
79 * 功能描述： 单片机内部CAN控制器初始化，包括GPIO(TX,RX），CAN外设时钟，波特率，过滤器等
80 * 输入参数： 无
81 * 输出参数： 无
82 * 返 回 值： int can_port:返回对应CAN控制器的通道号，类比socketcan中的套接口
83 * 修改日期             版本号        修改人           修改内容
84 * -----------------------------------------------
85 * 2020/05/13         V1.0             bert            创建
86 ***********************************************************************/
87 int CAN_Set_Controller( void )
88 {
89     /* STM32 GPIO结构体 */
90     GPIO_InitTypeDef       GPIO_InitStructure;
91     /* STM32 CAN结构体 */
92     CAN_InitTypeDef        CAN_InitStructure;
93     /* STM32 CAN过滤器结构体 */
94     CAN_FilterInitTypeDef  CAN_FilterInitStructure;
95
96     /*************************************************************/
97     /*CAN相关GPIO配置，此处为：CAN_TX, CAN_RX*/
98
99     /*使能GPIO时钟*/
100     RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD, ENABLE);
101     /*初始化管脚配置*/
102     GPIO_InitStructure.GPIO_Pin     = GPIO_Pin_0 ;
103     GPIO_InitStructure.GPIO_Mode    = GPIO_Mode_AF;
104     GPIO_InitStructure.GPIO_Speed   = GPIO_Speed_50MHz;
105     GPIO_InitStructure.GPIO_OType   = GPIO_OType_PP;
106     GPIO_InitStructure.GPIO_PuPd    = GPIO_PuPd_UP;
107     GPIO_Init(GPIOD, &GPIO_InitStructure);
108
109     GPIO_InitStructure.GPIO_Pin     = GPIO_Pin_1;
110     GPIO_InitStructure.GPIO_Mode    = GPIO_Mode_AF;
111     GPIO_InitStructure.GPIO_Speed   = GPIO_Speed_50MHz;
112     GPIO_InitStructure.GPIO_OType   = GPIO_OType_PP;
113     GPIO_InitStructure.GPIO_PuPd    = GPIO_PuPd_UP;
114     GPIO_Init(GPIOD, &GPIO_InitStructure);
115     /*将GPIO设置为CAN复用模式*/
116     GPIO_PinAFConfig(GPIOD, GPIO_PinSource0, GPIO_AF_CAN1);
117     GPIO_PinAFConfig(GPIOD, GPIO_PinSource1, GPIO_AF_CAN1);
118
119     /*************************************************************/
120     /*CAN控制器相关配置，此处为波特率，采样率等*/
121
122     /* 使能CAN时钟 */
123     RCC_APB1PeriphClockCmd(RCC_APB1Periph_CAN1, ENABLE);
124
125     /* 初始化CAN控制器工作模式*/
126     CAN_DeInit(CAN1);
127     CAN_StructInit(&CAN_InitStructure);
128     CAN_InitStructure.CAN_TTCM = DISABLE;
129     CAN_InitStructure.CAN_ABOM = DISABLE;
130     CAN_InitStructure.CAN_AWUM = DISABLE;
131     CAN_InitStructure.CAN_NART = DISABLE;
132     CAN_InitStructure.CAN_RFLM = DISABLE;
133     CAN_InitStructure.CAN_TXFP = DISABLE;
134     CAN_InitStructure.CAN_Mode = CAN_Mode_Normal;//CAN工作模式
135
136     /* 初始化CAN波特率 */
137     CAN_Baud_Process(500,&CAN_InitStructure);
138     CAN_Init(CAN1, &CAN_InitStructure);
139
140
141     /*************************************************************/
142     /* 初始化CAN过滤器 */
143     CAN_FilterInitStructure.CAN_FilterNumber = 0;                       /* CAN1滤波器号从0到13 */
144     CAN_FilterInitStructure.CAN_FilterMode = CAN_FilterMode_IdMask;     /* 滤波屏蔽模式 */
145     CAN_FilterInitStructure.CAN_FilterScale = CAN_FilterScale_32bit;
146     CAN_FilterInitStructure.CAN_FilterIdHigh = 0x0000;
147     CAN_FilterInitStructure.CAN_FilterIdLow = 0x0000;
148     CAN_FilterInitStructure.CAN_FilterMaskIdHigh = 0x0000;      /* 不屏蔽任何ID */
149     CAN_FilterInitStructure.CAN_FilterMaskIdLow = 0x0000;           /* 不屏蔽任何ID */
150     CAN_FilterInitStructure.CAN_FilterFIFOAssignment = 0;
151
152     CAN_FilterInitStructure.CAN_FilterActivation = ENABLE;
153     CAN_FilterInit(&CAN_FilterInitStructure);
154
155     /*************************************************************/
156     /* 设置完CAN后，返回当前设置的CAN的端口号，此处主要类比linux socketcan中的套接口 */
157
158
159     return CAN_PORT_CAN1;
160     /*************************************************************/
161 }
162
163 /**********************************************************************
164 * 函数名称： void CAN_Set_Interrupt(int can_port,  pCanInterrupt callback)
165 * 功能描述： 使能CAN中断处理，并传入应用的的回调函数，回调函数主要处理应用层的功能
166 * 输入参数： can_port,端口号
167 *            callback： 中断具体处理应用功能的回调函数
168 * 输出参数： 无
169 * 返 回 值： 无
170 * 修改日期             版本号        修改人           修改内容
171 * -----------------------------------------------
172 * 2020/05/13         V1.0             bert            创建
173 ***********************************************************************/
174 void CAN_Set_Interrupt(int can_port,  pCanInterrupt callback)
175 {
176     NVIC_InitTypeDef NVIC_InitStructure;
177
178     /* 根据CAN端口号配置中断 */
179     switch( can_port )
180     {
181         case CAN_PORT_CAN1:
182         {
183             /* 初始化回调接口函数 */
184             if ( NULL != callback )
185             {
186                 g_pCanInterrupt = callback;
187             }
188
189             /* 使用CAN0_RX中断，在linux socket can中类似创建接收线程 */
190             NVIC_InitStructure.NVIC_IRQChannel = CAN1_RX0_IRQn;
191             NVIC_PriorityGroupConfig(NVIC_PriorityGroup_4);
192             NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
193             NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
194             NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
195             NVIC_Init(&NVIC_InitStructure);
196             CAN_ITConfig(CAN1, CAN_IT_FMP0, ENABLE);
197         }
198         break;
199
200         default:
201             break;
202
203     }
204     return ;
205 }
206
207
208 /**********************************************************************
209 * 函数名称： void CAN_Read(int can_port, CanRxMsg* recv_msg)
210 * 功能描述： CAN读取接收寄存器，取出接收到的报文
211 * 输入参数： can_port,端口号
212 * 输出参数： recv_msg：接收报文
213 * 返 回 值： 无
214 * 修改日期             版本号        修改人           修改内容
215 * -----------------------------------------------
216 * 2020/05/13         V1.0             bert            创建
217 ***********************************************************************/
218 void CAN_Read(int can_port, CanRxMsg* recv_msg)
219 {
220     switch( can_port )
221     {
222         case CAN_PORT_CAN1:
223         {
224             /* 从FIFO中读取CAN报文 */
225             CAN_Receive(CAN1,CAN_FIFO0, recv_msg);
226         }
227         break;
228
229         default:
230             break;
231     }
232     return ;
233 }
234
235 /**********************************************************************
236 * 函数名称： void CAN_Write(int can_port, CanTxMsg send_msg)
237 * 功能描述： CAN报文发送接口，调用发送寄存器发送报文
238 * 输入参数： can_port,端口号
239 * 输出参数： send_msg：发送报文
240 * 返 回 值： 无
241 * 修改日期             版本号        修改人           修改内容
242 * -----------------------------------------------
243 * 2020/05/13         V1.0             bert            创建
244 ***********************************************************************/
245 void CAN_Write(int can_port, CanTxMsg send_msg)
246 {
247     unsigned char i;
248     uint8_t transmit_mailbox = 0;
249     CanTxMsg TxMessage;
250
251     switch( can_port )
252     {
253         case CAN_PORT_CAN1:
254         {
255             TxMessage.StdId = send_msg.StdId;         // 标准标识符为0x000~0x7FF
256             TxMessage.ExtId = 0x0000;             // 扩展标识符0x0000
257             TxMessage.IDE   = CAN_ID_STD;         // 使用标准标识符
258             TxMessage.RTR   = CAN_RTR_DATA;       // 设置为数据帧
259             TxMessage.DLC   = send_msg.DLC;       // 数据长度, can报文规定最大的数据长度为8字节
260
261             for(i=0; i<TxMessage.DLC; i++)
262             {
263                 TxMessage.Data[i] = send_msg.Data[i];
264             }
265             transmit_mailbox = CAN_Transmit(CAN1,&TxMessage);  /* 返回这个信息请求发送的邮箱号0,1,2或没有邮箱申请发送no_box */
266         }
267         break;
268
269         default:
270             break;
271     }
272     return ;
273 }
274
275 /**********************************************************************
276 * 函数名称： void CAN1_RX0_IRQHandler(void)
277 * 功能描述： CAN接收中断函数
278 * 输入参数： 无
279 * 输出参数： 无
280 * 返 回 值： 无
281 * 修改日期             版本号        修改人           修改内容
282 * -----------------------------------------------
283 * 2020/05/13         V1.0             bert            创建
284 ***********************************************************************/
285 void CAN1_RX0_IRQHandler(void)
286 {
287     /* 如果回调函数存在，则执行回调函数 */
288     if( g_pCanInterrupt != NULL)
289     {
290         g_pCanInterrupt();
291     }
292
293     /* 清除挂起中断 */
294     CAN_ClearITPendingBit(CAN1,CAN_IT_FMP0);
295 }
296
297
298 /**********************************************************************
299 * 名称：     can1_controller
300 * 功能描述： CAN1结构体初始化
301 * 修改日期             版本号        修改人           修改内容
302 * -----------------------------------------------
303 * 2020/05/13         V1.0             bert            创建
304 ***********************************************************************/
305 CAN_COMM_STRUCT can1_controller = {
306     .name                   = "can0",
307     .can_port               = CAN_PORT_CAN1,
308     .can_set_controller     = CAN_Set_Controller,
309     .can_set_interrput      = CAN_Set_Interrupt,
310     .can_read               = CAN_Read,
311     .can_write              = CAN_Write,
312 };
313
314
315 /**********************************************************************
316 * 函数名称： void CAN1_contoller_add(void)
317 * 功能描述： CAN结构体注册接口，应用层在使用can1_controller前调用
318 * 输入参数： 无
319 * 输出参数： 无
320 * 返 回 值： 无
321 * 修改日期             版本号        修改人           修改内容
322 * -----------------------------------------------
323 * 2020/05/13         V1.0             bert            创建
324 ***********************************************************************/
325 void CAN1_contoller_add(void)
326 {
327     /*将can1_controller传递给应用层*/
328     register_can_controller( &can1_controller );
329 }
330
331
332 /***********************************************************************
333 ****************End Of File*********************************************
334 ***********************************************************************/
335
