01 /**********************************************************************
02 * 文件名称： app_can.c
03 * 功能描述： CAN应用层代码，实现功能如下：
04 *            功能1：周期发送CAN报文；
05 *            功能2：接收报文，并转发报文
06 * 文件目的： 掌握CAN报文的基本发送和接收处理
07 * 修改日期             版本号        修改人           修改内容
08 * -----------------------------------------------
09 * 2020/05/13         V1.0             bert            创建
10 ***********************************************************************/
11
12
13 /**************头文件**************************************************/
14 #include "app_can.h"
15 #include "can_controller.h"
16 #include <string.h>
17
18
19 /**************宏定义**************************************************/
20 /* 本例程中测试周期发送的CAN报文ID */
21 #define TX_CAN_ID           0X101
22
23 /* 本例程中测试接收的CAN报文ID */
24 #define RX_CAN_ID           0x201
25
26 /* 本例程中接收到测试报文为RX_CAN_ID的报文，将ID改为RX_TO_TX_CAN_ID为ID的报文转发出去 */
27 #define RX_TO_TX_CAN_ID     0X301
28
29
30 /**************全局变量定义*********************************************/
31 /**
32 * CAN应用层调用结构体指针变量
33 */
34 static CAN_COMM_STRUCT gCAN_COMM_STRUCT;
35
36 /**
37 *CAN应用层接收CAN报文后存储的CAN消息
38 */
39 static CanRxMsg  g_CAN1_Rx_Message;
40
41 /**
42 *CAN应用层接收CAN报文标志，处理完后清零
43 */
44 static unsigned char g_CAN1_Rx_Flag =0;
45
46
47 /**************函数声明*************************************************/
48 /**
49 *CAN中断中回调函数，在MCU中中断处理函数有特定的函数定义，
50 *此处将中断函数放在can_controller.c中，将中断中的具体处理内容放到app_can.c应用中，
51 *此处作为回调是为了与linux socket编程中类比，在linux socket中使用回调传入的是接收线程函数
52 */
53 void CAN_RX_IRQHandler_Callback(void);
54
55
56
57
58 /***********************************************************************
59 ****************应用层代码**********************************************
60 ***********************************************************************/
61
62 /**********************************************************************
63 * 函数名称： int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
64 * 功能描述： 应用层进行CAN1结构体注册
65 * 输入参数： p_can_controller，CAN控制器抽象结构体
66 * 输出参数： 无
67 * 返 回 值： 无
68 * 修改日期             版本号        修改人           修改内容
69 * -----------------------------------------------
70 * 2020/05/13         V1.0             bert            创建
71 ***********************************************************************/
72 int register_can_controller(const pCAN_COMM_STRUCT p_can_controller)
73 {
74     /* 判断传入的p_can_controller为非空，目的是确认这个结构体是实体*/
75     if( p_can_controller != NULL )
76     {
77         /* 将传入的参数p_can_controller赋值给应用层结构体gCAN_COMM_STRUCT */
78
79         /*端口号，类比socketcan套接口*/
80         gCAN_COMM_STRUCT.can_port              = p_can_controller->can_port;
81         /*CAN控制器配置函数*/
82         gCAN_COMM_STRUCT.can_set_controller    = p_can_controller->can_set_controller;
83         /*CAN中断配置*/
84         gCAN_COMM_STRUCT.can_set_interrput     = p_can_controller->can_set_interrput;
85         /*CAN报文读函数*/
86         gCAN_COMM_STRUCT.can_read              = p_can_controller->can_read;
87         /*CAN报文发送函数*/
88         gCAN_COMM_STRUCT.can_write             = p_can_controller->can_write;
89         return 1;
90     }
91      return 0;
92 }
93
94 /**********************************************************************
95 * 函数名称： void app_can_init(void)
96 * 功能描述： CAN应用层初始化
97 * 输入参数： 无
98 * 输出参数： 无
99 * 返 回 值： 无
100 * 修改日期             版本号        修改人           修改内容
101 * -----------------------------------------------
102 * 2020/05/13         V1.0             bert            创建
103 ***********************************************************************/
104 void app_can_init(void)
105 {
106     /**
107     * 应用层进行CAN1结构体注册
108     */
109     CAN1_contoller_add();
110
111     /*
112     *调用can_set_controller进行CAN控制器配置，
113     *返回can_port，类比linux socketcan中的套接口，单片机例程中作为自定义CAN通道
114     */
115     gCAN_COMM_STRUCT.can_port = gCAN_COMM_STRUCT.can_set_controller();
116     /**
117     * 调用can_set_interrput配置CAN接收中断，类比socketcan中的接收线程
118     */
119     gCAN_COMM_STRUCT.can_set_interrput( gCAN_COMM_STRUCT.can_port, CAN_RX_IRQHandler_Callback );
120 }
121
122
123 /**********************************************************************
124 * 函数名称： void app_can_tx_test(void)
125 * 功能描述： CAN应用层报文发送函数，用于测试周期发送报文
126 * 输入参数： 无
127 * 输出参数： 无
128 * 返 回 值： 无
129 * 修改日期             版本号        修改人           修改内容
130 * -----------------------------------------------
131 * 2020/05/13         V1.0             bert            创建
132 ***********************************************************************/
133 void app_can_tx_test(void)
134 {
135     // 以10ms为基准，运行CAN测试程序
136
137     unsigned char i=0;
138
139     /* 发送报文定义 */
140     CanTxMsg TxMessage;
141
142     /* 发送报文中用一个字节来作为计数器 */
143     static unsigned char tx_counter = 0;
144
145     /* 以10ms为基准，通过timer计数器设置该处理函数后面运行代码的周期为1秒钟*/
146     static unsigned int timer =0;
147     if(timer++>100)
148     {
149         timer = 0;
150     }
151     else
152     {
153         return ;
154     }
155
156     /* 发送报文报文数据填充，此报文周期是1秒 */
157     TxMessage.StdId = TX_CAN_ID;          /* 标准标识符为0x000~0x7FF */
158     TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
159     TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
160     TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
161     TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
162
163     /* 填充数据，此处可以根据实际应用填充 */
164     TxMessage.Data[0] = tx_counter++;       /* 用来识别报文发送计数器 */
165     for(i=1; i<TxMessage.DLC; i++)
166     {
167         TxMessage.Data[i] = i;
168     }
169
170     /*  调用can_write发送CAN报文 */
171     gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
172
173 }
174
175
176 /**********************************************************************
177 * 函数名称： void app_can_rx_test(void)
178 * 功能描述： CAN应用层接收报文处理函数，用于处理中断函数中接收的报文
179 * 输入参数： 无
180 * 输出参数： 无
181 * 返 回 值： 无
182 * 修改日期             版本号        修改人           修改内容
183 * -----------------------------------------------
184 * 2020/05/13         V1.0             bert            创建
185 ***********************************************************************/
186 void app_can_rx_test(void)
187 {
188     unsigned char i=0;
189
190     /* 发送报文定义 */
191     CanTxMsg TxMessage;
192
193     /* 发送报文中用一个字节来作为计数器 */
194     static unsigned char rx_counter = 0;
195
196
197     if( g_CAN1_Rx_Flag == 1)
198     {
199         g_CAN1_Rx_Flag = 0;
200
201         /* 发送报文报文数据填充，此报文周期是1秒 */
202         TxMessage.StdId = RX_TO_TX_CAN_ID;    /* 标准标识符为0x000~0x7FF */
203         TxMessage.ExtId = 0x0000;             /* 扩展标识符0x0000 */
204         TxMessage.IDE   = CAN_ID_STD;         /* 使用标准标识符 */
205         TxMessage.RTR   = CAN_RTR_DATA;       /* 设置为数据帧  */
206         TxMessage.DLC   = 8;                  /* 数据长度, can报文规定最大的数据长度为8字节 */
207
208         /* 填充数据，此处可以根据实际应用填充 */
209         TxMessage.Data[0] = rx_counter++;      /* 用来识别报文发送计数器 */
210         for(i=1; i<TxMessage.DLC; i++)
211         {
212             TxMessage.Data[i] = g_CAN1_Rx_Message.Data[i];
213         }
214
215         /*  调用can_write发送CAN报文 */
216         gCAN_COMM_STRUCT.can_write(gCAN_COMM_STRUCT.can_port, TxMessage);
217     }
218 }
219
220 /**********************************************************************
221 * 函数名称： void CAN_RX_IRQHandler_Callback(void)
222 * 功能描述： CAN1接收中断函数；在linux中可以类比用线程，或定时器去读CAN数据
223 * 输入参数： 无
224 * 输出参数： 无
225 * 返 回 值： 无
226 * 修改日期             版本号        修改人           修改内容
227 * -----------------------------------------------
228 * 2020/05/13         V1.0             bert            创建
229 ***********************************************************************/
230 void CAN_RX_IRQHandler_Callback(void)
231 {
232     /* 接收报文定义 */
233     CanRxMsg RxMessage;
234
235     /* 接收报文清零 */
236     memset( &RxMessage, 0, sizeof(CanRxMsg) );
237
238     /* 通过can_read接口读取寄存器已经接收到的报文 */
239     gCAN_COMM_STRUCT.can_read(gCAN_COMM_STRUCT.can_port, &RxMessage);
240
241     /* 将读取到的CAN报文存拷贝到全局报文结构体g_CAN1_Rx_Message */
242     memcpy(&g_CAN1_Rx_Message, &RxMessage, sizeof( CanRxMsg ) );
243
244     /* 设置当前接收完成标志，判断当前接收报文ID为RX_CAN_ID，则设置g_CAN1_Rx_Flag=1*/
245     if( g_CAN1_Rx_Message.StdId == RX_CAN_ID )
246     {
247         g_CAN1_Rx_Flag = 1;
248     }
249 }
250
251
252 /***********************************************************************
253 ****************End Of File*********************************************
254 ***********************************************************************/
255
256
257
258
259
