#include "timer.h"
#include "my_printf.h"

static volatile unsigned int *CCM_CCGR3 							 ;
static volatile unsigned int *IOMUXC_SW_MUX_CTL_PAD_CSI_HSYNC        ;
static volatile unsigned int *GPIO4_GDIR							 ;
static volatile unsigned int *GPIO4_DR								 ;
extern unsigned int temp_negative_flag;


/**********************************************************************
 * 函数名称： ds18b20_gpio_init
 * 功能描述： 初始化GPIO管脚
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_gpio_init(void)
{
	unsigned int val;

	CCM_CCGR3								= (volatile unsigned int *)(0x20C4074);
	IOMUXC_SW_MUX_CTL_PAD_CSI_HSYNC			= (volatile unsigned int *)(0x20E01E0);
	GPIO4_GDIR								= (volatile unsigned int *)(0x20A8000 + 0x4);
	GPIO4_DR								= (volatile unsigned int *)(0x20A8000);
	
	/* 使能GPIO4
	 * set CCM to enable GPIO4
	 * CCM_CCGR3[CG6] 0x20C4074
	 * bit[13:12] = 0b11
	 */
	*CCM_CCGR3 |= (3<<12);
	
	/* 设置GPIO4_IO20用于GPIO
	 * set IOMUXC_SW_MUX_CTL_PAD_CSI_HSYNC
	 *      to configure GPIO4_IO120 as GPIO
	 * IOMUXC_SW_MUX_CTL_PAD_CSI_HSYNC  0x20E01E0
	 * bit[3:0] = 0b0101 alt5
	 */
	val = *IOMUXC_SW_MUX_CTL_PAD_CSI_HSYNC;
	val &= ~(0xf);
	val |= (5);
	*IOMUXC_SW_MUX_CTL_PAD_CSI_HSYNC = val;

}


/**********************************************************************
 * 函数名称： ds18b20_gpio_as_input
 * 功能描述： 设置GPIO4_20为输入引脚
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_gpio_as_input(void)
{	
	/*  
	 * 设置GPIO4_IO20作为input引脚
	 * set GPIO4_GDIR to configure GPIO4_IO20 as input
	 * GPIO4_GDIR  0x20A8000
	 * bit[1] = 0b0
	 */
	 *GPIO4_GDIR &= ~(1<<20);
}


/**********************************************************************
 * 函数名称： ds18b20_gpio_as_output
 * 功能描述： 设置GPIO4_20为输出引脚
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_gpio_as_output(void)
{		
	/*  
	 * 设置GPIO4_IO20作为output引脚
	 * set GPIO4_GDIR to configure GPIO4_IO20 as output
	 * GPIO4_GDIR  0x20A8000
	 * bit[1] = 0b1
	 */
	 *GPIO4_GDIR |= (1<<20);
}


/**********************************************************************
 * 函数名称： ds18b20_data_set
 * 功能描述： GPIO4_20管脚输出电平
 * 输入参数： 1-高电平 0-低电平
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_data_set(int val)
{
	if(val)
		*GPIO4_DR |= (1<<20);
	else
		*GPIO4_DR &= ~(1<<20);
}


/**********************************************************************
 * 函数名称： ds18b20_data_get
 * 功能描述： 读取GPIO4_20管脚电平
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 1-高电平 0-低电平
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static int ds18b20_data_get(void)
{
	if((*GPIO4_DR>>20) & 0x1)
		return 1;
	else
		return 0;
}


/**********************************************************************
 * 函数名称： ds18b20_data_set_val_for_time
 * 功能描述： 设置电平并延时一段时间
 * 输入参数： 电平(1--高电平  0--低电平) 延时时间 
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_data_set_val_for_time(int val, int us)
{
	ds18b20_gpio_as_output();
	ds18b20_data_set(val);
	gpt2_chan1_delay_us(us);
}


/**********************************************************************
 * 函数名称： ds18b20_init
 * 功能描述： 初始化ds18b20
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 0--初始化成功
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static int ds18b20_init(void)
{
	unsigned int val;
	ds18b20_data_set_val_for_time(1,6);
	ds18b20_data_set_val_for_time(0,500);
	ds18b20_gpio_as_input();
	gpt2_chan1_delay_us(80);
	
	val = ds18b20_data_get();
	gpt2_chan1_delay_us(400);
	
	return val;
}


/**********************************************************************
 * 函数名称： ds18b20_write_bit
 * 功能描述： 写入一位数据
 * 输入参数： 1或0
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_write_bit(int val)
{
	if (!val)
	{
		ds18b20_data_set_val_for_time(0, 60);
		ds18b20_gpio_as_input();
		gpt2_chan1_delay_us(2);
	}
	else
	{
		ds18b20_data_set_val_for_time(0, 2);
		ds18b20_gpio_as_input();
		gpt2_chan1_delay_us(60);
	}
}


/**********************************************************************
 * 函数名称： ds18b20_read_bit
 * 功能描述： 读取一位数据
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 1或0
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static int ds18b20_read_bit(void)
{
	int val;
	
	ds18b20_data_set(1);
	ds18b20_data_set_val_for_time(0, 2);
	ds18b20_gpio_as_input();
	gpt2_chan1_delay_us(10);
	val = ds18b20_data_get();
	gpt2_chan1_delay_us(50);
	return val;
}


/**********************************************************************
 * 函数名称： ds18b20_write_byte
 * 功能描述： 写入一字节数据
 * 输入参数： 1字节(8位)数据
 * 输出参数： 无
 * 返 回 值： 无
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static void ds18b20_write_byte(unsigned char data)
{
	/*
	优先传输最低位
	*/
	int i;
	for (i = 0; i < 8; i++)
	{ 
		ds18b20_write_bit(data & (1<<i));
	}
}


/**********************************************************************
 * 函数名称： ds18b20_read_byte
 * 功能描述： 读取一字节数据
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 1字节(8位)数据
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
static unsigned char ds18b20_read_byte(void)
{
	int i;
	unsigned char data = 0;

	for (i = 0; i < 8; i++)
	{
		if (ds18b20_read_bit() == 1)
			data |= (1<<i);
	}

	return data;
}


/**********************************************************************
 * 函数名称： ds18b20_read_rom
 * 功能描述： 读取ds18b20 64位序列号
 * 输入参数： 无
 * 输出参数： 无
 * 返 回 值： 0--读取成功 1--读取失败
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
int ds18b20_read_rom(void)
{
	int i;
	unsigned char rom[8];
	ds18b20_gpio_init();							//GPIO初始化
	if(ds18b20_init() != 0)						//ds18b20初始化
	{
		printf("ds18b20_initialization err!\n\r");
		return -1;
	}
	ds18b20_write_byte(0x33);						//读取64位序列号指令
	for (i = 0; i < 8; i++)							
	{
		rom[i] = ds18b20_read_byte();				//每一次读取8位数据，循环8次
		printf("%02x ", rom[i]);
	}
	printf("\n\r");
	return 0;
}


/**********************************************************************
 * 函数名称： ds18b20_data_read
 * 功能描述： 读取ds18b20数据
 * 输入参数： 无
 * 输出参数： 8位数据
 * 返 回 值： 0--读取成功 1--读取失败
 * 修改日期        版本号     修改人	      修改内容
 * -----------------------------------------------
 * 2020/03/23	     V1.0	  来木	      创建
 ***********************************************************************/
int ds18b20_data_read(void)
{
	unsigned char tempL=0,tempH=0;
	unsigned int integer;
	unsigned char decimal1,decimal2,decimal;

	ds18b20_gpio_init();							//GPIO管脚初始化

	if(ds18b20_init() != 0)							//ds18b20初始化
	{
		printf("ds18b20_initialization err!\n\r");
		return -1;
	}
	ds18b20_write_byte(0xcc);						//忽略rom指令，直接使用功能指令
	ds18b20_write_byte(0x44);						//温度转换指令
	gpt2_chan1_delay_us(1000000);					//转换需要时间，延时1s
	if(ds18b20_init() != 0)							//ds18b20初始化
	{
		printf("ds18b20_initialization err!\n\r");
		return -1;
	}
	ds18b20_write_byte(0xcc);						//忽略rom指令，直接使用功能指令
	ds18b20_write_byte(0xbe);						//读暂存器指令
	tempL = ds18b20_read_byte();					//读温度低8位
	tempH = ds18b20_read_byte();					//读温度高8位
	if(tempH>0x7f)      							//最高位为1时温度是负
	{
		tempL    = ~tempL;         						//补码转换，取反加一
		tempH    = ~tempH+1;      
		integer  = tempL/16+tempH*16;      			//整数部分
		decimal1 = (tempL&0x0f)*10/16; 				//小数第一位
		decimal2 = (tempL&0x0f)*100/16%10;			//小数第二位
		decimal  = decimal1*10+decimal2; 			//小数两位
		printf("temperature = %d.%d\n\r\n", integer, decimal);
    }
	integer  = tempL/16+tempH*16;      				//整数部分
	decimal1 = (tempL&0x0f)*10/16; 					//小数第一位
	decimal2 = (tempL&0x0f)*100/16%10;				//小数第二位
	decimal  = decimal1*10+decimal2; 				//小数两位
	printf("temperature = %d.%d\n\r\n", integer, decimal);

	return 0;
}
