package com.mn.icc.lib;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

public class ICCLib {
    private static final ICCLib ourInstance = new ICCLib();
    private ICCLib() { }
    public static ICCLib getDefault() {
        return ourInstance;
    }
    /**
     * 方法缓存表
     * key: 当前注册类的Class对象
     * value: 方法信息封装类的集合
     */
    private HashMap<Class, List<ICCMethod>> mMethodCaches = new HashMap();
    /**
     * 方法执行缓存表
     * key: 标签
     * value：方法执行信息封装类的集合
     */
    private HashMap<String, List<ICCMethodInvoke>> mMethodInvokeCaches = new HashMap();
    /**
     * 标签缓存表
     * key: 当前注册类的Class对象
     * value: 需要注销的标签集合
     */
    private HashMap<Class, List<String>> mTagCaches = new HashMap();

    /**
     * 注册
     * 三张缓存表初始化并赋值
     * @param object
     */
    public void register(Object object){
        //缓存
        // method.invoke();
        Class<?> objectClass = object.getClass();
        List<ICCMethod> iccMethods = findTaggedMethods(objectClass);

        List<String> tags = mTagCaches.get(objectClass);

        if (tags == null){
            tags = new ArrayList<>();
        }
        for (ICCMethod iccMethod : iccMethods) {
            String tag = iccMethod.getTag();
            if (!tags.contains(tag)){
                tags.add(tag);
            }

            List<ICCMethodInvoke> iccMethodInvokes = mMethodInvokeCaches.get(tag);
            if (iccMethodInvokes == null){
                iccMethodInvokes = new ArrayList<>();
            }
            iccMethodInvokes.add(new ICCMethodInvoke(object, iccMethod));
            mMethodInvokeCaches.put(tag, iccMethodInvokes);
        }
        mTagCaches.put(objectClass, tags);
    }

    private List<ICCMethod> findTaggedMethods(Class<?> objectClass) {
        List<ICCMethod> iccMethods = mMethodCaches.get(objectClass);
        if (iccMethods == null){
            iccMethods = new ArrayList<>();
            Method[] methods = objectClass.getDeclaredMethods();//objectClass 里 所有方法
            for (Method method : methods) {
                ICCTag iccTag = method.getAnnotation(ICCTag.class);
                if (iccTag != null){
                    method.setAccessible(true);
                    String[] tags = iccTag.value();
                    for (String tag : tags) {
                        iccMethods.add(new ICCMethod(tag, method));
                    }
                }
            }
            mMethodCaches.put(objectClass, iccMethods);
        }
        return iccMethods;
    }

    /**
     * 反注册
     * @param object 想要反注册的对象
     */
    public void unregister(Object object){
        List<String> tags = mTagCaches.get(object.getClass());
        if (tags == null || tags.size() == 0) {
            return ;
        }
        for (String tag : tags) {
            List<ICCMethodInvoke> iccMethodInvokes = mMethodInvokeCaches.get(tag);
            if (iccMethodInvokes != null && iccMethodInvokes.size() > 0){
                Iterator<ICCMethodInvoke> iterator = iccMethodInvokes.iterator();
                while (iterator.hasNext()){
                    ICCMethodInvoke iccMethodInvoke = iterator.next();
                    if (iccMethodInvoke.getObject() == object){
                        iterator.remove();
                    }
                }
            }
        }
    }

    /**
     * 发送消息
     * @param tag
     * @param params
     */
    public void post(String tag, Object... params){
        List<ICCMethodInvoke> iccMethodInvokes = mMethodInvokeCaches.get(tag);
        if (iccMethodInvokes == null || iccMethodInvokes.size() == 0) {
            return ;
        }
        for (ICCMethodInvoke iccMethodInvoke : iccMethodInvokes) {
            ICCMethod iccMethod = iccMethodInvoke.getMethod();
            //组装参数
            Class<?>[] parameterTypes = iccMethod.getMethod().getParameterTypes();
            Object[] realParams = new Object[parameterTypes.length];
            if (params != null){
                for (int i = 0; i < parameterTypes.length; i++) {
                    //1, i 不能超过params的长度
                    //2， 参数类型要一致
                    if (i< params.length && parameterTypes[i].isInstance(params[i])){
                        realParams[i] = params[i];
                    }else{
                        realParams[i] = null;
                    }
                }
            }

            try {
                iccMethod.getMethod().invoke(iccMethodInvoke.getObject(), realParams);
            }
            catch (IllegalAccessException e) {
                e.printStackTrace();
            }
            catch (InvocationTargetException e) {
                e.printStackTrace();
            }

        }

    }
}
